
/*
 * Copyright (c) 2005, Arnaud KLEIN
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * * Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 */ 
 
#include "XmlFileWriter.h" 
#include "SysTools.h"
#include "pflogx.h"
#include <fstream>
#include <string.h>


/* Constructor */
cXmlFileWriter::cXmlFileWriter(const string& file, const cLogEntryFormat& entryFormat, const vector<cLogEntry>& vectLogEntries) : m_file(file), m_entryFormat(entryFormat), m_vectLogEntries(vectLogEntries), m_outputStream(NULL), m_errorString() 
{
}


/* Write */
bool cXmlFileWriter::Write()
{
	bool bHasError=false;

	if (OpenFile())
	{
		// Write header
		if (WriteHeader())
		{
			// Write entries
			for(vector<cLogEntry>::const_iterator it=m_vectLogEntries.begin(); !bHasError && (it!=m_vectLogEntries.end()); it++)
				bHasError=!WriteEntry(*it);
		
			if (!bHasError)
			{
				// Write footer
				if (!WriteFooter())
					bHasError=true;
			}
		}
		else
			bHasError=true;
	

		CloseFile();
	}
	else
		bHasError=true;

	return !bHasError;
}


/* Open output file */
bool cXmlFileWriter::OpenFile()
{
	if (OutputOnStdOut())
		m_outputStream=&cout;
	else
	{
		// Construct new ofstream object and open file for writing
		m_outputStream=new (nothrow) ofstream(m_file.c_str());
		
		if (!m_outputStream || !static_cast<ofstream*>(m_outputStream)->is_open())
		{
			m_errorString="Could not open output file (" + cSysTools::GetLastSystemError() + ")";
			return false;
		}
	}
	
	return true;
}
	

/* Close output file */
bool cXmlFileWriter::CloseFile()
{
	if (!m_outputStream)
	{
		m_errorString="Output file not opened";
		return false;
	}
		
	if (OutputOnStdOut())
		m_outputStream=NULL;
	else
	{
		if (static_cast<ofstream*>(m_outputStream)->is_open())
			static_cast<ofstream*>(m_outputStream)->close();

		delete m_outputStream;
		m_outputStream=NULL;
	}

	return true;
}


/* Write a string to outfile file */
bool cXmlFileWriter::WriteString(const string& str)
{
	m_outputStream->write(str.c_str(), str.length());
	
	if (m_outputStream->fail())
	{
		m_errorString="Write error (" + cSysTools::GetLastSystemError() + ")";
		return false;
	}
	else
		return true;
}


/* Write file header */
bool cXmlFileWriter::WriteHeader()
{
	string strHeader="<?xml version=\"1.0\" encoding=\"" + string(XMLFILE_ENCODING) + "\"?>\n<pflogx version=\"" + string(PFLOGX_VERSION) + "\" >\n<logs>\n";
	return WriteString(strHeader);
}


/* Write file footer */
bool cXmlFileWriter::WriteFooter()
{
	string strFooter="</logs>\n</pflogx>\n";
	return WriteString(strFooter);
}


/* Write entry */
bool cXmlFileWriter::WriteEntry(const cLogEntry& entry)
{
	m_entryFormat.Format(entry);
	
	string strEntry="<log date=\"" + m_entryFormat.GetTime() + "\" " + 
	 				"if=\"" + m_entryFormat.GetIfName() + "\" " +
					"action=\"" + m_entryFormat.GetAction() + "\" " +
					"rule=\"" + m_entryFormat.GetRuleNumber() + "\" " +
					"direction=\"" + m_entryFormat.GetDirection() + "\" " +
					"protocol=\"" + m_entryFormat.GetProtocol() + "\" " +
					"src_adr=\"" + m_entryFormat.GetSrcAddress() + "\" " +
					"src_port=\"" + m_entryFormat.GetSrcPort() + "\" " +
					"dest_adr=\"" + m_entryFormat.GetDestAddress() + "\" " +
					"dest_port=\"" + m_entryFormat.GetDestPort() + "\" " +
					"/>\n";

	return WriteString(strEntry);
}

