
/*
 * Copyright (c) 2005, Arnaud KLEIN
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * * Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 */ 
 
#include "LogEntryParser.h"
#include <stdio.h>
#include <time.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>



/* Constructor */
cLogEntryParser::cLogEntryParser() : m_entry()
{
}
	

/* Parse given strings */
enum cLogEntryParser::eParseStringRet cLogEntryParser::ParseStrings(const string& time, const string& ifName, const string& ruleNumber, const string& action, const string& direction, const string& protocol, const string& destAddress, const string& srcAddress, const string& destPort, const string& srcPort)
{
	enum eParseStringRet eRet=PARSESTRING_NO_ERROR;
	
	Reset();

	// Pflog header
	eRet=ParsePflogStrings(time, ifName, ruleNumber, action, direction, protocol);
	if (eRet == PARSESTRING_NO_ERROR)
	{
		// Need to parse addresses and/or ports ?
		bool bParsePorts=(m_entry.GetProtocol() == IPPROTO_TCP) || (m_entry.GetProtocol() == IPPROTO_UDP);
		bool bParseAddresses=(m_entry.GetProtocol() == IPPROTO_IP) || (m_entry.GetProtocol() == IPPROTO_ICMP) || bParsePorts;
			
		// Addresses
		if (bParseAddresses)
		{
			eRet=ParseAddressesStrings(destAddress, srcAddress);
			if (eRet == PARSESTRING_NO_ERROR)
			{
				// Ports
				if (bParsePorts)
					eRet=ParsePortsStrings(destPort, srcPort);
			}
		}	
	}

	return eRet;
}	


/* Parse Pflog header strings */
enum cLogEntryParser::eParseStringRet cLogEntryParser::ParsePflogStrings(const string& time, const string& ifName, const string& ruleNumber, const string& action, const string& direction, const string& protocol)
{
	enum eParseStringRet eRet=PARSESTRING_NO_ERROR;

	// Check for mandatory fields
	if (!time.empty() && !ifName.empty() && !ruleNumber.empty() && !action.empty() && !direction.empty() && !protocol.empty())
	{
		if (ParseTime(time))
		{
			if (ParseRuleNumber(ruleNumber))
			{
				if (ParseAction(action))
				{
					if (ParseDirection(direction))
					{
						if (ParseProtocol(protocol))
						{
							m_entry.SetIfName(ifName);
							m_entry.SetEntryDataFlags(m_entry.GetEntryDataFlags() | cLogEntry::ENTRYDATA_PFLOG);
						}
						else
							eRet=PARSESTRING_PROTOCOL_ERROR;					
					}
					else
						eRet=PARSESTRING_DIRECTION_ERROR;
				}
				else
					eRet=PARSESTRING_ACTION_ERROR;
			}
			else
				eRet=PARSESTRING_RULENUMBER_ERROR;
		}
		else
			eRet=PARSESTRING_TIME_ERROR;
	}
	else
		eRet=PARSESTRING_MISSING_PFLOG_FIELDS;

	return eRet;
}


/* Parse addresses strings */
enum cLogEntryParser::eParseStringRet cLogEntryParser::ParseAddressesStrings(const string& destAddress, const string& srcAddress)
{
	enum eParseStringRet eRet=PARSESTRING_NO_ERROR;

	// Check for mandatory fields
	if (!destAddress.empty() && !srcAddress.empty())
	{
		struct in_addr inDestAddress, inSrcAddress;
		if (inet_aton(destAddress.c_str(), &inDestAddress) && inet_aton(srcAddress.c_str(), &inSrcAddress))
		{
			m_entry.SetDestAddress(inDestAddress);
			m_entry.SetSrcAddress(inSrcAddress);			
			m_entry.SetEntryDataFlags(m_entry.GetEntryDataFlags() | cLogEntry::ENTRYDATA_IP);
		}
		else
			eRet=PARSESTRING_ADDRESSES_ERROR;
	}
	else
		eRet=PARSESTRING_MISSING_IP_FIELDS;

	return PARSESTRING_NO_ERROR;
}


/* Parse ports strings */
enum cLogEntryParser::eParseStringRet cLogEntryParser::ParsePortsStrings(const string& destPort, const string& srcPort)
{
	enum eParseStringRet eRet=PARSESTRING_NO_ERROR;
	
	// Check for mandatory fields
	if (!destPort.empty() && !srcPort.empty())
	{
		unsigned int inDestPort, inSrcPort;
		if ((sscanf(destPort.c_str(), "%u", &inDestPort) == 1) && (sscanf(srcPort.c_str(), "%u", &inSrcPort) == 1))
		{
			m_entry.SetDestPort(static_cast<u_int16_t>(inDestPort));
			m_entry.SetSrcPort(static_cast<u_int16_t>(inSrcPort));			

			if (m_entry.GetProtocol() == IPPROTO_TCP)
				m_entry.SetEntryDataFlags(m_entry.GetEntryDataFlags() | cLogEntry::ENTRYDATA_TCP);
			else
				m_entry.SetEntryDataFlags(m_entry.GetEntryDataFlags() | cLogEntry::ENTRYDATA_UDP);
		}
		else
			eRet=PARSESTRING_PORTS_ERROR;
	}
	else
		eRet=PARSESTRING_PORTS_ERROR;
	
	return PARSESTRING_NO_ERROR;
}


/* Parse time */
bool cLogEntryParser::ParseTime(const string& time)
{
	bool bRet=false;
	struct tm timeM;
	struct timeval timeV;

	// We try to parse time with cOptions::DATEFORMAT_SECONDS format
	if (sscanf(time.c_str(), "%lu.%lu", &timeV.tv_sec, &timeV.tv_usec) == 2)
		bRet=true;
	else
	{
		// We try now to parse time with cOptions::DATEFORMAT_HUMAN format 
		const char* ptrTimeStr=strptime(time.c_str(), "%Y-%m-%d %T", &timeM);
		if (ptrTimeStr && ((ptrTimeStr - time.c_str()) > 0))
		{
			timeM.tm_isdst=-1;
			timeV.tv_sec=mktime(&timeM);

			// Parse microseconds
			if (sscanf(ptrTimeStr, ".%lu", &timeV.tv_usec) == 1)
				bRet=true;
		} 
	}

	if (bRet)
		m_entry.SetTime(timeV.tv_sec, timeV.tv_usec);

	return bRet;
}


/* Parse action */
bool cLogEntryParser::ParseAction(const string& action)
{
	if (action == "pass")
		m_entry.SetAction(PF_PASS);
	else if (action == "drop")
		m_entry.SetAction(PF_DROP);
	else
	{
		unsigned int inAction;
		if (sscanf(action.c_str(), "(%x)", &inAction) == 1)
			m_entry.SetAction(static_cast<u_int8_t>(inAction));
		else
			return false;
	}

	return true;
}


/* Parse rule number */
bool cLogEntryParser::ParseRuleNumber(const string& ruleNumber)
{
	unsigned int inRuleNumber;
	if (sscanf(ruleNumber.c_str(), "%u", &inRuleNumber) == 1)
	{
		m_entry.SetRuleNumber(static_cast<u_int32_t>(inRuleNumber));
		return true;
	}
	else
		return false;
}


/* Parse direction */
bool cLogEntryParser::ParseDirection(const string& direction)
{
	if (direction == "in")
		m_entry.SetDirection(PF_IN);
	else if (direction == "out")
		m_entry.SetDirection(PF_OUT);
	else if (direction == "in-out")
		m_entry.SetDirection(PF_INOUT);
	else
		return false;	
	
	return true;
}


/* Parse protocol */
bool cLogEntryParser::ParseProtocol(const string& protocol)
{
	if (protocol == "ip")
		m_entry.SetProtocol(IPPROTO_IP);
	else if (protocol == "icmp")
		m_entry.SetProtocol(IPPROTO_ICMP);
	else if (protocol == "tcp")
		m_entry.SetProtocol(IPPROTO_TCP);
	else if (protocol == "udp")
		m_entry.SetProtocol(IPPROTO_UDP);
	else if (protocol == "raw")
		m_entry.SetProtocol(IPPROTO_RAW);
	else
	{
		unsigned int inProtocol;
		if (sscanf(protocol.c_str(), "(%x)", &inProtocol) == 1)
			m_entry.SetProtocol(static_cast<u_int8_t>(inProtocol));
		else
			return false;
	}

	return true;
}


/* Reset object */
void cLogEntryParser::Reset()
{
	m_entry.Reset();
}
