
/*
 * Copyright (c) 2005, Arnaud KLEIN
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * * Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 */ 
 
#include "LogEntryFormat.h" 
#include <sys/types.h>
#include <sys/socket.h>
#include <stdio.h>
#include <time.h>
#include <netinet/in.h>
#include <arpa/inet.h>


#define STATIC_BUFFER_LEN	128


/* Constructor */
cLogEntryFormat::cLogEntryFormat(const cOptions& options) : m_options(options), m_time(), m_ifName(), m_ruleNumber(), m_action(), m_direction(), m_protocol(), m_destAddress(), m_srcAddress(), m_destPort(), m_srcPort()
{
}


/* Format given entry */
void cLogEntryFormat::Format(const cLogEntry& entry)
{
	Reset();

	// Time
	FormatTime(entry);
	
	// Pflog header
	if ((entry.GetEntryDataFlags() & cLogEntry::ENTRYDATA_PFLOG) == cLogEntry::ENTRYDATA_PFLOG)
	{
		m_ifName=entry.GetIfName();
		FormatRuleNumber(entry);
		FormatAction(entry);
		FormatDirection(entry);
		FormatProtocol(entry);
	}

	// IP header
	if ((entry.GetEntryDataFlags() & cLogEntry::ENTRYDATA_IP) == cLogEntry::ENTRYDATA_IP)
		FormatAddresses(entry);

	// TCP or UDP header
	if ((entry.GetEntryDataFlags() & (cLogEntry::ENTRYDATA_TCP | cLogEntry::ENTRYDATA_UDP)) != 0)
		FormatPorts(entry);
}


/* Format time */
void cLogEntryFormat::FormatTime(const cLogEntry& entry)
{
	char strBuffer[STATIC_BUFFER_LEN];

	
	// Format seconds
	time_t t=entry.GetTime().tv_sec;
	
	if (m_options.GetOutDateFormat() == cOptions::DATEFORMAT_SECONDS)
	{
		if (snprintf(strBuffer, STATIC_BUFFER_LEN, "%lu", static_cast<unsigned long>(t)) > 0)
			m_time=strBuffer;
	}
	else
	{
		// Convert time_t to struct tm
		struct tm* localTime=localtime(&t);
		
		// Convert time to string
		if (strftime(strBuffer, STATIC_BUFFER_LEN, "%Y-%m-%d %T", localTime) > 0)
			m_time=strBuffer;
	}
	
	
	// Format microseconds
	if (snprintf(strBuffer, STATIC_BUFFER_LEN, "%s.%lu", m_time.c_str(), entry.GetTime().tv_usec) > 0)
		m_time=strBuffer;
}


/* Format rule number */
void cLogEntryFormat::FormatRuleNumber(const cLogEntry& entry)
{
	char strBuffer[STATIC_BUFFER_LEN];
	if (snprintf(strBuffer, STATIC_BUFFER_LEN, "%u", entry.GetRuleNumber()) > 0)
		m_ruleNumber=strBuffer;
}


/* Format action */
void cLogEntryFormat::FormatAction(const cLogEntry& entry)
{
	if (entry.GetAction() == PF_PASS)
		m_action="pass";
	else if (entry.GetAction() == PF_DROP)
		m_action="drop";
	else
	{
		char strBuffer[STATIC_BUFFER_LEN];

		if (snprintf(strBuffer, STATIC_BUFFER_LEN, "(%x)", entry.GetAction()) > 0)
			m_action=strBuffer;
	}
}


/* Format direction */
void cLogEntryFormat::FormatDirection(const cLogEntry& entry)
{
	if (entry.GetDirection() == PF_IN)
		m_direction="in";
	else if (entry.GetDirection() == PF_OUT)
		m_direction="out";
	else if (entry.GetDirection() == PF_INOUT)
		m_direction="in-out";
}


/* Format protocol */
void cLogEntryFormat::FormatProtocol(const cLogEntry& entry)
{
	switch(entry.GetProtocol())
	{
		case IPPROTO_IP:
		{
			m_protocol="ip";
			break;
		}
		
		case IPPROTO_ICMP:
		{
			m_protocol="icmp";
			break;
		}
		
		case IPPROTO_TCP:
		{
			m_protocol="tcp";
			break;
		}
		
		case IPPROTO_UDP:
		{
			m_protocol="udp";
			break;
		}
		
		case IPPROTO_RAW:
		{
			m_protocol="raw";
			break;
		}
		
		default:
		{
			char strBuffer[STATIC_BUFFER_LEN];
			if (snprintf(strBuffer, STATIC_BUFFER_LEN, "(%x)", entry.GetProtocol()) > 0)
				m_protocol=strBuffer;
		}
	}
}


/* Format source and destination addresses */
void cLogEntryFormat::FormatAddresses(const cLogEntry& entry)
{
	const char* strBuffer=NULL;
	
	strBuffer=inet_ntoa(entry.GetSrcAddress());
	if (strBuffer)
		m_srcAddress=strBuffer;

	strBuffer=inet_ntoa(entry.GetDestAddress());
	if (strBuffer)
		m_destAddress=strBuffer;
}


/* Format source and destination ports */
void cLogEntryFormat::FormatPorts(const cLogEntry& entry)
{
	char strBuffer[STATIC_BUFFER_LEN];
	
	if (snprintf(strBuffer, STATIC_BUFFER_LEN, "%u", entry.GetSrcPort()) > 0)
		m_srcPort=strBuffer;

	if (snprintf(strBuffer, STATIC_BUFFER_LEN, "%u", entry.GetDestPort()) > 0)
		m_destPort=strBuffer;
}


/* Reset object */
void cLogEntryFormat::Reset()
{
	m_time.erase();
	m_ifName.erase();
	m_ruleNumber.erase(); 
	m_action.erase();
	m_direction.erase(); 
	m_protocol.erase();
	m_destAddress.erase(); 
	m_srcAddress.erase();
	m_destPort.erase();
	m_srcPort.erase();
}
