
/*
 * Copyright (c) 2005, Arnaud KLEIN
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * * Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 */ 
 
#include "EntriesFilter.h"
#include <algorithm>
#include <string.h>


/* Constructor */
cEntriesFilter::cEntriesFilter(const vector<cLogEntry>& vectLogEntries, const cOptions& options) : m_vectInputEntries(vectLogEntries), m_vectOutputEntries(), m_options(options) 
{
}


/* Apply filters */
void cEntriesFilter::ApplyFilters()
{
	// First step: apply filters
	for(vector<cLogEntry>::const_iterator it=m_vectInputEntries.begin(); it!=m_vectInputEntries.end(); it++)
	{
		const cLogEntry& entry=*it;
	
		// Action
		if (!IsInFilter<u_int8_t>(entry.GetAction(), m_options.GetFilterAction()))
			continue;

		// Direction
		if (!IsInFilter<u_int8_t>(entry.GetDirection(), m_options.GetFilterDirection()))
			continue;

		// Protocol
		if (!IsInFilter<u_int8_t>(entry.GetProtocol(), m_options.GetFilterProtocol()))
			continue;

		// Interface
		if (!IsInFilter<string>(entry.GetIfName(), m_options.GetFilterInterface()))
			continue;

		// Add entry to output list
		m_vectOutputEntries.push_back(entry);	
	}
	

	// Second step: sort entries
	if (m_options.GetReverseSort())
		sort(m_vectOutputEntries.begin(), m_vectOutputEntries.end(), sSortLogEntryDesc());
	else
		sort(m_vectOutputEntries.begin(), m_vectOutputEntries.end(), sSortLogEntryAsc());
		
	
	// Third step: remove duplicate entries
	vector<cLogEntry>::iterator new_end=unique(m_vectOutputEntries.begin(), m_vectOutputEntries.end(), sLogEntryCmp());
	m_vectOutputEntries.erase(new_end, m_vectOutputEntries.end());
}


/* Check if given filter is included in filters list */
template <class T>
bool cEntriesFilter::IsInFilter(T filter, const vector<T>& vectFilters) const
{
	if (vectFilters.empty())
		return true;

	for(typename vector<T>::const_iterator it=vectFilters.begin(); it!=vectFilters.end(); it++)
	{
		if (*it == filter)
			return true;
	}

	return false;
}


// ---------------------------------------------


/* Structre to compare two cLogEntry objects */
bool cEntriesFilter::sLogEntryCmp::operator()(const cLogEntry& e1, const cLogEntry& e2) const
{
	return ((e1.GetEntryDataFlags() == e2.GetEntryDataFlags()) && \
			(memcmp(&e1.GetTime(), &e2.GetTime(), sizeof(struct timeval)) == 0) && \
			(e1.GetIfName() == e2.GetIfName()) && \
			(e1.GetRuleNumber() == e2.GetRuleNumber()) && \
			(e1.GetAction() == e2.GetAction()) && \
			(e1.GetDirection() == e2.GetDirection()) && \
			(e1.GetProtocol() == e2.GetProtocol()) && \
			(memcmp(&e1.GetDestAddress(), &e2.GetDestAddress(), sizeof(struct in_addr)) == 0) && \
			(memcmp(&e1.GetSrcAddress(), &e2.GetSrcAddress(), sizeof(struct in_addr)) == 0) && \
			(e1.GetDestPort() == e2.GetDestPort()) && \
			(e1.GetSrcPort() == e2.GetSrcPort()));
}


/* Structure to compare time of two cLogEntry objects in ascending order */
bool cEntriesFilter::sSortLogEntryAsc::operator()(const cLogEntry& e1, const cLogEntry& e2) const
{
	const struct timeval& t1=e1.GetTime();
	const struct timeval& t2=e2.GetTime();
	
	if (t1.tv_sec == t2.tv_sec)
		return t1.tv_usec < t2.tv_usec;
	else
		return t1.tv_sec < t2.tv_sec;
}


/* Structure to compare time of two cLogEntry objects in descending order */
bool cEntriesFilter::sSortLogEntryDesc::operator()(const cLogEntry& e1, const cLogEntry& e2) const
{
	const struct timeval& t1=e1.GetTime();
	const struct timeval& t2=e2.GetTime();
	
	if (t2.tv_sec == t1.tv_sec)
		return t2.tv_usec < t1.tv_usec;
	else
		return t2.tv_sec < t1.tv_sec;
}
