/* peksystray
 * Copyright (c) 2003 Mohammed Sameer.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /* HAVE_CONFIG_H */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
/* Sun OpenWindows */
#include <X11/Xlib.h>
#include <X11/Xutil.h>

#define SYSTEM_TRAY_REQUEST_DOCK    0
#define SYSTEM_TRAY_BEGIN_MESSAGE   1
#define SYSTEM_TRAY_CANCEL_MESSAGE  2

/* Flags for _XEMBED_INFO */
#define XEMBED_MAPPED                   (1 << 0)

#define DEBUG(x) fprintf(stdout,"%s\n", x)

#define MAX_ICONS 4		/* Please don't change this otherwise it'll break */

Atom selection_atom, opcode_atom, data_atom;

int is_full = 0;		/* Can we accept more icons ? */
int no_tray_icon = 0;		/* No, of embedded icons */
Window dock[MAX_ICONS];
Window icon[MAX_ICONS];
Window icons[MAX_ICONS];

int vertical = 0;
int icon_size = 16;
int border_width = 0;
int multiple = 0;
Display *display = NULL;
char *display_name = NULL;

Window Root;
Window dockWindow;
Window iconWindow;

int screen;

void
quit (char *err)
{
  fprintf (stderr, err);
  exit (1);
}

void
display_help ()
{
  fprintf (stdout, ""
"%s - version %s\n"
"Copyright 2003, Mohammed Sameer <msameer@foolab.org>\n"
"\n"
"Usage: %s [OPTIONS]\n"
"\n"
"Options:\n"
" --help\t Display this help.\n"
" --version\t Display version number and exit.\n"
" --display DISPLAY\t The X display to connect to.\n"
" --icon-size SIZE\t Icon size. Default is 16.\n"
" --vertical\t Vertical layout. Default is horizontal.\n"
" --border SIZE\t Border width. Default is 0.\n"
" --multiple\t Open each icon in a separate window. Default is no.\n"
"\n"
"", PACKAGE, VERSION, PACKAGE);
  exit (0);
}

void
display_version ()
{
  fprintf (stdout, "%s\n", VERSION);
  exit (0);
}

void
parse_cmd_line (int argc, char *argv[])
{
  int x;
  if (argc == 1)
    {
      return;
    }
  for (x = 1; x < argc; x++)
    {
      DEBUG (argv[x]);
      if (!strcmp (argv[x], "--help"))
	{
	  display_help ();
	}
      else if (!strcmp (argv[x], "--version"))
	{
	  display_version ();
	}
      else if (!strcmp (argv[x], "--vertical"))
	{
	  vertical = 1;
	}
      else if (!strcmp (argv[x], "--multiple"))
	{
	  multiple = 1;
	}
      else if (!strcmp (argv[x], "--display"))
	{
	  if (x + 1 == argc)
	    {
	      quit ("--display require an argument");
	    }
	  else
	    {
	      display_name = argv[x + 1];
	      ++x;
	      continue;
	    }
	}
      else if (!strcmp (argv[x], "--icon-size"))
	{
	  if (x + 1 == argc)
	    {
	      quit ("--icon-size require an argument");
	    }
	  else
	    {
	      icon_size = atoi (argv[x + 1]);
	      ++x;
	      continue;
	    }
	}
      else if (!strcmp (argv[x], "--border"))
	{
	  if (x + 1 == argc)
	    {
	      quit ("--border require an argument");
	    }
	  else
	    {
	      border_width = atoi (argv[x + 1]);
	      ++x;
	      continue;
	    }
	}

      else
	{
	  fprintf (stderr, "Unknown command line argument: %s\n", argv[x]);
	}
    }

}

void
add_tray_icon (Window win)
{
  XEvent xevent;
  Window tmp;
  fprintf (stderr, "%s: %i\n", __FUNCTION__, win);
  if (multiple)
    {
      XClassHint *class_hint;
      XWMHints *hints;
      class_hint = XAllocClassHint ();

      dock[no_tray_icon] =
	XCreateSimpleWindow (display, DefaultRootWindow (display), 0, 0,
			     icon_size + border_width,
			     icon_size + border_width, border_width, 0, 0);
      icons[no_tray_icon] =
	XCreateSimpleWindow (display, dock[no_tray_icon], 0, 0,
			     icon_size + border_width,
			     icon_size + border_width, border_width, 0, 0);

      tmp = icons[no_tray_icon];

      if (class_hint == NULL)
	{
	  quit ("Failed to allocate class hint");
	}

      class_hint->res_class = PACKAGE;
      class_hint->res_name = PACKAGE;

      XSetClassHint (display, dock[no_tray_icon], class_hint);
      XFree (class_hint);
      hints = XAllocWMHints ();
      if (hints == NULL)
	{
	  quit ("Failed to allocate hints");
	}
      hints->flags = StateHint | WindowGroupHint | IconWindowHint;
      hints->initial_state = WithdrawnState;
      hints->icon_window = icons[no_tray_icon];
      hints->window_group = dock[no_tray_icon];

      XSetWMHints (display, dock[no_tray_icon], hints);
      XFree (hints);
      XSelectInput (display, dock[no_tray_icon], SubstructureNotifyMask);
      XSetWindowBackgroundPixmap (display, icons[no_tray_icon],
				  ParentRelative);
      XMapRaised (display, dock[no_tray_icon]);
      XFlush (display);
      XReparentWindow (display, win, tmp, border_width, border_width);
    }
  else
    {
      tmp = iconWindow;
      dock[no_tray_icon] = win;
      if (vertical)
	{
	  XReparentWindow (display, win, tmp, border_width,
			   no_tray_icon * icon_size);
	}
      else
	{
	  XReparentWindow (display, win, tmp, no_tray_icon * icon_size,
			   border_width);
	}
    }

  XMapRaised (display, win);
  xevent.xclient.window = win;
  xevent.xclient.type = ClientMessage;
  xevent.xclient.message_type = XInternAtom (display, "_XEMBED", False);
  xevent.xclient.format = 32;
  xevent.xclient.data.l[0] = CurrentTime;
  xevent.xclient.data.l[1] = SYSTEM_TRAY_REQUEST_DOCK;
  xevent.xclient.data.l[2] = 0;
  xevent.xclient.data.l[3] = tmp;
  xevent.xclient.data.l[4] = 0;
  XSendEvent (display, Root, False, NoEventMask, &xevent);

  icon[no_tray_icon] = win;

  no_tray_icon++;
}

void
rm_tray_icon (Window win)
{
  int i;

  fprintf (stderr, "%s: %i\n", __FUNCTION__, win);

  if (no_tray_icon == 0)
    {
      return;
    }
  for (i = 0; i < MAX_ICONS; i++)
    {
      if (icon[i] == win)
	{
	  if (multiple)
	    {
	      XDestroyWindow (display, dock[i]);
	      XDestroyWindow (display, icons[i]);
	    }
	  dock[i] = dock[no_tray_icon];
	  icons[i] = icons[no_tray_icon];
	  icon[i] = icon[no_tray_icon];
	  dock[no_tray_icon] = 0;
	  icons[no_tray_icon] = 0;
	  icon[no_tray_icon] = 0;
	  no_tray_icon--;
	  break;
	}
    }
}

void
handle_event (XEvent ev)
{
  XClientMessageEvent *event = (XClientMessageEvent *) & ev;
  int x;
  XWindowAttributes tray_icon_attr;

  fprintf (stderr, "%s\n", __FUNCTION__);

  switch (event->data.l[1])
    {
    case SYSTEM_TRAY_REQUEST_DOCK:
      {
	if (no_tray_icon == MAX_ICONS)
	  {
	    fprintf (stderr, "full\n");
	    return;
	  }

	XGetWindowAttributes (display, event->data.l[2], &tray_icon_attr);
	fprintf (stderr, "Width: %i\t Heught:%i\tBorder: %i\n",
		 tray_icon_attr.width, tray_icon_attr.height,
		 tray_icon_attr.border_width);

	XResizeWindow (display, event->data.l[2], icon_size, icon_size);

	add_tray_icon (event->data.l[2]);
	return;
      }
    case SYSTEM_TRAY_BEGIN_MESSAGE:
      {

	return;
      }
    case SYSTEM_TRAY_CANCEL_MESSAGE:
      {

	return;
      }
    default:
      break;
    }
}

void
handle_message_data (XEvent ev)
{
  fprintf (stderr, "%s\n", __FUNCTION__);
}

void
eventLoop (void)
{
  XEvent ev;

  while (1)
    {
      while (XPending (display))
	{
	  XNextEvent (display, &ev);

	  switch (ev.type)
	    {
	    case ReparentNotify:
	      {
		XReparentEvent *xev = (XReparentEvent *) & ev;
		fprintf (stderr, "ReparentNotify\n");
/* Handle this */
		if (xev->parent == dockWindow)
		  {

		  }
		else
		  {
		  }
		break;
	      }
	    case UnmapNotify:
	      {
		XUnmapEvent *xev = (XUnmapEvent *) & ev;
		fprintf (stderr, "UnmapNotify\n");
		if (xev->window != dockWindow)
		  {
		    rm_tray_icon (xev->window);
		  }
		break;
	      }
	    case DestroyNotify:
	      {
		XDestroyWindowEvent *xev = (XDestroyWindowEvent *) & ev;
		fprintf (stderr, "DestroyNotify\n");
		if (xev->window == dockWindow)
		  {
		    return;
		  }
		else
		  {
		    rm_tray_icon (xev->window);
		    break;
		  }
	      }
	    case ClientMessage:
	      {
		if (ev.xclient.message_type == opcode_atom)
		  {
		    handle_event (ev);
		  }
		else if (ev.xclient.message_type == data_atom)
		  {
		    handle_message_data (ev);
		  }
		break;
	      }
	    case SelectionClear:
	      {
		if (XGetSelectionOwner (display, selection_atom) ==
		    dockWindow)
		  {
		    XSetSelectionOwner (display, selection_atom, None,
					CurrentTime);
		  }
		return;
	      }
	    default:
	      break;
	    }
	}
      usleep (50000L);
    }
}

int
main (int argc, char *argv[])
{
  XClassHint *classHint;
  XWMHints *hints;
  int w, h;

/*  win[0] = 0;
  win[1] = 0;
  win[2] = 0;
  win[3] = 0;
*/
  parse_cmd_line (argc, argv);

  if ((display = XOpenDisplay (display_name)) == NULL)
    {
      quit ("Cannot open display\n");
    }

  h = icon_size;
  w = 64;

  if (border_width)
    {
      h += border_width;
    }

  if (vertical)
    {
      dockWindow =
	XCreateSimpleWindow (display, DefaultRootWindow (display), 0, 0,
			     h, w, border_width, 0, 0);
      iconWindow = XCreateSimpleWindow (display, dockWindow, 0, 0,
					h, w, border_width, 0, 0);

    }
  else
    {
      dockWindow =
	XCreateSimpleWindow (display, DefaultRootWindow (display), 0, 0,
			     w, h, border_width, 0, 0);
      iconWindow = XCreateSimpleWindow (display, dockWindow, 0, 0,
					w, h, border_width, 0, 0);

    }
  if ((classHint = XAllocClassHint ()) == NULL)
    {
      quit ("Failed to allocate class hint\n");
    }
  classHint->res_class = PACKAGE;
  classHint->res_name = PACKAGE;
  XSetClassHint (display, dockWindow, classHint);
  XFree (classHint);

  if ((hints = XAllocWMHints ()) == NULL)
    {
      quit ("Failed to allocate hints\n");
    }

  hints->flags = StateHint | WindowGroupHint | IconWindowHint;
  hints->icon_window = iconWindow;
  hints->initial_state = WithdrawnState;
  hints->window_group = dockWindow;

/*  hints->icon_window = dockIconWindow;
  hints->icon_x = 0;
  hints->icon_y = 0; */

  XSetWMHints (display, dockWindow, hints);
  XFree (hints);

/* Set the background */
  XSetWindowBackgroundPixmap (display, iconWindow, ParentRelative);

  selection_atom = XInternAtom (display, "_NET_SYSTEM_TRAY_S0", False);
  XSetSelectionOwner (display, selection_atom, dockWindow, CurrentTime);

  if (XGetSelectionOwner (display, selection_atom) == dockWindow)
    {
      XClientMessageEvent xev;
      int scr = DefaultScreen (display);
      Root = RootWindow (display, scr);
      xev.type = ClientMessage;
      xev.window = Root;
      xev.message_type = XInternAtom (display, "MANAGER", False);

      xev.format = 32;
      xev.data.l[0] = CurrentTime;
      xev.data.l[1] = selection_atom;
      xev.data.l[2] = dockWindow;
      xev.data.l[3] = 0;	/* manager specific data */
      xev.data.l[4] = 0;	/* manager specific data */

      XSendEvent (display,
		  Root, False, StructureNotifyMask, (XEvent *) & xev);

      opcode_atom = XInternAtom (display, "_NET_SYSTEM_TRAY_OPCODE", False);

      data_atom = XInternAtom (display,
			       "_NET_SYSTEM_TRAY_MESSAGE_DATA", False);

    }

/*  XSelectInput (display, dockWindow, ExposureMask, ButtonPressMask); */
  XSelectInput (display, dockWindow, SubstructureNotifyMask);
  if (!multiple)
    {
      XMapRaised (display, dockWindow);
    }
  XFlush (display);
  eventLoop ();
  XCloseDisplay (display);

  return 0;
}
