static char rcsid[] = "$Id: TestRecord.cc,v 1.11 2000/02/03 18:59:14 bmah Exp $";
//
// $Id: TestRecord.cc,v 1.11 2000/02/03 18:59:14 bmah Exp $
//
// TestRecord.cc
// Bruce A. Mah <bmah@ca.sandia.gov>
//
// This work was first produced by an employee of Sandia National
// Laboratories under a contract with the U.S. Department of Energy.
// Sandia National Laboratories dedicates whatever right, title or
// interest it may have in this software to the public. Although no
// license from Sandia is needed to copy and use this software,
// copying and using the software might infringe the rights of
// others. This software is provided as-is. SANDIA DISCLAIMS ANY
// WARRANTY OF ANY KIND, EXPRESS OR IMPLIED.
//
#include <stdio.h>
#include <stdlib.h>

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#ifdef HAVE_IPV6
#include <netinet/in.h>
#endif /* HAVE_IPV6 */
#include <arpa/inet.h>

#include "pc.h"
#include "TestRecord.h"

static const int buflen=1024;
static char buffer[buflen];
static char buffer2[buflen];

//
// TestRecord::htoa
//
// Input:  Pctest structure, needed for figuring out the right
// representations of IP addresses.
//
// Output:  Pointer to a statically-allocated buffer with an ASCII
// representation of the TestRecord.
//
// Make an ASCII representation of a TestRecord structure.
//
char *TestRecord::htoa(Pctest *pct) 
{

    // The way we build up the output is an artifact of various
    // method calls (i.e. Pctest::GetPrintableAddress) that used fixed,
    // statically-allocated buffers for returning their output.

    buffer[0] = '\0';
#ifdef HAVE_SNPRINTF
    snprintf(buffer2, buflen, 
#else
    sprintf(buffer2, 
#endif /* HAVE_SNPRINTF */
      "probe timestamp %ld.%06ld src %s ", tvstart.tv_sec, tvstart.tv_usec, pct->GetPrintableAddress(icmpDestAddress));
    strncat(buffer, buffer2, buflen);

#ifdef HAVE_SNPRINTF
    snprintf(buffer2, buflen, 
#else
    sprintf(buffer2, 
#endif /* HAVE_SNPRINTF */
      "dest %s ", pct->GetPrintableAddress(targetAddress));
    strncat(buffer, buffer2, buflen);

#ifdef HAVE_SNPRINTF
    snprintf(buffer2, buflen, 
#else
    sprintf(buffer2, 
#endif /* HAVE_SNPRINTF */
      "hops %d bytes %d icmpsrc %s icmptype %d icmpcode %d rtt %ld.%06ld", hops, size, pct->GetPrintableAddress(icmpSourceAddress), icmpType, icmpCode, tv.tv_sec, tv.tv_usec);
    strncat(buffer, buffer2, buflen);

    return buffer;
}

//
// TestRecord::atoh
//
// Input:
//
// Output:
//
// Parse the ASCII representation described above and make up a new
// TestRecord with demarshalled parameters.  The caller "owns"
// this object and is responsible for deallocating it.
//
TestRecord *TestRecord::atoh(char *s) 
{

    TestRecord *tr;

    char srcChars[256], destChars[256], icmpsrcChars[256];
    float tvstartFloat, tvFloat;
    int hops, size;
    int icmpType, icmpCode;

    if (sscanf(s, "probe timestamp %f src %s dest %s hops %d bytes %d icmpsrc %s icmptype %d icmpcode %d rtt %f", &tvstartFloat, srcChars, destChars, &hops, &size, icmpsrcChars, &icmpType, &icmpCode, &tvFloat) == 9) {

	tr = new TestRecord;
	tr->size = size;
	tr->hops = hops;
	tr->tvstart.tv_sec = (long) tvstartFloat;
	tr->tvstart.tv_usec = (long) ((tvstartFloat - ((long) tvstartFloat)) * 1000000.0);
	tr->tv.tv_sec = (long) tvFloat;
	tr->tv.tv_usec = (long) ((tvFloat - ((long) tvFloat)) * 1000000.0);

	tr->targetAddress = new char[sizeof(in_addr)];
	((in_addr *) tr->targetAddress)->s_addr = inet_addr(destChars);
	tr->targetAddressLength= sizeof(in_addr);

	tr->icmpSourceAddress = new char[sizeof(in_addr)];
	((in_addr *) tr->icmpSourceAddress)->s_addr = inet_addr(icmpsrcChars);
	tr->icmpSourceAddressLength= sizeof(in_addr);

	tr->icmpDestAddress = new char[sizeof(in_addr)];
	((in_addr *) tr->icmpDestAddress)->s_addr = inet_addr(srcChars);
	tr->icmpDestAddressLength= sizeof(in_addr);

	tr->icmpType = icmpType;
	tr->icmpCode = icmpCode;

	return tr;

    }
    else {
	fprintf(stderr, "Syntax error: %s\n", s);
	return NULL;
    }

}

#ifdef HAVE_IPV6
//
// TestRecord::atoh6
//
// Input:
//
// Output:
//
// Parse the ASCII representation described above and make up a new
// TestRecord with demarshalled parameters.  The caller "owns"
// this object and is responsible for deallocating it.
//
TestRecord *TestRecord::atoh6(char *s) 
{

    TestRecord *tr;

    char srcChars[256], destChars[256], icmpsrcChars[256];
    float tvstartFloat, tvFloat;
    int hops, size;
    int icmpType, icmpCode;

    if (sscanf(s, "probe timestamp %f src %s dest %s hops %d bytes %d icmpsrc %s icmptype %d icmpcode %d rtt %f", &tvstartFloat, srcChars, destChars, &hops, &size, icmpsrcChars, &icmpType, &icmpCode, &tvFloat) == 9) {

	tr = new TestRecord;
	tr->size = size;
	tr->hops = hops;
	tr->tvstart.tv_sec = (long) tvstartFloat;
	tr->tvstart.tv_usec = (long) ((tvstartFloat - ((long) tvstartFloat)) * 1000000.0);
	tr->tv.tv_sec = (long) tvFloat;
	tr->tv.tv_usec = (long) ((tvFloat - ((long) tvFloat)) * 1000000.0);

	tr->targetAddress = new char[sizeof(in6_addr)];
	inet_pton(AF_INET6, destChars, (void *) tr->targetAddress);
	tr->targetAddressLength= sizeof(in6_addr);

	tr->icmpSourceAddress = new char[sizeof(in6_addr)];
	inet_pton(AF_INET6, icmpsrcChars, (void *) tr->icmpSourceAddress);
	tr->icmpSourceAddressLength= sizeof(in6_addr);

	tr->icmpDestAddress = new char[sizeof(in6_addr)];
	inet_pton(AF_INET6, srcChars, (void *) tr->icmpDestAddress);
	tr->icmpDestAddressLength= sizeof(in6_addr);

	tr->icmpType = icmpType;
	tr->icmpCode = icmpCode;

	return tr;

    }
    else {
	fprintf(stderr, "Syntax error: %s\n");
	return NULL;
    }

}
#endif /* HAVE_IPV6 */

