static char rcsid[] = "$Id: PctestIpv6Udp.cc,v 1.13 2000/05/05 22:43:27 bmah Exp $";
//
// $Id: PctestIpv6Udp.cc,v 1.13 2000/05/05 22:43:27 bmah Exp $
//
// PctestIpv6Udp.cc
// Bruce A. Mah <bmah@ca.sandia.gov>
//
// This work was first produced by an employee of Sandia National
// Laboratories under a contract with the U.S. Department of Energy.
// Sandia National Laboratories dedicates whatever right, title or
// interest it may have in this software to the public. Although no
// license from Sandia is needed to copy and use this software,
// copying and using the software might infringe the rights of
// others. This software is provided as-is. SANDIA DISCLAIMS ANY
// WARRANTY OF ANY KIND, EXPRESS OR IMPLIED.
//
// Class of IPv6 tests using UDP
//

//
// Solaris needs some "extra stuff" to get msg_control in recvmsg(2)
// according to Erik Nordmark <Erik.Nordmark@eng.sun.com>.  His quick
// fix to do this is:
#ifdef NEED_XOPEN
#define _XOPEN_SOURCE 500
#define __EXTENSIONS__
#endif /* NEED_XOPEN */

#include <sys/types.h>
#include <sys/param.h>
#include <sys/socket.h>
#include <sys/uio.h>
#include <netdb.h>
#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <netinet/ip6.h>
#include <netinet/udp.h>
#include <netinet/icmp6.h>
#include <arpa/inet.h>

#include "pc.h"
#include "PctestIpv6Udp.h"
#include "TestRecord.h"

//
// GetSocketOut
//
// Input:  None
//
// Output:  In return value, returns socket number.
//
// Get output socket of an appropriate type.  We'll make this virtual 
// shortly.
//
int PctestIpv6Udp::GetSocketOut() {

    int sock;
    int rc;
    
    sock = socket(AF_INET6, SOCK_DGRAM, 0);
    if (sock < 0) {
	perror("socket");
	return sock;
    }
    
#ifdef linux
    // Linux needs SO_BSDCOMPAT enabled on our UDP socket, to avoid
    // getting ICMP errors when we send packets out.
    int bsdcompatOption;
    bsdcompatOption = 1;
    rc = setsockopt(sock, SOL_SOCKET, SO_BSDCOMPAT, &bsdcompatOption,
		    sizeof(bsdcompatOption));
    if (rc < 0) {
	perror("setsockopt(SO_BSDCOMPAT)");
	return rc;
    }
#endif /* linux */

    return sock;
}

//
// PctestIpv6Udp::Test
//
// Input:
//
// Output:
//
// A negative icmpCode indicates a timeout.
//
int PctestIpv6Udp::Test(TestRecord &tr)
{

    struct timeval tvBefore, tvAfter;
    struct timeval timeout;
    int rc;			// syscall return code
    fd_set readFds;		// reading file descriptors
    int done = 0;

    // If the requested sending size is too small, then return an
    // error.  The caller should have figured out the minimum sending
    // size by calling Pctest::GetMinSize().
    if (tr.size < GetMinSize()) {
	return -1;
    }

    // Make up a UDP packet to send out.
    int udpPayloadSize = tr.size - sizeof(ip6_hdr) - sizeof(udphdr);
    char *udpPayload;
    udpPayload = GeneratePayload(udpPayloadSize);
    if (udpPayload == NULL) {
	fprintf(stderr, "Couldn't allocate space for payload\n");
	return -1;
    }

    targetSocketAddress.sin6_port = htons(destPort++);

    // Set TTL.
    rc = setsockopt(socketOut, IPPROTO_IPV6, IPV6_UNICAST_HOPS, (char *) &tr.hops, sizeof(tr.hops));
    if (rc < 0) {
	perror("setsockopt(IPV6_UNICAST_HOPS)");
	return rc;
    }

    // Set timeout value and socket select parameters
    extern int Timeout;
    timeout.tv_sec = Timeout;
    timeout.tv_usec = 0;
    FD_ZERO(&readFds);
    FD_SET(socketIn, &readFds);

    // Timestamp before 
    gettimeofday(&tvBefore, NULL);

    // Send UDP packet
    rc = sendto(socketOut, udpPayload, udpPayloadSize, 0,
		(struct sockaddr *) &targetSocketAddress,
		sizeof(struct sockaddr_in6));
    delete [] udpPayload;
    if (rc < 0) {
	perror("sendto");
	return rc;
    }

    // We need to check the socket until we get a valid packet.
    // So we might end up doing this select/read several times.
    do {

	// Select and wait for an ICMP response or a timeout
	rc = select(FD_SETSIZE, &readFds, NULL, NULL, &timeout);
	if (rc < 0) {
	    perror("select");
	    return rc;
	}

	// Timestamp after and update test record timestamp fields
	gettimeofday(&tvAfter, NULL);
	tr.tv.tv_sec = tvAfter.tv_sec - tvBefore.tv_sec - syscallTime.tv_sec;
	tr.tv.tv_usec = tvAfter.tv_usec - tvBefore.tv_usec - syscallTime.tv_usec;
	while (tr.tv.tv_usec < 0) {
	    tr.tv.tv_usec += 1000000;
	    tr.tv.tv_sec--;
	}

	// Read response from socket
	if (rc == 1) {
	    IF_DEBUG(2, fprintf(stderr, "icmp packet received\n"));

	    const int icmpPacketSize = 1500;
	    char icmpPacket[icmpPacketSize];
	    struct msghdr msg;	// msghdr is for recvmsg
	    struct iovec iov[1];
	    int controlsize;
	    struct cmsghdr *cmsgptr;

	    // Fill in the message header so we can read all the
	    // metadata from the ICMP packet.  A lot harder than
	    // with ICMPv4 since we had the IP header to work with.
	    msg.msg_name = (char *) &icmpSourceSocketAddress;
	    msg.msg_namelen = sizeof(icmpSourceSocketAddress);
	    iov[0].iov_base = icmpPacket;
	    iov[0].iov_len = icmpPacketSize;
	    msg.msg_iov = iov;
	    msg.msg_iovlen = 1;

	    // Solaris 8 (which has native IPv6) doesn't define 
            // CMSG_SPACE for now.  According to Erik Nordmark
	    // <Erik.Nordmark@eng.sun.com> it'll be added once
	    // draft-ietf-ipngwg-2292bis becomes an RFC.  Until
	    // then, he has a small hack to fix this, slightly
	    // modified by bmah.
#ifdef __sun__
#ifndef CMSG_SPACE
#define CMSG_SPACE(length) \
        (_CMSG_DATA_ALIGN(sizeof(struct cmsghdr)) + _CMSG_HDR_ALIGN(length))
#endif /* CMSG_SPACE */
#endif /* __sun__ */
	    controlsize = CMSG_SPACE(sizeof(in6_pktinfo));
	    msg.msg_control = new char[controlsize];
	    msg.msg_controllen = controlsize;
	    msg.msg_flags = 0;

	    rc = recvmsg(socketIn, &msg, 0);
	    if (rc < 0) {
		perror("read");
		return rc;
	    }

	    // Now parse the packet, doing a little error checking along
	    // the way.  By the end, we'll have ipHeader and icmpHeader
	    // pointing to valid structures within the packet, and
	    // ipHeader2 pointing to the IP header of the generating
	    // IP packet..
	    ip6_hdr *ipHeader2;
	    icmp6_hdr *icmpHeader;
	    udphdr *udpHeader;

	    // Find the ICMPv6 header
	    icmpHeader = (icmp6_hdr *) icmpPacket;
	    IF_DEBUG(3, fprintf(stderr, "icmp type = %d, code = %d\n", 
				icmpHeader->icmp6_type, icmpHeader->icmp6_code));

	    // Check ICMP type.  Most types (such as echo request/reply,
	    // router adverts, etc.) we ignore.
	    if ((icmpHeader->icmp6_type != ICMP6_TIME_EXCEEDED) && 
		(icmpHeader->icmp6_type != ICMP6_DST_UNREACH)) {
		IF_DEBUG(3, fprintf(stderr, "ignoring icmp packet\n"));
		goto donepacket;
	    }
	    
	    // Check for a valid (to us) IP header within the packet.
	    // For "time exceeded" or "destination unreachable", this
	    // header will be 8 bytes past the ICMP header.
	    ipHeader2 = (ip6_hdr *) ((char *) icmpHeader + sizeof(icmp6_hdr));

	    // Check to be sure that we have enough of the packet to hold
	    // a valid IP header? XXX

	    // Additional checking here...must be UDP
	    if (ipHeader2->ip6_nxt != IPPROTO_UDP) {
		IF_DEBUG(3, fprintf(stderr, "ignoring icmp packet for non-udp\n"));
		goto donepacket;
	    }

	    // Align UDP header template, check port numbers and
	    // payload length for us. XXX
	    udpHeader = (udphdr *) (((char *) ipHeader2) + sizeof(ip6_hdr));

	    // Check destination UDP port number (we don't know the
	    // source) and UDP (header+payload) length
	    if ((udpHeader->uh_dport != targetSocketAddress.sin6_port) || 
		(ntohs(udpHeader->uh_ulen) != udpPayloadSize + sizeof(udphdr))) {
		IF_DEBUG(3, fprintf(stderr, "ignoring icmp packet for unknown udp packet\n"));
		goto donepacket;
	    }

	    // Fill in return fields
	    tr.targetAddress = new char[sizeof(in6_addr)];
	    memcpy(tr.targetAddress, &targetAddress, sizeof(in6_addr));
	    tr.targetAddressLength = sizeof(in6_addr);

	    // See if there's any ancillary data...if so it should
	    // be the receiving address for the ICMPv6 packet.  Hopefully
	    // that's the same as the source address for the original
	    // UDP packet we sent.
	    tr.icmpDestAddress = new char[sizeof(in6_addr)];
	    memset(tr.icmpDestAddress, 0, sizeof(in6_addr));
	    cmsgptr = CMSG_FIRSTHDR(&msg);
	    if (cmsgptr) {
		if ((cmsgptr->cmsg_level == IPPROTO_IPV6) && 
		    (cmsgptr->cmsg_type == IPV6_PKTINFO)) {
		    memcpy(tr.icmpDestAddress, CMSG_DATA(cmsgptr), sizeof(in6_addr));
		}
	    }
	    tr.icmpDestAddressLength = sizeof(in6_addr);

	    tr.icmpSourceAddress = new char[sizeof(in6_addr)];
	    memcpy(tr.icmpSourceAddress, &icmpSourceSocketAddress.sin6_addr, sizeof(in6_addr));
	    tr.icmpSourceAddressLength = sizeof(in6_addr);

	    tr.icmpType = icmpHeader->icmp6_type;
	    tr.icmpCode = icmpHeader->icmp6_code;

	    done = 1;

	  donepacket:
	    if (msg.msg_control) {
		delete [] (char *) msg.msg_control;
		msg.msg_control = NULL;
	    }
	    
	}
	else {

	    IF_DEBUG(2, fprintf(stderr, "timeout\n"));

	    tr.targetAddress = new char[sizeof(in6_addr)];
	    memcpy(tr.targetAddress, &targetAddress, sizeof(in6_addr));
	    tr.targetAddressLength = sizeof(in6_addr);

	    tr.icmpDestAddress = new char[sizeof(in6_addr)];
	    memset(tr.icmpDestAddress, 0, sizeof(in6_addr));
	    tr.icmpDestAddressLength = sizeof(in6_addr);

	    tr.icmpSourceAddress = new char[sizeof(in6_addr)];
	    memset(tr.icmpSourceAddress, 0, sizeof(in6_addr));
	    tr.icmpSourceAddressLength = sizeof(in6_addr);

	    tr.icmpType = -1;
	    tr.icmpCode = -1;

	    done = 1;
	}

    } while (!done);

    return 0;

}

