/* $Id: pkgquery.c,v 1.3 2004/09/29 03:53:58 igor Exp $ */

/*
 * Copyright (c) 2004 Igor Boehm <igor@bytelabs.org>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF MIND, USE, DATA OR PROFITS, WHETHER
 * IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "fail.h"
#include "macros.h"
#include "pkgquery.h"

enum {
	PROP_0,
	PROP_CASE_SENS,
	PROP_QNAME,
	PROP_QCOMMENT,
	PROP_QEDESC,
	PROP_QSALL,
	PROP_QBDEP,
#ifdef __OpenBSD__
	PROP_QLDEP,
#endif				/* __OpenBSD__ */
	PROP_QRDEP
};

/* PROTOTYPES */
static void     pkgquery_class_init(PkgQueryClass *);
static void     pkgquery_finalize(GObject *);
static void     pkgquery_set_property(GObject *, guint, const GValue *, GParamSpec *);
static void     pkgquery_get_property(GObject *, guint, GValue *, GParamSpec *);


/* VARIABLES */
static gpointer parent_class = NULL;


/**
 * Register Type.
 */
GType
pkgquery_get_type(void)
{
	static GType    pkgquery_type = 0;

	if (!pkgquery_type) {
		static const GTypeInfo pkgquery_info =
		{
			sizeof(PkgQueryClass),
			NULL,	/* base_init */
			NULL,	/* base_finalize */
			(GClassInitFunc) pkgquery_class_init,	/* class initialiser */
			NULL,	/* class_finalize */
			NULL,	/* class_data */
			sizeof(PkgQuery),
			16,	/* n_preallocs */
			NULL,	/* instance initialiser */
			NULL
		};

		pkgquery_type = g_type_register_static(G_TYPE_OBJECT, "PkgQuery",
						       &pkgquery_info, 0);
	}
	return pkgquery_type;
}


/**
 * Class initialiser.
 */
static void
pkgquery_class_init(PkgQueryClass * klass)
{
	GObjectClass   *gobject_class;

	gobject_class = G_OBJECT_CLASS(klass);
	parent_class = g_type_class_peek_parent(klass);

	/* override base class methods */
	gobject_class->set_property = pkgquery_set_property;
	gobject_class->get_property = pkgquery_get_property;

	gobject_class->finalize = pkgquery_finalize;

	/* Set up properties */
	g_object_class_install_property(gobject_class,
					PROP_CASE_SENS,
					g_param_spec_boolean("case_sens",
							     "Case-Sens",
			      "Wheather the search is case sensitive or not.",
							     FALSE,
							  G_PARAM_READWRITE));
	g_object_class_install_property(gobject_class,
					PROP_QNAME,
					g_param_spec_string("q_name",
							    "q-name",
						   "Distribution name query.",
							    NULL,
							  G_PARAM_READWRITE));
	g_object_class_install_property(gobject_class,
					PROP_QCOMMENT,
					g_param_spec_string("q_comment",
							    "q-comment",
							    "Comment query.",
							    NULL,
							  G_PARAM_READWRITE));
	g_object_class_install_property(gobject_class,
					PROP_QEDESC,
					g_param_spec_string("q_edesc",
							    "q-edesc",
						"Extended Description query.",
							    NULL,
							  G_PARAM_READWRITE));
	g_object_class_install_property(gobject_class,
					PROP_QSALL,
					g_param_spec_string("q_sall",
							    "q-sall",
							  "Search all query.",
							    NULL,
							  G_PARAM_READWRITE));
	g_object_class_install_property(gobject_class,
					PROP_QRDEP,
					g_param_spec_string("q_rdep",
							    "q-rdep",
							 "Run depends query.",
							    NULL,
							  G_PARAM_READWRITE));
	g_object_class_install_property(gobject_class,
					PROP_QBDEP,
					g_param_spec_string("q_bdep",
							    "q-bdep",
						       "Build depends query.",
							    NULL,
							  G_PARAM_READWRITE));
#ifdef __OpenBSD__
	g_object_class_install_property(gobject_class,
					PROP_QLDEP,
					g_param_spec_string("q_ldep",
							    "q-ldep",
						     "Library depends query.",
							    NULL,
							  G_PARAM_READWRITE));
#endif				/* __OpenBSD__ */
}

static void
pkgquery_finalize(GObject * obj)
{
	PkgQuery       *query;

	query = PKGQUERY(obj);
	FREE(query->q_name);
	FREE(query->q_comment);
	FREE(query->q_edesc);
	FREE(query->q_sall);
	FREE(query->q_rdep);
	FREE(query->q_bdep);
#ifdef __OpenBSD__
	FREE(query->q_ldep);
#endif				/* __OpenBSD__ */
	/* DONT FORGET TO CALL PARENTS FINALIZE */
	G_OBJECT_CLASS(parent_class)->finalize(obj);
}

static void
pkgquery_set_property(GObject * obj, guint prop_id, const GValue * value, GParamSpec * pspec)
{
	PkgQuery       *query;

	query = PKGQUERY(obj);
	switch (prop_id) {
	case PROP_CASE_SENS:
		pkgquery_set_case_sensitive(query, g_value_get_boolean(value));
		break;
	case PROP_QNAME:
		pkgquery_set_qname(query, g_value_get_string(value));
		break;
	case PROP_QCOMMENT:
		pkgquery_set_qcomment(query, g_value_get_string(value));
		break;
	case PROP_QEDESC:
		pkgquery_set_qedesc(query, g_value_get_string(value));
		break;
	case PROP_QSALL:
		pkgquery_set_qsall(query, g_value_get_string(value));
		break;
	case PROP_QBDEP:
		pkgquery_set_qbdep(query, g_value_get_string(value));
		break;
	case PROP_QRDEP:
		pkgquery_set_qrdep(query, g_value_get_string(value));
		break;
#ifdef __OpenBSD__
	case PROP_QLDEP:
		pkgquery_set_qldep(query, g_value_get_string(value));
		break;
#endif
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID(obj, prop_id, pspec);
		break;
	}
}

static void
pkgquery_get_property(GObject * obj, guint prop_id, GValue * value, GParamSpec * pspec)
{
	PkgQuery       *query;

	query = PKGQUERY(obj);
	switch (prop_id) {
	case PROP_CASE_SENS:
		g_value_set_boolean(value, query->case_sens);
		break;
	case PROP_QNAME:
		g_value_set_string(value, query->q_name);
		break;
	case PROP_QCOMMENT:
		g_value_set_string(value, query->q_comment);
		break;
	case PROP_QEDESC:
		g_value_set_string(value, query->q_edesc);
		break;
	case PROP_QSALL:
		g_value_set_string(value, query->q_sall);
		break;
	case PROP_QBDEP:
		g_value_set_string(value, query->q_bdep);
		break;
	case PROP_QRDEP:
		g_value_set_string(value, query->q_rdep);
		break;
#ifdef __OpenBSD__
	case PROP_QLDEP:
		g_value_set_string(value, query->q_ldep);
		break;
#endif				/* __OpenBSD__ */
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID(obj, prop_id, pspec);
		break;
	}
}

PkgQuery       *
pkgquery_clone(PkgQuery * obj)
{
	PkgQuery       *new;

	new = g_object_new(TYPE_PKGQUERY, NULL);
	if (pkgquery_get_qname(obj))
		g_object_set(new, "q-name", pkgquery_get_qname(obj), NULL);
	if (pkgquery_get_qcomment(obj))
		g_object_set(new, "q-comment", pkgquery_get_qcomment(obj), NULL);
	if (pkgquery_get_qedesc(obj))
		g_object_set(new, "q-edesc", pkgquery_get_qedesc(obj), NULL);
	if (pkgquery_get_qsall(obj))
		g_object_set(new, "q-sall", pkgquery_get_qsall(obj), NULL);
	if (pkgquery_get_qbdep(obj))
		g_object_set(new, "q-bdep", pkgquery_get_qbdep(obj), NULL);
	if (pkgquery_get_qrdep(obj))
		g_object_set(new, "q-rdep", pkgquery_get_qrdep(obj), NULL);
#ifdef __OpenBSD__
	if (pkgquery_get_qldep(obj))
		g_object_set(new, "q-ldep", pkgquery_get_qldep(obj), NULL);
#endif				/* __OpenBSD__ */
	return new;
}

void
pkgquery_set_qname(PkgQuery * obj, const char *str)
{
	FREE(obj->q_name);
	obj->q_name = (str) ? strdup(str) : NULL;
}

void
pkgquery_set_qcomment(PkgQuery * obj, const char *str)
{
	FREE(obj->q_comment);
	obj->q_comment = (str) ? strdup(str) : NULL;
}

void
pkgquery_set_qedesc(PkgQuery * obj, const char *str)
{
	FREE(obj->q_edesc);
	obj->q_edesc = (str) ? strdup(str) : NULL;
}

void
pkgquery_set_qsall(PkgQuery * obj, const char *str)
{
	FREE(obj->q_sall);
	obj->q_sall = (str) ? strdup(str) : NULL;
}

void
pkgquery_set_qrdep(PkgQuery * obj, const char *str)
{
	FREE(obj->q_rdep);
	obj->q_rdep = (str) ? strdup(str) : NULL;
}

void
pkgquery_set_qbdep(PkgQuery * obj, const char *str)
{
	FREE(obj->q_bdep);
	obj->q_bdep = (str) ? strdup(str) : NULL;
}

char           *
pkgquery_get_qname(PkgQuery * obj)
{
	return obj->q_name;
}

char           *
pkgquery_get_qcomment(PkgQuery * obj)
{
	return obj->q_comment;
}

char           *
pkgquery_get_qedesc(PkgQuery * obj)
{
	return obj->q_edesc;
}

char           *
pkgquery_get_qsall(PkgQuery * obj)
{
	return obj->q_sall;
}

char           *
pkgquery_get_qrdep(PkgQuery * obj)
{
	return obj->q_rdep;
}

char           *
pkgquery_get_qbdep(PkgQuery * obj)
{
	return obj->q_bdep;
}

#ifdef __OpenBSD__
void
pkgquery_set_qldep(PkgQuery * obj, const char *str)
{
	FREE(obj->q_ldep);
	obj->q_ldep = (str) ? strdup(str) : NULL;
}

char           *
pkgquery_get_qldep(PkgQuery * obj)
{
	return obj->q_ldep;
}
#endif				/* __OpenBSD__ */


Boolean
pkgquery_is_case_sensitive(PkgQuery * obj)
{
	return obj->case_sens;
}

void
pkgquery_set_case_sensitive(PkgQuery * obj, Boolean sens)
{
	obj->case_sens = sens;
}



void
pkgquery_print(PkgQuery * obj)
{
	pb_message("---- PkgQuery element START ----");
	pb_message(" CASE SENSITIVE: ", (pkgquery_is_case_sensitive(obj)) ? "Y" : "N");
	pb_message(" QNAME: %s", pkgquery_get_qname(obj));
	pb_message(" QCOMMENT: %s", pkgquery_get_qcomment(obj));
	pb_message(" QEXT-DESC: %s", pkgquery_get_qedesc(obj));
	pb_message(" QSEARCH-ALL: %s", pkgquery_get_qsall(obj));
	pb_message(" QB-DEP: %s", pkgquery_get_qbdep(obj));
	pb_message(" QR-DEP: %s", pkgquery_get_qrdep(obj));
#ifdef __OpenBSD__
	pb_message(" QL-DEP: %s", pkgquery_get_qldep(obj));
#endif
	pb_message("---- PkgQuery element END ----");
}

/**
 * Returns the amount of non-NULL entries in a PkgQuery
 */
size_t
pkgquery_get_number_of_entries(PkgQuery * obj)
{
	size_t          n_entries;

	n_entries = 0;
	if (pkgquery_get_qname(obj))
		n_entries++;
	if (pkgquery_get_qcomment(obj))
		n_entries++;
	if (pkgquery_get_qedesc(obj))
		n_entries++;
	if (pkgquery_get_qsall(obj))
		n_entries++;
	if (pkgquery_get_qbdep(obj))
		n_entries++;
	if (pkgquery_get_qrdep(obj))
		n_entries++;
#ifdef __OpenBSD__
	if (pkgquery_get_qldep(obj))
		n_entries++;
#endif
	return n_entries;
}

Boolean
pkgquery_is_empty(PkgQuery * obj)
{
	if (!(obj->q_name) && !(obj->q_comment) && !(obj->q_edesc) &&
	    !(obj->q_sall) && !(obj->q_rdep) && !(obj->q_bdep)
#ifdef __OpenBSD__
	    && !(obj->q_ldep)
#endif				/* __OpenBSD__ */
	)
		return TRUE;
	else
		return FALSE;
}

void
pkgquery_clear(PkgQuery * obj)
{
	FREE(obj->q_name);
	FREE(obj->q_comment);
	FREE(obj->q_edesc);
	FREE(obj->q_sall);
	FREE(obj->q_rdep);
	FREE(obj->q_bdep);
#ifdef __OpenBSD__
	FREE(obj->q_ldep);
#endif				/* __OpenBSD__ */
}
