/* $Id: pkgcmd.c,v 1.24 2005/03/13 18:36:36 igor Exp $ */

/*
 * Copyright (c) 2004 Igor Boehm <igor@bytelabs.org>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF MIND, USE, DATA OR PROFITS, WHETHER
 * IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include <sys/types.h>
#include <sys/wait.h>

#include <errno.h>
#ifndef __OpenBSD__
#include <libutil.h>
#endif
#include <limits.h>
#include <pwd.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#ifdef __OpenBSD__
#include <util.h>
#endif

#include <glib.h>

#include "macros.h"
#include "fail.h"
#include "pkgobject.h"
#include "pbutil.h"
#include "pkgcmd.h"

/* PROTOTYPES */
static void     pb_set_process_output(const char *);
static void     pb_get_install_command(PkgObject *, PkgCmdInstallType, char *[]);
static Boolean  pb_exec_su(char *, GError **);

/* IN HERE WE HOLD THE ROOT PWD */
static char     password[_PASSWORD_LEN] = "";
static Boolean  password_is_valid = FALSE;
static Boolean	auth_once = TRUE;
static Boolean	use_sudo = FALSE;
/* IN HERE WE HOLD THE PROCESS OUTPUT */
static char     output[BUFSIZ] = "";
/* MUTEX PROTECTING ACCESS TO password[] */
static GStaticMutex password_mutex = G_STATIC_MUTEX_INIT;
/* MUTEX PROTECTING ACCESS TO output[] */
static GStaticMutex output_mutex = G_STATIC_MUTEX_INIT;
/* MUTEX PROTECTING ACCESS TO THE pkg_perform_operation() */
static GStaticMutex install_mutex = G_STATIC_MUTEX_INIT;
/* MUTEX PROTECTING ACCESS TO THE pkg_perform_operation() */
static GStaticMutex stop_mutex = G_STATIC_MUTEX_INIT;
static pid_t    child_pid = -1;
extern char    *__progname;
/* URI for installation of Packages  */
static char	pkg_uri[_POSIX_SSIZE_MAX];


/**
 * This function should be called from the outside in order
 * to install|make|uninstall a package|port.
 */
Boolean
pkg_perform_operation(InstallParams * param)
{
	GError         *tmp_err;
	char           *str;
	Boolean         retval;
	GQuark          pb_quark;

	if (!g_static_mutex_trylock(&install_mutex)) {
		pb_quark = g_quark_from_string(__progname);
		g_set_error(&param->err, pb_quark, PB_PKGCMD_ERROR_ACQUIRE_MUTEX,
			    "Failed to acquire install mutex.");
		return FALSE;
	}
	tmp_err = NULL;
	retval = TRUE;
	pb_get_install_command(param->pkg, param->inst_type, &str);
	if (!(retval = pb_exec_su(str, &tmp_err))) {
		if (param->inst_type != PB_PKGCMD_DELF_PKG ||
		    strncmp(tmp_err->message, "Can't remove", 12))
			g_propagate_error(&param->err, tmp_err);
	}
	/* DO WE ASK THE USER FOR A PWD EACH AND EVERY TIME ? */
	if (!auth_once)
		pb_root_pwd_clear(); 
	FREE(str);
	g_static_mutex_unlock(&install_mutex);
	return retval;
}

/**
 * Stop the install|uninstall process.
 */
void
pkg_stop_operation(int sig)
{
	GError         *err;
	char            buf[512];

	if (!g_static_mutex_trylock(&stop_mutex))
		return;
	err = NULL;
	if (child_pid > 0) {
		switch (sig) {
		case SIGINT:
		case SIGTERM:
			snprintf(buf, sizeof(buf), "%s -s KILL %d", KILL_BIN, child_pid);
			if (pb_get_verbose_level())
				pb_message("CAUGHT SIGNAL: KILLING pid %d", child_pid);
			if (!pb_exec_su(buf, &err))
				pb_warning(err->message);
			/* DUMP WHATEVER IS LEFT IN THE OUTPUT BUFFER */
			pb_clear_process_output();
			/* RESET THE CHILD PID */
			child_pid = -1;
			break;
		}
	}
	g_static_mutex_unlock(&stop_mutex);
}

static Boolean
pb_exec_su(char *ecmd, GError ** error)
{
	int             fdpty, status;
	int             nr, auth_tries;
	fd_set          rfds;
	pid_t           pid;
	char            buf[256];
	GQuark          pb_quark;
	struct timeval  tv;

	auth_tries = 0;
	pb_quark = g_quark_from_string(__progname);
	pid = forkpty(&fdpty, NULL, NULL, NULL);
	if (pid == 0) {
	  	if (use_sudo) {
		  	/* EXECUTE SUDO PROCESS */
		  	if (execl(SUDO_BIN, SUDO_BIN, "sh", "-c", ecmd, (char *) NULL) == -1) {
				pb_error(1, strerror(errno));
				/* NEVER REACHED */
			}
		} else {
			/* EXECUTE SU PROCESS */
			if (execl(SU_BIN, SU_BIN, "-", "root", "-c", ecmd, (char *) NULL) == -1) {
				pb_error(1, strerror(errno));
				/* NEVER REACHED */
			}
		}
	} else if (pid == -1) {
		g_set_error(error, pb_quark, PB_PKGCMD_ERROR_FORK,
			    "Failed to fork install process.");
	} else {
		child_pid = pid;
		if ((nr = read(fdpty, buf, sizeof(buf)) - 1) == -1 && nr != 0) {
			g_set_error(error, pb_quark, PB_PKGCMD_ERROR_READ_FD,
			      "Error while reading from file descriptor: %s.",
				    strerror(errno));
			return FALSE;
		} 
		/* MAKE SURE IT IS NULL TERMINATED */
		buf[sizeof(buf) - 1] = '\0';
		if (!strncmp(buf, "Password:", 9)) {
			write(fdpty, password, strlen(password));
		}
		FD_ZERO(&rfds);
		FD_SET(fdpty, &rfds);
		while (!waitpid(pid, &status, WNOHANG) && !*error) {
			tv.tv_sec = 0;
			tv.tv_usec = 100;
			if (select(fdpty + 1, &rfds, NULL, NULL, &tv) < 0) {
				g_set_error(error, pb_quark, PB_PKGCMD_ERROR_READ_FD,
					    "Could not read from pipe: %s.",
					    strerror(errno));
				return FALSE;
			}
			if (FD_ISSET(fdpty, &rfds)) {
				bzero(buf, sizeof(buf));
				if ((nr = read(fdpty, buf, sizeof(buf) - 1)) != -1 && nr != 0) {
					/* MAKE SURE IT IS NULL TERMINATED */
				  	buf[sizeof(buf) - 1] = '\0';
					
					/* XXX: SUDO WORKAROUND 
					 * IF WE USE SUDO WE FIRST GET THE USUAL SUDO
					 * LECTURE BEFORE SUDO IS ASKING FOR A PWD. 
					 * THAT's WHY WE DIDN'T CATCH IT EARLIER SO DO
					 * IT NOW
					 */
					if (use_sudo && !*error && auth_tries > 0) {
						g_set_error(error, pb_quark, PB_PKGCMD_ERROR_AUTH,
						    "Wrong password. Authentication failed.");
						password_is_valid = FALSE;
					} else if (use_sudo && !auth_tries && !strncmp(buf, "Password:", 9)) {
					  	write(fdpty, password, strlen(password));
						auth_tries++;	
					} else {
						pb_set_process_output(buf);
					}
					/* REMOVE LEADING|TRAILING WHITESPACES */
					if (pb_get_verbose_level()) {
						/*
						 * XXX: HACK WHENEVER SOMETHING
						 * IS PULLED VIA FTP
						 */
						if (strstr(buf, "% |"))
							pb_printf("*");
						else
							pb_printf("%s", buf);
					}
					pb_strstrip(buf);
					/**
					 * ERROR HANDLING
					 */
					
					/* TSET HAS BEEN CALLED -> e.g. NOT USING THE DEFAULT ROOT SHELL */
					if (!*error && !strncmp(buf, "Terminal type?", 14)) {
						write(fdpty, TERM_TYPE"\n", strlen(TERM_TYPE)+1);
					}	
					/* WRONG PASSWORD */
					if (!*error && !strncmp(buf, "Sorry", 5)) {
						g_set_error(error, pb_quark, PB_PKGCMD_ERROR_AUTH,
							    "Wrong password. Authentication failed.");
						/* INCORRECT PASSWORD */
						password_is_valid = FALSE;
					} else if (!*error) { 
						/* CORRECT PASSWORD */  
						password_is_valid = TRUE;
					}
					/* PORT NOT AVAILABLE */
					if (!*error && !strncmp(buf, "Can't find", 10)) {
						g_set_error(error, pb_quark, PB_PKGCMD_ERROR_PKG_UNAVAILABLE,
							    "This Package is not available. Try to install the port.");
					}
					/* REMOVE DEPENDENCIES FIRST */
					if (!*error && !strncmp(buf, "Can't remove", 12)) {
						g_set_error(error, pb_quark, PB_PKGCMD_ERROR_PKG_UNAVAILABLE,
							    buf);
					}
					/* CONFLICTS WITH INSTALLED PACKAGE */
					if (!*error && strstr(buf, "conflicts with installed package")) {
						g_set_error(error, pb_quark, PB_PKGCMD_ERROR_PKG_CONFLICT,
							    buf);
					}
					/* OPERATION NOT PERMITTED */
					if (!*error && strstr(buf, "Operation not permitted")) {
						g_set_error(error, pb_quark, PB_PKGCMD_ERROR_PERM,
							    buf);
					}
				}
			} else
				FD_SET(fdpty, &rfds);
		}
		if (WIFEXITED(status) || *error) {
			if (!*error && WEXITSTATUS(status)) {
				g_set_error(error, pb_quark, PB_PKGCMD_ERROR_TERM,
					    "The installation process terminated: "
					    "Look at the process output for further "
					    "information.");
			}
			return (*error == NULL);
		}
	}
	return (*error == NULL);
}

void
pb_clear_process_output(void)
{
	g_static_mutex_lock(&output_mutex);
	bzero(output, sizeof(output));
	output[0] = '\0';
	g_static_mutex_unlock(&output_mutex);
}

/**
 * This function is thread safe and can be used in order to set the output
 * from the installation process.
 */
static void
pb_set_process_output(const char *str)
{
	g_static_mutex_lock(&output_mutex);
	if (strlen(str) + strlen(output) < sizeof(output)) {
		/* WE MAY APPEND */
		STRLCAT(output, str, sizeof(output));
	} else {
		/* THIS IS WAY TOO MUCH CLEAR OUT FIRST  */
		bzero(output, sizeof(output));
		STRLCPY(output, str, sizeof(output));
	}
	g_static_mutex_unlock(&output_mutex);
}

/**
 * This function is thread safe and can be used to get the output from
 * the installation process. After this function is called, all chars
 * are zeroed out from the char array holding the output.
 */
void
pb_grep_process_output(char **str)
{
	if (!g_static_mutex_trylock(&output_mutex)){
		*str = NULL;
		return;
	}	  
	if (output && output[0] != '\0')
		*str = strdup(output);
	bzero(output, sizeof(output));
	output[0] = '\0';
	g_static_mutex_unlock(&output_mutex);
}

/**
 * Set the password char array. This function is THREAD SAFE.
 */
void
pb_root_pwd_set(const char *pwd)
{
	g_static_mutex_lock(&password_mutex);
	if (pwd[strlen(pwd) - 1] == '\n') {
		STRLCPY(password, pwd, sizeof(password));
	} else
		snprintf(password, sizeof(password), "%s\n", pwd);
	g_static_mutex_unlock(&password_mutex);
}

/**
 * Clear the password char array. This function is THREAD SAFE.
 */
void
pb_root_pwd_clear(void)
{
	g_static_mutex_lock(&password_mutex);
	bzero(password, sizeof(password));
	password_is_valid = FALSE;
	g_static_mutex_unlock(&password_mutex);
}

/**
 * We should be able to check from the outside if
 * the superuser password has been correct.
 */
Boolean
pb_root_pwd_is_valid(void)
{
	return password_is_valid; 
}

/**
 * Set weather we want to ask the user for a pwd 
 * each and every time since we delete the pwd
 * from memory or if we should keep the pwd in 
 * memory. 
 */
void
pb_auth_once_set(Boolean val)
{
	auth_once = val;
}

/**
 * Should we use sudo(8) instead of su(1).
 */
void
pb_auth_use_sudo(Boolean val)
{
	use_sudo = val;
}

Boolean
pb_auth_type_is_sudo(void)
{
	return use_sudo;
}

/**
 * Get a copy of the current PKG_PATH setting.
 */
void
pb_pkg_path_get(char **dest)
{
	if (pkg_uri)
	  *dest = strdup(pkg_uri);
	else
	  *dest = NULL;
}

/**
 * Set the current PKG_PATH.
 */
void
pb_pkg_path_set(const char *src)
{
  	bzero(pkg_uri, sizeof(pkg_uri));
	STRLCPY(pkg_uri, src, sizeof(pkg_uri));
}

/**
 * Put together the command depending on users choice. This
 * is a little nasty because of all the #ifdefs.
 */
static void
pb_get_install_command(PkgObject * pkg, PkgCmdInstallType type, char **cmd)
{
	static char     str[_POSIX_SSIZE_MAX];
#ifdef __OpenBSD__
	static char     str_extra[_POSIX_SSIZE_MAX];
#endif

	bzero(str, sizeof(str));
	switch (type) {
		/*
		 * INSTALLING PORT
		 */
	case PB_PKGCMD_INST_PORT:
#ifdef __OpenBSD__
		snprintf(str, sizeof(str), "cd %s%s && ", PORTS_DIR,
			 pkgobject_get_port_path(pkg));
		/* TAKE CARE OF FLAVOURS AND MULTIPACKAGES */
		bzero(str_extra, sizeof(str_extra));
		if (pkgobject_get_flavour(pkg)) {
			STRLCAT(str_extra, "env FLAVOR=\"", sizeof(str_extra));
			STRLCAT(str_extra, pkgobject_get_flavour(pkg), sizeof(str_extra));
			STRLCAT(str_extra, "\" ", sizeof(str_extra));
		}
		if (pkgobject_get_multipackage(pkg)) {
			STRLCAT(str_extra, "env SUBPACKAGE=\"", sizeof(str_extra));
			STRLCAT(str_extra, pkgobject_get_multipackage(pkg), sizeof(str_extra));
			STRLCAT(str_extra, "\" ", sizeof(str_extra));
		}
		if (strlen(str_extra)) {
			STRLCAT(str, str_extra, sizeof(str));
			STRLCAT(str, " make install BULK=Yes ", sizeof(str));
		} else {
			STRLCAT(str, " make install BULK=Yes ", sizeof(str));
		}
#else
		snprintf(str, sizeof(str), "cd %s && make install && make clean",
			 pkgobject_get_port_path(pkg));
#endif
		if (pb_get_verbose_level()) {
			pb_message("INSTALLING PORT %s", pkgobject_get_distrib_name(pkg));
			pb_message("COMMAND: %s", str);
		}
		break;
		/*
		 * INSTALLING PACKAGE
		 */
	case PB_PKGCMD_INST_PKG:
		snprintf(str, sizeof(str),
#ifdef __OpenBSD__
			 "%s %s%s%s",
#else
			 "%s %s%s/%s%s",
#endif
			 PKG_ADD,
			 pkg_uri,
#ifndef __OpenBSD__
			 pkgobject_get_category(pkg),
#endif
			 pkgobject_get_distrib_name(pkg), SUFFIX);
		if (pb_get_verbose_level()) {
			pb_message("INSTALLING PACKAGE %s", pkgobject_get_distrib_name(pkg));
			pb_message("COMMAND: %s", str);
		}
		break;
		/*
		 * DELETING PACKAGE
		 */
	case PB_PKGCMD_DEL_PKG:
		snprintf(str, sizeof(str), "%s %s",
			 PKG_DELE,
			 pkgobject_get_distrib_name(pkg));
		if (pb_get_verbose_level()) {
			pb_message("DELETING PORT %s", pkgobject_get_distrib_name(pkg));
			pb_message("COMMAND: %s", str);
		}
		break;
	case PB_PKGCMD_DELF_PKG:
		snprintf(str, sizeof(str), "%s %s",
			 PKG_DELE_FORCE,
			 pkgobject_get_distrib_name(pkg));
		if (pb_get_verbose_level()) {
			pb_message("DELETING PORT %s", pkgobject_get_distrib_name(pkg));
			pb_message("COMMAND: %s", str);
		}
		break;
	default:
		pb_error(1, "Unknown PkgCmdInstallType.");
		/* NOT REACHED */
	}
	if (str)
		*cmd = strdup(str);
	else
		*cmd = NULL;
}
