/* $Id: gtkutil.c,v 1.15 2004/11/14 17:25:19 igor Exp $ */

/*
 * Copyright (c) 2004 Igor Boehm <igor@bytelabs.org>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF MIND, USE, DATA OR PROFITS, WHETHER
 * IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include <pwd.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <gtk/gtk.h>

#include "macros.h"
#include "fail.h"
#include "pkgobject.h"
#include "gtkutil.h"
#include "pkgcmd.h"


/* PROTOTYPES */
static void     retrieve_pwd(GtkWidget *, gpointer);
static void     retrieve_path(GtkWidget *, gpointer);


/**
 * Set the text in a statusbar.
 */
Boolean
pb_set_statusbar_text(GtkStatusbar * bar, char *fmt,...)
{
	char            buf[LENSTATUSTEXT];
	va_list         ap;

	va_start(ap, fmt);
	vsnprintf(buf, sizeof(buf), fmt, ap);
	va_end(ap);
	/* remove anything which was on top */
	gtk_statusbar_pop(bar, -1);
	gtk_statusbar_push(bar, 0, buf);
	return TRUE;
}

/**
 * Creates a custom Button with with an image from a file or
 * from stock with a custom text.
 */
GtkWidget      *
pb_create_custom_button(const char *image_filename, const char *label_text)
{
	GtkWidget      *button;
	GtkWidget      *box;
	GtkWidget      *label;
	GtkWidget      *image;
	GtkWidget      *align;
	GtkStockItem    item;

	/* First look and see if we want a stock image */
	if (gtk_stock_lookup(image_filename, &item)) {
		image = gtk_image_new_from_stock(image_filename,
						 GTK_ICON_SIZE_BUTTON);
	} else {
		image = gtk_image_new_from_file(image_filename);
	}
	button = g_object_new(GTK_TYPE_BUTTON, NULL);
	box = g_object_new(GTK_TYPE_HBOX, NULL);
	/* Create a label for the button */
	if (strchr(label_text, '_')) {
		label = gtk_label_new_with_mnemonic(label_text);
	} else {
		label = gtk_label_new(label_text);
	}
	align = gtk_alignment_new(0.5, 0.5, 0.0, 0.0);
	/* Pack the image and label into the box */
	gtk_box_pack_start(GTK_BOX(box), image, FALSE, FALSE, 3);
	gtk_box_pack_end(GTK_BOX(box), label, FALSE, FALSE, 3);

	gtk_container_add(GTK_CONTAINER(button), align);
	gtk_container_add(GTK_CONTAINER(align), box);
	gtk_widget_show_all(align);
	return button;
}

/**
 * Append a MenuItem with accel key and signal handler to a menu.
 */
GtkWidget      *
pb_menu_shell_append(GtkMenu * menu, const char *text_or_stockid, const char *sig,
		     GCallback callback_func, gpointer callback_data,
		     GtkAccelGroup * accel, int accel_key)
{
	GtkStockItem    item;
	GtkWidget      *m_entry;

	if (gtk_stock_lookup(text_or_stockid, &item)) {
		m_entry = gtk_image_menu_item_new_from_stock(text_or_stockid, NULL);
	} else {
		if (strchr(text_or_stockid, '_'))
			m_entry = gtk_menu_item_new_with_mnemonic(text_or_stockid);
		else
			m_entry = gtk_menu_item_new_with_label(text_or_stockid);
	}
	gtk_menu_shell_append(GTK_MENU_SHELL(menu), m_entry);
	g_signal_connect(m_entry, sig, callback_func, callback_data);
	/* IS THERE A ACCEL KEY */
	if (accel)
		gtk_widget_add_accelerator(m_entry, sig, accel, accel_key,
					 GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE);
	return GTK_WIDGET(m_entry);
}

/**
 * Creates a password dialog. The entered password is stored in the
 * variable password which is defined in pkgcmd.h.
 */
GtkWidget      *
pb_create_pwd_dialog(GtkWidget *parent)
{
	GtkDialog      *pwd_dialog;
	GtkLabel       *dialog_text, *pwd_text;
	GtkImage       *dialog_icon;
	GtkHBox        *dialog_hbox, *table_hbox;
	GtkEntry       *pwd_entry;
	GtkWidget      *table;
	GtkWidget      *b_proceed;

	/* CREATE DIALOG WINDOW */
	pwd_dialog = GTK_DIALOG(
				gtk_dialog_new_with_buttons(
							    "Enter Password",	/* Titel */
							    GTK_WINDOW(parent),	/* Parent Window */
					       GTK_DIALOG_DESTROY_WITH_PARENT,	/* FLAG */
					GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT,
							    NULL));
	gtk_dialog_set_has_separator(pwd_dialog, FALSE);
	gtk_window_set_modal(GTK_WINDOW(pwd_dialog), TRUE);
	/* CUSTOM BUTTON */
	b_proceed = pb_create_custom_button(GTK_STOCK_JUMP_TO, "_Proceed");
	gtk_dialog_add_action_widget(pwd_dialog, b_proceed, GTK_RESPONSE_OK);
	GTK_WIDGET_SET_FLAGS(GTK_WIDGET(b_proceed), GTK_CAN_DEFAULT);
	gtk_widget_grab_default(GTK_WIDGET(b_proceed));
	/* Create table */
	table = gtk_table_new(2, 1, FALSE);
	gtk_table_set_row_spacings(GTK_TABLE(table), 3);
	gtk_table_set_col_spacings(GTK_TABLE(table), 3);
	pwd_text = g_object_new(GTK_TYPE_LABEL, "label", "Password:",
				NULL);
	pwd_entry = g_object_new(GTK_TYPE_ENTRY,
				 "activates-default", TRUE,
				 "max-length", _PASSWORD_LEN,
				 NULL);
	gtk_entry_set_visibility(pwd_entry, FALSE);
	gtk_entry_set_invisible_char(pwd_entry, '*');

	gtk_table_attach(GTK_TABLE(table),
			 GTK_WIDGET(pwd_text), 0, 1, 0, 1,
			 (GtkAttachOptions) (GTK_FILL),
			 (GtkAttachOptions) (GTK_FILL), 3, 0);
	gtk_table_attach(GTK_TABLE(table),
			 GTK_WIDGET(pwd_entry), 1, 2, 0, 1,
			 (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
			 (GtkAttachOptions) (0), 3, 0);
	if (pb_auth_type_is_sudo()) {
		dialog_text = g_object_new(GTK_TYPE_LABEL,
					   "label", "\
<big><b>Please authenticate yourself with your password</b></big> \
and press <b>Proceed</b> to continue.",
					   "wrap", TRUE,
					   "use-markup", TRUE,
				   	NULL);
	} else {
		dialog_text = g_object_new(GTK_TYPE_LABEL,
					   "label", "\
<big><b>This process requires administrator privileges.</b></big>\n \
Please enter the root password and press <b>Proceed</b> to continue.",
					   "wrap", TRUE,
					   "use-markup", TRUE,
				   	NULL);
	}
	dialog_icon = g_object_new(GTK_TYPE_IMAGE,
				   "stock", GTK_STOCK_DIALOG_WARNING,
				   "icon-size", GTK_ICON_SIZE_DIALOG,
				   "xalign", 0.5,
				   "yalign", 1.0,
				   NULL);
	dialog_hbox = g_object_new(GTK_TYPE_HBOX,
				   "border-width", 8,
				   NULL);
	table_hbox = g_object_new(GTK_TYPE_HBOX,
				  "border-width", 8,
				  NULL);
	/* Pack everything in there */
	gtk_box_pack_start(GTK_BOX(dialog_hbox), GTK_WIDGET(dialog_icon),
			   FALSE, FALSE, 6);
	gtk_box_pack_start(GTK_BOX(dialog_hbox), GTK_WIDGET(dialog_text),
			   TRUE, TRUE, 6);
	gtk_box_pack_end(GTK_BOX(table_hbox), GTK_WIDGET(table),
			 FALSE, FALSE, 6);
	gtk_box_pack_start(GTK_BOX((pwd_dialog)->vbox), GTK_WIDGET(dialog_hbox),
			   FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX((pwd_dialog)->vbox), GTK_WIDGET(table_hbox),
			   FALSE, FALSE, 0);
	gtk_widget_show_all(GTK_WIDGET((pwd_dialog)->vbox));
	/* connect the b_proceed button */
	g_signal_connect(GTK_OBJECT(b_proceed),
			 "clicked",
			 G_CALLBACK(retrieve_pwd),
			 pwd_entry);
	return GTK_WIDGET(pwd_dialog);
}

GtkWidget      *
pb_create_warning_dialog(GtkWidget *parent, const char *str)
{
	GtkDialog      *dialog;
	GtkHBox        *dialog_hbox;
	GtkImage       *dialog_icon;
	GtkLabel       *dialog_text;
	char            text[_POSIX_SSIZE_MAX];

	snprintf(text, sizeof(text), "<big><b>%s</b></big>", str);
	dialog = GTK_DIALOG(
			    gtk_dialog_new_with_buttons("Warning",
					 ((parent) ? GTK_WINDOW(parent) : NULL),
					       GTK_DIALOG_DESTROY_WITH_PARENT,
					    GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
							NULL));
	gtk_dialog_set_default_response(dialog, GTK_RESPONSE_ACCEPT);
	dialog_hbox = g_object_new(GTK_TYPE_HBOX, "border-width", 8, NULL);
	dialog_icon = g_object_new(GTK_TYPE_IMAGE,
				   "stock", GTK_STOCK_DIALOG_WARNING,
				   "icon-size", GTK_ICON_SIZE_DIALOG,
				   "xalign", 0.5,
				   "yalign", 1.0,
				   NULL);
	gtk_box_pack_start(GTK_BOX(dialog_hbox), GTK_WIDGET(dialog_icon),
			   FALSE, FALSE, 0);
	dialog_text = g_object_new(GTK_TYPE_LABEL,
				   "wrap", TRUE,
				   "use-markup", TRUE,
				   "label", text,
				   NULL);
	gtk_box_pack_start(GTK_BOX(dialog_hbox), GTK_WIDGET(dialog_text),
			   TRUE, TRUE, 0);
	gtk_box_pack_start(GTK_BOX(dialog->vbox), GTK_WIDGET(dialog_hbox),
			   FALSE, FALSE, 0);
	gtk_dialog_set_has_separator(dialog, FALSE);
	gtk_widget_show_all(GTK_WIDGET(dialog->vbox));
	return GTK_WIDGET(dialog);
}

/**
 * Callback function for the password dialog. It receives the GtkEntry
 * widget which contains the root password and copies it into the
 * password array which has been defined in pkgcmd.h. After that the
 * widget is destroyed right away!
 */
static void
retrieve_pwd(GtkWidget *widget, gpointer data)
{
	pb_root_pwd_set(gtk_entry_get_text(GTK_ENTRY(data)));
	/* make sure we get rid of it */
	gtk_widget_destroy(GTK_WIDGET(data));
}


/**
 * Callback function for the PKG_PATH dialog.
 */
static void
retrieve_path(GtkWidget *widget, gpointer data)
{
	pb_pkg_path_set(gtk_entry_get_text(GTK_ENTRY(data)));
}

/**
 * Creates a window with a Progress bar and returns pointer to window
 * and the Progressbar.
 */
GtkWidget      *
pb_create_progressbar(GtkWidget *parent, GtkProgressBar ** pbar,
		      GtkTextView ** view, GtkButton ** cbutton,
		      GtkButton ** sbutton, char *title, char *strlabel,
		      char *pstr)
{
	GtkWidget      *window;
	GtkWidget      *scroll;
	GtkImage       *image;
	GtkWidget      *vbox;
	GtkWidget      *hbox, *bbox;
	GtkWidget      *label;
	GtkWidget      *separator;
	char            buf[_POSIX_SSIZE_MAX];

	snprintf(buf, sizeof(buf), "<big><b>%s</b></big>", strlabel);
	window = g_object_new(GTK_TYPE_WINDOW,
			      "default-width", 500,
			      "default-height", 300,
			      "type", GTK_WINDOW_TOPLEVEL,
			      "title", title,
			      "modal", TRUE,
			      NULL);
	gtk_window_set_transient_for(GTK_WINDOW(window),
				     GTK_WINDOW(parent));
	gtk_window_set_position(GTK_WINDOW(window),
				GTK_WIN_POS_CENTER_ON_PARENT);
	vbox = g_object_new(GTK_TYPE_VBOX, NULL);
	gtk_container_add(GTK_CONTAINER(window), vbox);

	hbox = g_object_new(GTK_TYPE_HBOX, NULL);
	gtk_widget_show(hbox);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 6);
	image = g_object_new(GTK_TYPE_IMAGE,
			     "stock", GTK_STOCK_DIALOG_INFO,
			     "icon-size", GTK_ICON_SIZE_DIALOG,
			     "xalign", 0.5,
			     "yalign", 1.0,
			     NULL);
	gtk_box_pack_start(GTK_BOX(hbox), GTK_WIDGET(image), FALSE, FALSE, 0);
	label = g_object_new(GTK_TYPE_LABEL,
			     "label", buf,
			     "use-markup", TRUE,
			     "justify", GTK_JUSTIFY_LEFT,
			     NULL);
	gtk_widget_show(label);
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
	*pbar = g_object_new(GTK_TYPE_PROGRESS_BAR,
			     "pulse-step", 0.02,
			     NULL);
	scroll = g_object_new(GTK_TYPE_SCROLLED_WINDOW,
			      "hscrollbar-policy", GTK_POLICY_AUTOMATIC,
			      "shadow-type", GTK_SHADOW_IN,
			      NULL);
	gtk_widget_show(scroll);
	gtk_box_pack_start(GTK_BOX(vbox), scroll, TRUE, TRUE, 6);
	*view = g_object_new(GTK_TYPE_TEXT_VIEW,
			     "cursor-visible", FALSE,
			     "editable", FALSE,
			     "left-margin", 10,
			     NULL);
	gtk_widget_show(GTK_WIDGET(*view));
	gtk_container_add(GTK_CONTAINER(scroll), GTK_WIDGET(*view));
	if (pstr)
		g_object_set(pbar, "text", pstr, NULL);
	gtk_widget_show(GTK_WIDGET(*pbar));
	gtk_box_pack_start(GTK_BOX(vbox), GTK_WIDGET(*pbar), FALSE, FALSE, 0);

	separator = g_object_new(GTK_TYPE_HSEPARATOR, NULL);
	gtk_widget_show(separator);
	gtk_box_pack_start(GTK_BOX(vbox), separator, FALSE, FALSE, 0);

	bbox = g_object_new(GTK_TYPE_HBUTTON_BOX,
			    "layout-style", GTK_BUTTONBOX_END,
			    NULL);
	gtk_box_set_spacing(GTK_BOX(bbox), 10);
	gtk_container_set_border_width(GTK_CONTAINER(bbox), 5);
	gtk_widget_show(bbox);
	gtk_box_pack_start(GTK_BOX(vbox), bbox, FALSE, FALSE, 0);

	*cbutton = GTK_BUTTON(pb_create_custom_button(GTK_STOCK_JUMP_TO, "_Done"));
	gtk_widget_show(GTK_WIDGET(*cbutton));

	*sbutton = GTK_BUTTON(pb_create_custom_button(GTK_STOCK_CANCEL, "_Cancel"));
	gtk_widget_show(GTK_WIDGET(*cbutton));
	gtk_container_add(GTK_CONTAINER(bbox), GTK_WIDGET(*sbutton));
	gtk_container_add(GTK_CONTAINER(bbox), GTK_WIDGET(*cbutton));
	gtk_widget_show_all(vbox);
	return window;
}


/**
 * GSourceFunc - this one can be used in g_timeout_add().
 * 		The parameter must be a GTK_PROGRESS_BAR.
 */
gboolean
pb_update_progress(gpointer data)
{
	gtk_progress_bar_pulse(GTK_PROGRESS_BAR(data));
	return TRUE;
}

/**
 * GSourceFunc - this one can be used in g_timeout_add().
 * 		The parameter must be a GtkTextView.
 */
gboolean
pb_update_textview(gpointer data)
{
	GtkTextView    *view;
	GtkTextBuffer  *buf;
	GtkTextIter     start, end;
	char           *text;
	int             lcnt;

	text = NULL;
	pb_grep_process_output(&text);
	if (text && strlen(text)) {
		view = (GtkTextView *) data;
		buf = gtk_text_view_get_buffer(view);
		/* XXX: A little hack whenever stuff is fetched... */
#ifndef __OpenBSD__
		if (strstr(text, "%") && strstr(text, "Receiving")) {
#else
		if (strstr(text, "% |")) {
#endif
			lcnt = gtk_text_buffer_get_line_count(buf);
			if (lcnt > 1) {
				gtk_text_buffer_get_iter_at_line(buf, &start, (lcnt - 2));
				gtk_text_buffer_get_end_iter(buf, &end);
				gtk_text_buffer_delete(buf, &start, &end);
			}
		}
		gtk_text_buffer_get_end_iter(buf, &end);
		gtk_text_buffer_insert(buf, &end, text, -1);
		gtk_text_view_scroll_to_iter(view, &end, 0.0, FALSE, 0.0, 0.0);
	}
	FREE(text);
	return TRUE;
}

/**
 * This function creates the search widget for the ports search.
 */
GtkWidget      *
pb_create_search_window(GtkWidget *parent, GtkEntry ** entry,
			GtkWidget **tog_case, GtkButton ** search,
			GtkButton ** clear, GtkButton ** close,
			GtkOptionMenu ** optionm, const char *title)
{
	GtkWidget      *window;
	GtkWidget      *vbox;
	GtkWidget      *tlabel1, *tlabel2;
	GtkWidget      *bbox;
	GtkWidget      *table;
	GtkMenu        *menu;

	window = g_object_new(GTK_TYPE_WINDOW,
			      "type", GTK_WINDOW_TOPLEVEL,
			      "title", title,
			      NULL);
	gtk_window_set_transient_for(GTK_WINDOW(window),
				     GTK_WINDOW(parent));
	gtk_window_set_position(GTK_WINDOW(window),
				GTK_WIN_POS_CENTER_ON_PARENT);

	vbox = g_object_new(GTK_TYPE_VBOX, NULL);
	gtk_widget_show(vbox);
	gtk_container_add(GTK_CONTAINER(window), vbox);

	/* TABLE */
	table = gtk_table_new(2, 2, FALSE);
	gtk_widget_show(table);
	gtk_table_set_row_spacings(GTK_TABLE(table), 3);
	gtk_table_set_col_spacings(GTK_TABLE(table), 3);
	gtk_box_pack_start(GTK_BOX(vbox), table, TRUE, TRUE, 6);

	tlabel1 = g_object_new(GTK_TYPE_LABEL, "label", "Search:", NULL);
	gtk_widget_show(tlabel1);
	gtk_table_attach(GTK_TABLE(table), tlabel1, 0, 1, 0, 1,
			 (GtkAttachOptions) (GTK_FILL),
			 (GtkAttachOptions) (GTK_FILL), 3, 0);
	gtk_misc_set_alignment(GTK_MISC(tlabel1), 0, 0.5);

	tlabel2 = g_object_new(GTK_TYPE_LABEL, "label", "Look in:", NULL);
	gtk_widget_show(tlabel2);
	gtk_table_attach(GTK_TABLE(table), tlabel2, 0, 1, 1, 2,
			 (GtkAttachOptions) (GTK_FILL),
			 (GtkAttachOptions) (GTK_FILL), 3, 0);
	gtk_misc_set_alignment(GTK_MISC(tlabel2), 0, 0.5);


	*entry = g_object_new(GTK_TYPE_ENTRY, "activates-default", TRUE, NULL);
	gtk_widget_show(GTK_WIDGET(*entry));
	gtk_table_attach(GTK_TABLE(table),
			 GTK_WIDGET(*entry), 1, 2, 0, 1,
			 (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
			 (GtkAttachOptions) (0), 3, 0);

	/* Create Option Menu: Keep this in sync with the enum in gtkutil.h */
	menu = g_object_new(GTK_TYPE_MENU, NULL);
	gtk_menu_shell_append(GTK_MENU_SHELL(menu),
			      gtk_menu_item_new_with_label("All fields"));
	gtk_menu_shell_append(GTK_MENU_SHELL(menu),
			      gtk_menu_item_new_with_label("Package Name"));
	gtk_menu_shell_append(GTK_MENU_SHELL(menu),
			      gtk_menu_item_new_with_label("Comment"));
	gtk_menu_shell_append(GTK_MENU_SHELL(menu),
			      gtk_menu_item_new_with_label("Description"));
	gtk_menu_shell_append(GTK_MENU_SHELL(menu),
			    gtk_menu_item_new_with_label("Run dependencies"));
#ifdef __OpenBSD__
	gtk_menu_shell_append(GTK_MENU_SHELL(menu),
			gtk_menu_item_new_with_label("Library dependencies"));
#endif				/* __OpenBSD__ */
	gtk_menu_shell_append(GTK_MENU_SHELL(menu),
			  gtk_menu_item_new_with_label("Build dependencies"));
	*optionm = g_object_new(GTK_TYPE_OPTION_MENU, "menu", menu, NULL);
	gtk_table_attach(GTK_TABLE(table),
			 GTK_WIDGET(*optionm), 1, 2, 1, 2,
			 (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
			 (GtkAttachOptions) (0), 3, 0);

	/* Check Button */
	*tog_case = gtk_check_button_new_with_mnemonic("C_ase sensitive");
	gtk_widget_show(*tog_case);
	gtk_box_pack_start(GTK_BOX(vbox), *tog_case, TRUE, TRUE, 3);
	/* BUTTONBOX */
	bbox = g_object_new(GTK_TYPE_HBUTTON_BOX, NULL);
	gtk_box_set_spacing(GTK_BOX(bbox), 10);
	gtk_container_set_border_width(GTK_CONTAINER(bbox), 5);
	gtk_widget_show(GTK_WIDGET(bbox));
	gtk_box_pack_end(GTK_BOX(vbox), GTK_WIDGET(bbox), FALSE, FALSE, 0);
	/* BUTTONS */
	*search = g_object_new(GTK_TYPE_BUTTON,
			       "use-underline", TRUE,
			       "use-stock", TRUE,
			       "label", GTK_STOCK_FIND,
			       NULL);
	gtk_widget_show(GTK_WIDGET(*search));
	gtk_box_pack_start(GTK_BOX(bbox), GTK_WIDGET(*search), TRUE, TRUE, 0);
	GTK_WIDGET_SET_FLAGS(GTK_WIDGET(*search), GTK_CAN_DEFAULT);
	gtk_widget_grab_default(GTK_WIDGET(*search));

	*clear = g_object_new(GTK_TYPE_BUTTON,
			      "use-underline", TRUE,
			      "use-stock", TRUE,
			      "label", GTK_STOCK_CLEAR,
			      NULL);
	gtk_widget_show(GTK_WIDGET(*clear));
	gtk_box_pack_start(GTK_BOX(bbox), GTK_WIDGET(*clear), TRUE, TRUE, 0);

	*close = g_object_new(GTK_TYPE_BUTTON,
			      "use-underline", TRUE,
			      "use-stock", TRUE,
			      "label", GTK_STOCK_CLOSE,
			      NULL);
	gtk_widget_show(GTK_WIDGET(*close));
	gtk_box_pack_start(GTK_BOX(bbox), GTK_WIDGET(*close), TRUE, TRUE, 0);
	return window;
}

GtkWidget      *
pb_create_path_dialog(GtkWidget *parent)
{
	GtkDialog      *path_dialog;
	GtkHBox        *dialog_hbox;
	GtkEntry       *path_entry;
	GtkLabel       *path_label;
	GtkWidget      *b_proceed;
	char	       *path_txt;
	
	/* CREATE DIALOG WINDOW */
	path_dialog = GTK_DIALOG(
				 gtk_dialog_new_with_buttons(
							  "Edit Package Path",	/* Titel */
							   GTK_WINDOW(parent),	/* Parent Window */
					       GTK_DIALOG_DESTROY_WITH_PARENT,	/* FLAG */
					GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT,
							     NULL));
	gtk_dialog_set_has_separator(path_dialog, FALSE);
	gtk_window_set_modal(GTK_WINDOW(path_dialog), TRUE);
	/* CUSTOM BUTTON */
	b_proceed = pb_create_custom_button(GTK_STOCK_OK, "_OK");
	gtk_dialog_add_action_widget(path_dialog, b_proceed, GTK_RESPONSE_OK);
	GTK_WIDGET_SET_FLAGS(GTK_WIDGET(b_proceed), GTK_CAN_DEFAULT);
	gtk_widget_grab_default(GTK_WIDGET(b_proceed));

	path_label = g_object_new(GTK_TYPE_LABEL, 
	    	"label", "Package Path:",
		NULL);  
	path_entry = g_object_new(GTK_TYPE_ENTRY,
				  "activates-default", TRUE,
				  NULL);
	pb_pkg_path_get(&path_txt);
	gtk_entry_set_text(path_entry, path_txt);
	FREE(path_txt);
	dialog_hbox = g_object_new(GTK_TYPE_HBOX,
				   NULL);
	/* Pack everything in there */
	gtk_box_pack_start(GTK_BOX(dialog_hbox), GTK_WIDGET(path_label),
			   TRUE, TRUE, 6);
	gtk_box_pack_start(GTK_BOX(dialog_hbox), GTK_WIDGET(path_entry),
			   TRUE, TRUE, 6);
	gtk_box_pack_start(GTK_BOX((path_dialog)->vbox), GTK_WIDGET(dialog_hbox),
			   FALSE, FALSE, 0);
	gtk_widget_show_all(GTK_WIDGET((path_dialog)->vbox));
	/* connect the b_proceed button */
	g_signal_connect(GTK_OBJECT(b_proceed),
			 "clicked",
			 G_CALLBACK(retrieve_path),
			 path_entry);
	return GTK_WIDGET(path_dialog);
}
