/* $Id: fail.c,v 1.5 2004/09/29 03:53:58 igor Exp $ */

/*
 * Copyright (c) 2004 Igor Boehm <igor@bytelabs.org>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF MIND, USE, DATA OR PROFITS, WHETHER
 * IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include <limits.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>

#include "fail.h"

/* ERROR MESSAGES */
#define	ERR_MSG		"ERROR: "
#define MSG_MSG		"MESSAGE: "
#define WRN_MSG		"WARNING: "
#define VRB_MSG		"VERBOSE: "

typedef enum {
	WARNING,
	MESSAGE,
	ERROR,
	VERBOSE,
	NONE,
	N_MESSAGES
} MsgType;

/* verbose level */
int             i_verbose;

/* PROTOTYPES */
static void     pb_errprefix(FILE *, MsgType);
static void     pb_writemsg(FILE *, MsgType, const char *, va_list);

void
pb_set_verbose_level(int level)
{
	i_verbose = level;
}

int
pb_get_verbose_level(void)
{
	return i_verbose;
}

static void
pb_errprefix(FILE * stream, MsgType type)
{
	switch (type) {
		case WARNING:
		fprintf(stream, "%s", WRN_MSG);
		break;
	case ERROR:
		fprintf(stream, "%s", ERR_MSG);
		break;
	case MESSAGE:
		fprintf(stream, "%s", MSG_MSG);
		break;
	case VERBOSE:
		fprintf(stream, "%s", VRB_MSG);
		break;
	case NONE:
		break;
	default:
		break;
	}
}

static void
pb_writemsg(FILE * stream, MsgType type, const char *fmt, va_list ap)
{
	pb_errprefix(stream, type);
	vfprintf(stream, fmt, ap);
	if (type != NONE)
		fprintf(stream, "\n");
	fflush(stream);
}


void
pb_warning(const char *fmt,...)
{
	va_list         ap;

	va_start(ap, fmt);
	pb_writemsg(stderr, WARNING, fmt, ap);
	va_end(ap);
}

void
pb_message(const char *fmt,...)
{
	va_list         ap;

	va_start(ap, fmt);
	pb_writemsg(stdout, MESSAGE, fmt, ap);
	va_end(ap);
}

void
pb_printf(const char *fmt,...)
{
	va_list         ap;

	if (pb_get_verbose_level()) {
		va_start(ap, fmt);
		pb_writemsg(stdout, NONE, fmt, ap);
		va_end(ap);
	}
}

void
pb_error(const int eval, const char *fmt,...)
{
	va_list         ap;

	va_start(ap, fmt);
	pb_writemsg(stderr, ERROR, fmt, ap);
	va_end(ap);
	exit(eval);
}
