/**
 ** Copyright (c) 1996  En Garde Systems, Inc. (EGS) 
 **
 ** Users and possessors of this source code are hereby granted a
 ** nonexclusive, royalty-free copyright and design patent license to
 ** use this code in individual software provided that the above copyright
 ** notice and this paragraph are included in their entirety.  License is
 ** not granted for commercial resale, in whole or in part, without prior
 ** written permission from EGS.  This source is provided "AS IS"
 ** without express or implied warranty of any kind.
 **
 ** In the case of superceding copyright notices, only the modifications to
 ** the original software are covered by this notice.
 **
 ** For further information contact:
 **     E-Mail:     info@EnGarde.com
 **
 **     Surface Mail:   ATTN: PICS Research
 **             En Garde Systems, Inc.
 **             2101 White Cloud NE
 **             Albuquerque, NM 87112
 **             (505) 275-8655
 **/

/*
 * This file should display a connection just as it was recorded
 */

#include <stdio.h>
#include <alloca.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <netdb.h>
#include <netinet/in_systm.h>
#include <netinet/in.h>
#include <net/if.h>
#include <netinet/if_ether.h>
#include <netinet/ip.h>
#include <netinet/ip_icmp.h>
#include <netinet/tcp.h>
#include <netinet/udp.h>

#include <sgtty.h>

#include "pcap.h"
#include "parse.h"

extern struct conn_list_struct *ConnectionList;
u_long src, dst;
u_short src_port, dst_port;
int playspeed=1,oldplayspeed;
int reverseFlag=0;

static void display_handler();
static struct timeval *subtract_timeval();
static struct timeval *add_timeval();
static void write_ctos(); /* Client to server */
static void write_stoc(); /* Server to Client */
static void initialize_screen(); /* Clear screen, setup windows, etc */
static void dump_handler();
extern int ignore_user;

display_connection(input_filename, con_id)
char *input_filename;
int con_id;
{
  char ebuf[PCAP_ERRBUF_SIZE];
  pcap_t *pcp;
  struct conn_list_struct *clp;
  int i;
  extern void summarize_handler();

  if ((pcp = pcap_open_offline(input_filename, ebuf)) == NULL) {
    fprintf(stderr, "Can't open input file: %s\n", ebuf);
    exit(1);
  }

  if (pcap_dispatch(pcp, 0, summarize_handler, NULL) < 0) {
    pcap_perror(pcp, "Can't read from input log");
    pcap_close(pcp);
    exit(1);
  }

  pcap_close(pcp);

  /* Should be summarized, figure out which connection the user mentioned */
  for (i=0, clp = ConnectionList; (i < con_id && clp); i++) clp=clp->next;
  if (clp==NULL) {
    fprintf(stderr, "Connection ID out of range for input file.\n");
    exit(1);
  }

  src = clp->src;
  dst = clp->dst;
  src_port = clp->src_port;
  dst_port = clp->dst_port;

  if ((pcp = pcap_open_offline(input_filename, ebuf)) == NULL) {
    fprintf(stderr, "Can't open input file: %s\n", ebuf);
    exit(1);
  }

  if (pcap_dispatch(pcp, 0, display_handler, NULL) < 0) {
    pcap_perror(pcp, "Can't read from input log");
    pcap_close(pcp);
    exit(1);
  }

  pcap_close(pcp);
  return;
}

char init_user=0;
struct sgttyb init_mode, new_mode;

void
input_init()
{
  ioctl(0, TIOCGETP, &init_mode);
  new_mode = init_mode;
  new_mode.sg_flags |= CBREAK;
  new_mode.sg_flags &= ~ECHO;
  ioctl(0, TIOCSETP, &new_mode);
}

extern time_t time_offset;

static void
display_handler(user_data, pkth, p)
u_char *user_data;
struct pcap_pkthdr *pkth;
u_char *p;
{
  struct ether_header *ep;
  struct ip *ip;
  struct tcphdr *tcph;
  u_char *abuf, *abuf2;
  int flags, length;
  static struct timeval last_time;
  static int offset_set=0;
  struct timeval ct;

  fd_set fdset;
  struct timeval timeout;
  char inchar;
  char outbuf[64];

  /* tjk timing */
  static struct timeval session_start, cur_delta;

  /* set up cbreak mode, to handle user input */
  if (!init_user) {
    input_init();
    init_user=1;
  }

  ep = (struct ether_header *)p;
  p += sizeof(struct ether_header);

  if (ntohs(ep->ether_type) != ETHERTYPE_IP)
    return; /* Skip ARPs and other nasties */

  ip = (struct ip *)p;

  if ((int)ip & (sizeof(long)-1)) {
    /* needs to be aligned */
    abuf = (u_char *)alloca(pkth->caplen);

    bcopy((char *)ip, (char *)abuf, MIN(pkth->len, pkth->caplen));
    ip = (struct ip *)abuf;
  }
  p += ip->ip_hl * 4;

  if ((ntohs(ip->ip_off) & 0x1fff) ||  (ip->ip_p!=IPPROTO_TCP)) 
    return; /* Skip UDP or non-0 offsetted packets */

  tcph = (struct tcphdr *)p;
  if ((int)tcph & (sizeof(long)-1)) {
    /* needs to be aligned */
    abuf2 = (u_char *)alloca(pkth->caplen);

    bcopy((char *)tcph, (char *)abuf2, MIN(pkth->len, pkth->caplen));
    tcph = (struct tcphdr *)abuf2;
  }

  length = htons(ip->ip_len) - (ip->ip_hl * 4);
  length -= (tcph->th_off * 4); /* Subtract TCP header length */

  p += tcph->th_off * 4;

  /* check for time offsetting */
  if (time_offset && time_offset > pkth->ts.tv_sec)
	return;

  /* 
   * Okay, now figure out where and how we should display the packet 
   * The following could be confusing. We're looking for a packet from
   * the destination of a session, because it contains the data the server
   * sent to the user. This misses passwords, but we can deal with this
   * later.
   */
  if ((ntohl(ip->ip_src.s_addr) == src && ntohs(tcph->th_sport) == src_port &&
       (ntohl(ip->ip_dst.s_addr) == dst && ntohs(tcph->th_dport) == dst_port))
      ||
      (ntohl(ip->ip_src.s_addr) == dst && ntohs(tcph->th_sport) == dst_port &&
       ntohl(ip->ip_dst.s_addr) == src && ntohs(tcph->th_dport) == src_port)) {

    if (offset_set==0) {
      /* tjk timing */
      bcopy(&pkth->ts, &session_start, sizeof(struct timeval));
      last_time.tv_sec = 0;
      last_time.tv_usec = 0;
      cur_delta.tv_sec = 0;
      cur_delta.tv_usec = 0;
      offset_set = 1;
      initialize_screen();
    }
    
    for (;;) {
      int sel_out;
      struct timeval t1,t2,t3;

      bcopy(subtract_timeval(&pkth->ts, &session_start),
			&t1, sizeof(struct timeval));
      bcopy(subtract_timeval(&pkth->ts, &last_time),
			&t2, sizeof(struct timeval));
      last_time = pkth->ts;

	  timeout = t2;

	  while (playspeed && (timeout.tv_usec > 0 || timeout.tv_sec > 0)) {
		FD_ZERO(&fdset);
		FD_SET(0, &fdset);
		if (timeout.tv_sec >= 1) {
		  if (playspeed > 1) {
			t3.tv_usec = 1000000/playspeed;
			t3.tv_sec = 0;
		  } else { /* playspeed == 1 */
			t3.tv_usec = 0;
			t3.tv_sec = 1;
		  }
		  if (select(FD_SETSIZE, &fdset, 0, 0, &t3))
			break; /* break if we got user input */
 
		  /* update the clock */
		  t3.tv_usec = 0;
		  t3.tv_sec=1;
		  bcopy(add_timeval(&t3,&cur_delta), &cur_delta,
				sizeof(struct timeval));
		  write_ctos(add_timeval(&cur_delta, &session_start), p, 0);
 
		  timeout.tv_sec--;
		} else { /* timeout.tv_sec == 0 */
		  t3.tv_usec = (timeout.tv_usec)/playspeed;
		  t3.tv_sec = 0;
		  if (select(FD_SETSIZE, &fdset, 0, 0, &t3))
			break; /* break if we got user input */
		  timeout.tv_usec=0;
		  timeout.tv_sec=0;
		}
	  }

	  cur_delta = t1;
	  if (FD_ISSET(0, &fdset)) {
		read(0, &inchar, 1);
		switch(inchar) {
		  case '0':
		  case '1':
		  case '2':
		  case '3':
		  case '4':
		  case '5':
		  case '6':
		  case '7':
		  case '8':
		  case '9':
			playspeed=inchar-48;
			break;
		  case 'p':
			if (playspeed) {
			  oldplayspeed = playspeed;
			  playspeed=0;
			} else {
			  playspeed=oldplayspeed;
			}
			break;
		  case '>':
			if (playspeed < 10) playspeed++;
			break;
		  case '<':
			if (playspeed > 0) playspeed--;
			break;
		  case 'Q':
		  case 'q':
			cleanup_output();
			break;
		  default:
			break;
		}
      }
      /* after select -- DEFINITELY output packet, unless paused */
      if (playspeed) {
		/* Display the packet */

		/* 
		 * Here is the intelligence to figure out which side to display the
		 * data on. We should always display the lower number port as being
		 * the server, unless the user specifies the "reverse" flag
		 */
		if (ntohl(ip->ip_src.s_addr) == src && 
			ntohs(tcph->th_sport) == src_port &&
			ntohl(ip->ip_dst.s_addr) == dst && 
			ntohs(tcph->th_dport) == dst_port) {
		  /* Client to server */
		  if (reverseFlag)
			write_stoc(add_timeval(&cur_delta,&session_start), p, length);
		  else
			write_ctos(add_timeval(&cur_delta,&session_start), p, length);
		} else {
		  /* Server to client */
		  if (reverseFlag)
			write_ctos(add_timeval(&cur_delta,&session_start), p, length);
		  else
			write_stoc(add_timeval(&cur_delta,&session_start), p, length);
		  /* update clock, no data */
		  write_ctos(add_timeval(&cur_delta, &session_start), p, 0);
		}
		break;
      } else {
		write_ctos(add_timeval(&cur_delta, &session_start), p, 0);
	  }
    }
  }
  return;
}

static struct timeval *
subtract_timeval(first, second)
  /* First - second */
struct timeval *first;
struct timeval *second;
{
  long u_sec;
  long sec;
  static struct timeval ret;

  u_sec = first->tv_usec - second->tv_usec;
  sec = first->tv_sec - second->tv_sec;

  if (u_sec < 0) {
	sec--;
	u_sec = 1000000 + u_sec;
  }
  if (u_sec > 1000000) {
	sec++;
	u_sec -= 1000000;
  }
  ret.tv_sec = sec;
  ret.tv_usec = u_sec;
  return(&ret);
}

static struct timeval *
add_timeval(first, second)
struct timeval *first;
struct timeval *second;
{
  long u_sec;
  long sec;
  static struct timeval ret;

  u_sec = first->tv_usec + second->tv_usec;
  sec = first->tv_sec + second->tv_sec;

  if (u_sec > 1000000) {
	sec++;
	u_sec -= 1000000;
  }
  ret.tv_sec = sec;
  ret.tv_usec = u_sec;
  return(&ret);
}

static void
write_ctos(ts, p, length) /* Client to server */
struct timeval *ts;
char *p;
int length; /* length of p, 0 to do diffs, -1 to do complete redraw */
{
  static char former_time[50];
  static char former_speed[5];
  static char former_buffer[40];
  static int presets=0;
  static int former_buflen = 0;
  int len, diff, i;
  char *x, *y, s[2048], *pp;
  char new_time[50];
  char new_speed[5];
  char new_buffer[80], printable[80];
  char temp[80];
  time_t seconds;
  struct tm *tm;
  int diffs=0, mode;

  if (ignore_user)
	return;

  if (presets==0 || length==-1) {
	strcpy(former_time, "XXX XX XXXXXXXX XXX XXXX");
	strcpy(former_speed, "XX");
	presets = 1;
	diffs = 1;
	printf("\0337\033[1;24r\033[25;1H\033[2K");
	/* Save cursor position, Restrict scrolling, clear bottom two lines */
  }

  /* Save the cursor position, only if necessary (output mode, etc) */

  /* Put the new time into the buffer */
  seconds = ts->tv_sec;
  tm = localtime(&seconds);

  strftime(new_time, 49, "%h %d %H:%M:%S %Z %Y", tm);
  if (strcmp(new_time, former_time)) {
	if (diffs==0) {
	  printf("\0337");
	  diffs=1;
	}
	printf("\033[1m");
	for (x=former_time, y=new_time; (*x!=0 && *y!=0); ) {
	  if (*x != *y) {
		printf("\033[26;%dH", 1+(y-new_time));
		while (*x != *y) {
		  putchar(*y);
		  x++;
		  y++;
		}
	  } else {
		x++;
		y++;
	  }
	}
	printf("\033[0m");
	strcpy(former_time, new_time);
  }

  /* Draw Output Speed */
  sprintf(new_speed, "%2d", playspeed);
  if (strcmp(new_speed, former_speed)) {
	if (diffs==0) {
	  printf("\0337");
	  diffs=1;
	}
	if (length == -1) 
	  printf("\033[26;28H\033[1m+%s.0\033[0m", new_speed);
	else
	  printf("\033[26;29H\033[1m%s\033[0m", new_speed);
	strcpy(former_speed, new_speed);
  }

  /* Draw Client buffer */
  if (length==-1) { /* Redraw the buffer from scratch */
	if (diffs==0) {
	  printf("\0337");
	  diffs=1;
	}
	printf("\033[26;35H\033[7m%-46s\033[0m", former_buffer);
  }
	
  if (length > 0) {
	if (diffs==0) {
	  printf("\0337");
	  diffs=1;
	}
	/* Buffer length is 46 characters */

	if (length > 46) {
	  len = 46;
	  strncpy(s, p+(length-46), len); /* Get last 46 characters */
	  *(s+len)=0;
	} else {
	  len = length;
	  strncpy(s, p, len);
	  *(s+len)=0;
	}

	mode=0;
	pp=printable;
	for (x = s, y=&new_buffer[0]; x<(s+len); x++) {
	  /* Edit buffer and replace things */
	  if (!isprint(*x)) {
		switch(*x) {
		  case 9: /* Tab */
			if (mode==0) {
			  *y++='\016'; /* Invoke graphics set */
			  mode=1;
			}
			*y++=0142;
			*pp++='~'; /* Something was printed here */
			break;
		  case 10: /* NL */
			if (mode==0) {
			  *y++='\016'; /* Invoke graphics set */
			  mode=1;
			}
			*y++=0150;
			*pp++='~'; /* Something was printed here */
			break;
		  case 11: /* VT */
			if (mode==0) {
			  *y++='\016'; /* Invoke graphics set */
			  mode=1;
			}
			*y++=0151;
			*pp++='~'; /* Something was printed here */
			break;
		  case 13: /* CR */
			if (mode==0) {
			  *y++='\016'; /* Invoke graphics set */
			  mode=1;
			}
			*y++=0144;
			*pp++='~'; /* Something was printed here */
			break;
		  default:
#if 0
			if (mode==1) {
			  *y++='\017'; /* Invoke normal character set */
			  mode=0;
			}
			*y++ = '~';
#endif
			break;
			/* Ignore non-printables */
		}
		*x='~'; /* replace with non-print, in case... */
	  } else {
		if (mode==1) {
		  *y++='\017'; /* Invoke normal character set */
		  mode=0;
		}
		*y++=*x;
		*pp++=*x; /* Something was printed here */
	  }
	}
	if (mode==1) {
	  *y++='\017';
	  mode=0;
	}
	*y=0;
	*pp=0;
	len=strlen(printable);

	/* 
	 *   y         contains an annotated string (changes character sets, etc)
	 *   printable contains the printables ONLY string
	 */


	if (len == 46) {
	  printf("\033[26;34H\033[7m%s\033[0m", new_buffer);
	  former_buflen = 46;
	  strncpy(former_buffer, printable, len);
	  *(former_buffer+len)=0;
	} else {
	  if ((former_buflen + len) > 46) {
		diff = (former_buflen + len - 46);
		printf("\033[26;34H");
		for (i=0;i<diff;i++)
		  printf("\033[P"); /* Backspace */
		printf("\033[26;%dH", 34+former_buflen-(diff));
		printf("\033[7m%s\033[0m", new_buffer);
		former_buflen = 46;
		strncpy(temp, former_buffer+diff, 46-diff);
		*(temp+(46-diff))=0;
		strcat(temp, printable, len);
		*(temp+46)=0;
		strcpy(former_buffer, temp);
	  } else {
		printf("\033[26;%dH", 34+former_buflen);
		printf("\033[7m%s\033[0m", new_buffer);
		former_buflen += len;
		strncat(former_buffer, printable, len);
		*(former_buffer+former_buflen)=0;
	  }
	}
  }

  /* Restore the saved cursor position */
  if (diffs)
    printf("\0338"); 
  fflush(stdout);
  return;
}

static void
write_stoc(ts, p, length) /* Server to client */
struct timeval *ts;
char *p;
int length;
{
  char *x, *y, *beg, value;
  int i;
  int redraw_status=0;
  enum { STATE_NOTHING, STATE_ESCAPE, STATE_BRACKET, STATE_ARGUMENTS } state;
  
  if (ignore_user) {
	write(1, p, length);
	return;
  }
  
  state = STATE_NOTHING;
  value = 0;
  for (i=0,x=p;i<length;i++,x++) {
	switch(state) {
	  case STATE_NOTHING:
		if (*x==033) {
		  state=STATE_ESCAPE;
		  beg=x;
		  value=0;
		}
		break;
	  case STATE_ESCAPE:
		if (*x=='[')
		  state=STATE_BRACKET;
		else if (*x!=033) {
		  state=STATE_NOTHING;
		  beg=NULL;
		  value=0;
		}
		break;
	  case STATE_BRACKET:
		if (*x==';' || (*x>=060 && *x<=071))
		  state=STATE_ARGUMENTS;
		else {
		  state=STATE_NOTHING;
		  value=*x;
		}
		break;
	  case STATE_ARGUMENTS:
		if (*x==';' || (*x>=060 && *x<=071))
		  break; /* Still in arguments */
		else {
		  state=STATE_NOTHING;
		  value=*x;
		}
		break;
	}
	if (value) {
	  switch(value) {
		case 'J':
		  if (*(beg+2)=='0' || *(beg+2)=='2' || *(beg+2)=='J')
			redraw_status=1;
		  break;
		case 'r':
		  *x = 030; /* CAN */
		  break;
		case 'H':
		  if (atoi(beg+2)>24) {
			*(beg+2) = '2';
			*(beg+3) = '4';
		  }
		  break;
		default:
		  break; /* Who cares */
	  }
	  beg=NULL;
	  value=0;
	  state=STATE_NOTHING;
	}
  } /* of for */
		  
  write(1, p, length);
  if (redraw_status) 
	write_ctos(ts, NULL, -1);
  return;
}

static void
initialize_screen() /* Clear screen, setup windows, etc */
{
  /* Assume vt100, already in cbreak mode */
  fd_set fdset;
  int i;
  struct timeval to;
  int lines;

  if (ignore_user)
	lines=24;
  else
	lines=26;
  for (;;) {
	printf("\033[2J\033[H");
	for (i=1;i<=lines;i++) 
	  printf("\033[%d;1H%d", i, i);
	  
	printf("\033[7;20HYour Terminal must have %d lines.", lines);
	printf("\033[9;22HAdjuste your screen properly,");
	printf("\033[11;26HAnd hit any key.");
	fflush(stdout);
	FD_ZERO(&fdset);
	FD_SET(0, &fdset);
	to.tv_sec = 2;
	to.tv_usec = 0;
	if (select(FD_SETSIZE, &fdset, NULL, NULL, &to) > 0)
	  break;
  }
  /* Clear screen */
  printf("\033[2J\033[H");

  if (ignore_user) {
	fflush(stdout);
	return;
  }

  /* Restrict scrolling */
  printf("\033[1;24r");

  /* DRAW STATUS LINE */
  /* Draw Time Line */
  printf("\033[26;1HMar 13 12:11:02 CST 1996");

  /* Draw Output Speed */
  printf("\033[26;28H\033[1m+ 1.0\033[0m");

  /* Draw Client buffer */
  printf("\033[26;34H\033[7m                                               \033[0m"); /* 46 */

  /* Go back to main window */
  printf("\033[H");

  /* Set alternate character set */
  printf("\033)0");
  fflush(stdout);

  return;
}


static FILE *sfp=NULL, *cfp=NULL;
static char dumpbase[1024];

dump_connection(input_filename, con_id)
char *input_filename;
int con_id;
{
  char ebuf[PCAP_ERRBUF_SIZE];
  pcap_t *pcp;
  struct conn_list_struct *clp;
  int i;
  extern void summarize_handler();

  printf("Enter basename for output files (server side will append '.srv', client '.cli'\n");
  printf(":");
  fflush(stdout);
  gets(dumpbase);

  if ((pcp = pcap_open_offline(input_filename, ebuf)) == NULL) {
    fprintf(stderr, "Can't open input file: %s\n", ebuf);
    exit(1);
  }

  if (pcap_dispatch(pcp, 0, summarize_handler, NULL) < 0) {
    pcap_perror(pcp, "Can't read from input log");
    pcap_close(pcp);
    exit(1);
  }

  pcap_close(pcp);

  /* Should be summarized, figure out which connection the user mentioned */
  for (i=0, clp = ConnectionList; (i < con_id && clp); i++) clp=clp->next;
  if (clp==NULL) {
    fprintf(stderr, "Connection ID out of range for input file.\n");
    exit(1);
  }

  src = clp->src;
  dst = clp->dst;
  src_port = clp->src_port;
  dst_port = clp->dst_port;

  if ((pcp = pcap_open_offline(input_filename, ebuf)) == NULL) {
    fprintf(stderr, "Can't open input file: %s\n", ebuf);
    exit(1);
  }

  if (pcap_dispatch(pcp, 0, dump_handler, NULL) < 0) {
    pcap_perror(pcp, "Can't read from input log");
    pcap_close(pcp);
    exit(1);
  }

  pcap_close(pcp);
  fclose(sfp);
  fclose(cfp);
  printf("NOTE: You may want to run:\n");
  printf("tr -d '\\015' <basename.srv     - and/or -\n");
  printf("tr '\\015' '\\012' <basename.cli\n");
  
  return;
}

static void
dump_handler(user_data, pkth, p)
u_char *user_data;
struct pcap_pkthdr *pkth;
u_char *p;
{
  struct ether_header *ep;
  struct ip *ip;
  struct tcphdr *tcph;
  u_char *abuf, *abuf2;
  int flags, length;
  char inchar;
  char outbuf[64];
  char buf[1024], buf2[1024];

  if (sfp==NULL) {
	struct in_addr in;

	sprintf(buf, "%s.srv", dumpbase);
	if ((sfp = fopen(buf, "w+")) == NULL) {
	  perror("Can't open server file");
	  exit(1);
	}
	sprintf(buf, "%s.cli", dumpbase);
	if ((cfp = fopen(buf, "w+")) == NULL) {
	  perror("Can't open client file");
	  exit(1);
	}
	
	in.s_addr = src;
	sprintf(buf, "Server side: %s (%d) -> ", inet_ntoa(in), src_port);
	in.s_addr = dst;
	sprintf(buf2, "%s (%d)\n", inet_ntoa(in), dst_port);
	strcat(buf, buf2);
	fputs(buf, sfp);
	in.s_addr = src;
	sprintf(buf, "Client side: %s (%d) -> ", inet_ntoa(in), src_port);
	in.s_addr = dst;
	sprintf(buf2, "%s (%d)\n", inet_ntoa(in), dst_port);
	strcat(buf, buf2);
	fputs(buf, cfp);
  }

  ep = (struct ether_header *)p;
  p += sizeof(struct ether_header);

  if (ntohs(ep->ether_type) != ETHERTYPE_IP)
    return; /* Skip ARPs and other nasties */

  ip = (struct ip *)p;

  if ((int)ip & (sizeof(long)-1)) {
    /* needs to be aligned */
    abuf = (u_char *)alloca(pkth->caplen);

    bcopy((char *)ip, (char *)abuf, MIN(pkth->len, pkth->caplen));
    ip = (struct ip *)abuf;
  }
  p += ip->ip_hl * 4;

  if ((ntohs(ip->ip_off) & 0x1fff) ||  (ip->ip_p!=IPPROTO_TCP)) 
    return; /* Skip UDP or non-0 offsetted packets */

  tcph = (struct tcphdr *)p;
  if ((int)tcph & (sizeof(long)-1)) {
    /* needs to be aligned */
    abuf2 = (u_char *)alloca(pkth->caplen);

    bcopy((char *)tcph, (char *)abuf2, MIN(pkth->len, pkth->caplen));
    tcph = (struct tcphdr *)abuf2;
  }

  length = htons(ip->ip_len) - (ip->ip_hl * 4);
  length -= (tcph->th_off * 4); /* Subtract TCP header length */

  p += tcph->th_off * 4;

  /* 
   * Okay, now figure out where and how we should display the packet 
   * The following could be confusing. We're looking for a packet from
   * the destination of a session, because it contains the data the server
   * sent to the user. This misses passwords, but we can deal with this
   * later.
   */
  if ((ntohl(ip->ip_src.s_addr) == src && ntohs(tcph->th_sport) == src_port &&
       (ntohl(ip->ip_dst.s_addr) == dst && ntohs(tcph->th_dport) == dst_port))
      ||
      (ntohl(ip->ip_src.s_addr) == dst && ntohs(tcph->th_sport) == dst_port &&
       ntohl(ip->ip_dst.s_addr) == src && ntohs(tcph->th_dport) == src_port)) {

	  /* Display the packet */
	  if (ntohl(ip->ip_src.s_addr) == src && 
		  ntohs(tcph->th_sport) == src_port &&
		  ntohl(ip->ip_dst.s_addr) == dst && 
		  ntohs(tcph->th_dport) == dst_port) {
		/* Client to server */
		fwrite(p, 1, length, cfp);
	  } else {
		/* Server to client */
		fwrite(p, 1, length, sfp);
	  }
  }
  return;
}
