/**
 ** Copyright (c) 1996  En Garde Systems, Inc. (EGS) 
 **
 ** Users and possessors of this source code are hereby granted a
 ** nonexclusive, royalty-free copyright and design patent license to
 ** use this code in individual software provided that the above copyright
 ** notice and this paragraph are included in their entirety.  License is
 ** not granted for commercial resale, in whole or in part, without prior
 ** written permission from EGS.  This source is provided "AS IS"
 ** without express or implied warranty of any kind.
 **
 ** In the case of superceding copyright notices, only the modifications to
 ** the original software are covered by this notice.
 **
 ** For further information contact:
 **     E-Mail:     info@EnGarde.com
 **
 **     Surface Mail:   ATTN: PICS Research
 **             En Garde Systems, Inc.
 **             2101 White Cloud NE
 **             Albuquerque, NM 87112
 **             (505) 275-8655
 **/

/*
 * The purpose of this code is to read stdin, copy it to the output file
 * and make comments on it "New connection, End connection, UDP stuff...
 */

#include <stdio.h>
#include <syslog.h>
#include <alloca.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <netdb.h>
#include <netinet/in_systm.h>
#include <netinet/in.h>
#include <net/if.h>
#include <netinet/if_ether.h>
#include <netinet/ip.h>
#include <netinet/ip_icmp.h>
#include <netinet/tcp.h>
#include <netinet/udp.h>

#include <sgtty.h>

#include "pcap.h"

#define COMMENT_LOG_FACILITY LOG_DAEMON
#define COMMENT_LOG_LEVEL LOG_NOTICE

void comment_input();
static char *print_tcp_header();
static char *print_udp_header();
static char *print_icmp_header();
static char *print_timestamp();
char *get_hostname();
char *get_portname();

static pcap_dumper_t *pcap_output=NULL;

void
parse_stdin(output_filename)
char *output_filename;
{
  pcap_t *pcap_input;
  char errbuf[PCAP_ERRBUF_SIZE];

  if ((pcap_input = pcap_open_offline("-", errbuf)) == NULL) {
	fprintf(stderr, "Can't open stdin data: %s\n", errbuf);
	exit(1);
  }

  if (output_filename == NULL)
	pcap_output=NULL;
  else
	if ((pcap_output = pcap_dump_open(pcap_input, output_filename)) == NULL) {
	  pcap_perror(pcap_input, "Can't open output file");
	  pcap_close(pcap_input);
	  exit(1);
	}

  openlog("logit", LOG_PID | LOG_NOWAIT, COMMENT_LOG_FACILITY);

  if (pcap_dispatch(pcap_input, 0, comment_input, (u_char *)pcap_output) < 0) {
	pcap_perror(pcap_input, "Can't dispatch packets");
	pcap_close(pcap_input);
	exit(1);
  }

  pcap_close(pcap_input);
  if (pcap_output!=NULL)
	pcap_dump_close(pcap_output);

  closelog();

  return;
}

#ifndef MIN
#define MIN(a,b) (((a) < (b)) ? a : b)
#endif

void
comment_input(user_data, pkth, p)
u_char *user_data;
struct pcap_pkthdr *pkth;
u_char *p;
{

  struct ether_header *ep;
  struct ip *ip;
  struct tcphdr *tcph;
  struct udphdr *udph;
  struct icmp *icmph;
  u_char *abuf, *abuf2;
  char   logbuf[1024];

  if (user_data!=NULL)
	pcap_dump(user_data, pkth, p); /* Save a copy */

  ep = (struct ether_header *)p;
  p += sizeof(struct ether_header);

  if (ntohs(ep->ether_type) != ETHERTYPE_IP) 
	return; /* SKIP ARPs and other nasties */

  ip = (struct ip *)p;

  if ((int)ip & (sizeof(long)-1)) {
	/* Needs to be aligned */
	abuf = (u_char *)alloca(pkth->caplen);

	bcopy((char *)ip, (char *)abuf, MIN(pkth->len, pkth->caplen));
	ip = (struct ip *)abuf;
  }

  p += ip->ip_hl * 4;

  if (ntohs(ip->ip_off) & 0x1fff) 
	return; /* No frags XXX-used against a firewall, and are logged there */

  logbuf[0]=0;
  switch(ip->ip_p) {
	case IPPROTO_UDP:
	  udph = (struct udphdr *)p;
	  if ((int)udph & (sizeof(long)-1)) {
		/* Needs to be aligned */
		abuf2 = (u_char *)alloca(pkth->caplen);

		bcopy((char *)udph, (char *)abuf2, MIN(pkth->len, pkth->caplen));
		udph = (struct udphdr *)abuf2;
	  }
	  sprintf(logbuf,"%s\n", print_udp_header(pkth, ip, udph));
	  break;
	case IPPROTO_TCP:
	  tcph = (struct tcphdr *)p;
	  if ((int)tcph & (sizeof(long)-1)) {
		/* needs to be aligned */
		abuf2 = (u_char *)alloca(pkth->caplen);

		bcopy((char *)tcph, (char *)abuf2, MIN(pkth->len, pkth->caplen));
		tcph = (struct tcphdr *)abuf2;
	  }

	  if (tcph->th_flags == TH_SYN) 
	    sprintf(logbuf,"%sNew connection\n", print_tcp_header(pkth, ip, tcph));
	  break;
	case IPPROTO_ICMP:
	  icmph = (struct icmp *)p;
	  if ((int)icmph & (sizeof(long)-1)) {
		/* needs to be aligned */
		abuf2 = (u_char *)alloca(pkth->caplen);

		bcopy((char *)icmph, (char *)abuf2, MIN(pkth->len, pkth->caplen));
		icmph = (struct icmp *)abuf2;
	  }
	  switch (icmph->icmp_type) {
		case ICMP_SOURCEQUENCH:
 		  sprintf(logbuf,"%sSource Quench\n", print_icmp_header(pkth, ip, icmph));
		  break;
		case ICMP_REDIRECT:
 		  sprintf(logbuf,"%sRedirect\n", print_icmp_header(pkth, ip, icmph));
		  break;
		case ICMP_ECHO:
 		  sprintf(logbuf,"%sEcho Request\n", print_icmp_header(pkth, ip, icmph));
		  break;
		case ICMP_TSTAMP:
 		  sprintf(logbuf,"%sTime Stamp\n", print_icmp_header(pkth, ip, icmph));
		  break;
		default:
		  break;
	  }
	  break; /* XXX-ignore ICMP? */
	default:
	  break;
  }
  if (strlen(logbuf) > 0) {
	syslog(COMMENT_LOG_LEVEL, logbuf);
	printf("Logging: %s", logbuf);
  }
  return;
}

static char *
print_tcp_header(pkth, ip, tcph)
struct pcap_pkthdr *pkth;
struct ip *ip;
struct tcphdr *tcph; /* Followed by data */
{
  static char head_buf[1024];

  sprintf(head_buf, "%s[TCP] %s (%s) -> %s (%s) : ",
  	print_timestamp(pkth),
  	get_hostname(ntohl(ip->ip_src.s_addr)),
	get_portname(ntohs(tcph->th_sport), "tcp"),
	get_hostname(ntohl(ip->ip_dst.s_addr)),
	get_portname(ntohs(tcph->th_dport), "tcp"));
  return(head_buf);
}

static char *
print_udp_header(pkth, ip, udph)
struct pcap_pkthdr *pkth;
struct ip *ip;
struct udphdr *udph; /* Followed by data */
{
  static char head_buf[1024];

  sprintf(head_buf, "%s[UDP] %s (%s) -> %s (%s) : ",
  		 print_timestamp(pkth),
  		 get_hostname(ntohl(ip->ip_src.s_addr)),
		 get_portname(ntohs(udph->uh_sport), "udp"),
		 get_hostname(ntohl(ip->ip_dst.s_addr)),
		 get_portname(ntohs(udph->uh_dport), "udp"));
  return(head_buf);
}

static char *
print_icmp_header(pkth, ip, icmph)
struct pcap_pkthdr *pkth;
struct ip *ip;
struct icmp *icmph; /* Followed by data */
{
  static char head_buf[1024];

  sprintf(head_buf, "%s[ICMP] %s -> %s : ",
		print_timestamp(pkth),
   		get_hostname(ntohl(ip->ip_src.s_addr)),
		get_hostname(ntohl(ip->ip_dst.s_addr)));
  return(head_buf);
}

static char *
print_timestamp(pkth)
struct pcap_pkthdr *pkth;
{
  struct tm *tm;
  static char timebuf[41];

  tm = localtime((time_t *)&pkth->ts.tv_sec);
  strftime(timebuf, 39, "%D %T ", tm);

  return(timebuf);
}

/* The following can return up to two hostnames before data is overwritten */
char *
get_hostname(ipaddr)
u_long ipaddr;
{
  static char hostbuf[2][40];
  static int whichhost=0;
  struct hostent *he;
  struct in_addr in;

  if ((he = gethostbyaddr((char *)&ipaddr, 4, AF_INET)) != NULL) {
	strncpy(hostbuf[whichhost], he->h_name, 39);
  } else {
	in.s_addr = ipaddr;
	strncpy(hostbuf[whichhost], inet_ntoa(in), 39);
  }
  whichhost=1-whichhost;

  return(hostbuf[1-whichhost]);
}

char *
get_portname(port, protocol)
u_short port;
char *protocol;
{
  static char portname[2][40];
  static int whichport = 0;
  struct servent *se;
  
  if ((se = getservbyport(port, protocol)) != NULL)
	strncpy(portname[whichport], se->s_name, 39);
  else
	sprintf(portname[whichport], "%d", port);
  whichport = 1-whichport;

  return(portname[1-whichport]);
}

void
cleanup_output()
{
  extern char init_user;
  extern struct sgttyb init_mode;
  if (init_user) {
    ioctl(0,TIOCSETP, &init_mode);
	printf("\033[1;r\n");
  }
  if (pcap_output!=NULL)
	pcap_dump_close(pcap_output);
  fprintf(stderr,"Parse: clean exit\n");
  exit(0);
}
