/*
    ParaGUI - crossplatform widgetset
    Copyright (C) 2000,2001  Alexander Pipelka

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Alexander Pipelka
    pipelka@teleweb.at

    Last Update:      $Author: pipelka $
    Update Date:      $Date: 2001/06/12 11:07:45 $
    Source File:      $Source: /usr/local/CVSROOT/linux/paragui/themes/themeloader.cpp,v $
    CVS/RCS Revision: $Revision: 1.12.2.19 $
    Status:           $State: Exp $
*/

#include "xmlparse.h"
#include "theme_priv.h"
#include <stdio.h>
#include <fstream.h>
#include <sys/stat.h>
#ifndef _WIN32
#include <unistd.h>
#endif

#include "xmltchar.h"
#include "unzip.h"

#include "paragui.h"
#include <string.h>

typedef struct _PARSE_INFO {
	int depth;
	int mode;
	THEME_THEME* theme;
	std::string str_currentWidget;
	std::string str_currentObject;
	THEME_WIDGET* p_currentWidget;
	THEME_OBJECT* p_currentObject;
	const char* path;
	std::string themename;
	unzFile zip_file;
	int xml_file_size;
} PARSE_INFO;

#define THEMEMODE_NONE			0
#define THEMEMODE_THEME			1
#define THEMEMODE_WIDGET		2
#define THEMEMODE_OBJECT		3

#define BUFFSIZE	512

char buff[BUFFSIZE];

char *ReadZippedThemeXML( const char *xmltheme, PARSE_INFO *info );
char *LoadFromZip( PARSE_INFO *info, const char *lpszSourceFile );

char* mkFileName(PARSE_INFO* info, const char* file) {
	static char filename[256];
	
	if(info->path) {
		strcpy(filename, info->path);
		strcat(filename, PG_FILE_SEPARATOR);
		strcat(filename, file);
	}
	else {
		strcpy(filename, file);
	}
	
	return filename;
}

void splitColor(SDL_Color*c, Uint32 v) {

#if SDL_BYTEORDER == SDL_LIL_ENDIAN
	c->r = (v >> 16) & 0xFF;
	c->g = (v >> 8) & 0xFF;
	c->b = v & 0xFF;
#else
	c->r = (v >> 8) & 0xFF;
	c->g = (v >> 16) & 0xFF;
	c->b = (v >> 24) & 0xFF;
#endif

}

void parseGlobProps(PARSE_INFO* info, const XML_Char* name, const XML_Char** atts) {

	if(strcmp(name, "theme") == 0) {
		// create new theme template
		info->theme = new THEME_THEME;

#ifdef DEBUG
		cout << "created new theme ..." << endl;
#endif
	}
	else {
		cerr << "UNKNOWN PROP: " << name << endl;
	}
}

void parseThemeProps(PARSE_INFO* info, const XML_Char* prop, const XML_Char** atts) {
	const XML_Char* val = atts[1];
	int i=0;
	
	if(tcscmp(T(prop), T("title")) == 0) {
		info->theme->title = val;

#ifdef DEBUG
		cout << "Title: " << val << endl;
#endif
	}
	else if (tcscmp(T(prop), T("description")) == 0) {
		info->theme->description = val;

#ifdef DEBUG
		cout << "Description: " << val << endl;
#endif
	}
	else if (tcscmp(T(prop), T("author")) == 0) {
		info->theme->author = val;

#ifdef DEBUG
		cout << "Author: " << val << endl;
#endif
	}
	else if (tcscmp(T(prop), T("email")) == 0) {
		info->theme->email = val;

#ifdef DEBUG
		cout << "Email: " << val << endl;
#endif
	}
	else if (tcscmp(T(prop), T("widget")) == 0) {

#ifdef DEBUG
		cout << "creating new widget ..." << endl;
#endif

		THEME_WIDGET* widget = new THEME_WIDGET;
		widget->type = "";
		info->p_currentWidget = widget;
		info->theme->widget.insert(info->theme->widget.end(), widget);
		info->mode = THEMEMODE_WIDGET;
	}
	else if(tcscmp(T(prop), T("font")) == 0) {
		THEME_FONT* font = new THEME_FONT;
		font->size = 12;
		
		for(i=0; atts[i]; i += 2) {
			if(tcscmp(T(atts[i]), T("name")) == 0) {
				font->name = atts[i+1];
			}
			else if(tcscmp(T(atts[i]), T("value")) == 0) {
				font->value = atts[i+1];
			}
			else if(tcscmp(T(atts[i]), T("size")) == 0) {
				font->size = atoi(atts[i+1]);
			}
			else {
				cerr << "UNKNOWN FONT ATTRIBUTE: " << atts[i] << endl;
			}
		}

#ifdef DEBUG
		cout << "Font: " << font->name.c_str() << "=" << font->value.c_str() << endl;
		cout << "Loading defaultfont " << mkFileName(info, font->value.c_str()) << "... " << endl;
#endif

		// load the font file

 		// O.K., MacOS version of SDL_ttf / FreeType library is not compatible
 		// with others about loading font ---- on MacOS, we need to use system
 		// font and not the font files here and there..! So we should not use
 		// mkFileName function here.
 		// Hm...
 		// Masahiro Minami<elsur@aaa.letter.co.jp>
 		// 01/05/07

#ifndef macintosh
		if( info->zip_file != NULL ) {
			char font_filename[255];
			strcpy( font_filename, LoadFromZip( info, font->value.c_str() ) );
			if( !font_filename[0] ) {
				strcpy( font_filename, mkFileName( info, font->value.c_str() ) );	
			}

			font->font = TTF_OpenFont(
				font_filename,
				font->size
				);
			if( !font->font ){
				font->font = TTF_OpenFont(
					mkFileName(info, font->value.c_str()),
					font->size
					);
			} else {
				unlink( font_filename );
			}

		} else {
			font->font = TTF_OpenFont(
				mkFileName(info, font->value.c_str()),
				font->size
				);
		}
#else
 		font->font = TTF_OpenFont( font->value.c_str(), font->size );
#endif	// macintosh
		
#ifdef DEBUG
		cout << (font->font ? "ok." : "failed!") << endl;
#endif

		font = info->theme->AddToCache(font);
		info->theme->defaultfont = font;
	}
	else {
		cerr << "UNKNOWN THEME ATTRIBUTE: " << prop << endl;
	}

}

void parseWidgetProps(PARSE_INFO* info, const XML_Char* prop, const XML_Char** atts) {
	std::string val=atts[1];

	if(tcscmp(T(prop), T("type")) == 0) {
		info->p_currentWidget->type = val;

#ifdef DEBUG
		cout << "Type: " << val.c_str() << endl;
#endif
	}
	else if(tcscmp(T(prop), T("object")) == 0) {

#ifdef DEBUG
		cout << "creating new object ..." << endl;
#endif

		THEME_OBJECT* object = new THEME_OBJECT;
		object->type = "";
		object->name = "";
		info->p_currentObject = object;
		info->p_currentWidget->object.insert(info->p_currentWidget->object.end(), object);
		info->mode = THEMEMODE_OBJECT;
	}
	else {
		cerr << "UNKNOWN WIDGET ATTRIBUTE: " << prop << endl;
	}
}

void parseObjectProps(PARSE_INFO* info, const XML_Char* prop, const XML_Char** atts) {
	std::string val=atts[1];
	int i;
	THEME_OBJECT* object = info->p_currentObject;

	//
	//	TYPE
	//
		
	if(tcscmp(T(prop), T("type")) == 0) {
		object->type = val;

#ifdef DEBUG
		cout << "Type: " << val.c_str() << endl;
#endif
	}

	//
	//	NAME
	//
		
	else if(tcscmp(T(prop), T("name")) == 0) {
		object->name = val;

#ifdef DEBUG
		cout << "Name: " << val.c_str() << endl;
#endif
	}

	//
	//	FILENAME
	//
		
	else if(tcscmp(T(prop), T("filename")) == 0) {
		THEME_FILENAME* filename = new THEME_FILENAME;
		filename->hasColorKey = false;

		for(i=0; atts[i]; i += 2) {
			if(tcscmp(T(atts[i]), T("name")) == 0) {
				filename->name = atts[i+1];
			}
			else if(tcscmp(T(atts[i]), T("value")) == 0) {
				filename->value = atts[i+1];
			}
			else if(tcscmp(T(atts[i]), T("colorkey")) == 0) {
				sscanf(atts[i+1], "0x%08x", (unsigned int*)(&filename->colorkey));
				filename->hasColorKey = true;
			}
			else {
				cerr << "UNKNOWN FILENAME ATTRIBUTE: " << atts[i] << endl;
			}
		}

		THEME_FILENAME* cache = info->theme->FindInSurfaceCache(filename->value.c_str());
		if(cache != NULL) {
			filename->surface = cache->surface;
			object->filename.insert(info->p_currentObject->filename.end(), filename);
			return;
		}

#ifdef DEBUG
		cout << "Filename: " << filename->name.c_str() << "=" << filename->value.c_str() << endl;
		cout << "Loading surface " << mkFileName(info, filename->value.c_str()) << "... ";
#endif
		
		// load the image file
		if( info->zip_file != NULL ) {
			char surface_filename[255];
			strcpy( surface_filename, LoadFromZip( info, filename->value.c_str() ) );
			if( !surface_filename[0] ) {
				strcpy( surface_filename, mkFileName( info, filename->value.c_str() ) );	
			}
			filename->surface = PG_LoadImage(
				surface_filename
				);
			if( !filename->surface ) {
				filename->surface = PG_LoadImage(
					mkFileName(info, filename->value.c_str())
					);
			} else {
				unlink( surface_filename );
			}
		} else {
			filename->surface = PG_LoadImage(
				mkFileName(info, filename->value.c_str())
				);
		}
		
		if(filename->surface == NULL) {
			delete filename;
			return;
		}

		// set the colorkey (if there is any)
		if(filename->hasColorKey && filename->surface) {
			SDL_Color c;
			splitColor(&c, filename->colorkey);
			Uint32 key = SDL_MapRGB(filename->surface->format, c.r, c.g, c.b);
			SDL_SetColorKey(filename->surface, SDL_SRCCOLORKEY, key);
		}

#ifdef DEBUG
		cout << (filename->surface ? "ok." : "failed!") << endl;
#endif

		info->theme->AddToCache(filename);		
		object->filename.insert(info->p_currentObject->filename.end(), filename);
	}

	//
	//	FONT
	//

	else if(tcscmp(T(prop), T("font")) == 0) {
		THEME_FONT* font = new THEME_FONT;
		font->size = 12;
		
		for(i=0; atts[i]; i += 2) {
			if(tcscmp(T(atts[i]), T("name")) == 0) {
				font->name = atts[i+1];
			}
			else if(tcscmp(T(atts[i]), T("value")) == 0) {
				font->value = atts[i+1];
			}
			else if(tcscmp(T(atts[i]), T("size")) == 0) {
				font->size = atoi(atts[i+1]);
			}
			else {
				cerr << "UNKNOWN FONT ATTRIBUTE: " << atts[i] << endl;
			}
		}

#ifdef DEBUG
		cout << "Font: " << font->name.c_str() << "=" << font->value.c_str() << endl;
		cout << "Loading font " << mkFileName(info, font->value.c_str()) << "... ";
#endif
	
		// load the font file
#ifndef macintosh
		if( info->zip_file != NULL ) {
			char font_filename[255];
			strcpy( font_filename, LoadFromZip( info, font->value.c_str() ) );
			if( !font_filename[0] ) {
				strcpy( font_filename, mkFileName( info, font->value.c_str() ) );	
			}
			font->font = TTF_OpenFont(
				font_filename,
				font->size
				);
			if( !font->font ) {
				font->font = TTF_OpenFont(
					mkFileName(info, font->value.c_str()),
					font->size
					);
			} else {
				unlink( font_filename );
			}

		} else {
			font->font = TTF_OpenFont(
				mkFileName(info, font->value.c_str()),
				font->size
				);
		}
#else
 		font->font = TTF_OpenFont( font->value.c_str(), font->size );
#endif
		
#ifdef DEBUG
		cout << (font->font ? "ok." : "failed!") << endl;
#endif

		font = info->theme->AddToCache(font);
		object->font = font;
	}
		
	//
	//	PROPERTY
	//
		
	else if(tcscmp(T(prop), T("property")) == 0) {
		THEME_PROPERTY* property = new THEME_PROPERTY;
		
		for(i=0; atts[i]; i += 2) {
			if(tcscmp(T(atts[i]), T("name")) == 0) {
				property->name = atts[i+1];
			}
			else if(tcscmp(T(atts[i]), T("value")) == 0) {
				if(tcscmp(T(atts[i+1]), T("TILE")) == 0) {
					property->value = BKMODE_TILE;
				}
				else if(tcscmp(T(atts[i+1]), T("STRETCH")) == 0) {
					property->value = BKMODE_STRETCH;
				}
				else {
					property->value = atoi(atts[i+1]);
				}
			}
			else {
				cerr << "UNKNOWN PROPERTY ATTRIBUTE: " << atts[i] << endl;
			}
		}

#ifdef DEBUG
		cout << "Property: " << property->name.c_str() << "=" << property->value << endl;
#endif

		object->property.insert(object->property.end(), property);
	}
	
	//
	//	COLOR
	//
		
	else if(tcscmp(T(prop), T("color")) == 0) {
		THEME_PROPERTY* property = new THEME_PROPERTY;
		
		for(i=0; atts[i]; i += 2) {
			if(tcscmp(T(atts[i]), T("name")) == 0) {
				property->name = atts[i+1];
			}
			else if(tcscmp(T(atts[i]), T("value")) == 0) {
				sscanf(atts[i+1], "0x%08x", (unsigned int*)(&property->value));
			}
			else {
				cerr << "UNKNOWN COLOR ATTRIBUTE: " << atts[i] << endl;
			}
		}

#ifdef DEBUG
		cout << "Color: " << property->name.c_str() << "=" << property->value << endl;
#endif

		object->property.insert(info->p_currentObject->property.end(), property);
	}

	//
	//	GRADIENT
	//
		
	else if(tcscmp(T(prop), T("gradient")) == 0) {
		THEME_GRADIENT* gradient = new THEME_GRADIENT;
		Uint32 c;
		std::string val;

		for(i=0; atts[i]; i += 2) {

			if(tcscmp(T(atts[i]), T("name")) == 0) {
				gradient->name = atts[i+1];

#ifdef DEBUG
				cout << "Gradient: " << gradient->name.c_str() << endl;
#endif
			}
			else if(tcscmp(T(atts[i]), T("color0")) == 0) {
				val = atts[i+1];
				sscanf(val.c_str(), "0x%08x", &c);
				gradient->color[0].val = c;
				splitColor(&(gradient->color[0].c), c);

#ifdef DEBUG
				cout << "Color0: " << atts[i+1] << " = " << c << endl;
#endif
			}
			else if(tcscmp(T(atts[i]), T("color1")) == 0) {
				val = atts[i+1];
				sscanf(val.c_str(), "0x%08x", &c);
				gradient->color[1].val = c;
				splitColor(&(gradient->color[1].c), c);

#ifdef DEBUG
				cout << "Color1: " << atts[i+1] << " = " << c << endl;
#endif
			}
			else if(tcscmp(T(atts[i]), T("color2")) == 0) {
				val = atts[i+1];
				sscanf(val.c_str(), "0x%08x", &c);
				gradient->color[2].val = c;
				splitColor(&(gradient->color[2].c), c);

#ifdef DEBUG
				cout << "Color2: " << atts[i+1] << " = " << c << endl;
#endif
			}
			else if(tcscmp(T(atts[i]), T("color3")) == 0) {
				val = atts[i+1];
				sscanf(val.c_str(), "0x%08x", &c);
				gradient->color[3].val = c;
				splitColor(&(gradient->color[3].c), c);

#ifdef DEBUG
				cout << "Color3: " << atts[i+1] << " = " << c << endl;
#endif
			}
			else {
				cerr << "UNKNOWN PROPERTY ATTRIBUTE: " << atts[i] << endl;
			}
		}
		
		for(i=0; i<4; i++) {
			gradient->gradient.colors[i] = gradient->color[i].c;
		}
		
		object->gradient.insert(info->p_currentObject->gradient.end(), gradient);
	}

	//
	//	STRING
	//
	else if(tcscmp(T(prop), T("string")) == 0) {
		THEME_STRING* str = new THEME_STRING;

		for(i=0; atts[i]; i += 2) {

			if(tcscmp(T(atts[i]), T("name")) == 0) {
				str->name = atts[i+1];
			}
			else if(tcscmp(T(atts[i]), T("value")) == 0) {
				str->value = atts[i+1];
			}
			else {
				cerr << "UNKNOWN STRING ATTRIBUTE: " << atts[i] << endl;
			}
		}

		object->strings.insert(info->p_currentObject->strings.end(), str);
	}

	//
	//	UNKNOWN OBJECT ATTRIBUTE
	//
		
	else {
		cerr << "UNKNOWN OBJECT ATTRIBUTE: " << prop << endl;
	}
}


void handlerStart(void* userData, const XML_Char *name, const XML_Char** atts) {
	PARSE_INFO* info = (PARSE_INFO*)userData;
	
	info->depth++;

	switch(info->mode) {
		case THEMEMODE_NONE:
			parseGlobProps(info, name, atts);
			info->mode = THEMEMODE_THEME;
			break;
						
		case THEMEMODE_THEME:
			parseThemeProps(info, name, atts);
			break;
		
		case THEMEMODE_WIDGET:
			parseWidgetProps(info, name, atts);
			break;
			
		case THEMEMODE_OBJECT:
			parseObjectProps(info, name, atts);
			break;		
	}
}

void handlerEnd(void* userData, const XML_Char* name) {
	PARSE_INFO* info = (PARSE_INFO*)userData;

	if(tcscmp(T(name), T("object")) == 0) {
		info->mode = THEMEMODE_WIDGET;
		
#ifdef DEBUG
		cout << "object finished!" << endl;
#endif
	}
	else if(tcscmp(T(name), T("widget")) == 0) {
		info->mode = THEMEMODE_THEME;
		
#ifdef DEBUG
		cout << "widget finished!" << endl;
#endif
	}
	else if(tcscmp(T(name), T("theme")) == 0) {
		info->mode = THEMEMODE_NONE;
		
#ifdef DEBUG
		cout << "theme finished!" << endl;
#endif
	}

	info->depth--;
}

// simple file check

bool FileExists(const char *filename) {
	struct stat st;
	int ret;
	ret = stat(filename, &st);
	if(ret == -1) return false;

#ifdef WIN32
	if(st.st_mode & _S_IFREG) {
#else
	if(S_ISREG(st.st_mode)) {
#endif
		return true;
	} else {
		return false;
	}  
}

/**
 * Given a search path and the name of a file that contains the theme definition, it loads
 * the Theme into ParaGUI's system
 *
 * @param path The search path to locate the theme file
 * @param xmltheme The file name for the theme. The extension ".theme" is assumed.
 */


PG_Theme* PARAGUI_LoadTheme(const char* path, const char* xmltheme) {
	std::string filename = path;
	// create new parse info
	PARSE_INFO* info = new PARSE_INFO;
	bool is_zip = false;

	// init parseinfo
	info->path = path;
	info->theme = NULL;
	info->p_currentWidget = NULL;
	info->p_currentObject = NULL;
	info->depth = 0;
	info->mode = THEMEMODE_NONE;
	info->zip_file = 0;
	
  	filename += PG_FILE_SEPARATOR;
	filename += xmltheme;

	// first try to open an uncompressed theme

	filename = path + (std::string)PG_FILE_SEPARATOR + xmltheme + (std::string)THEME_SUFFIX;
	if(!FileExists(filename.c_str())) {

		// try to open a compressed theme file

		filename = path + (std::string)PG_FILE_SEPARATOR + xmltheme + (std::string)".zip";
		if(FileExists(filename.c_str())) {
			is_zip = true;	
		}
		else {
			return NULL;
		}
	}
	
	info->themename = xmltheme;

	// create a parser
	XML_Parser p = XML_ParserCreate(NULL);
	
	// set userdata (parseinfo)
	XML_SetUserData(p, (void*)info);
	XML_SetElementHandler(p, handlerStart, handlerEnd);

	if( !is_zip ) {

		// create an input-stream

#if ! defined( macintosh ) && ! defined(__MWERKS__)
		// CodeWarrior ( pro 4 ) compiler doesn't like this
		// Masahiro Minami<elsur@aaa.letter.co.jp>
		// 01/05/05
		ifstream file(filename.c_str(), ios::in);
#else
		ifstream file(filename.c_str());
#endif
	
		if(!file) {
			XML_ParserFree(p);
			return NULL;
		}

		// do the parsing
		while (file) {
			int done;

			file.read(buff, BUFFSIZE);
			done = file.eof();
		
			if (! XML_Parse(p, buff, file.gcount(), done)) {
				cerr << "Parse error at line " << XML_GetCurrentLineNumber(p) << ":" << endl << XML_ErrorString(XML_GetErrorCode(p)) << endl;
				XML_ParserFree(p);
				return NULL;
			}

			if(done) { 
				break; 
			}
		} 
	} else {
		// do the parsing
		char *buffer = ReadZippedThemeXML( filename.c_str(), info );
		if( buffer != NULL ) {

			if (! XML_Parse(p, buffer, info->xml_file_size, 1)) {
				cerr << "Parse error at line " << XML_GetCurrentLineNumber(p) << ":" << endl << XML_ErrorString(XML_GetErrorCode(p)) << endl;
				XML_ParserFree(p);
				delete [] buffer;
				return NULL;
			}
			delete [] buffer;
			unzClose( info->zip_file );
		} 
	}

		
	// free the parser
	XML_ParserFree(p);
	PG_Theme *theme = info->theme;
	delete info;
	return theme;
}

void PARAGUI_UnloadTheme(PG_Theme* theme) {
	delete ((THEME_THEME*)theme);
}

/*! Loads a theme from a zip file
 *
 *  @param xmltheme filename of theme to read in via zip file.
 *  @param info a PARSE_INFO structure containing data about theme.
 *  @see LoadTheme()
 *
 */
char *ReadZippedThemeXML( const char *filename, PARSE_INFO *info ) {
	char *buffer = NULL;
	char theme_ext[255];
	unz_file_info file_info;

	strcpy( (char *)theme_ext, info->themename.c_str() );
	strcat( (char *)theme_ext, THEME_SUFFIX );
	
	info->zip_file = unzOpen( filename );
	if(!info->zip_file) {
	  cerr << "Cannot open zip file " << filename << endl;
	  exit(1);
	}
	if( unzLocateFile( info->zip_file, theme_ext, 2 ) != UNZ_OK ) {
		unzClose( info->zip_file );
#ifdef DEBUG
		cerr << "Unable to locate: " << theme_ext << " in " << filename << endl;
#endif
		return NULL;
	}
	unzGetCurrentFileInfo( info->zip_file, &file_info, NULL, 0, NULL, 0, NULL, 0 );

	info->xml_file_size = file_info.uncompressed_size;
	
	buffer = new char [ file_info.uncompressed_size ];
	if( buffer == NULL ) {
		unzClose( info->zip_file );
		cerr << "Unable to allocate a buffer for reading in the theme info." << endl << "Needed: " << file_info.uncompressed_size << " bytes of memory." << endl;
	}

	if( unzOpenCurrentFile(info->zip_file) != UNZ_OK ) {
		unzClose( info->zip_file );
		cerr << "Unable to read in theme xml file. Check for a corrupt archive." << endl;
		delete [] buffer;
		return NULL;
	}
	else if (unzReadCurrentFile(info->zip_file, buffer, file_info.uncompressed_size) != (int)file_info.uncompressed_size ) {
		unzClose( info->zip_file );
		cerr << "Error during read from archive. Check for a corrupt archive." << endl;
		delete [] buffer;
		return NULL;
	}
	else if ( unzCloseCurrentFile(info->zip_file)	== UNZ_CRCERROR ) {
		unzClose( info->zip_file );
		cerr << "There was a CRC Error on the data read from the archive. Check for a corrupt archive." << endl;
		delete [] buffer;
		return NULL;
	}

	return buffer;
}

/*! Loads an object from a zip file
 *
 * @param info a PARSE_INFO structure containing data about a theme
 * @param lpszSourcefile filename to load from zip file
 *
 */
char *LoadFromZip( PARSE_INFO *info, const char *lpszSourceFile ) {
	static char filename[255];
	char *p = NULL;
	char buffer[ BUFFSIZE ];
	int bytes; 
	char sep = PG_FILE_SEPARATOR[0];

	// create the file in PARAGUI_TEMPDIR

	strcpy( filename, PARAGUI_TEMPDIR);
	strcat( filename, "/");

	// strip the path

	if( strchr( lpszSourceFile, sep ) != NULL ) {
		p = (char *)&lpszSourceFile[0] + strlen( lpszSourceFile ) - 1;
		while( *--p != sep );
		p++;
		strcat( filename, p );
	}
	else {
		strcat( filename, lpszSourceFile );
	}

	if( unzLocateFile( info->zip_file, lpszSourceFile, 2 ) == UNZ_OK ) {
#ifdef DEBUG
		cout << "Found it in ZIP" << endl;
#endif
		ofstream fp(filename, ofstream::binary);
	        if(!fp) {
		  cerr << "Failed to open file " << filename << " for writing." << endl;
		  exit(1);
		}
		unzOpenCurrentFile( info->zip_file );

		while( !unzeof( info->zip_file ) ) {
			bytes = unzReadCurrentFile( info->zip_file, buffer, BUFFSIZE );
			fp.write(buffer, bytes);
		}

		unzCloseCurrentFile( info->zip_file );
	}

	return( filename );
}
