/*
    ParaGUI - crossplatform widgetset
    Copyright (C) 2000,2001  Alexander Pipelka
 
    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.
 
    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.
 
    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 
    Alexander Pipelka
    pipelka@teleweb.at
 
    Last Update:      $Author: pipelka $
    Update Date:      $Date: 2001/06/28 14:12:49 $
    Source File:      $Source: /usr/local/CVSROOT/linux/paragui/src/pgwidgetlist.cpp,v $
    CVS/RCS Revision: $Revision: 1.1.2.18 $
    Status:           $State: Exp $
*/

#include "pgwidgetlist.h"
#include "pgapplication.h"

PG_WidgetList::PG_WidgetList(PG_Widget* parent, const PG_Rect& r, const char* style) : PG_StaticFrame(parent, r, style) {
	my_widgetCount = 0;
	my_firstWidget = 0;
	my_enableScrollbar = true;
	my_widthScrollbar = 5;

	my_rectScrollbar.SetRect(
		r.my_width - my_widthScrollbar,
		0,
		my_widthScrollbar,
		r.my_height);

	my_objScrollbar = new PG_ScrollBar(this, PG_IDWIDGETLIST_SCROLL, my_rectScrollbar, PG_SB_VERTICAL, style);
	my_objScrollbar->SetRange(0,0);
	my_widthScrollbar = my_objScrollbar->Width();
	my_objScrollbar->MoveWindow(r.my_width - my_widthScrollbar, 0);
	my_objScrollbar->SizeWindow(my_widthScrollbar, r.my_height);

	my_rectList.SetRect(
		my_bordersize,
		my_bordersize,
		r.my_width - my_widthScrollbar - my_bordersize*2,
		r.my_height - my_bordersize*2);

	LoadThemeStyle(style);
}

PG_WidgetList::~PG_WidgetList() {
	DeleteAll();
	//delete my_objScrollbar;
}

void PG_WidgetList::LoadThemeStyle(const char* widgettype) {
	PG_StaticFrame::LoadThemeStyle(widgettype);
	my_objScrollbar->LoadThemeStyle(widgettype);

	my_widthScrollbar = my_objScrollbar->Width();
	my_objScrollbar->MoveWindow(my_width - my_widthScrollbar, 0);
}

void PG_WidgetList::eventBlit(SDL_Surface* srf, const PG_Rect& src, const PG_Rect& dst) {
	PG_Rect r = my_rectList;
	r.my_xpos -= my_bordersize;
	r.my_ypos -= my_bordersize;
	r.my_width += my_bordersize*2;
	r.my_height += my_bordersize*2;

	r.my_xpos += my_xpos;
	r.my_ypos += my_ypos;

	PG_StaticFrame::eventBlit(srf, src, dst);
	DrawBorder(r, my_bordersize);
}

void PG_WidgetList::eventSizeWindow(int w, int h) {

	PG_StaticFrame::eventSizeWindow(w,h);

	my_rectList.SetRect(
		my_bordersize,
		my_bordersize,
		w - my_widthScrollbar - my_bordersize*2,
		h - my_bordersize*2);

	my_rectScrollbar.SetRect(
		w - my_widthScrollbar,
		0,
		my_widthScrollbar,
		h);

	my_objScrollbar->MoveWindow(my_rectScrollbar.my_xpos, my_rectScrollbar.my_ypos);
	my_objScrollbar->SizeWindow(my_rectScrollbar.my_width, my_rectScrollbar.my_height);
	
	CheckScrollBar();
}

bool PG_WidgetList::eventScrollPos(int id, PG_Widget* widget, unsigned long data) {
	if(widget == my_objScrollbar) {
		ScrollToY(data);
		return true;
	}
	return PG_StaticFrame::eventScrollPos(id, widget, data);
}

bool PG_WidgetList::eventScrollTrack(int id, PG_Widget* widget, unsigned long data) {
	if(widget == my_objScrollbar) {
		ScrollToY(data);
		return true;
	}
	return PG_StaticFrame::eventScrollTrack(id, widget, data);
}

void PG_WidgetList::eventShow() {
	CheckScrollBar();
}

/**  */
void PG_WidgetList::AddWidget(PG_Widget* w) {
	
	if(w == NULL) {
		return;
	}

	w->SetVisible(false);
	AddChild(w);

	if(my_widgetList.empty()) {
		my_listheight = w->Height();
	} else {
		my_listheight += w->Height();
	}

	w->MoveWindow(w->my_xpos + my_bordersize, my_listheight - w->my_height + my_bordersize);

	my_widgetList.insert(my_widgetList.end(), w);
	my_widgetCount++;
	
	UpdateScrollBarPos();
	
	if(IsVisible()) {
		CheckScrollBar();
		w->SetVisible(true);
		Update();
	}

	GetChildList()->BringToFront(my_objScrollbar);
}

void PG_WidgetList::DeleteAll() {
	std::vector<PG_Widget*>::iterator list = my_widgetList.begin();
	PG_Widget* w = NULL;

	while(list != my_widgetList.end()) {
		w = *list;
		delete w;
		my_widgetList.erase(list);
		list = my_widgetList.begin();
	}

	RemoveAll();
}

void PG_WidgetList::RemoveAll() {
	my_listheight = 0;
	my_widgetCount = 0;
	my_widgetList.clear();
}

Sint32 PG_WidgetList::ScrollToY(Sint32 position) {
	PG_Rect r;

	if(position > Sint32(my_listheight-Height())) {
		position  = my_listheight-Height();
	}

	if(position < 0)
		position = 0;

	// don't blit recursive, restore background
	//Blit(false, true);

	Sint32 py = my_ypos;
	py -= position;

	for(int i=0; i<my_widgetCount; i++) {
		r = *(my_widgetList[i]);
		r.my_xpos -= my_bordersize;

		if(py < -2000) {
			r.my_ypos = -2000;
		}
		else if (py > 2000) {
			r.my_ypos = 2000;
		}
		else {
			r.my_ypos = py;
		}

		my_widgetList[i]->MoveRect(r.my_xpos + my_bordersize, r.my_ypos + my_bordersize);

		// blit recursive, don't restore background (done before)
		//my_widgetList[i]->Blit(true, false);

		py += my_widgetList[i]->Height();
	}

	my_firstWidget = position;
	//Blit();
	//my_objScrollbar->Blit();

	//Update(false);
	Update();
	return position;
}

PG_Widget* PG_WidgetList::GetWidgetFromPos(Sint32 y) {
	Uint32 dy = 0;
	Uint32 min_dy = 100000000;
	PG_Widget* result = NULL;

	for(int i=0; i<my_widgetCount; i++) {
		dy = abs(0- (my_widgetList[i]->my_ypos - my_ypos));

		if(dy < min_dy) {
			min_dy = dy;
			result = my_widgetList[i];
		}
	}

	return result;
}

bool PG_WidgetList::RemoveWidget(PG_Widget* w) {
	std::vector<PG_Widget*>::iterator wi = my_widgetList.begin();
	PG_Widget* widget = NULL;

	while (wi != my_widgetList.end()) {
		widget = *wi;

		if(widget == w) {

			my_widgetList.erase(wi);
			w->SetVisible(false);
			w->MoveRect(0,-2000);
			my_listheight -= w->Height();
			my_widgetCount--;

			if(my_listheight >= (Uint32)Height()) {
				ScrollToY(my_firstWidget);
			}
			else {
				ScrollToY(0);
			}

			UpdateScrollBarPos();

			if(IsVisible()) {
				CheckScrollBar();
				Update();
			}

			return true;
		};

		wi++;
	}
	return false;
}

bool PG_WidgetList::RemoveWidgetAt(int index) {
	return RemoveWidget(FindWidget(index));
}

bool PG_WidgetList::DeleteWidget(PG_Widget* w) {

	if(!RemoveWidget(w)) {
		return false;
	}
	
	delete w;
	
	return true;	
}

bool PG_WidgetList::DeleteWidgetAt(int index) {
	return DeleteWidget(FindWidget(index));
}

PG_Widget* PG_WidgetList::FindWidget(int index) {

	if((index < 0) || (index >= (int)my_widgetList.size())) {
		return NULL;
	}

	return my_widgetList[index];
}

int PG_WidgetList::GetWidgetCount() {
	return my_widgetCount;
}

void PG_WidgetList::UpdateScrollBarPos() {
	int ls = 0;

	if(my_widgetCount != 0) {

		ls = my_listheight / my_widgetCount;

		if(ls == 0) {
			ls = 1;
		}

		my_objScrollbar->SetLineSize(ls);
		my_objScrollbar->SetRange(0, my_listheight- Height());
		my_objScrollbar->SetPageSize(Height());
		my_objScrollbar->SetPosition(my_objScrollbar->GetPosition());
	}

}

void PG_WidgetList::CheckScrollBar() {

	if(my_listheight > (Uint32)Height()) {
		my_objScrollbar->SetVisible(my_enableScrollbar);
	} else {
		my_objScrollbar->SetVisible(false);
	}

}

void PG_WidgetList::EnableScrollBar(bool enable) {
	my_enableScrollbar = enable;
	CheckScrollBar();

	if(!my_enableScrollbar) {
		my_widthScrollbar = 0;
		SizeWindow(Width(), Height());
	}
}

void PG_WidgetList::ScrollTo(PG_Widget* widget) {
	if(my_widgetCount == 0) {
		return;
	}

	int ypos = widget->y - FindWidget(0)->y;

	ypos = ScrollToY(ypos);
	my_objScrollbar->SetPosition(ypos);
}

void PG_WidgetList::ScrollTo(int index) {
	PG_Widget* w = FindWidget(index);

	if(w == NULL) {
		return;
	}

	ScrollTo(w);
}
