/*
   ParaGUI - crossplatform widgetset
   Copyright (C) 2000,2001  Alexander Pipelka

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

   Alexander Pipelka
   pipelka@teleweb.at

   Last Update:      $Author: pipelka $
   Update Date:      $Date: 2001/06/28 10:58:30 $
   Source File:      $Source: /usr/local/CVSROOT/linux/paragui/src/pgwidget.cpp,v $
   CVS/RCS Revision: $Revision: 1.1.2.58 $
   Status:           $State: Exp $
 */

#include <iostream>
#include "pgwidget.h"
#include "pgapplication.h"


bool PG_Widget::bBulkUpdate = false;
PG_RectList PG_Widget::widgetList;


PG_Widget::PG_Widget(PG_Widget* parent, const PG_Rect& rect) :
PG_DrawObject(rect, false),
my_widgetParent(parent)
{
	InitWidget();
}

PG_Widget::PG_Widget(PG_Widget* parent, const PG_Rect& rect, bool bObjectSurface) :
PG_DrawObject(rect, bObjectSurface),
my_widgetParent(parent)
{
	InitWidget();
}

void PG_Widget::InitWidget() {
	my_mutexProcess = SDL_CreateMutex();
	my_id = -1;
	my_transparency = 0;
	my_font = PG_Application::GetDefaultFont();

	my_childList = NULL;
	my_haveTooltip = false;
	my_fadeSteps = 10;

	AddToWidgetList();

	bSetCaptureOnShow = false;

	SetTextColor(0x00FFFFFF);
	
	if (my_widgetParent != NULL) {
		my_xpos = my_widgetParent->my_xpos + my_xpos;
		my_ypos = my_widgetParent->my_ypos + my_ypos;
		my_widgetParent->AddChild(this);
	}

	my_mouseInside = false;
}

PG_Widget::~PG_Widget() {

	Hide();

	// remove all child widgets
	if(my_childList != NULL) {

		PG_RectList::iterator i = my_childList->begin();
		while(i != my_childList->end()) {
			PG_Widget* w = *i;
			RemoveChild(w);
			delete (w);
			i = my_childList->begin();
		}

		my_childList->clear();
	}
	
	// remove myself from my parent's childlist (if any parent)

	if (GetParent() != NULL) {
		GetParent()->RemoveChild(this);
	}

	RemoveFromWidgetList();

	// unlock process mutex
	SDL_DestroyMutex(my_mutexProcess);

	// remove childlist
	delete my_childList;
	my_childList = NULL;
}

void PG_Widget::RemoveFromWidgetList() {
	widgetList.Remove(this);
}

void PG_Widget::AddToWidgetList() {
	if(!GetParent()) {
		widgetList.Add(this);
	}
}

/** Check if we can accept the event */

bool PG_Widget::AcceptEvent(const SDL_Event * event) {

	if (!IsVisible()) {
		return false;
	}

	switch (event->type) {
		case SDL_MOUSEMOTION:
			if ((event->motion.x < my_rectClip.my_xpos) || (event->motion.x > (my_rectClip.my_xpos + my_rectClip.my_width - 1))) {
				if (my_mouseInside) {
					my_mouseInside = false;
					eventMouseLeave();
				}
				return false;
			}
			if ((event->motion.y < my_rectClip.my_ypos) || (event->motion.y > (my_rectClip.my_ypos + my_rectClip.my_height - 1))) {
				if (my_mouseInside) {
					my_mouseInside = false;
					eventMouseLeave();
				}
				return false;
			}
			if (!my_mouseInside) {
				my_mouseInside = true;
				eventMouseEnter();
				return false;
			}
			break;

		case SDL_MOUSEBUTTONUP:
		case SDL_MOUSEBUTTONDOWN:
			if ((event->button.x < my_rectClip.my_xpos) || (event->button.x > (my_rectClip.my_xpos + my_rectClip.my_width - 1)))
				return false;

			if ((event->button.y < my_rectClip.my_ypos) || (event->button.y > (my_rectClip.my_ypos + my_rectClip.my_height - 1)))
				return false;

			break;
	}

	return true;		// accept the event as default
}


/**  */
void PG_Widget::eventMouseEnter() {
}


/**  */
void PG_Widget::eventMouseLeave() {
	my_mouseInside = false;

	if(my_childList == NULL) {
		return;
	}

	for(PG_RectList::iterator i = my_childList->begin(); i != my_childList->end(); i++) {
		if((*i)->my_mouseInside) {
			(*i)->eventMouseLeave();
		};
	}
}

/**  */
void PG_Widget::eventShow() {
}

/**  */
void PG_Widget::eventHide() {
}

/**  */
PG_Point PG_Widget::ClientToScreen(int sx, int sy) {
	PG_Point p;

	p.x = sx + my_xpos;
	p.y = sy + my_ypos;

	return p;
}

/**  */
SDL_Surface *PG_Widget::GetWidgetSurface() {
	return my_srfObject;
}

/**  */
PG_Point PG_Widget::ScreenToClient(int x, int y) {
	PG_Point p;

	p.x = x - my_xpos;
	p.y = y - my_ypos;

	return p;
}

void PG_Widget::AddChild(PG_Widget * child) {

	// remove our new child from previous lists
	if(child->GetParent()) {
		child->GetParent()->RemoveChild(child);
	}
	else {
		child->RemoveFromWidgetList();
	}

	child->my_widgetParent = this;

	if (my_childList == NULL) {
		my_childList = new PG_RectList;
	}

	my_childList->Add(child);
}

/**  */
bool PG_Widget::MoveWindow(int x, int y) {

	if (GetParent() != NULL) {
		x += GetParent()->my_xpos;
		y += GetParent()->my_ypos;
	}

	if(!IsVisible()) {
		MoveRect(x, y);
		return true;
	}

	// delta x,y
	int dx = x - my_xpos;
	int dy = y - my_ypos;

	// calculate vertical update rect

	PG_Rect vertical(0, 0, abs(dx), my_height + abs(dy));

	if(dx >= 0) {
		vertical.my_xpos = my_xpos;
	}
	else {
		vertical.my_xpos = my_xpos + my_width + dx;
	}

	vertical.my_ypos = my_ypos;

	// calculate vertical update rect

	PG_Rect horizontal(0, 0, my_width + abs(dx), abs(dy));

	horizontal.my_xpos = my_xpos;

	if(dy >= 0) {
		horizontal.my_ypos = my_ypos;
	}
	else {
		horizontal.my_ypos = my_ypos + my_height + dy;
	}

	// move rectangle and store new background
	MoveRect(x, y);

	if(vertical.my_xpos + vertical.my_width > my_srfScreen->w) {
		vertical.my_width = my_srfScreen->w - vertical.my_xpos;
	}
	if(vertical.my_ypos + vertical.my_height > my_srfScreen->h) {
		vertical.my_height = my_srfScreen->h - vertical.my_ypos;
	}

	if(horizontal.my_xpos + horizontal.my_width > my_srfScreen->w) {
		horizontal.my_width = my_srfScreen->w- horizontal.my_xpos;
	}
	if(horizontal.my_ypos + horizontal.my_height > my_srfScreen->h) {
		horizontal.my_height = my_srfScreen->h - horizontal.my_ypos;
	}

	UpdateRect(vertical);
	UpdateRect(horizontal);
	UpdateRect(my_rectClip);

	PG_UpdateRects(my_srfScreen, 1, &my_rectClip);
	PG_UpdateRects(my_srfScreen, 1, &vertical);
	PG_UpdateRects(my_srfScreen, 1, &horizontal);

	return true;
}

bool PG_Widget::SizeWindow(int w, int h) {
	bool v = IsVisible();

	if (v) {
		SetVisible(false);
	}

	PG_DrawObject::SizeWindow(w, h);

	if (v) {
		SetVisible(true);
	}
	return true;
}
/**  */
bool PG_Widget::ProcessEvent(const SDL_Event * event) {
	bool processed = false;

	SDL_mutexP(my_mutexProcess);

	// i will send that event to my children

	if(my_childList != NULL) {
		PG_RectList::iterator list = my_childList->begin();

		while (!processed && (list != my_childList->end())) {
			processed = (*list)->ProcessEvent(event);
			list++;
		}
	}

	// lets see if i can process that event

	if(!processed) {
		processed = PG_MessageObject::ProcessEvent(event);
	}

	SDL_mutexV(my_mutexProcess);

	return processed;
}

/**  */
int PG_Widget::GetID() {
	return my_id;
}

/**  */
void PG_Widget::SetID(int id) {
	my_id = id;
}

bool PG_Widget::RemoveChild(PG_Widget * child) {
	if(my_childList == NULL) {
		return false;
	}

	return my_childList->Remove(child);
}

bool PG_Widget::IsMouseInside() {
	PG_Point p;

	SDL_GetMouseState(&p.x, &p.y);	
	my_mouseInside = IsInside(p);

	return my_mouseInside;
}

/**  */
bool PG_Widget::Redraw(bool update) {

	PG_DrawObject::Redraw(false);

	if(my_childList != NULL) {
		PG_RectList::iterator list = my_childList->begin();

		while (list != my_childList->end()) {
			(*list)->Redraw(false);
			list++;
		}
	}

	if (update) {
		Update();
	}
	return true;
}

void PG_Widget::SetVisible(bool visible) {

	PG_DrawObject::SetVisible(visible);

	if(my_childList != NULL) {
		PG_RectList::iterator list = my_childList->begin();

		while (list != my_childList->end()) {
			(*list)->SetVisible(visible);
			if(visible) {
				(*list)->eventShow();
			}
			else {
				(*list)->eventHide();
			}
			list++;
		}
	}
}

/**  */
void PG_Widget::Show(bool fade) {

	widgetList.BringToFront(this);

	SetVisible(true);
	
	eventShow();

	if (fade) {
		FadeIn();
	}
	
	if (bSetCaptureOnShow) {
		SetCapture();
	}

	if(IsMouseInside()) {
		eventMouseEnter();
	}

	SDL_SetClipRect(my_srfScreen, NULL);
	Update();

	return;
}

/**  */
void PG_Widget::Hide(bool fade) {

	if(!IsVisible()) {
		return;
	}

	RecalcClipRect();

	eventMouseLeave();

	if (fade) {
		FadeOut();
	}

	SetVisible(false);
	eventHide();
	
	ReleaseCapture();
	ReleaseInputFocus();
	
	RestoreBackground();
	//Update();
	UpdateRect(my_rectClip);
	SDL_SetClipRect(my_srfScreen, NULL);
	PG_UpdateRects(my_srfScreen, 1, &my_rectClip);

	return;
}

/**  */
void PG_Widget::MoveRect(int x, int y) {
	int dx = x - my_xpos;
	int dy = y - my_ypos;

	// recalc cliprect
	RecalcClipRect();

	my_xpos = x;
	my_ypos = y;
	my_rectClip.my_xpos += dx;
	my_rectClip.my_ypos += dy;

	// recalc cliprect
	RecalcClipRect();

	if(my_childList != NULL) {
		PG_RectList::iterator list = my_childList->begin();

		while (list != my_childList->end()) {
			(*list)->MoveRect((*list)->my_xpos + dx, (*list)->my_ypos + dy);
			list++;
		}
	}

	eventMoveWindow(x, y);
}

void PG_Widget::Blit(bool recursive, bool restore) {

	if(PG_Application::GetBulkMode() && !bBulkUpdate) {
		return;
	}

	if(!IsVisible()) {
		return;
	}

	// update the drawobject
	RecalcClipRect();

	PG_DrawObject::Blit(restore);

	if(!recursive) {
		return;
	}

	if(my_childList != NULL) {
		my_childList->Blit();
	}
}

/**  */
void PG_Widget::Update(bool doBlit) {

	if(PG_Application::GetBulkMode() && !bBulkUpdate) {
		return;
	}

	if(doBlit) {
		Blit();
	}
	else {
		// recalc cliprect
		RecalcClipRect();
	}

	// reblit all widgets that overlap myself

	// check if other children of my parent overlap myself

	int index = 0;
	if(GetParent() != NULL) {
		PG_RectList* children = GetParent()->GetChildList();
		if(children) {
			index = children->FindIndexOf(this);
			if(index != -1) {
				// get a list with all objects in front of me
				PG_RectList frontlist = children->Intersect(&my_rectClip, index+1, -1);
				SDL_SetClipRect(my_srfScreen, &my_rectClip);
				frontlist.Blit(my_rectClip);
			}
		}
	}

	// find the toplevel widget
	PG_Widget* obj = this;
	while(obj->GetParent()) {
		obj = obj->GetParent();
	}

	// find my index
	index = widgetList.FindIndexOf(obj);

	if(index != -1) {
		// get a list with all objects in front of me
		PG_RectList frontlist = widgetList.Intersect(&my_rectClip, index+1, -1);
		SDL_SetClipRect(my_srfScreen, &my_rectClip);
		frontlist.Blit(my_rectClip);
	}

	PG_DrawObject::Update(false);
	SDL_SetClipRect(my_srfScreen, NULL);
}

/**  */
void PG_Widget::SetChildTransparency(Uint8 t) {
	if(my_childList == NULL) {
		return;
	}

	PG_RectList::iterator list = my_childList->begin();

	while (list != my_childList->end()) {
		(*list)->SetTransparency(t);
		list++;
	}
	Update();
}

PG_RectList* PG_Widget::GetChildList() {
	return my_childList;
}

int PG_Widget::GetChildCount() {
	return my_childList ? my_childList->size() : 0;
}

void PG_Widget::SetFont(TTF_Font * newfont) {

	if(newfont == NULL) {
		return;
	}
	
	my_font = newfont;
}

bool PG_Widget::IsDisplayRectValid() {

	if (GetParent() == NULL) {
		return PG_DrawObject::IsDisplayRectValid();
	}

	PG_Rect i = IntersectRect(*GetParent());

	return (i.my_width != 0) && (i.my_height != 0);
}

void PG_Widget::StartWidgetDrag() {
	SDL_GetMouseState(&my_ptDragStart.x, &my_ptDragStart.y);
	my_ptDragStart.x -= my_xpos;
	my_ptDragStart.y -= my_ypos;
}

void PG_Widget::WidgetDrag(int x, int y) {

	x -= my_ptDragStart.x;
	y -= my_ptDragStart.y;

	if(x < 0) x=0;
	if(y < 0) y=0;
	if(x > (my_srfScreen->w - my_width -1)) x = (my_srfScreen->w - my_width -1);
	if(y > (my_srfScreen->h - my_height -1)) y = (my_srfScreen->h - my_height -1);

	MoveWindow(x,y);
}

void PG_Widget::EndWidgetDrag(int x, int y) {
	WidgetDrag(x,y);
	my_ptDragStart.x = 0;
	my_ptDragStart.y = 0;
}

void PG_Widget::SetTextColor(SDL_Color c) {
	my_textcolor = c;
}

void PG_Widget::SetTextColor(Uint32 color) {
	my_textcolor.r = (color >> 16) & 0xFF;
	my_textcolor.g = (color >> 8) & 0xFF;
	my_textcolor.b = color & 0xFF;
}

void PG_Widget::HideAll() {
	for(Uint16 i=0; i<widgetList.size(); i++) {
		widgetList[i]->Hide();
	}
}

void PG_Widget::BulkUpdate() {
	bBulkUpdate = true;

	for(Uint16 i=0; i<widgetList.size(); i++) {
		if(widgetList[i]->IsVisible()) {
			widgetList[i]->Update();
		}
	}

	bBulkUpdate = false;
}

void PG_Widget::BulkBlit() {
	bBulkUpdate = true;
	widgetList.Blit();
	bBulkUpdate = false;
}

void PG_Widget::LoadThemeStyle(const char* widgettype, const char* objectname) {
	PG_Theme* t = PG_Application::GetTheme();
	TTF_Font* font = t->FindFont(widgettype, objectname);
	SDL_Color* c;
	
	if(font != NULL) {
		SetFont(font);
	}

	c = t->FindColor(widgettype, objectname, "textcolor");
	if(c != NULL) SetTextColor(*c);

	c = t->FindColor(widgettype, objectname, "bordercolor0");
	if(c != NULL) {
		my_colorBorder[0][0] = *c;
	}

	c = t->FindColor(widgettype, objectname, "bordercolor1");
	if(c != NULL) {
		my_colorBorder[1][0] = *c;
	}	
}

void PG_Widget::LoadThemeStyle(const char* widgettype) {
}

void PG_Widget::FadeOut() {
	PG_Rect r(0, 0, my_width, my_height);

	// blit the widget to screen (invisible)
	Blit();

	// create a temp surface
	SDL_Surface* srfFade = SDL_CreateRGBSurface(
					SDL_SWSURFACE,
					my_width, my_height,
					my_srfScreen->format->BitsPerPixel,
					my_srfScreen->format->Rmask,
		 			my_srfScreen->format->Gmask,
		 			my_srfScreen->format->Bmask,
		 			0 //my_srfScreen->format->Amask
					);

	// blit the widget to temp surface
	PG_BlitSurface(my_srfScreen, *this, srfFade, r);

	int d = (255-my_transparency)/ my_fadeSteps;

	for(int i=my_transparency; i<255; i += d) {
		RestoreBackground();
		SDL_SetAlpha(srfFade, SDL_SRCALPHA, 255-i);
		SDL_BlitSurface(srfFade, NULL, my_srfScreen, this);
		Update(false);
	}

	RestoreBackground();
	SDL_SetAlpha(srfFade, SDL_SRCALPHA, 0);
	SDL_BlitSurface(srfFade, NULL, my_srfScreen, this);
	SetVisible(false);
	Update(false);

	SDL_FreeSurface(srfFade);
}

void PG_Widget::FadeIn() {

	// blit the widget to screen (invisible)
	Blit();

	PG_Rect src(
			0,
			0,
			(my_xpos < 0) ? my_width + my_xpos : my_width,
			(my_ypos < 0) ? my_height + my_ypos : my_height);

	// create a temp surface
	SDL_Surface* srfFade = SDL_CreateRGBSurface(
					SDL_SWSURFACE,
					w, h,
					my_srfScreen->format->BitsPerPixel,
					my_srfScreen->format->Rmask,
		 			my_srfScreen->format->Gmask,
		 			my_srfScreen->format->Bmask,
		 			0 //my_srfScreen->format->Amask
					);

	// blit the widget to temp surface
	PG_BlitSurface(my_srfScreen, my_rectClip, srfFade, src);

	int d = (255-my_transparency)/ my_fadeSteps;

	for(int i=255; i>my_transparency; i -= d) {
		RestoreBackground();
		SDL_SetAlpha(srfFade, SDL_SRCALPHA, 255-i);
		PG_BlitSurface(srfFade, src, my_srfScreen, my_rectClip);
		PG_UpdateRects(my_srfScreen, 1, &my_rectClip);
	}

	Update();

	SDL_FreeSurface(srfFade);
}

void PG_Widget::SetFadeSteps(int steps) {
	my_fadeSteps = steps;
}

bool PG_Widget::Action(PG_ACTION action) {
	int x = my_xpos + my_width / 2;
	int y = my_ypos + my_height / 2;

	switch(action) {
		case PG_ACT_ACTIVATE:
			SDL_WarpMouse(x,y);
			eventMouseEnter();
			break;

		case PG_ACT_DEACTIVATE:
			eventMouseLeave();
			break;

		case PG_ACT_OK:
			SDL_MouseButtonEvent button;
			button.button = 1;
			button.x = x;
			button.y = y;
			eventMouseButtonDown(&button);
			SDL_Delay(200);
			eventMouseButtonUp(&button);
			Action(PG_ACT_ACTIVATE);
			break;

		default:
		break;
	}

	return false;
}

bool PG_Widget::RestoreBackground(PG_Rect* clip) {

	if(clip == NULL) {
		//RecalcClipRect();
		clip = &my_rectClip;
	}

/*	if(GetTransparency() == 0) {
		return true;
	}
*/
	if(GetParent() == NULL) {
		PG_Application::RedrawBackground(*clip);

		int index = widgetList.FindIndexOf(this);

		if(index != -1) {
			SDL_SetClipRect(my_srfScreen, clip);
			PG_RectList backlist = widgetList.Intersect(clip, 0, index);
			backlist.Blit(*clip);
		}
		return true;
	}

	SDL_SetClipRect(my_srfScreen, clip);
	GetParent()->RestoreBackground(clip);
	GetParent()->Blit(false, false);
	//SDL_SetClipRect(my_srfScreen, NULL);

	return true;
}

PG_Widget* PG_Widget::FindWidgetFromPos(int x, int y) {
	PG_Point p;
	p.x = x;
	p.y = y;

	return widgetList.IsInside(p);
}

void PG_Widget::UpdateRect(const PG_Rect& r) {
	SDL_Surface* screen = PG_Application::GetScreen();

	PG_Application::RedrawBackground(r);

	PG_RectList list = widgetList.Intersect((PG_Rect*)&r);
	SDL_SetClipRect(screen, (PG_Rect*)&r);
	list.Blit(r);
	SDL_SetClipRect(screen, NULL);
}

void PG_Widget::UpdateScreen() {
	UpdateRect(
		PG_Rect(0, 0, PG_Application::GetScreenWidth(), PG_Application::GetScreenHeight())
		);
}

void PG_Widget::BringToFront() {
	widgetList.BringToFront(this);
	Update();
}

void PG_Widget::RecalcClipRect() {
	static PG_Rect pr;

	if (my_widgetParent != NULL) {
		pr = my_widgetParent->GetClipRect();
	} else {
		pr.SetRect(
			0,
			0,
			PG_Application::GetScreenWidth(),
			PG_Application::GetScreenHeight());
	}

	PG_Rect ir = IntersectRect(pr);
	SetClipRect(ir);
}

SDL_Surface* PG_Widget::GetScreenSurface() {
	return my_srfScreen;
}

PG_Widget* PG_Widget::GetParent() {
	return my_widgetParent;
}

TTF_Font* PG_Widget::GetFont() {
	return my_font;
}

SDL_Color PG_Widget::GetTextColor() {
	return my_textcolor;
}

PG_RectList* PG_Widget::GetWidgetList() {
	return &widgetList;
};
