/*
    ParaGUI - crossplatform widgetset
    Copyright (C) 2000,2001  Alexander Pipelka
 
    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.
 
    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 
    Alexander Pipelka
    pipelka@teleweb.at
 
    Last Update:      $Author: pipelka $
    Update Date:      $Date: 2001/05/09 18:28:11 $
    Source File:      $Source: /usr/local/CVSROOT/linux/paragui/src/Attic/pgbutton.cpp,v $
    CVS/RCS Revision: $Revision: 1.1.2.22 $
    Status:           $State: Exp $
*/

#include <iostream.h>
#include <string.h>
#include "pgbutton.h"
#include "pgapplication.h"

PG_UserButton::PG_UserButton(PG_Widget* parent, int btnid, const PG_Rect& r, char* text) : PG_Widget(parent, r) {
	srf_normal = NULL;
	srf_high = NULL;
	srf_down = NULL;

	srf_icon[0] = NULL;
	srf_icon[1] = NULL;
	
	free_icons = false;
	
	labeltext = "";
	if(text != NULL) {
		labeltext = text;
	}

	togglemode = false;
	isPressed = false;
	id = btnid;
	my_state = BTN_STATE_NORMAL;
	
	my_pressShift = 1;

	my_bordersize[0] = 1;
	my_bordersize[1] = 1;
	my_bordersize[2] = 1;

	my_transparency[0] = 0;
	my_transparency[1] = 0;
	my_transparency[2] = 0;
}

PG_UserButton::~PG_UserButton() {

	FreeSurfaces();
	FreeIcons();
}

void PG_UserButton::SetBorderColorColor(int b, Uint32 color) {
	my_colorBorder[b][0].r = (color >> 16) & 0xFF;
	my_colorBorder[b][0].g = (color >> 8) & 0xFF;
	my_colorBorder[b][0].b = color & 0xFF;
}
	
/**  */
void PG_UserButton::eventSizeWindow(int w, int h) {

	FreeSurfaces();

	OnButtonSurface(&srf_normal, BTN_STATE_NORMAL, w, h);
	if(srf_normal) {
		if(my_transparency[0] > 0) {
			SDL_SetAlpha(srf_normal, SDL_SRCALPHA, 255-my_transparency[0]);
		}
	}

	OnButtonSurface(&srf_down, BTN_STATE_PRESSED, w, h);
	if(srf_down) {
		if(my_transparency[1] > 0) {
			SDL_SetAlpha(srf_down, SDL_SRCALPHA, 255-my_transparency[1]);
		}
	}

	OnButtonSurface(&srf_high, BTN_STATE_HIGH, w, h);
	if(srf_high) {
		if(my_transparency[2] > 0) {
			SDL_SetAlpha(srf_high, SDL_SRCALPHA, 255-my_transparency[2]);
		}
	}

	return;
}

/**  */
void PG_UserButton::FreeSurfaces() {
	if(srf_normal) {
		DeleteThemedSurface(srf_normal);
		srf_normal = NULL;
	}

	if(srf_high) {
		DeleteThemedSurface(srf_high);
		srf_high = NULL;
	}

	if(srf_down) {
		DeleteThemedSurface(srf_down);
		srf_down = NULL;
	}
}

void PG_UserButton::FreeIcons() {

	if(!free_icons) {
		return;
	}
	
	if(srf_icon[0]){
		SDL_FreeSurface(srf_icon[0]);
		srf_icon[0] = NULL;
	}

	if(srf_icon[1]){
		SDL_FreeSurface(srf_icon[1]);
		srf_icon[1] = NULL;
	}
	
	free_icons = false;
}

/**  */
void PG_UserButton::eventMouseEnter() {
	my_state = BTN_STATE_HIGH;

	Update();
	PG_Widget::eventMouseEnter();
}

/**  */
void PG_UserButton::eventMouseLeave() {

	if(my_state == BTN_STATE_HIGH) {
		my_state = BTN_STATE_NORMAL;
	}

	PG_Widget::eventMouseLeave();

	Update();
}

/**  */
bool PG_UserButton::eventMouseButtonDown(const SDL_MouseButtonEvent* button) {

	if(button->button == 1) {
		my_state = BTN_STATE_PRESSED;
		SetCapture();

		Update();
		return true;
	}

	return false;
}

/**  */
bool PG_UserButton::eventMouseButtonUp(const SDL_MouseButtonEvent* button) {
	if(button->button != 1) {
		return false;
	}

	if(togglemode) {
		if(!isPressed) {
			my_state = BTN_STATE_PRESSED;
			isPressed = true;
		} else {
			my_state = BTN_STATE_HIGH;
			isPressed = false;
		}
	} else {
		my_state = BTN_STATE_NORMAL;
		isPressed = false;
	}

	if(!IsMouseInside()) {
		my_state = BTN_STATE_NORMAL;
		ReleaseCapture();
		Update();
		return false;
	}

	ReleaseCapture();
	Update();

	SendMessage(GetParent(), MSG_BUTTONCLICK, id, 0);
	return true;
}

/**  */
bool PG_UserButton::SetIcon(char* filenameup, char* filenamedown, Uint32 colorkey) {

	FreeIcons();
	
	srf_icon[0] = PG_LoadImage(PG_Application::GetRelativePath(filenameup));

	if(srf_icon[0] == NULL) {
		cerr << "Failed to load '" << PG_Application::GetRelativePath(filenameup) << "'\n";
		return false;
	}

	SDL_SetColorKey(srf_icon[0], SDL_SRCCOLORKEY, colorkey);

	if(filenamedown == NULL) {
		srf_icon[1] = NULL;
	} else {
		srf_icon[1] = PG_LoadImage(PG_Application::GetRelativePath(filenamedown));

		if(srf_icon[1] == NULL) {
			cerr << "Failed to load '" << PG_Application::GetRelativePath(filenamedown) << "'\n";
			return false;
		}
		SDL_SetColorKey(srf_icon[1], SDL_SRCCOLORKEY, colorkey);

	}

	free_icons = true;
	
	Redraw();
	return true;
}

/**  */
bool PG_UserButton::SetIcon(SDL_Surface* icon_up, SDL_Surface* icon_down){

	if(!icon_up && !icon_down) {
		return false;
	}

	FreeIcons();
	
	srf_icon[0] = icon_up;
	srf_icon[1] = icon_down;
	
	free_icons = false;

	return true;
}

/**  */
void PG_UserButton::SetBorderSize(int norm, int pressed, int high) {

	if(norm >= 0) {
		my_bordersize[BTN_STATE_NORMAL] = norm;
	}

	if(pressed >= 0) {
		my_bordersize[BTN_STATE_PRESSED] = pressed;
	}

	if(high >= 0) {
		my_bordersize[BTN_STATE_HIGH] = high;
	}
}

/**  */
void PG_UserButton::OnButtonSurface(SDL_Surface** surface, int newstate, int w, int h) {}

/**  */
void PG_UserButton::SetToggle(bool bToggle) {
	togglemode = bToggle;
}

/**  */
void PG_UserButton::SetPressed(bool pressed) {
	if(!togglemode)
		return;

	isPressed = pressed;
	my_state = (isPressed ? BTN_STATE_PRESSED : BTN_STATE_NORMAL);

	Update();
}

/**  */
void PG_UserButton::SetTransparency(int norm, int pressed, int high) {
	if(norm >= 0) {
		my_transparency[0] = norm;
	}

	if(pressed >= 0) {
		my_transparency[1] = pressed;
	}

	if(high >= 0) {
		my_transparency[2] = high;
	}
}

void PG_UserButton::SetText(char* text) {
	labeltext = text;
}

const char* PG_UserButton::GetText() {
	return labeltext.c_str();
}


PG_Button::PG_Button(PG_Widget* parent, int id, const PG_Rect& r, char* text, const char* style) : PG_UserButton(parent, id, r, text) {

	for(int i=0; i<3; i++) {
		my_background[i] = NULL;
		my_backMode[i] = BKMODE_TILE;
		my_backBlend[i] = 0;
	}

	LoadThemeStyle(style);
}

PG_Button::~PG_Button() {
}

void PG_Button::LoadThemeStyle(const char* widgettype) {
	LoadThemeStyle("Button", "Button");
	LoadThemeStyle(widgettype, "Button");
}

void PG_Button::LoadThemeStyle(const char* widgettype, const char* objectname) {
	int b;
	const char* s = NULL;
	PG_Theme* t = PG_Application::GetTheme();
	
	b = t->FindProperty(widgettype, objectname, "textcolor");
	if(b != -1) SetTextColor((Uint32)b);

	SetIcon(
		t->FindSurface(widgettype, objectname, "iconup"),
		t->FindSurface(widgettype, objectname, "icondown")
		);

	PG_Gradient* g;
	g = t->FindGradient(widgettype, objectname, "gradient0");
	if(g) {
		my_gradState[0] = *g;
	}
	
	g = t->FindGradient(widgettype, objectname, "gradient1");
	if(g) {
		my_gradState[1] = *g;
	}

	g = t->FindGradient(widgettype, objectname, "gradient2");
	if(g) {
		my_gradState[2] = *g;
	}

	SetBackground(
		0,
		t->FindSurface(widgettype, objectname, "background0"),
		t->FindProperty(widgettype, objectname, "backmode0")
		);

	SetBackground(
		1,
		t->FindSurface(widgettype, objectname, "background1"),
		t->FindProperty(widgettype, objectname, "backmode1")
		);

	SetBackground(
		2,
		t->FindSurface(widgettype, objectname, "background2"),
		t->FindProperty(widgettype, objectname, "backmode2")
		);

	b = t->FindProperty(widgettype, objectname, "blend0");
	if(b != -1) my_backBlend[0] = b;

	b = t->FindProperty(widgettype, objectname, "blend1");
	if(b != -1) my_backBlend[1] = b;

	b = t->FindProperty(widgettype, objectname, "blend2");
	if(b != -1) my_backBlend[2] = b;

	b = t->FindProperty(widgettype, objectname, "shift");
	if(b != -1) my_pressShift = b;

	b = t->FindProperty(widgettype, objectname, "bordersize");
	if(b != -1) {
		SetBorderSize(b, b, b);
	}

	SetBorderSize(
		t->FindProperty(widgettype, objectname, "bordersize0"),
		t->FindProperty(widgettype, objectname, "bordersize1"),
		t->FindProperty(widgettype, objectname, "bordersize2")
		);

	SetTransparency(		
		t->FindProperty(widgettype, objectname, "transparency0"),
		t->FindProperty(widgettype, objectname, "transparency1"),
		t->FindProperty(widgettype, objectname, "transparency2")
		);
		
	s = t->FindString(widgettype, objectname, "label");
	if(s != NULL) {
		SetText((char*)s);
	}

	PG_Widget::LoadThemeStyle(widgettype, objectname);
	SizeWindow(Width(), Height());
}

/**  */
void PG_Button::OnButtonSurface(SDL_Surface** surface, int newstate, int w, int h) {
	PG_Rect r(0, 0, w, h);

	// remove the old button surface (if there are no more references)
	DeleteThemedSurface(*surface);

	// create a new one
	*surface = CreateThemedSurface(
					r,
					&my_gradState[newstate],
					my_background[newstate],
					my_backMode[newstate],
					my_backBlend[newstate]);

}

/**  */
void PG_Button::SetGradient(int state, PG_Gradient& gradient) {
	my_gradState[state] = gradient;
}

void PG_Button::SetBackground(int state, SDL_Surface* background, int mode) {

	if(!background) {
		return;
	}
	
	my_background[state] = background;
	my_backMode[state] = mode;
}

/**  */
bool PG_UserButton::GetPressed() {
	return (my_state == BTN_STATE_PRESSED);
}

void PG_UserButton::eventBlit(SDL_Surface* srf, const PG_Rect& src, const PG_Rect& dst) {
	PG_Rect rect = *this;
	PG_Rect r;
	PG_Rect my_src, my_dst;

	Uint8 t = 0;

	// get the right surface for the current state
	switch(my_state) {
		case BTN_STATE_NORMAL:
			if(srf_normal){
				t = my_transparency[0];
				srf = srf_normal;
			}
			break;

		case BTN_STATE_PRESSED:
			if(srf_down) {
				t = my_transparency[1];
				srf = srf_down;
			}
			break;

		case BTN_STATE_HIGH:
			if(srf_high) {
				t = my_transparency[2];
				srf = srf_high;
			}
			break;
	}

	// blit it
	
	if(t != 255) {
		SDL_SetAlpha(srf, SDL_SRCALPHA, 255-t);
		PG_BlitSurface(srf, src, my_srfScreen, dst);
	}

	int shift = (((my_state == BTN_STATE_PRESSED) || (togglemode && isPressed)) ? 1 : 0) * my_pressShift;

	r.my_xpos = rect.my_xpos + rect.my_width/2 + shift;
	r.my_ypos = rect.my_ypos + rect.my_height/2 + shift;
	r.my_height = 0;

	// check for icon srf
	SDL_Surface* iconsrf = (my_state == BTN_STATE_PRESSED) ? ((srf_icon[1] == 0) ? srf_icon[0] : srf_icon[1]) : srf_icon[0];

	if(iconsrf) {

		int dx = (rect.my_width - iconsrf->w)/2;
		int dy = (rect.my_height - iconsrf->h)/2;

		r.my_xpos = rect.my_xpos + dx + shift;
		r.my_ypos = rect.my_ypos + dy + shift;
		r.my_width = iconsrf->w;
		r.my_height = iconsrf->h;

		// calc new cliprect for icon
		GetClipRects(my_src, my_dst, r);

		// blit the icon
		PG_BlitSurface(iconsrf, my_src, my_srfScreen, my_dst);
	}

	// draw the text

	int w, h;
	GetTextSize(labeltext.c_str(), w, h, GetFont());

	int tx = (my_width - w)/2 + shift;
	int ty = (my_height - h)/2 + shift;

	if(iconsrf) {
		ty = my_height - h;
	}

	DrawText(tx, ty, labeltext.c_str(), GetTextColor(), GetFont());

	int i0, i1;

	if(!togglemode) {
		i0 = (my_state == BTN_STATE_PRESSED) ? 1 : 0;
		i1 = (my_state == BTN_STATE_PRESSED) ? 0 : 1;
	} else {
		i0 = (isPressed) ? 1 : 0;
		i1 = (isPressed) ? 0 : 1;
	}


	DrawBorder(PG_Rect(0, 0, Width(), Height()), my_bordersize[my_state], i1);
}
