/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Author: Charles Kerr <charles@rebelbase.com>
 *
 * Copyright (C) 2002  Charles Kerr <charles@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*********************
**********************  Includes
*********************/

#include <config.h>

#include <glib.h>
#include <gdk/gdkkeysyms.h>
#include <gtk/gtk.h>

#include <pan/base/argset.h>
#include <pan/base/debug.h>
#include <pan/base/pan-i18n.h>
#include <pan/base/pan-glib-extensions.h>

#include <pan/gui.h>
#include <pan/prefs.h>
#include <pan/queue.h>
#include <pan/task-manager.h>
#include <pan/util.h>

#include <pan/xpm/pan-pixbufs.h>

/*********************
**********************  Defines / Enumerated types
*********************/

/*********************
**********************  Macros
*********************/

/*********************
**********************  Structures / Typedefs
*********************/

typedef struct
{
	GtkWidget    * window;
	GtkWidget    * top;
	GtkWidget    * list;
	GtkWidget    * online_tb;

	guint          timeout_id;

	gboolean       titlebar_needs_refresh;
	gboolean       dampen_move_feedback_loop;
}
ManagerUI;

/*********************
**********************  Variables
*********************/

extern GtkTooltips * ttips;

static ManagerUI * manager_ui = NULL;

/*********************
**********************  BEGINNING OF SOURCE
*********************/

/************
*************  PRIVATE ROUTINES
************/

/**
***  Internal Utility
**/

static GSList *
get_selected_task_list (ManagerUI * ui)
{
	GSList * tasks = NULL;
	GtkCList * clist = GTK_CLIST(ui->list);
	GList * l;
	for (l=clist->selection; l!=NULL; l=l->next)
	{
		const int old_row = GPOINTER_TO_INT (l->data);
		Task * const task = gtk_clist_get_row_data (clist, old_row);
		tasks = g_slist_prepend (tasks, task);
	}
	return g_slist_reverse (tasks);
}

/**
*** Helper functions for user interactions
**/

static void
select_task (ManagerUI * ui, gboolean first)
{
	GtkCList * clist;
	gint       row;
	debug_enter ("select_task");

	/* sanity clause */
	g_return_if_fail (ui!=NULL);

	clist = GTK_CLIST(ui->list);
	row = first ? 0 : clist->rows-1;
	
	gtk_clist_freeze(clist);
	gtk_clist_unselect_all (clist);
	gtk_clist_select_row (clist, row, 0);
	clist->focus_row = row;
	gtk_clist_thaw(clist);

	debug_exit("select_task");
}


/**
***  User Interactions
**/

static int
online_status_changed_idle (gpointer user_data)
{
	GtkToggleButton * tb = GTK_TOGGLE_BUTTON (user_data);
	if (tb != NULL)
	{
		const gboolean is_online = queue_is_online ();
		gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), GTK_WIDGET(tb),
			is_online ? _("Pan is Online") : _("Pan is Offline"), "");
		if (gtk_toggle_button_get_active (tb) != is_online)
			gtk_toggle_button_set_active (tb, is_online);
	}

	return 0;
}

static void
online_status_changed_cb (gpointer call_obj,
                          gpointer call_arg,
                          gpointer user_data)
{
	ManagerUI * ui = (ManagerUI*) user_data;
	gui_queue_add (online_status_changed_idle, ui->online_tb);
}

static void
online_toggled_cb (GtkToggleButton  * tb,
                  gpointer           user_data)
{
	gboolean online = gtk_toggle_button_get_active (tb);

	if (online != queue_is_online())
		queue_set_online (online);
}

static void
up_clicked_cb (GtkButton   * button,
               gpointer      user_data)
{
        ManagerUI * ui = (ManagerUI*) user_data;
	GSList * tasks = get_selected_task_list (ui);

	if (tasks != NULL) {
		const int index = gtk_clist_find_row_from_data (GTK_CLIST(ui->list), tasks->data);
		if (index > 0)
			queue_move_tasks (tasks, index-1);
		else
			g_slist_free (tasks);
	}
}

static void
down_clicked_cb (GtkButton   * button,
                 gpointer      user_data)
{
        ManagerUI * ui = (ManagerUI*) user_data;
	GtkCList * clist = GTK_CLIST (ui->list);
	GSList * tasks = get_selected_task_list (ui);

	if (tasks != NULL) {
		const int index = gtk_clist_find_row_from_data (clist, tasks->data);
		if (index < clist->rows - 1)
			queue_move_tasks (tasks, index+1);
		else
			g_slist_free (tasks);
	}
}

static void
top_clicked_cb (GtkButton    * button,
                gpointer       user_data)
{
        ManagerUI * ui = (ManagerUI*) user_data;
	queue_move_tasks (get_selected_task_list(ui), 0);
}

static void
bottom_clicked_cb (GtkButton   * button,
                   gpointer      user_data)
{
        ManagerUI * ui = (ManagerUI*) user_data;
	queue_move_tasks (get_selected_task_list(ui), -1);
}

static void
stop_clicked_cb (GtkButton   * button,
                   gpointer      user_data)
{
        ManagerUI * ui = (ManagerUI*) user_data;
	queue_abort_tasks (get_selected_task_list(ui));
}

static void
delete_clicked_cb (GtkButton   * button,
                   gpointer      user_data)
{
        ManagerUI * ui = (ManagerUI*) user_data;
	queue_remove_tasks (get_selected_task_list(ui));
}


static void
requeue_cb (GtkButton   * button,
            gpointer      user_data)
{
	ManagerUI * ui = (ManagerUI*) user_data;
	queue_requeue_failed_tasks (get_selected_task_list(ui));
}


/**
***  Display
**/

static int
get_secs_left (const Task * task)
{
	int retval = 0;

	if (task->sock)
	{
		double elapsed_time = time(0) - task->sock->byte_count_start_time;
		double progress = status_item_get_progress_of_100 (STATUS_ITEM(task)) / 100.0;
		double safe_progress = MAX(progress, 0.01); /* don't divide by 0 */
		const double estimated_total_time = elapsed_time/safe_progress;
		retval = estimated_total_time - elapsed_time;
	}

	return retval;
}

static void
get_percent_str (const Task   * task,
                 gchar        * buf,
                 gint           buf_size)
{
	int percent = 0;

	if (task != NULL)
		percent = status_item_get_progress_of_100 (STATUS_ITEM(task));

	if (percent == 0)
		*buf = '\0';
	else
		g_snprintf (buf, buf_size, "%d%%", percent);
}

static void
get_xfer_str (const Task    * task,
              gchar         * buf,
              gint            buf_size)
{
	if (!task->sock)
	{
		*buf = '\0';
	}
	else
	{
		int secs_left = get_secs_left (task);
		const int h = secs_left / 3600;
		const int m = (secs_left % 3600) / 60;
		const int s = secs_left % 60;
		const double KBps = pan_socket_get_xfer_rate_KBps (task->sock);
		g_snprintf (buf, buf_size, _("%d:%02d:%02d (%.2f KB/s)"), h, m, s, KBps);
	}
}

static void
get_status_str (const Task     * task,
                gchar          * buf,
                gint             buf_size)
{
	const char* status_str;
	QueueTaskStatus status = queue_get_task_status(task);

	switch (status)
	{
		case QUEUE_TASK_STATUS_NOT_QUEUED:
			status_str = _("Not Queued");
			break;
		case QUEUE_TASK_STATUS_QUEUED:
			status_str = _("Queued");
			break;
		case QUEUE_TASK_STATUS_RUNNING:
			status_str = _("Running");
			break;
		case QUEUE_TASK_STATUS_FAILED:
			status_str = _("Failed");
			break;
		case QUEUE_TASK_STATUS_ABORTING:
			status_str = _("Aborting");
			break;
		case QUEUE_TASK_STATUS_CONNECTING:
			status_str = _("Connecting");
			break;
		default:
			status_str = _("???");
			break;
	}

	if (!queue_is_online() && status==QUEUE_TASK_STATUS_QUEUED)
		g_snprintf (buf, buf_size, _("Offline"));
	else if (!task->tries)
		g_snprintf (buf, buf_size, "%s", status_str);
	else
		g_snprintf (buf, buf_size, _("%s (%d tries)"), status_str, task->tries);
}


static void
add_tasks_to_list (ManagerUI     * ui,
                   GSList        * tasks,
                   int             index)
{
	GSList * task_it;
	GtkCList * list = GTK_CLIST(ui->list);
	debug_enter ("add_tasks_to_list");

	for (task_it=tasks; task_it!=NULL; task_it=task_it->next)
	{
		Task * task = TASK(task_it->data);
		int row;
		char statusbuf[64];
		char percentbuf[8];
		char xferbuf[64];
		char * description = status_item_describe(STATUS_ITEM(task));
		char * text [5];
		char * freeme[5] = { NULL, NULL, NULL, NULL, NULL };

		get_xfer_str (task, xferbuf, sizeof(xferbuf));
		get_status_str (task, statusbuf, sizeof(statusbuf));
		get_percent_str (task, percentbuf, sizeof(percentbuf));

		text[0] = (char*) pan_utf8ize (statusbuf, -1, freeme+0);
		text[1] = (char*) pan_utf8ize (percentbuf, -1, freeme+1);
		text[2] = (char*) pan_utf8ize (task->server->name, -1, freeme+2);
		text[3] = (char*) pan_utf8ize (xferbuf, -1, freeme+3);
		text[4] = (char*) pan_utf8ize (description, -1, freeme+4);

		pan_lock ();
		row = gtk_clist_insert (list, index++, text);
		gtk_clist_set_row_data (list, row, task);
		pan_object_ref (PAN_OBJECT (task));
		pan_unlock ();

		g_free (description);

		/* cleanup */
		for (row=0; row<G_N_ELEMENTS(freeme); ++row)
			if (freeme[row] != NULL)
				g_free (freeme[row]);
	}

	debug_exit ("add_tasks_to_list");
}


static void
ui_populate (ManagerUI *ui)
{
	guint i;
	GPtrArray * tasks;
	GSList * slist = NULL;

	/* add the tasks to the gui */
	tasks = queue_get_tasks ();
	for (i=0; i<tasks->len; ++i)
		slist = g_slist_prepend (slist, g_ptr_array_index(tasks,i));
	slist = g_slist_reverse (slist);
	add_tasks_to_list (ui, slist, 0);

	/* cleanup */
	g_slist_foreach (slist, (GFunc)pan_object_unref, NULL);
	g_slist_free (slist);
	g_ptr_array_free (tasks, TRUE);
}

/**
***  Queue Changes
**/

static int
tasks_added_idle (gpointer argset_gpointer)
{
	ArgSet * argset;
	ManagerUI * ui;
	GSList * tasks;
	int index;
	debug_enter ("tasks_added_idle");

	argset = (ArgSet*) argset_gpointer;
	ui = (ManagerUI*) argset_get (argset, 0);
	tasks = (GSList*) argset_get (argset, 1);
	index = GPOINTER_TO_INT (argset_get (argset, 2));

	/* insert the task */
	add_tasks_to_list (ui, tasks, index);

	/* cleanup */
	g_slist_foreach (tasks, (GFunc)pan_object_unref, NULL);
	g_slist_free (tasks);
	argset_free (argset);
	debug_exit ("tasks_added_idle");
	return 0;
}

static void
tasks_added_cb (gpointer call_obj, gpointer call_arg, gpointer client_arg)
{
	GSList * tasks;
	int index;
	ManagerUI * ui;
	debug_enter ("tasks_added_cb");

	tasks = (GSList*) call_obj;
	index = GPOINTER_TO_INT(call_arg);
	ui = (ManagerUI*) client_arg;

	tasks = g_slist_copy (tasks);
	g_slist_foreach (tasks, (GFunc)pan_object_ref, NULL);
	gui_queue_add (tasks_added_idle, argset_new3 (ui, tasks, GINT_TO_POINTER(index)));

	debug_exit ("tasks_added_cb");
}

static int
tasks_removed_idle (gpointer argset_gpointer)
{
	ArgSet * argset;
	ManagerUI * ui;
	GSList * tasks;
	GtkCList * clist;
	GSList   * l;
	debug_enter ("tasks_removed_idle");

	argset = (ArgSet*) argset_gpointer;
	ui = (ManagerUI*) argset_get (argset, 0);
	tasks = (GSList*) argset_get (argset, 1);
	clist = GTK_CLIST(ui->list);

	/* remove tasks */
	pan_lock ();
	gtk_clist_freeze (clist);
	for (l=tasks; l!=NULL; l=l->next) {
		const int index = gtk_clist_find_row_from_data (clist, l->data);
		if (index != -1)
		{
			gtk_clist_remove (clist, index);
			pan_object_unref (PAN_OBJECT (l->data));
		}
	}
	gtk_clist_thaw (clist);
	pan_unlock ();

	/* cleanup */
	g_slist_foreach (tasks, (GFunc)pan_object_unref, NULL);
	g_slist_free (tasks);
	argset_free (argset);
	debug_exit ("tasks_removed_idle");
	return 0;
}

static void
tasks_removed_cb (gpointer call_obj, gpointer call_arg, gpointer client_arg)
{
	GSList * tasks;
	ManagerUI * ui;
	debug_enter ("tasks_removed_cb");

	ui = (ManagerUI*) client_arg;
	tasks = (GSList*) call_obj;
	tasks = g_slist_copy (tasks);
	g_slist_foreach (tasks, (GFunc)pan_object_ref, NULL);
	gui_queue_add (tasks_removed_idle, argset_new2 (ui, tasks));

	debug_exit ("tasks_removed_cb");
}

static void
tasks_moved_cb (gpointer call_obj, gpointer call_arg, gpointer client_arg)
{
	int i;
	GSList * l;
	GSList * tasks = (GSList*) call_obj;
	ManagerUI * ui = (ManagerUI*) client_arg;
	GtkCList * clist = GTK_CLIST(ui->list);
	int index = GPOINTER_TO_INT(call_arg);

	/* extra ref to make sure the tasks don't get deleted while moving them */
	g_slist_foreach (tasks, (GFunc)pan_object_ref, NULL);

	/* remove the old rows */
	pan_lock ();
	gtk_clist_freeze (clist);
	for (l=tasks; l!=NULL; l=l->next) {
		const int index = gtk_clist_find_row_from_data (clist, l->data);
		if (index != -1)
			gtk_clist_remove (clist, index);
	}
	pan_unlock ();

	/* insert the new rows */
	add_tasks_to_list (ui, tasks, index);

	/* select the new rows */
	pan_lock ();
	for (l=tasks, i=index; l!=NULL; l=l->next, ++i)
		gtk_clist_select_row (clist, i, -1);
	gtk_clist_thaw (clist);
	pan_unlock ();

	/* clear extra ref */
	g_slist_foreach (tasks, (GFunc)pan_object_unref, NULL);
}

static gboolean
gui_key_press_cb (GtkWidget      * widget,
                  GdkEventKey    * event,
                  gpointer         data)
{
	ManagerUI * ui;
	GtkCList  * clist;
	debug_enter ("gui_key_press_cb");

	ui    = (ManagerUI*) data;
	clist = GTK_CLIST (ui->list);

        switch (event->keyval)
        {
                case GDK_Home:
                        if (event->state & GDK_SHIFT_MASK)
				top_clicked_cb (NULL, ui);
			else
				select_task (ui, TRUE);
			break;
                case GDK_End:
                        if (event->state & GDK_SHIFT_MASK)
				bottom_clicked_cb (NULL, ui);
			else
				select_task (ui, FALSE);
			break;
                default:
			break;
        }

	debug_exit ("gui_key_press_cb");
        return TRUE;
}

static void
update_titlebar_lock (ManagerUI * ui)
{
	int running, queued, failed;
	guint i;
	GPtrArray * tasks;
	gchar* pch;

	g_return_if_fail (ui!=NULL);

	/* get information about the tasks */
	running = queued = failed = 0;
	tasks = queue_get_tasks ();
	for (i=0; i<tasks->len; ++i) {
		Task * t = TASK(g_ptr_array_index(tasks,i));
		QueueTaskStatus status = queue_get_task_status (t);
	
		if (status==QUEUE_TASK_STATUS_QUEUED)
			++queued;
		else if (status==QUEUE_TASK_STATUS_RUNNING)
			++running;
		else if (status==QUEUE_TASK_STATUS_ABORTING || status==QUEUE_TASK_STATUS_FAILED)
			++failed;
	}

	/* create a titlebar */
	if (failed)
		pch = g_strdup_printf (_("Pan %s Task Manager (%d Queued, %d Running, %d Failed)"), VERSION, queued, running, failed);
	else if (queued || running)
		pch = g_strdup_printf (_("Pan %s Task Manager (%d Queued, %d Running)"), VERSION, queued, running);
	else 
		pch = g_strdup_printf (_("Pan %s Task Manager"), VERSION);

	/* show the titlebar */
	pan_lock ();
	gtk_window_set_title (GTK_WINDOW(ui->window), pch);
	pan_unlock ();
	g_free (pch);

	/* cleanup */
	pan_g_ptr_array_foreach (tasks, (GFunc)pan_object_unref, NULL);
	g_ptr_array_free (tasks, TRUE);
}

static int
periodic_update_timer (gpointer user_data)
{
	ManagerUI * ui = (ManagerUI*) user_data;
	GtkCList * clist = GTK_CLIST(ui->list);
	gint row;

	debug0 (DEBUG_QUEUE, "task manager periodic ui refresh");

	/* update the ui */
	pan_lock ();
	for (row=0; row<clist->rows; ++row)
	{
		if (gtk_clist_row_is_visible (clist, row))
		{
			char buf[256];
			char * pch;
			char * old_text;
			const char * new_text;
			char * freeme = NULL;
			Task * task;

			task = gtk_clist_get_row_data(clist, row);

			old_text = NULL;
			gtk_clist_get_text (clist, row, 0, &old_text);
			get_status_str (task, buf, sizeof(buf));
			new_text = pan_utf8ize (buf, -1, &freeme);
			if (pan_strcmp (new_text, old_text))
				gtk_clist_set_text (clist, row, 0, new_text);
			g_free (freeme);

			old_text = NULL;
			gtk_clist_get_text (clist, row, 1, &old_text);
			get_percent_str (task, buf, sizeof(buf));
			new_text = pan_utf8ize (buf, -1, &freeme);
			if (pan_strcmp (new_text, old_text))
				gtk_clist_set_text (clist, row, 1, new_text);
			g_free (freeme);

			old_text = NULL;
			gtk_clist_get_text (clist, row, 2, &old_text);
			new_text = pan_utf8ize (task->server->name, -1, &freeme);
			if (pan_strcmp (new_text, old_text))
				gtk_clist_set_text (clist, row, 2, new_text);
			g_free (freeme);

			old_text = NULL;
			gtk_clist_get_text (clist, row, 3, &old_text);
			get_xfer_str (task, buf, sizeof(buf));
			new_text = pan_utf8ize (buf, -1, &freeme);
			if (pan_strcmp (new_text, old_text))
				gtk_clist_set_text (clist, row, 3, new_text);
			g_free (freeme);

			old_text = NULL;
			gtk_clist_get_text (clist, row, 4, &old_text);
			pch = status_item_describe(STATUS_ITEM(task));
			new_text = pan_utf8ize (pch, -1, &freeme);
			if (pan_strcmp (new_text, old_text))
				gtk_clist_set_text (clist, row, 4, new_text);
			g_free (freeme);
			g_free (pch);
		}
	}
	pan_unlock ();

	if (ui->titlebar_needs_refresh)
	{
		update_titlebar_lock (ui);
		ui->titlebar_needs_refresh = FALSE;
	}

	return 1;
}

static gint
window_delete_event_cb (GtkWidget * w, GdkEvent * e, gpointer data)
{
	ManagerUI * ui = (ManagerUI*) data;
	gui_save_window_size (ui->window, "task_manager_2");
	gui_save_column_widths (ui->list, "task_manager_2");
	return FALSE;
} 
static void
ui_destroy_cb (GtkObject * o, gpointer user_data)
{
	ManagerUI *ui = (ManagerUI*) user_data;

	pan_callback_remove (queue_get_online_status_changed_callback(), online_status_changed_cb, ui);
	pan_callback_remove (queue_tasks_added, tasks_added_cb, ui);
	pan_callback_remove (queue_tasks_removed, tasks_removed_cb, ui);
	pan_callback_remove (queue_tasks_moved, tasks_moved_cb, ui);

	g_source_remove (ui->timeout_id);

	pan_warn_if_fail (ui == manager_ui);
	g_free (manager_ui);
	manager_ui = NULL;
}

/**
***
**/

static ManagerUI*
task_manager_build_ui (GtkWidget * window)
{
	//gboolean b;
	ManagerUI * ui = g_new0 (ManagerUI, 1);
	GtkWidget * w;
	GtkWidget * toolbar;
	GtkWidget * vbox;
	GdkPixbuf * pixbuf;
	GtkIconSet * icon_set;
	GtkWidget * image;
	char* list_titles [5];

	ui->window = window;
	vbox = gtk_vbox_new (FALSE, GUI_PAD_SMALL);

	/* button bar */
	toolbar = gtk_toolbar_new ();

	/* online button */
	pixbuf = gdk_pixbuf_new_from_inline (-1, icon_network, FALSE, NULL);
	icon_set = gtk_icon_set_new_from_pixbuf (pixbuf);
	g_object_unref (G_OBJECT(pixbuf));
	image = gtk_image_new_from_icon_set (icon_set, GTK_ICON_SIZE_SMALL_TOOLBAR);
	gtk_icon_set_unref (icon_set);
	w = gtk_toolbar_append_element (GTK_TOOLBAR(toolbar),
	                                GTK_TOOLBAR_CHILD_TOGGLEBUTTON, NULL,
	                                _("Online"),
					queue_is_online ? _("Pan is Online") : _("Pan is Offline"),
					NULL,
	                                image, G_CALLBACK(online_toggled_cb), ui);
	ui->online_tb = w;
	pan_callback_add (queue_get_online_status_changed_callback(),
	                  online_status_changed_cb, ui);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), queue_is_online());

        /* edit per-server maximum */
	pixbuf = gdk_pixbuf_new_from_inline (-1, icon_server, FALSE, NULL);
	icon_set = gtk_icon_set_new_from_pixbuf (pixbuf);
	g_object_unref (G_OBJECT(pixbuf));
	image = gtk_image_new_from_icon_set (icon_set, GTK_ICON_SIZE_SMALL_TOOLBAR);
	gtk_icon_set_unref (icon_set);
	gtk_toolbar_append_element (GTK_TOOLBAR(toolbar),
	                            GTK_TOOLBAR_CHILD_BUTTON, NULL,
				    _("Server"), _("Set Per-Server Connection Limits"), NULL,
				    image, G_CALLBACK(prefs_spawn_to_news_connections), ui);

	gtk_toolbar_insert_space (GTK_TOOLBAR(toolbar), -1);

	/* up */
	gtk_toolbar_insert_stock (GTK_TOOLBAR(toolbar), GTK_STOCK_GO_UP,
	                          _("Move Selected Task(s) Up"), NULL,
	                          G_CALLBACK(up_clicked_cb), ui, -1);
	/* top */
	gtk_toolbar_insert_stock (GTK_TOOLBAR(toolbar), GTK_STOCK_GOTO_TOP,
	                          _("Move Selected Task(s) to Top"), NULL,
	                          G_CALLBACK(top_clicked_cb), ui, -1);

	gtk_toolbar_insert_space (GTK_TOOLBAR(toolbar), -1);

	/* down */
	gtk_toolbar_insert_stock (GTK_TOOLBAR(toolbar), GTK_STOCK_GO_DOWN,
	                          _("Move Selected Task(s) Down"), NULL,
	                          G_CALLBACK(down_clicked_cb), ui, -1);
	/* bottom */
	gtk_toolbar_insert_stock (GTK_TOOLBAR(toolbar), GTK_STOCK_GOTO_BOTTOM,
	                          _("Move Selected Task(s) to Bottom"), NULL,
	                          G_CALLBACK(bottom_clicked_cb), ui, -1);

	gtk_toolbar_insert_space (GTK_TOOLBAR(toolbar), -1);

	/* redo */
	gtk_toolbar_insert_stock (GTK_TOOLBAR(toolbar), GTK_STOCK_REDO,
	                          _("Restart selected stopped/failed Task(s)"), NULL,
	                          G_CALLBACK(requeue_cb), ui, -1);

        /* cancel */
	gtk_toolbar_insert_stock (GTK_TOOLBAR(toolbar), GTK_STOCK_STOP,
	                          _("Stop Selected Task(s)"), NULL,
	                          G_CALLBACK(stop_clicked_cb), ui, -1);

	/* remove */
	gtk_toolbar_insert_stock (GTK_TOOLBAR(toolbar), GTK_STOCK_DELETE,
	                          _("Delete Selected Task(s)"), NULL,
	                          G_CALLBACK(delete_clicked_cb), ui, -1);

	/* add button bar to the queued window */
	gtk_box_pack_start (GTK_BOX(vbox), toolbar, FALSE, FALSE, 0);

        /* queue list */
        list_titles[0] = _("Status");
	list_titles[1] = _("% Done");
	list_titles[2] = _("Server");
        list_titles[3] = _("Time Remaining");
	list_titles[4] = _("Description");
	ui->list = w = gtk_clist_new_with_titles (5, list_titles);
	gtk_clist_set_column_width (GTK_CLIST(w), 0, 200);
	gtk_clist_set_column_width (GTK_CLIST(w), 4, 1500);
	gui_restore_column_widths (ui->list, "task_manager_2");

	gtk_clist_set_selection_mode (GTK_CLIST(w), GTK_SELECTION_MULTIPLE);
	w = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW(w), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_container_add (GTK_CONTAINER (w), ui->list);
	gtk_box_pack_start (GTK_BOX(vbox), w, TRUE, TRUE, 0);

	pan_callback_add (queue_tasks_added, tasks_added_cb, ui);
	pan_callback_add (queue_tasks_removed, tasks_removed_cb, ui);
	pan_callback_add (queue_tasks_moved, tasks_moved_cb, ui);

	ui->titlebar_needs_refresh = TRUE;
	ui->timeout_id = pan_timeout_add (2000, periodic_update_timer, ui);
	ui->top = vbox;
        g_signal_connect (ui->top, "destroy", G_CALLBACK(ui_destroy_cb), ui);
	return ui;
}

/**
***  Titlebar dirty
**/

static void
update_titlebar_cb (gpointer call_obj, gpointer call_arg, gpointer user_data)
{
	((ManagerUI*)(user_data))->titlebar_needs_refresh = TRUE;
}
static void
remove_update_titlebar_callbacks_cb (GtkObject * o, gpointer user_data)
{
	pan_callback_remove (queue_tasks_added, update_titlebar_cb, user_data);
	pan_callback_remove (queue_tasks_removed, update_titlebar_cb, user_data);
	pan_callback_remove (queue_task_status_changed, update_titlebar_cb, user_data);
}

/************
*************  PUBLIC ROUTINES
************/

void
task_manager_spawn (void)
{
	/* There can be only one */
	if (manager_ui != NULL)
	{
		pan_lock();
		if (!GTK_WIDGET_MAPPED(GTK_WIDGET(manager_ui->window)))
			gtk_widget_map ( GTK_WIDGET(manager_ui->window));
		gdk_window_raise (manager_ui->window->window);
		gdk_window_show (manager_ui->window->window);
		pan_unlock();
		return;
	}

	manager_ui = task_manager_build_ui (gtk_window_new(GTK_WINDOW_TOPLEVEL));
	ui_populate (manager_ui);

	pan_callback_add (queue_tasks_added, update_titlebar_cb, manager_ui);
	pan_callback_add (queue_tasks_removed, update_titlebar_cb, manager_ui);
	pan_callback_add (queue_task_status_changed, update_titlebar_cb, manager_ui);
	g_signal_connect (manager_ui->window, "delete_event",
	                  G_CALLBACK(window_delete_event_cb), manager_ui);
	g_signal_connect (manager_ui->window, "destroy",
	                  G_CALLBACK(remove_update_titlebar_callbacks_cb), manager_ui);
	g_signal_connect (manager_ui->window, "key_press_event",
	                  G_CALLBACK(gui_key_press_cb), manager_ui);
	gtk_window_set_title (GTK_WINDOW(manager_ui->window), _("Pan - Task Manager"));
	gtk_container_add (GTK_CONTAINER(manager_ui->window), manager_ui->top);

	gtk_window_set_default_size (GTK_WINDOW (manager_ui->window), 600, 400);
	gui_restore_window_size (manager_ui->window, "task_manager_2");
	gtk_widget_show_all (manager_ui->window);
}

