/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999, 2000, 2001  Pan Development Team <pan@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

/*********************
**********************  Includes
*********************/

#include <config.h>

#include <stdlib.h>
#include <string.h>

#include <glib.h>

#include <pan/base/acache.h>
#include <pan/base/article.h>
#include <pan/base/debug.h>
#include <pan/base/log.h>
#include <pan/base/pan-glib-extensions.h>
#include <pan/base/pan-i18n.h>

#include <pan/nntp.h>
#include <pan/queue.h>
#include <pan/task-bodies.h>

/*********************
**********************  Defines / Enumerated types
*********************/

/*********************
**********************  Macros
*********************/

/*********************
**********************  Structures / Typedefs
*********************/

/*********************
**********************  Private Function Prototypes
*********************/

static int task_bodies_run (Task* item);

static gchar* task_bodies_describe (const StatusItem* item);

/*********************
**********************  Variables
*********************/

/***********
************  Extern
***********/

/***********
************  Public
***********/

/***********
************  Private
***********/

/*********************
**********************  BEGINNING OF SOURCE
*********************/

static void
task_bodies_destructor (PanObject *o)
{
	TaskBodies * task;
	debug_enter ("task_bodies_destructor");

	/* task-bodies dtor: unref articles + group */
 	task = TASK_BODIES(o);
	if (task->articles->len) {
		guint i;
		const gchar ** message_ids = g_new0 (const gchar*, task->articles->len);
		for (i=0; i<task->articles->len; ++i)
			message_ids[i] = article_get_message_id (ARTICLE(g_ptr_array_index(task->articles,i)));
		acache_file_checkin (message_ids, task->articles->len);
		g_free (message_ids);
	}
	group_unref_articles (task->group, STATUS_ITEM(task));

	/* task-bodies dtor: cleanup fields */
	g_ptr_array_free (task->articles, TRUE);
	task->articles = NULL;
	task->working_index = 0;

	/* destroy parent class */
	task_destructor (o);

	debug_exit ("task_bodies_destructor");
}

/************
*************  PUBLIC ROUTINES
************/

PanObject*
task_bodies_new (Group             * group,
                 const GPtrArray   * articles)
{
	gint i;
	TaskBodies* bodies = NULL;
	gboolean needs_socket;
	debug_enter ("task_bodies_new");

	/* sanity clause */
	g_return_val_if_fail (group_is_valid(group), NULL);
	g_return_val_if_fail (articles!=NULL, NULL);
	g_return_val_if_fail (articles->len!=0, NULL);

	/* check out all the articles */
	if (articles->len) {
		guint i;
		const gchar ** message_ids = g_new0 (const gchar*, articles->len);
		for (i=0; i<articles->len; ++i)
			message_ids[i] = article_get_message_id (ARTICLE(g_ptr_array_index(articles,i)));
		acache_file_checkout (message_ids, articles->len);
		g_free (message_ids);
	}

	/* do we need a connection? */
	needs_socket = FALSE;
	for (i=0; i<articles->len && !needs_socket; ++i)
		if (!article_has_body(ARTICLE(g_ptr_array_index(articles,i))))
			needs_socket = TRUE;

	/* create the object */
	group_ref_articles (group, NULL);
       	bodies = g_new0 (TaskBodies, 1);
        debug1 (DEBUG_PAN_OBJECT, "task_bodies_new: %p", bodies);

	/* initialize the parent class */
	task_constructor (TASK(bodies), task_bodies_destructor,
	                                task_bodies_describe,
	                                task_bodies_run,
	                                group->server, TRUE,
	                                needs_socket);
	TASK(bodies)->gets_bodies = TRUE;
	TASK(bodies)->type = TASK_TYPE_BODIES;

	/* initialize the task-bodies */
	bodies->group = group;
	bodies->articles = pan_g_ptr_array_dup ((GPtrArray*)articles);
	bodies->working_index = 0;

	debug_exit ("task_bodies_new");
	return PAN_OBJECT(bodies);
}

/*****
******
*****/

static gchar*
task_bodies_describe (const StatusItem* si)
{
	TaskBodies *item = TASK_BODIES(si);

	return g_strdup_printf (_("Downloading %d articles from group %s"),
		item->articles->len,
		group_get_readable_name(item->group));
}

/*****
******
*****/

static int
task_bodies_run (Task* task)
{
	gint retval;
	gint steps = (int)TASK_BODIES(task)->articles->len;
	debug_enter ("task_bodies_run");

	if (!task->needs_socket)
	{
		retval = TASK_SUCCESS;
	}
	else
	{
		status_item_emit_init_steps (STATUS_ITEM(task), steps);

		retval = nntp_download_bodies (STATUS_ITEM(task),
					       TASK_BODIES(task)->group,
					       task->sock,
					       &task->hint_abort,
					       TASK_BODIES(task)->articles,
					       &TASK_BODIES(task)->working_index,
					       FALSE, FALSE);

		if (retval == TASK_SUCCESS) {
			log_add_va (LOG_INFO, ("Fetched %u bodies from `%s'"),
				steps,
				group_get_readable_name(TASK_BODIES(task)->group));
		}
	}

	debug_exit ("task_bodies_run");
	return retval;
}
