/*
 * Pan - A Newsreader for X
 * Copyright (C) 2002  Charles Kerr <charles@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>

#include <string.h>
#include <locale.h>

#include <glib.h>
#include <gtk/gtk.h>

#include <pan/base/pan-i18n.h>
#include <pan/base/pan-glib-extensions.h>

#include <pan/pan-charset-picker.h>

#define PAN_DEFAULT_CHARSET	"ISO-8859-1"

struct CharsetStruct {
    const char *charset, *name;
} charsets[] = {
	{"ISO-8859-4",   N_("Baltic")},
	{"ISO-8859-13",  N_("Baltic")},
	{"ISO-8859-2",   N_("Central European")},
	{"gb2312",       N_("Chinese Simplified")},
	{"big5",         N_("Chinese Traditional")},
	{"ISO-8859-5",   N_("Cyrillic")},
	{"windows-1251", N_("Cyrillic")},
	{"KOI8-R",       N_("Cyrillic")},
	{"KOI8-U",       N_("Cyrillic, Ukrainian")},
	{"ISO-8859-7",   N_("Greek")},
	{"ISO-2022-jp",  N_("Japanese")},
	{"euc-kr",       N_("Korean")},
	{"ISO-8859-9",   N_("Turkish")},
	{"ISO-8859-1",   N_("Western")},
	{"ISO-8859-15",  N_("Western, New")},
};



/**
 ** PUBLIC FUNCTIONS
 **/


GtkWidget *
pan_charset_picker_new (const char *charset)
{
	GtkWidget  * menu;
	GtkWidget  * w;
	gint         i;
	gint         sel_idx;
	const char * default_charset = NULL;

	if (!is_nonempty_string (charset)) 
		default_charset = get_charset_from_locale ();
	else	
		default_charset = charset;

	menu = gtk_menu_new ();
	sel_idx=-1;
	for (i=0; i<G_N_ELEMENTS(charsets); i++)
	{
		GtkWidget * menu_child;
		char        buf[512];

		g_snprintf (buf, sizeof buf, "%s (%s)", charsets[i].name, 
			charsets[i].charset);
		menu_child = gtk_menu_item_new_with_label (buf);
		g_object_set_data (G_OBJECT(menu_child), "charset", charsets+i);
		gtk_menu_shell_append (GTK_MENU_SHELL(menu), menu_child);
		if (g_strcasecmp (charsets[i].charset, default_charset) == 0)
			sel_idx = i;
	}

	w = gtk_option_menu_new ();
	gtk_option_menu_set_menu (GTK_OPTION_MENU(w), menu);
	if (sel_idx != -1)
		gtk_option_menu_set_history (GTK_OPTION_MENU(w), sel_idx);

	return w;
}

char *
pan_charset_picker_free (GtkWidget * w)
{
	return NULL;
}

const char *
pan_charset_picker_get_charset (GtkWidget * w)
{
	GtkWidget                  * menu;
	GtkWidget                  * menu_item;
	const struct CharsetStruct * charset;

	g_return_val_if_fail (w!=NULL, PAN_DEFAULT_CHARSET);

	menu = gtk_option_menu_get_menu (GTK_OPTION_MENU(w));
	menu_item = gtk_menu_get_active (GTK_MENU(menu));
	charset = (const struct CharsetStruct *) g_object_get_data (G_OBJECT(menu_item), "charset");

	return charset ? charset->charset : PAN_DEFAULT_CHARSET;
}
