/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999, 2000, 2001  Pan Development Team <pan@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>

#include <ctype.h>
#include <errno.h>
#include <stdio.h>
#include <string.h>
#include <locale.h>

#include <glib.h>
#include <gtk/gtk.h>

#include <gmime/gmime-charset.h>

#include <pan/base/argset.h>
#include <pan/base/debug.h>
#include <pan/base/gnksa.h>
#include <pan/base/log.h>
#include <pan/base/pan-i18n.h>
#include <pan/base/pan-glib-extensions.h>
#include <pan/base/serverlist.h>
#include <pan/base/util-file.h>
#include <pan/base/util-wrap.h>
#include <pan/base/pan-callback.h>

#include <pan/identities/identity-manager.h>

#include <pan/pan-charset-picker.h>
#include <pan/articlelist.h>
#include <pan/globals.h>
#include <pan/grouplist.h>
#include <pan/gui.h>
#include <pan/gui-headers.h>
#include <pan/message-check-ui.h>
#include <pan/message-send.h>
#include <pan/message-window.h>
#include <pan/prefs.h>
#include <pan/print.h>
#include <pan/queue.h>
#include <pan/task-body.h>
#include <pan/text.h> /* for colors of fg, bg, and quoted text */
#include <pan/util.h>

#include <pan/xpm/pan-pixbufs.h>

/* #define ATTACHMENTS */

typedef struct
{
	GtkWidget * window;

	GtkWidget * read_info_pane;

	GtkWidget * from_om;
	GtkWidget * custom_headers_text;
	GtkWidget * user_agent_tb;
	GtkWidget * organization;
	GtkWidget * charset_om;
	GtkWidget * newsgroups;
	GtkWidget * users;
	GtkWidget * subject;
        GtkWidget * followup_to;
        GtkWidget * reply_to;

        GtkWidget * do_wrap_togglebutton;

	GtkTextBuffer * body_buffer;
	GtkWidget * body_view;

	/* Attachment */
	GtkWidget * file_clist;
	GtkWidget * file_add_button;
	GtkWidget * file_remove_button;
	GtkWidget * attachment_lines_per_part_sb;
	GtkWidget * single_part_rb;
	GtkWidget * multi_part_rb;
	GtkWidget * encoding_mime_tb;
	GtkWidget * encoding_uuenc_tb;
	GtkWidget * encoding_yenc_tb;

	Article * article;
	ComposeType type;

	char * identity_name;
	char * last_attribution_text;
}
Compose;

extern GtkTooltips *ttips;

static void rot13_cb                       (gpointer, int, GtkWidget*);
static void wrap_cb                        (gpointer, int, GtkWidget*);
static void save_cb                        (gpointer, int, GtkWidget*);
static void send_now_cb                    (gpointer, int, GtkWidget*);
static void send_later_cb                  (gpointer, int, GtkWidget*);
static void message_window_save_cb         (gpointer, int, GtkWidget*);
static void message_window_cut_cb          (gpointer, int, GtkWidget*);
static void message_window_copy_cb         (gpointer, int, GtkWidget*);
static void message_window_paste_cb        (gpointer, int, GtkWidget*);
static void message_window_close           (gpointer, int, GtkWidget*);
static void compose_ext_editor_cb          (gpointer, int, GtkWidget*);
static void identities_changed_cb          (gpointer, gpointer, gpointer);

static void compose_ext_editor_cb2          (GtkWidget*, gpointer);
static void rot13_cb2                       (GtkWidget*, gpointer);
static void wrap_cb2                        (GtkWidget*, gpointer);
static void send_now_cb2                    (GtkWidget*, gpointer);
static void send_later_cb2                  (GtkWidget*, gpointer);

static GtkWidget * extra_headers_page      (Compose*);
static GtkWidget * create_body_pane_nolock (Compose*);

/***
****  SHUTDOWN
***/

static int
window_delete_event_cb (GtkWidget * w, GdkEvent * e, gpointer data)
{
	Compose * compose = (Compose*) data;
	gui_save_window_size (compose->window, "compose");
	return FALSE;
}

static void
window_destroy_cb (GtkWidget * window, Compose * compose)
{
	pan_callback_remove (identity_manager_get_identities_changed_callback(),
	                     identities_changed_cb, compose);

	if (compose->article != NULL)
		group_unref_articles (compose->article->group, NULL);
	g_free (compose->identity_name);
	g_free (compose->last_attribution_text);
	g_free (compose);
}

/***
****  EXPERIMENTAL WRAP CODE
***/

static gboolean experimental_toggle_active = TRUE;

static void
wrap_tb_toggled_cb (GtkToggleButton * togglebutton, gpointer user_data)
{
	experimental_toggle_active = gtk_toggle_button_get_active (togglebutton);
}


static void
text_inserted_cb (GtkTextBuffer   * text_buffer,
                  GtkTextIter     * insert_pos,
		  char            * text,
		  int               text_len,
		  gpointer          user_data)
{
	static gboolean dampen_feedback = FALSE;
	debug_enter ("text_inserted_cb");

	if (!dampen_feedback
		&& experimental_toggle_active
		&& wrap_column<gtk_text_iter_get_line_offset(insert_pos))
	{
		int text_len;
		int pos;
		int del_start_pos;
		int del_end_pos;
		char * text;
		char * pch;
		char * del_start_pch;
		GtkTextMark * mark;
		GtkTextIter line_pos;
		GtkTextIter del_start_iter;
		GtkTextIter del_end_iter;

		/* find start of line */
		line_pos = *insert_pos;
		gtk_text_iter_set_line (&line_pos, gtk_text_iter_get_line(insert_pos));

		/* get the contents of the line */
		text = gtk_text_buffer_get_text (text_buffer, &line_pos, insert_pos, FALSE);
		text_len = text ? strlen(text) : 0;

		/* try to find the last space before the wrap column */
		del_start_pos=pos=0;
		for (pch=text; *pch; pch=g_utf8_next_char (pch), pos++) {
			gunichar ch;
			if (pos > wrap_column)
				break;
			ch =  g_utf8_get_char (pch);
			if (g_unichar_isspace(ch)) {
				del_start_pos = del_end_pos = pos;
				del_start_pch = pch;
			}
		}

		/* if there's no place to break... */
		if (del_start_pos == 0)
			return;

		/* find the end of the whitespace */
		for (pch=del_start_pch; *pch; pch=g_utf8_next_char (pch), del_end_pos++) {
			gunichar ch = g_utf8_get_char (pch);
			if (!g_unichar_isspace(ch))
				break;
		}

		/* replace the space(s) with a linefeed */
		dampen_feedback = TRUE;
		mark = gtk_text_buffer_create_mark (text_buffer, "blah", insert_pos, FALSE);
		del_start_iter = del_end_iter = line_pos;
		gtk_text_iter_forward_chars (&del_start_iter, del_start_pos);
		gtk_text_iter_forward_chars (&del_end_iter, del_end_pos);
		gtk_text_buffer_delete (text_buffer, &del_start_iter, &del_end_iter);
		gtk_text_buffer_insert (text_buffer, &del_start_iter, "\n", 1);
		gtk_text_buffer_get_iter_at_mark (text_buffer, insert_pos, mark);
		gtk_text_buffer_delete_mark (text_buffer, mark);
		dampen_feedback = FALSE;

		g_free (text);
	}

	debug_exit ("text_inserted_cb");
}

static void
experimental_wrap_handler (GtkTextBuffer * buffer, GtkToggleButton * tb)
{
	pan_lock ();
	gtk_toggle_button_set_active (tb, experimental_toggle_active);
	g_signal_connect (buffer, "insert-text", G_CALLBACK(text_inserted_cb), NULL);
	g_signal_connect (tb, "toggled", G_CALLBACK(wrap_tb_toggled_cb), NULL);
	pan_unlock ();
}

/***
****
***/

static GtkItemFactoryEntry edit_menu_entries [] =
{
	{N_("/_File"), NULL, NULL, 0, "<Branch>"},
	{N_("/_File/_Save Changes"), "S", save_cb, 0, "<StockItem>", GTK_STOCK_SAVE},
	{N_("/_File/Save _As..."), "<control>S", message_window_save_cb, 0, "<StockItem>", GTK_STOCK_SAVE_AS},
	{N_("/_File/---"), NULL, NULL, 0, "<Separator>"},
	{N_("/_File/_Close"), "<control>W", message_window_close, 0, "<StockItem>", GTK_STOCK_CLOSE},
	{N_("/_Edit"), NULL, NULL, 0, "<Branch>"},
	{N_("/_Edit/Cu_t"), "<control>X", message_window_cut_cb, 0, "<StockItem>", GTK_STOCK_CUT},
	{N_("/_Edit/_Copy"), "<control>C", message_window_copy_cb, 0, "<StockItem>", GTK_STOCK_COPY},
	{N_("/_Edit/_Paste"), "<control>V", message_window_paste_cb, 0, "<StockItem>", GTK_STOCK_PASTE},
	{N_("/_Edit/---"), NULL, NULL, 0, "<Separator>"},
	{N_("/_Edit/_Rot13 Selected Text"), "<shift><control>R", rot13_cb, 0, "<StockItem>", GTK_STOCK_REFRESH},
	{N_("/_Edit/Edit with E_xternal Editor"), "<control>E", compose_ext_editor_cb, 0, "<StockItem>", GTK_STOCK_JUMP_TO}
};

static GtkItemFactoryEntry post_menu_entries [] =
{
	{N_("/_File/Send Now"), NULL, send_now_cb, 0, "<ImageItem>", icon_stock_mail_send},
	{N_("/_File/Send Later"), NULL, send_later_cb, 0, "<ImageItem>", icon_stock_timer},
	{N_("/_File/---"), NULL, NULL, 0, "<Separator>"},
	{N_("/_File/Save As..."), "<control>S", message_window_save_cb, 0, "<StockItem>", GTK_STOCK_SAVE_AS},
	{N_("/_File/---"), NULL, NULL, 0, "<Separator>"},
	{N_("/_File/_Close"), "<control>W", message_window_close, 0, "<StockItem>", GTK_STOCK_CLOSE},
	{N_("/_Edit"), NULL, NULL, 0, "<Branch>"},
	{N_("/_Edit/Cu_t"), "<control>X", message_window_cut_cb, 0, "<StockItem>", GTK_STOCK_CUT},
	{N_("/_Edit/_Copy"), "<control>C", message_window_copy_cb, 0, "<StockItem>", GTK_STOCK_COPY},
	{N_("/_Edit/_Paste"), "<control>V", message_window_paste_cb, 0, "<StockItem>", GTK_STOCK_PASTE},
	{N_("/_Edit/---"), NULL, NULL, 0, "<Separator>"},
	{N_("/_Edit/_Rot13 Selected Text"), "<shift><control>R", rot13_cb, 0, "<StockItem>", GTK_STOCK_REFRESH},
	{N_("/_Edit/Edit with E_xternal Editor"), "<control>E", compose_ext_editor_cb, 0, "<StockItem>", GTK_STOCK_JUMP_TO}
};

static char*
menu_translate (const char* path, gpointer data)
{
	return gettext (path);
}

static GtkWidget*
create_main_menu (Compose * compose)
{
	int entries_qty;
	GtkItemFactory * factory;
	GtkItemFactoryEntry * entries;
	GtkAccelGroup * accel_group;
	GtkWidget * menubar;

	/* sanity clause */
	g_return_val_if_fail (compose!=NULL, NULL);

	/* figure out which menu we're building */
	if (compose->type==EDIT_ORIGINAL) {
		entries = edit_menu_entries;
		entries_qty = G_N_ELEMENTS (edit_menu_entries);
	}
	else {
		entries = post_menu_entries;
		entries_qty = G_N_ELEMENTS (post_menu_entries);
	}

        /* build the menu */
        accel_group = gtk_accel_group_new ();
	gtk_window_add_accel_group (GTK_WINDOW(compose->window), accel_group);
	g_object_unref (accel_group);
	factory = gtk_item_factory_new (GTK_TYPE_MENU_BAR, "<compose>", accel_group);
	gtk_item_factory_set_translate_func (factory, menu_translate, NULL, NULL);
	gtk_item_factory_create_items (factory, entries_qty, entries, compose);
	menubar = gtk_item_factory_get_widget (factory, "<compose>");

	g_assert (GTK_IS_WIDGET(menubar));
	return menubar;
}

static GtkWidget*
create_image_from_inline (const guint8 * data)
{
	GtkWidget * image;
	GdkPixbuf * pixbuf;
	GtkIconSet * icon_set;

	pixbuf = gdk_pixbuf_new_from_inline (-1, data, FALSE, NULL);
	icon_set = gtk_icon_set_new_from_pixbuf (pixbuf);
	g_object_unref (G_OBJECT(pixbuf));
	image = gtk_image_new_from_icon_set (icon_set, GTK_ICON_SIZE_SMALL_TOOLBAR);
	gtk_icon_set_unref (icon_set);

	return image;
}

static GtkWidget*
create_toolbar (Compose * compose)
{
	GtkWidget * image;
	GtkWidget * toolbar;

	/* sanity clause */
	g_return_val_if_fail (compose!=NULL, NULL);

	toolbar = gtk_toolbar_new ();

	/* send now */
	image = create_image_from_inline (icon_stock_mail_send);
	gtk_toolbar_append_element (GTK_TOOLBAR(toolbar),
	                            GTK_TOOLBAR_CHILD_BUTTON, NULL,
	                            _("Send Now"), _("Send this Message Now"), NULL,
	                            image, G_CALLBACK(send_now_cb2), compose);

	/* send later */
	image = create_image_from_inline (icon_stock_timer);
	gtk_toolbar_append_element (GTK_TOOLBAR(toolbar),
	                            GTK_TOOLBAR_CHILD_BUTTON, NULL,
	                            _("Send Later"), _("Send this Message Later"), NULL,
	                            image, G_CALLBACK(send_later_cb2), compose);

	gtk_toolbar_insert_space (GTK_TOOLBAR(toolbar), -1);

	/* rewrap */
	image = gtk_image_new_from_stock (GTK_STOCK_JUSTIFY_FILL, GTK_ICON_SIZE_SMALL_TOOLBAR);
	gtk_toolbar_append_element (GTK_TOOLBAR(toolbar),
	                            GTK_TOOLBAR_CHILD_BUTTON, NULL,
	                            _("Rewrap"), _("Rewrap all the text in the Message Body"), NULL,
	                            image, G_CALLBACK(wrap_cb2), compose);


	/* line wrap togglebutton */
	image = gtk_image_new_from_stock (GTK_STOCK_JUSTIFY_FILL, GTK_ICON_SIZE_SMALL_TOOLBAR);
        compose->do_wrap_togglebutton =
	gtk_toolbar_append_element (GTK_TOOLBAR(toolbar),
	                            GTK_TOOLBAR_CHILD_TOGGLEBUTTON, NULL,
	                            _("Wrap Text"), _("Turn line wrap on/off"), NULL,
	                            image, NULL, NULL);

	gtk_toolbar_insert_space (GTK_TOOLBAR(toolbar), -1);

	/* rot13 */
	image = gtk_image_new_from_stock (GTK_STOCK_REFRESH, GTK_ICON_SIZE_SMALL_TOOLBAR);
	gtk_toolbar_append_element (GTK_TOOLBAR(toolbar),
	                            GTK_TOOLBAR_CHILD_BUTTON, NULL,
	                            _("Rot13"), _("Rot13 Selected Text"), NULL,
	                            image, G_CALLBACK(rot13_cb2), compose);

	/* external editor */
	image = gtk_image_new_from_stock (GTK_STOCK_JUMP_TO, GTK_ICON_SIZE_SMALL_TOOLBAR);
	gtk_toolbar_append_element (GTK_TOOLBAR(toolbar),
	                            GTK_TOOLBAR_CHILD_BUTTON, NULL,
	                            _("Editor"), _("Edit with an External Editor"), NULL,
	                            image, G_CALLBACK(compose_ext_editor_cb2), compose);

	return toolbar;
};


/**
 *  1. If the user selected an area, we use that.
 *  2. If the user didn't select an area, use the whole body.
 *  3. Quote fresh text with "> "
 *  4. Requote older quoted text with ">"
 *  5. Strip the signature if user didn't select it.
 *  6. Wrap everything.
 */
static char*
create_reply_body (Article * article)
{
	gboolean marked;
	GString * out = g_string_new (NULL);
	char * pch;

	/* body */
	pch = g_strdup (article_get_extra_header (article, PAN_REPLY_PORTION));
	marked = pch != NULL;
	if (pch == NULL)
		pch = article_get_body (article);
	g_strchomp (pch);
	if (is_nonempty_string(pch)) {
		char ** sd = g_strsplit (pch, "\n", -1);
		gint i;
		for (i=0; sd!=NULL && sd[i]!=NULL; ++i) {
			const char * l = sd[i];
			if (!marked && (!strcmp(l,"-- ") || !strcmp(l,"-- \r")))
				break;
			g_string_sprintfa (out,
				"%s%s\n", (*l=='>' ? ">" : "> "), l);
		}
		g_strfreev (sd);
	}
	g_free (pch);

	/* return the result */
	pch = out->str;
	g_string_free (out, FALSE);
	return pch;
}

/***
****
***/

static gboolean
is_posting (ComposeType type)
{
	return type==NNTP_POST
		|| type==NNTP_REPLY
		|| type==EMAIL_AND_POST_REPLY;
}

static char*
make_reply_string (const char* string)
{
	char * retval = NULL;

	if (string != NULL)
	{
		if (g_strncasecmp ("Re: ", string, 4))
			retval = g_strconcat ("Re: ", string, NULL);
		else
			retval = g_strdup (string);
	}

	return retval;
}

static void
rot13_cb (gpointer user_data, int action, GtkWidget * w)
{
	Compose * compose = (Compose*) user_data;
	GtkTextBuffer * buffer = compose->body_buffer;
	GtkTextIter start;
	GtkTextIter end;

	if (gtk_text_buffer_get_selection_bounds (buffer, &start, &end))
	{
		char * str = gtk_text_buffer_get_text (buffer, &start, &end, FALSE);
		rot13_inplace (str);
		gtk_text_buffer_delete (buffer, &start, &end);
		gtk_text_buffer_insert (buffer, &start, str, strlen(str));
	}
}
static void
rot13_cb2 (GtkWidget * w, gpointer user_data)
{
	rot13_cb (user_data, 0, w);
}

static void
wrap_cb (gpointer user_data, int action, GtkWidget * w)
{
	Compose * compose = (Compose*) user_data;
	GtkTextBuffer * buffer = compose->body_buffer;
	char * body;
	char * new_body;
	gboolean b;
	GtkTextIter start;
	GtkTextIter end;
	debug_enter ("wrap_cb");

	/* get the current body */
	gtk_text_buffer_get_bounds (buffer, &start, &end);
	body = gtk_text_buffer_get_text (buffer, &start, &end, FALSE);
	new_body = fill_body (body, wrap_column);

	/* turn off our own wrapping while we fill the body pane */
	b = experimental_toggle_active;
	experimental_toggle_active = FALSE;
	update_body_pane (compose->body_buffer, new_body, FALSE);
	experimental_toggle_active = b;

	/* cleanup */
	g_free (body);
	g_free (new_body);
	debug_exit ("wrap_cb");
}

static void
wrap_cb2 (GtkWidget * w, gpointer user_data)
{
	wrap_cb (user_data, 0, w);
}

/**
***
**/

static void
message_window_cut_cb (gpointer user_data, int action, GtkWidget * w)
{
	Compose * compose = (Compose*) user_data;
	GtkClipboard * clipboard = gtk_clipboard_get (GDK_NONE);
	gtk_text_buffer_cut_clipboard (compose->body_buffer, clipboard, TRUE);
}

static void
message_window_copy_cb (gpointer user_data, int action, GtkWidget * w)
{
	Compose * compose = (Compose*) user_data;
	GtkClipboard * clipboard = gtk_clipboard_get (GDK_NONE);
	gtk_text_buffer_copy_clipboard (compose->body_buffer, clipboard);
}

static void
message_window_paste_cb (gpointer user_data, int action, GtkWidget * w)
{
	Compose * compose = (Compose*) user_data;
	gtk_text_buffer_paste_clipboard (compose->body_buffer,
	                                 gtk_clipboard_get (GDK_NONE),
	                                 NULL, TRUE);
}

static void
message_window_destroy (Compose * compose)
{
	pan_lock();
	gtk_widget_destroy (compose->window);
	pan_unlock();
}

static void
message_window_close (gpointer user_data, int action, GtkWidget * widget)
{
	Compose * compose = (Compose*) user_data;
	message_window_destroy (compose);
}


static void
update_title_with_subject (GtkWidget * widget, Compose * compose)
{
	gtk_window_set_title (GTK_WINDOW(compose->window),
	                      gtk_entry_get_text (GTK_ENTRY(widget)));
}


/***
****
****   ATTACHMENTS
****
***/

typedef enum
{
	ENCODING_BASE64,
	ENCODING_UUENC,
	ENCODING_YENC
}
PanEncoding;

#if 0
static PanEncoding
get_encoding_type (Compose * compose)
{
	PanEncoding retval;

	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(compose->encoding_mime_tb)))
		retval = ENCODING_BASE64;
	else if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(compose->encoding_uuenc_tb)))
		retval = ENCODING_UUENC;
	else if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(compose->encoding_yenc_tb)))
		retval = ENCODING_YENC;
	else
		pan_warn_if_reached ();

	return retval;
}
#endif

static gulong
get_estimated_encoded_size (gulong raw_size, PanEncoding encoding)
{
	switch (encoding) {
		case ENCODING_BASE64: raw_size *= 1.33; break;
		case ENCODING_UUENC: raw_size *= 1.35; break;
		case ENCODING_YENC: raw_size *= 1.03; break;
	}
				    
	return raw_size;
}

static void
refresh_attachment_page (Compose * mw)
{
	size_t raw_size = 0;
	size_t lines_per_part = 0;
	GString * gstr = g_string_new (NULL);

	/* get lines-per-part */
	lines_per_part = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON(mw->attachment_lines_per_part_sb));

	/* get the unencoded size */
	if (1) {
		gint i;
		GtkCList * clist = GTK_CLIST(mw->file_clist);
		for (i=0; i<clist->rows; ++i) {
			const char * filename = gtk_clist_get_row_data (clist, i);
			raw_size += get_filesize (filename);
		}
	}

	/* update mime label */
	if (1) {
		gulong encoded_size = get_estimated_encoded_size (raw_size, ENCODING_BASE64);
		gulong lines = encoded_size / 45;
		gulong parts = (lines/lines_per_part) + 1;

		g_string_assign (gstr, _("Mime BASE64 Encoding - Single Articles Only "));
		if (lines!=0 && parts==1)
			g_string_sprintfa (gstr, _("(%lu lines in 1 article)"), lines);
		else if (lines!=0)
			g_string_sprintfa (gstr, _("(%lu lines in %lu articles)"), lines, parts);
		gtk_label_set_text (GTK_LABEL(GTK_BIN(mw->encoding_mime_tb)->child), gstr->str);

		if (parts>1 && gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(mw->encoding_mime_tb)))
			gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(mw->encoding_uuenc_tb), TRUE);
		gtk_widget_set_sensitive (mw->encoding_mime_tb, parts<2);
	}

	/* update uuenc label */
	if (1) {
		gulong encoded_size = get_estimated_encoded_size (raw_size, ENCODING_UUENC);
		gulong lines = encoded_size / 45;
		gulong parts = (lines/lines_per_part) + 1;

		g_string_assign (gstr, _("UUEncoding - Universally Accepted "));
		if (lines!=0 && parts==1)
			g_string_sprintfa (gstr, _("(%lu lines in 1 article)"), lines);
		else if (lines!=0)
			g_string_sprintfa (gstr, _("(%lu lines in %lu articles)"), lines, parts);

		gtk_label_set_text (GTK_LABEL(GTK_BIN(mw->encoding_uuenc_tb)->child), gstr->str);
	}

	/* update yenc label */
	if (1) {
		gulong encoded_size = get_estimated_encoded_size (raw_size, ENCODING_YENC);
		gulong lines = encoded_size / 45;
		gulong parts = (lines/lines_per_part) + 1;

		g_string_assign (gstr, _("Yenc - Smaller files, less universal "));
		if (lines!=0 && parts==1)
			g_string_sprintfa (gstr, _("(%lu lines in 1 article)"), lines);
		else if (lines!=0)
			g_string_sprintfa (gstr, _("(%lu lines in %lu articles)"), lines, parts);
		gtk_label_set_text (GTK_LABEL(GTK_BIN(mw->encoding_yenc_tb)->child), gstr->str);
	}
}

static void
attachments_lines_per_part_changed (GtkWidget * w, Compose * mw)
{
	refresh_attachment_page (mw);
}

static void
attachments_filesel_response_cb (GtkDialog * dialog, int response, gpointer user_data)
{
	if (response == GTK_RESPONSE_OK)
	{
		Compose * mw = (Compose*) user_data;
		int row;
		char * text[2];
		char line_buf[32];
		const char * selected_filename = gtk_file_selection_get_filename (GTK_FILE_SELECTION(dialog));
		GtkCList * clist = GTK_CLIST(mw->file_clist);

		/* insert the file */
		g_snprintf (line_buf, sizeof(line_buf), "%lu", (gulong)(get_filesize(selected_filename)/1024));
		text[0] = (char*) selected_filename;
		text[1] = line_buf;
		row = gtk_clist_insert (clist, -1, text);
		gtk_clist_set_row_data (clist, row, g_strdup(selected_filename));

		/* update the line counts */
		refresh_attachment_page (mw);
	}

	gtk_widget_destroy (GTK_WIDGET(dialog));
}

static void
attachment_add_button_clicked_cb (GtkButton * button, gpointer user_data)
{
	GtkWidget * w = gtk_file_selection_new(_("Select the file to attach."));
#if 0
	gtk_file_selection_set_show_dotfiles (GTK_FILE_SELECTION(w), TRUE);
#endif
	g_signal_connect (w, "response", G_CALLBACK(attachments_filesel_response_cb), user_data);
    	gtk_widget_show (w);
}

static void
attachment_remove_button_clicked_cb (GtkButton * button,
                                     gpointer user_data)
{
	GList * l;
	Compose * mw = (Compose*) user_data;
	GtkCList * clist = GTK_CLIST (mw->file_clist);

	l = clist->selection;
	if (l != NULL)
	{
		/* remove the file */
		gint row = GPOINTER_TO_INT (l->data);
		char * filename = gtk_clist_get_row_data (clist, row);
		g_free (filename);
		gtk_clist_remove (clist, row);

		/* update the line counts */
		refresh_attachment_page (mw);
	}
}

static void
attachment_clist_selection_changed_cb (GtkCList * clist,
                                       gint row,
                                       gint column,
                                       GdkEventButton * event,
                                       gpointer user_data)
{
	Compose * mw = (Compose*) user_data;
	gboolean has_selection = clist->selection != NULL;
	gtk_widget_set_sensitive (mw->file_remove_button, has_selection);
}

static GtkWidget *
attachment_page (Compose *mw)
{
	char * titles[2];
	GtkAdjustment * a;
	GtkWidget * w;
	GtkWidget * h;
	GtkWidget * v;
	GtkWidget * frame;
	GtkWidget * top;

	top = gtk_vbox_new (FALSE, GUI_PAD);


	/**
	***  Files Frame
	**/

	frame = gtk_frame_new (_("Files to Attach"));
	gtk_container_set_border_width (GTK_CONTAINER(frame), GUI_PAD_SMALL);

	v = gtk_vbox_new (FALSE, 0);
	gtk_container_set_border_width (GTK_CONTAINER(v), GUI_PAD);
	gtk_container_add (GTK_CONTAINER(frame), v);
	gtk_box_pack_start (GTK_BOX(top), frame, TRUE, TRUE, 0);

	/* Files clist */
	titles[0] = _("Filename");
	titles[1] = _("Kilobytes");
	mw->file_clist = w = gtk_clist_new_with_titles (2, titles);
	gtk_widget_set_usize (GTK_WIDGET(w), -1, 160);
	gtk_clist_set_column_width (GTK_CLIST(w), 0, 400);
	gtk_clist_set_column_width (GTK_CLIST(w), 1, 80);
	g_signal_connect (w, "select_row",
	                  G_CALLBACK(attachment_clist_selection_changed_cb), mw);
	g_signal_connect (w, "unselect_row",
	                  G_CALLBACK(attachment_clist_selection_changed_cb), mw);
        w = gtk_scrolled_window_new (NULL, NULL);
	//gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW(w), GTK_SHADOW_IN);
        gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW(w),
                GTK_POLICY_AUTOMATIC,
                GTK_POLICY_AUTOMATIC);
        gtk_container_add (GTK_CONTAINER(w), mw->file_clist);
	gtk_box_pack_start (GTK_BOX(v), w, TRUE, TRUE, GUI_PAD_SMALL);

	h = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX(v), h, FALSE, FALSE, 0);

	/* "Add" button */
	mw->file_add_button = w = gtk_button_new_from_stock (GTK_STOCK_ADD);
	gtk_button_set_relief (GTK_BUTTON(w), GTK_RELIEF_NONE);
	g_signal_connect (w, "clicked", G_CALLBACK(attachment_add_button_clicked_cb), mw);
	gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), w,
		_("Add a File to the Attachment List"), "");
	gtk_box_pack_start (GTK_BOX(h), w, FALSE, FALSE, 0);

	/* "Remove" button */
	mw->file_remove_button = w = gtk_button_new_from_stock (GTK_STOCK_CUT);
	gtk_widget_set_sensitive (w, FALSE);
	gtk_button_set_relief (GTK_BUTTON(w), GTK_RELIEF_NONE);
	g_signal_connect (w, "clicked", G_CALLBACK(attachment_remove_button_clicked_cb), mw);
	gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), w,
		_("Remove a File from the Attachment List"), "");
	gtk_box_pack_end (GTK_BOX(h), w, FALSE, FALSE, 0);

	/**
	***  Parts Frame
	**/

	frame = gtk_frame_new (_("Parts"));
	gtk_container_set_border_width (GTK_CONTAINER(frame), GUI_PAD_SMALL);
	gtk_box_pack_start (GTK_BOX(top), frame, FALSE, FALSE, 0);

	h = gtk_hbox_new (FALSE, GUI_PAD);
	gtk_container_set_border_width (GTK_CONTAINER(h), GUI_PAD_BIG);
	gtk_container_add (GTK_CONTAINER(frame), h);

	w = gtk_label_new (_("Lines Per Article:"));
	gtk_box_pack_start (GTK_BOX(h), w, FALSE, FALSE, 0);
	a = GTK_ADJUSTMENT(gtk_adjustment_new (5000, 100, 15000, 100, 10, 10));
	mw->attachment_lines_per_part_sb = w = gtk_spin_button_new (a, 0, 0);
 	g_signal_connect (a, "value_changed",
	                  G_CALLBACK(attachments_lines_per_part_changed), mw);
	gtk_widget_set_usize (GTK_WIDGET(w), 70, 0);
	gtk_box_pack_start (GTK_BOX(h), w, FALSE, FALSE, 0);


	/**
	***  Encoding Frame
	**/

	frame = gtk_frame_new (_("Estimated Article Size"));
	gtk_container_set_border_width (GTK_CONTAINER(frame), GUI_PAD_SMALL);
	gtk_box_pack_start (GTK_BOX(top), frame, FALSE, FALSE, 0);

	v = gtk_vbox_new (FALSE, 0);
	gtk_container_set_border_width (GTK_CONTAINER(v), GUI_PAD_BIG);
	gtk_container_add (GTK_CONTAINER(frame), v);

	w = gtk_radio_button_new_with_label (NULL, _("UUEncoded (universally accepted)"));
	mw->encoding_uuenc_tb = w;
	gtk_box_pack_start (GTK_BOX(v), w, FALSE, FALSE, 0);
	w = gtk_radio_button_new_with_label_from_widget (GTK_RADIO_BUTTON(w), _("Yenc (30 percent smaller than UUEnc but less universal)"));
	mw->encoding_yenc_tb = w;
	gtk_box_pack_start (GTK_BOX(v), w, FALSE, FALSE, 0);
	w = gtk_radio_button_new_with_label_from_widget (GTK_RADIO_BUTTON(w), _("Mime (single-part posts only)"));
	mw->encoding_mime_tb = w;
	gtk_box_pack_start (GTK_BOX(v), w, FALSE, FALSE, 0);


	refresh_attachment_page (mw);
	return top;
}


/***
****
****   EXTRA HEADERS
****
***/

static const gchar *
determine_charset_for_message (const Compose *mw)
{
	const gchar * charset = get_charset_from_locale ();
      
	if (mw && mw->article) {
		charset = article_get_extra_header (mw->article, PAN_CHARSET);

		if (charset == NULL) {
			charset = group_get_default_charset (mw->article->group);
		}

	}

	return charset;
}

static GtkWidget *
extra_headers_page (Compose * mw)
{
	GtkWidget * table;
	GtkWidget * scroll;
	GtkWidget * frame;
	GtkWidget * vbox;
	GtkWidget * w;
	GtkWidget * eventbox; 
	int row = 0;

	/**
	***  The top table: common headers
	**/

	table = gtk_table_new (5, 2, FALSE);
	gtk_table_set_row_spacings (GTK_TABLE(table), GUI_PAD_SMALL);
	gtk_table_set_col_spacings (GTK_TABLE(table), GUI_PAD_SMALL);

	w = gtk_label_new (_("<b>Followup-To:</b>"));
	gtk_label_set_use_markup (GTK_LABEL(w), TRUE);
	gtk_misc_set_alignment (GTK_MISC(w), 0.0, 0.5);
	eventbox = gtk_event_box_new ();
	gtk_container_add (GTK_CONTAINER(eventbox), w);
	gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), eventbox,
		_("The newsgroup or newsgroups where replies to "
		  "your posted message should go.  This is only "
		  "needed if it differs from the \"Post To Groups\" "
		  "header. "
		  "\nTo direct all replies to your email address, "
		  "use \"Followup-To: poster\""), "");
	gtk_table_attach (GTK_TABLE(table), eventbox, 0, 1, row, row+1,
			  GTK_FILL, GTK_FILL, GUI_PAD, 0);
	mw->followup_to = gtk_entry_new ();
	gtk_table_attach (GTK_TABLE(table), mw->followup_to,
			  1, 2, row, row+1,
			  GTK_FILL | GTK_EXPAND, GTK_FILL, GUI_PAD, 0);

	++row;

	w = gtk_label_new (_("<b>Reply-To:</b>"));
	gtk_label_set_use_markup (GTK_LABEL(w), TRUE);
	gtk_misc_set_alignment (GTK_MISC(w), 0.0, 0.5);
	eventbox = gtk_event_box_new ();
	gtk_container_add (GTK_CONTAINER(eventbox), w);
	gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), eventbox,
		_("The e-mail account where mail replies to "
		  "your posted message should go.  This is only "
		  "needed if it differs from the \"From\" "
		  "header."), "");
	gtk_table_attach (GTK_TABLE(table), eventbox,
			  0, 1, row, row+1,
			  GTK_FILL, GTK_FILL, GUI_PAD, 0);
	mw->reply_to = gtk_entry_new ();
	gtk_table_attach (GTK_TABLE(table), mw->reply_to,
			  1, 2, row, row+1,
			  GTK_FILL | GTK_EXPAND, GTK_FILL, GUI_PAD, 0);

	++row;

	w = gtk_label_new (_("<b>Organization:</b>"));
	gtk_label_set_use_markup (GTK_LABEL(w), TRUE);
	gtk_misc_set_alignment (GTK_MISC(w), 0.0, 0.5);
	eventbox = gtk_event_box_new ();
	gtk_container_add (GTK_CONTAINER(eventbox), w);
	gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), eventbox,
		_("The organization you're associated with."), "");
	gtk_table_attach (GTK_TABLE(table), eventbox,
			  0, 1, row, row+1,
			  GTK_FILL, GTK_FILL, GUI_PAD, 0);
	mw->organization = gtk_entry_new ();
	gtk_table_attach (GTK_TABLE(table), mw->organization,
			  1, 2, row, row+1,
			  GTK_FILL|GTK_EXPAND, GTK_FILL, GUI_PAD, 0);

	++row;

	w = gtk_label_new (_("<b>Charset:</b>"));
	gtk_label_set_use_markup (GTK_LABEL(w), TRUE);
	gtk_misc_set_alignment (GTK_MISC(w), 0.0, 0.5);
	gtk_table_attach (GTK_TABLE(table), w,
	                  0, 1, row, row+1,
	                  GTK_FILL, GTK_FILL, GUI_PAD, 0);
	mw->charset_om = pan_charset_picker_new (determine_charset_for_message(mw));
	gtk_table_attach (GTK_TABLE(table), mw->charset_om,
	                  1, 2, row, row+1,
	                  GTK_FILL|GTK_EXPAND, GTK_FILL, GUI_PAD, 0); 

	/**
	***  The extra headers edit field
	**/

	w = gtk_text_view_new ();

	gtk_text_view_set_wrap_mode (GTK_TEXT_VIEW(w), GTK_WRAP_NONE);
	gtk_text_view_set_editable (GTK_TEXT_VIEW(w), TRUE);
	mw->custom_headers_text = w;
	scroll = gtk_scrolled_window_new (NULL, NULL);
        gtk_container_set_border_width (GTK_CONTAINER(scroll), GUI_PAD_SMALL);
	gtk_container_add (GTK_CONTAINER(scroll), w);
        gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scroll),
                                        GTK_POLICY_AUTOMATIC,
                                        GTK_POLICY_AUTOMATIC);

	w = gtk_check_button_new_with_label (_("Don't add the \"User-Agent\" identification header"));
	mw->user_agent_tb = w;

	w = gtk_vbox_new (FALSE, 0);
	gtk_container_set_border_width (GTK_CONTAINER(w), GUI_PAD);
	gtk_box_pack_start (GTK_BOX(w), scroll, TRUE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX(w), mw->user_agent_tb, FALSE, TRUE, 0);

	frame = gtk_frame_new (_("Custom Headers"));
	gtk_container_set_border_width (GTK_CONTAINER(frame), GUI_PAD);
	gtk_container_add (GTK_CONTAINER(frame), w);

	/**
	***  Tie the top and bottom together
	**/

	vbox = gtk_vbox_new (FALSE, GUI_PAD);
	gtk_container_set_border_width (GTK_CONTAINER(vbox), GUI_PAD);
	gtk_box_pack_start (GTK_BOX(vbox), table, FALSE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX(vbox), frame, TRUE, TRUE, 0);
	return vbox;
}

/*--------------------------------------------------------------------
 * Newsgroups:
 * CC:
 * Subject:
 *--------------------------------------------------------------------*/
static GtkWidget*
create_post_info_pane (Compose *mw)
{
	GtkWidget * w;
	GtkWidget * main_page;
	GtkWidget * notebook;

	pan_lock();
	
	/*-----
	 * Headers Page
	 * ---- */
	
	main_page = gtk_table_new (5, 2, FALSE);
	gtk_container_set_border_width (GTK_CONTAINER(main_page), GUI_PAD);
	gtk_table_set_row_spacings (GTK_TABLE(main_page), GUI_PAD_SMALL);
	gtk_table_set_col_spacings (GTK_TABLE(main_page), GUI_PAD_SMALL);

	/* set the From: field */
	w = gtk_label_new ( _("<b>From:</b>") );
	gtk_label_set_use_markup (GTK_LABEL(w), TRUE);
	gtk_misc_set_alignment (GTK_MISC(w), 0.0, 0.5);
	gtk_table_attach (GTK_TABLE(main_page), w, 0, 1, 0, 1,
			  GTK_FILL, 0, GUI_PAD, 0);
	mw->from_om = gtk_option_menu_new ();
	gtk_table_attach (GTK_TABLE(main_page), mw->from_om, 1, 2, 0, 1,
			  GTK_FILL|GTK_EXPAND, 0, GUI_PAD, 0);

	/* set the subject of the new message */
	w = gtk_label_new ( _("<b>Subject:</b>") );
	gtk_label_set_use_markup (GTK_LABEL(w), TRUE);
	gtk_misc_set_alignment (GTK_MISC(w), 0.0, 0.5);
	gtk_table_attach (GTK_TABLE(main_page), w, 0, 1, 1, 2,
			  GTK_FILL, GTK_FILL, GUI_PAD, 0);
	mw->subject = gtk_entry_new ();
	gtk_table_attach (GTK_TABLE(main_page), mw->subject,
	                  1, 2, 1, 2,
			  GTK_FILL|GTK_EXPAND, 0, GUI_PAD, 0);
	g_signal_connect (mw->subject, "changed",
	                  G_CALLBACK(update_title_with_subject), mw);

	/* set the Post To Groups: */
	w = gtk_label_new ( _("<b>Post To Groups:</b>") );
	gtk_label_set_use_markup (GTK_LABEL(w), TRUE);
	gtk_misc_set_alignment (GTK_MISC(w), 0.0, 0.5);
	gtk_table_attach (GTK_TABLE(main_page), w,
	                  0, 1, 2, 3,
			  GTK_FILL, 0, GUI_PAD, 0);
	mw->newsgroups = gtk_entry_new ();
	gtk_table_attach (GTK_TABLE(main_page), mw->newsgroups,
	                  1, 2, 2, 3,
			  GTK_FILL|GTK_EXPAND, GTK_FILL, GUI_PAD, 0);


	/* set the Mail To: */
	w = gtk_label_new ( _("<b>Mail To:</b>") );
	gtk_label_set_use_markup (GTK_LABEL(w), TRUE);
	gtk_misc_set_alignment (GTK_MISC(w), 0.0, 0.5);
	gtk_table_attach (GTK_TABLE(main_page), w, 0, 1, 3, 4,
			  GTK_FILL, 0, GUI_PAD, 0);
	mw->users = gtk_entry_new();
	gtk_table_attach (GTK_TABLE(main_page), mw->users,
	                  1, 2, 3, 4,
			  GTK_FILL|GTK_EXPAND, GTK_FILL, GUI_PAD, 0);

	/* the body pane */
	w = create_body_pane_nolock (mw);
	gtk_table_attach (GTK_TABLE(main_page), w, 0, 2, 4, 5,
	                  GTK_FILL, GTK_EXPAND|GTK_FILL|GTK_SHRINK, GUI_PAD, 0);


	/*-----
	 * Fill the notebook
	 * ---- */
	
	notebook = gtk_notebook_new ();

        gtk_notebook_append_page (GTK_NOTEBOOK (notebook), main_page,
	                          gtk_label_new (_("Message")));
        gtk_notebook_append_page (GTK_NOTEBOOK (notebook), extra_headers_page(mw),
	                          gtk_label_new (_("More Headers")));
	gtk_notebook_append_page (GTK_NOTEBOOK (notebook), attachment_page(mw),
	                          gtk_label_new (_("Attachments")));

	pan_unlock();

	/* listen for changes to the identity manager */
	pan_callback_add (identity_manager_get_identities_changed_callback(),
		identities_changed_cb, mw);

	return notebook;
}

/***
****
**** IDENTITIES
****
***/

/*
 * Quick helper functions for controlling the info pane
 */

static void
clear_gtk_entry_if_same (GtkEntry * entry, const char * string)
{
	const char * field = gtk_entry_get_text (entry);

	if (!pan_strcmp (field, string))
		pan_gtk_entry_set_text (GTK_WIDGET(entry), "");
}

static void
set_gtk_entry_if_blank (GtkEntry * entry, const char * string)
{
	const char * field = gtk_entry_get_text (entry);

	if (is_nonempty_string(string) && !is_nonempty_string(field))
		pan_gtk_entry_set_text (GTK_WIDGET(entry), string);
}

/***
**** ATTRIBUTION LINE
***/

/*
 * Generates an attribution line
 */

static char*
article_gen_attribution_string (const Article * a, const char * attribution_line)
{
	char author[128];
	char * pch = NULL;
	char * retval = NULL;
	const char * cpch = NULL;
	GString * gstr = NULL;

	/* get the unsubstituted string */
	gstr = g_string_new (attribution_line);

	/* substitutions: message-id */
	cpch = article_get_message_id (a);
	pan_g_string_replace (gstr, "%i", cpch);

	/* substitutions: date */
	pch = a==NULL ? g_strdup("") : rfc822_date_generate(a->date);
	pan_g_string_replace (gstr, "%d", pch);
	g_free (pch);

	/* substitutions: author */
	if (1) {
		char author[512] = { '\0' };
		article_get_author_str (a, author, sizeof(author));
		pan_g_string_replace (gstr, "%a", author);
	}

	/* substitutions: author name */
	*author = '\0';
	if (a != NULL)
		article_get_short_author_str (a, author, sizeof(author));
	pan_g_string_replace (gstr, "%n", author);

	/* cleanup & return the result */
	retval = gstr->str;
	g_string_free (gstr, FALSE);

	return retval;
}

/*
 * Remove the attribution line 
 */

static void
remove_attribution_line (Compose * compose)
{
	g_return_if_fail (compose!=NULL);

	if (is_nonempty_string (compose->last_attribution_text))
	{
		char * text;
		char * index;
		GtkTextIter start;
		GtkTextIter end;

		/* get the text buffer contents */
		gtk_text_buffer_get_bounds (compose->body_buffer, &start, &end);
		text = gtk_text_buffer_get_text (compose->body_buffer, &start, &end, FALSE);

		/* if we find the attribution line, remove it */
		index = pan_strstr (text, compose->last_attribution_text);
		if (index) {
			index += 2+strlen (compose->last_attribution_text);
			update_body_pane (compose->body_buffer, index, FALSE);
		}

		replace_gstr (&compose->last_attribution_text, NULL);
		g_free (text);
	}
}

/*
 * Add the attribution line
 */

static void
prepend_attribution_line (Compose      * compose,
                          const char   * attribution_line)
{
	/* sanity clause */
	g_return_if_fail (compose!=NULL);
	g_return_if_fail (attribution_line!=NULL);

	if (compose->article!=NULL)
	{
		char * pch;
		char * text;
		GtkTextIter start;
		GtkTextIter end;

		/* get the text buffer contents */
		gtk_text_buffer_get_bounds (compose->body_buffer, &start, &end);
		text = gtk_text_buffer_get_text (compose->body_buffer, &start, &end, FALSE);

		/* add the attribution */
		pch = article_gen_attribution_string (compose->article, attribution_line);
		if (is_nonempty_string (pch))
		{
			char * out = g_strdup_printf ("%s\n\n%s", pch, text);

			update_body_pane (compose->body_buffer, out, FALSE);

			replace_gstr (&compose->last_attribution_text, pch);

			g_free (out);
		}

		g_free (text);
	}
}

/***
**** SIGNATURE
***/

static void
remove_signature (GtkTextBuffer * body_buffer)
{
	GtkTextIter start;
	GtkTextIter end;
	char * text;
	char * index;

	/* sanity clause */
	g_return_if_fail (body_buffer!=NULL);
	g_return_if_fail (GTK_IS_TEXT_BUFFER(body_buffer));

	/* get the text buffer */
	gtk_text_buffer_get_bounds (body_buffer, &start, &end);
	text = gtk_text_buffer_get_text (body_buffer, &start, &end, FALSE);

	if (!strncmp (text, "-- \r\n", 5) ||
	    !strncmp (text, "-- \n", 4))
		index = text;
	else
	{
		index = pan_strstr (text, "\n-- \n");
		if (index == NULL)
			index = pan_strstr (text, "\n-- \r"); 
	}	

	if (index)
	{
		*index = 0;
		update_body_pane (body_buffer, text, FALSE);
	}

	g_free (text);
}

static void
append_signature (GtkTextBuffer * body_buffer, const char *sig_file)
{
	char * sig = NULL;
	char * pch;

	/* sanity clause */
	g_return_if_fail (body_buffer!=NULL);
	g_return_if_fail (GTK_IS_TEXT_BUFFER(body_buffer));
	g_return_if_fail (sig_file!=NULL);

	/* since we're applying a signature, remove any signature anyway */
	remove_signature (body_buffer);

        /* load the signature */	
	pch = g_strdup (sig_file);
	g_strstrip (pch);
	if (is_nonempty_string (pch))
	{
		GError * err = NULL;

		if (*pch=='~')
			replace_gstr (&pch, g_strconcat (g_get_home_dir(), pch+1, NULL));


		if (!g_file_get_contents (pch, &sig, NULL, &err)) {
			log_add_va (LOG_ERROR, _("Couldn't read signature file '%s': %s"), pch, err->message);
			g_error_free (err);
		}
		else {
	
			gchar * freeme;

			pan_utf8ize (sig, -1, &freeme);
			if (freeme != NULL) {
				g_free (sig);
				sig = freeme;
			}
		}
	}
	g_free (pch);

	/* append the signature */
	if (sig!=NULL)
	{
		GtkTextIter start;
		GtkTextIter end;
		char * out;
		char * text;
		gboolean delimiter_exists;

		/* get the text buffer */
		gtk_text_buffer_get_bounds (body_buffer, &start, &end);
		text = gtk_text_buffer_get_text (body_buffer, &start, &end, FALSE);

		/* we insert "-- " if it's not already there */
		delimiter_exists = (
			    (!strncmp (sig, "-- \r\n", 5))
			 || (!strncmp (sig, "-- \n", 4))
			 || (pan_strstr(sig, "\n-- \r")!=NULL)
			 || (pan_strstr(sig, "\n-- \n")!=NULL));
		out = g_strdup_printf ("%s%s%s", text, 
			delimiter_exists ? "" : "\n-- \n", sig);

		update_body_pane (body_buffer, out, FALSE);

		g_free (out);
		g_free (sig);
		g_free (text);
	}
}

/***
**** CUSTOM HEADERS
****
****/

static void
remove_custom_headers (GtkWidget * header_view, const Identity * id)
{
	int             i;
	char          * text;
	char         ** lines;
	GString       * new_body;
	GtkTextIter     start;
	GtkTextIter     end;
	GtkTextBuffer * buffer;

	g_return_if_fail (GTK_IS_TEXT_VIEW(header_view));
	g_return_if_fail (id!=NULL);
	g_return_if_fail (id->custom_headers!=NULL);

	buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW(header_view));
	gtk_text_buffer_get_bounds (buffer, &start, &end);
	text = gtk_text_buffer_get_text (buffer, &start, &end, FALSE);
	lines = g_strsplit (text, "\n", -1);
       	new_body = g_string_new (NULL);
	for (i=0; lines != NULL && lines[i] != NULL; i++)
	{
		char * delimit;

		g_strstrip (lines[i]);
		delimit = strchr (lines[i], ':');

		if (delimit != NULL)
		{
			gint      h;
			gboolean  match;
			char   * n =  g_strndup (lines[i], delimit - lines[i]);
			char   * v = g_strdup (delimit + 1);

			g_strstrip (n);
			g_strstrip (v);

			/*
			 * If it's a custom header from this ID, 
			 * don't add it again.
			 */

			match = FALSE;
			for (h = 0; h < id->custom_headers->len; h++)
			{
				Header * header = g_ptr_array_index (id->custom_headers, h);

				if (!pan_strcmp (n, header->name))
				{
					match = TRUE;
					break;
				}
			}
			if (!match)
				g_string_sprintfa (new_body, "%s: %s\n", 
					n, v);

			g_free (v);
			g_free (n);
		}
	}

	/* rebuild text buffer */
	gtk_text_buffer_delete (buffer, &start, &end);
	gtk_text_buffer_insert (buffer, &start, new_body->str, new_body->len);

	/* cleanup */
	g_strfreev (lines);
	g_string_free (new_body, TRUE);
}

static void
add_custom_headers (GtkWidget * header_view, const Identity * id)
{
	int             i;
	char          * text;
	GString       * new_body;
	GtkTextIter     start;
	GtkTextIter     end;
	GtkTextBuffer * buffer;

	/* sanity clause */
	g_return_if_fail (GTK_IS_TEXT_VIEW(header_view));
	g_return_if_fail (id!=NULL);
	g_return_if_fail (id->custom_headers!=NULL);

	/* To avoid duplicates, delete them first */
	remove_custom_headers (header_view, id);

	/* get the text into a GString */
	buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW(header_view));
	gtk_text_buffer_get_bounds (buffer, &start, &end);
	text = gtk_text_buffer_get_text (buffer, &start, &end, FALSE);
	new_body = g_string_new (text);

	/* append the custom headers */
	for (i=0; i<id->custom_headers->len; i++) {
		Header * h = g_ptr_array_index (id->custom_headers, i);
		g_string_sprintfa (new_body, "%s: %s\n", h->name, h->value);
	}

	/* rebuild text buffer */
	gtk_text_buffer_delete (buffer, &start, &end);
	gtk_text_buffer_insert (buffer, &start, new_body->str, new_body->len);

	/* cleanup */
	g_free (text);
	g_string_free (new_body, TRUE);
}

/***
**** SETTING IDENTITIES 
***/

static void
revoke_identity (Compose * mw, const Identity * id)
{
	/* sanity clause */
	g_return_if_fail (mw!=NULL);
	g_return_if_fail (id!=NULL);

	clear_gtk_entry_if_same (GTK_ENTRY(mw->organization), id->organization);

	clear_gtk_entry_if_same (GTK_ENTRY(mw->reply_to), id->reply_to);

	remove_attribution_line (mw);

	if (is_nonempty_string (id->signature))
		remove_signature (mw->body_buffer);

	remove_custom_headers (mw->custom_headers_text, id);
}

static void
apply_identity (Compose * mw, const Identity * id)
{
	g_return_if_fail (mw!=NULL);
	g_return_if_fail (id!=NULL);

	set_gtk_entry_if_blank (GTK_ENTRY(mw->organization), id->organization);

	set_gtk_entry_if_blank (GTK_ENTRY(mw->reply_to), id->reply_to);

	if (is_nonempty_string (id->attribution) && 
	    mw->type != NNTP_POST && mw->type != EDIT_ORIGINAL)
		prepend_attribution_line (mw, id->attribution);

	if (is_nonempty_string (id->signature))
		append_signature (mw->body_buffer, id->signature);

	add_custom_headers (mw->custom_headers_text, id);
}


static void
set_identity_cb (GtkMenuItem * item, gpointer data)
{
	Compose * mw = (Compose *) data;
	char          * new_id_name;
	Identity      * old_id;
	Identity      * new_id;

	g_return_if_fail (item);
	g_return_if_fail (data);

	new_id_name = (char *) gtk_object_get_data (GTK_OBJECT(item),
		"identity_name");

	new_id = identity_manager_get_identity (new_id_name);
	old_id = identity_manager_get_identity (mw->identity_name);

	if (old_id!=NULL)
	{
		revoke_identity (mw, old_id);
		pan_object_unref (PAN_OBJECT(old_id));
	}

	if (new_id!=NULL)
	{
		apply_identity (mw, new_id);
		replace_gstr (&mw->identity_name, g_strdup(new_id_name));
		pan_object_unref (PAN_OBJECT(new_id));
	}
}

static char *
determine_default_identity (Compose *mw)
{
	char       * name = NULL;
	Identity    * id = NULL;

	g_return_val_if_fail (mw!=NULL, NULL);

	/* If we're replying/forwarding via email, use that identity */
	if (mw->type == EMAIL_REPLY ||
	    mw->type == EMAIL_FORWARD)
	{
		id = identity_manager_get_default (ID_MAIL_DEFAULT);
	}

	if (id == NULL)
	{
		/* Check we've already posted with an identity */
		Group * g = mw->article ? mw->article->group :
			                  articlelist_get_group ();

		if (g != NULL && is_nonempty_string(g->identity_name))
		{
			id = identity_manager_get_identity (g->identity_name);
		}
	}

	/*
	 * If it's a new message, see if there's already an author specified.
	 * If so, use the identity with the same name/address.
	 *
	 * This is mainly here to find the right identity for feedback email.
	 *
	 * Note: this isn't bullet proof: several ids could have a matching
	 * author and email addresses. Would be better if we could 
	 * distinguish whether we have are writing an email or an article.
	 */

	if (id == NULL && mw->type == EDIT_ORIGINAL && mw->article != NULL)
	{
		id = identity_manager_get_identity_by_author (mw->article->author_real, mw->article->author_addr);
	}

	/* Still nothing. Just use the default identity for news */
	if (id == NULL)
	{
		id = identity_manager_get_default (ID_NEWS_DEFAULT);
	}

	if (id != NULL)
	{
		name = g_strdup (id->name);
		pan_object_unref (PAN_OBJECT(id));
	}

	return name;
}

static void
populate_identity_menu (Compose * mw, const char * id_name)
{
	gint        i;
	gint        index = 0;
	GtkWidget * menu = gtk_menu_new ();
	GPtrArray * identities = g_ptr_array_new ();
	const Identity  * id = NULL;

	/* get all identities */
	identity_manager_get_identities (identities);

	for (i=0; i<identities->len; ++i)
	{
		GtkWidget * item;
		char buf[512];
		gboolean have_addr;
		gboolean have_name;

		id = IDENTITY(g_ptr_array_index(identities,i));

		/* get default id */
		if (!pan_strcmp (id->name, id_name) || i == 0)
			index = i;

		have_addr = is_nonempty_string (id->author_addr);
		have_name = is_nonempty_string (id->author_real);

	        if (have_addr && have_name)
			g_snprintf (buf, sizeof(buf), "\"%s\" <%s>", id->author_real, id->author_addr);
		else if (have_addr)
			g_snprintf (buf, sizeof(buf), "%s", id->author_addr);
		else if (have_name)
			g_snprintf (buf, sizeof(buf), "%s", id->author_real);


		item = gtk_menu_item_new_with_label (buf);
		gtk_object_set_data_full (GTK_OBJECT(item), "identity_name", g_strdup(id->name), g_free);
		g_signal_connect (item, "activate", G_CALLBACK(set_identity_cb), mw);
		gtk_widget_show (item);
		gtk_menu_append (GTK_MENU(menu), item);
	}

	gtk_option_menu_set_menu (GTK_OPTION_MENU(mw->from_om), menu);
	gtk_option_menu_set_history (GTK_OPTION_MENU(mw->from_om), index);
	gtk_widget_show_all (GTK_WIDGET(mw->from_om));

	/* remember the id name */
	id = IDENTITY(g_ptr_array_index(identities, index));
	replace_gstr (&mw->identity_name, g_strdup(id->name));

	pan_g_ptr_array_foreach (identities, (GFunc)pan_object_unref, NULL);
	g_ptr_array_free (identities, TRUE);
}


static void 
identities_changed_cb (gpointer a, gpointer b, gpointer c)
{
	Compose * mw = (Compose *) c;

	populate_identity_menu (mw, mw->identity_name);

}

static void
populate_post_info_pane (Compose *mw)
{
	char * p;
	const char * cp;
	Article * a;
	char * default_id_name;

	g_return_if_fail (mw != NULL);

	a = mw->article;

	/**
	***  Populate the extra headers
	**/

	{
		GString * gstr = g_string_new (NULL);

		/* start with "X-Comment-To:" header for Fido users. */
		if (1) {
			char author[512] = { '\0' };
			article_get_author_str (a, author, sizeof(author));
			g_string_sprintfa (gstr, "X-Comment-To: %s\n", *author ? author : "ALL");
		}

		/* add all the extra headers. */
		if (a!=NULL && group_is_folder(a->group)) {
			guint i;
			GPtrArray * h = article_get_all_headers (a);
			for (i=0; i!=h->len; i+=2) {
				char* key = (char*)g_ptr_array_index(h,i);
				char* val = (char*)g_ptr_array_index(h,i+1);
				if (article_header_is_extra (key))
					g_string_sprintfa (gstr, "%s: %s\n", key, val);
			}
			g_ptr_array_free (h, TRUE);
		}

		/* add the headers */
		if (1) {
			GtkTextView * view = GTK_TEXT_VIEW(mw->custom_headers_text);
			GtkTextBuffer * buffer = gtk_text_view_get_buffer (view);
			GtkTextIter start;
			GtkTextIter end;
			char * freeme;
			const char * text = pan_utf8ize (gstr->str, gstr->len, &freeme);
			gtk_text_buffer_get_bounds (buffer, &start, &end);
			gtk_text_buffer_delete (buffer, &start, &end);
			gtk_text_buffer_insert (buffer, &start, text, g_utf8_strlen(text,-1));
			g_free (freeme);
		}

		/* cleanup */
		g_string_free (gstr, TRUE);
	}

	/**
	***  Subject
	**/

	cp = a != NULL ? a->subject : NULL;
	p = NULL;
	if (cp != NULL) {
		if (mw->type == EDIT_ORIGINAL)
			p = g_strdup (cp);
		else if (mw->type == EMAIL_FORWARD)
			p = g_strdup_printf ("[%s] %s", a->group->name, cp);
		else
			p = make_reply_string (cp);
	}
	if (is_nonempty_string(p))
		pan_gtk_entry_set_text (mw->subject, p);
	g_free (p);


	/**
	***  Populate the other pieces
	**/

	if (mw->type == EDIT_ORIGINAL)
	{
		char * pch;
		const char * header;

		/* newsgroups */
		g_assert (mw->newsgroups!=NULL);
		header = article_get_extra_header (a, HEADER_NEWSGROUPS);
		if (is_nonempty_string(header))
			pan_gtk_entry_set_text (mw->newsgroups, header);

		/* mail-to */
		g_assert (mw->users!=NULL);
		header = article_get_extra_header (a, PAN_MAIL_TO);
		if (is_nonempty_string(header))
			pan_gtk_entry_set_text (mw->users, header);

		/* followup-to */
		g_assert (mw->followup_to!=NULL);
		header = article_get_extra_header (a, HEADER_FOLLOWUP_TO);
		if (is_nonempty_string(header))
			pan_gtk_entry_set_text (mw->followup_to, header);

		/* editor */
		g_assert (mw->body_view!=NULL);
		pch = article_get_body (a);
		update_body_pane (mw->body_buffer, pch, FALSE);
		g_free (pch);

	}
	else /* a reply */
	{
		char * p;
		gboolean default_newsgroups = TRUE;

		/**
		***  BODY
		**/

		g_assert (mw->body_view!=NULL);
		if (a != NULL) {
			p = create_reply_body (a);
			update_body_pane (mw->body_buffer, p, FALSE);
			g_free (p);
		} 

		/**
		***   NEWSGROUPS
		***   MAILTO
		**/

		p = NULL;

		/* are we posting an article? */
		if (a!=NULL && is_posting(mw->type))
		{
			const char * followup = article_get_extra_header (a, HEADER_FOLLOWUP_TO);
			const char * newsgroups = article_get_extra_header (a, HEADER_NEWSGROUPS);

			/* if user set followup-to, use that;
			 * otherwise, use newsgroups;
			 * otherwise, use the currently-selected group */
			if (is_nonempty_string(followup))
			{
				default_newsgroups = FALSE;

				/* Followup-To: poster */
				if (!g_strcasecmp ("poster", followup)) /* explicit reply by mail */
				{
					GtkWidget * w;
					const char * replyto = article_get_extra_header (a, HEADER_REPLY_TO);

					w = gtk_message_dialog_new (GTK_WINDOW(mw->window),
					                            GTK_DIALOG_DESTROY_WITH_PARENT,
					                            GTK_MESSAGE_INFO,
					                            GTK_BUTTONS_CLOSE,
					                            _("``Followup-To: poster'': sending email to author."));
					g_signal_connect_swapped (GTK_OBJECT(w), "response",
					                          G_CALLBACK (gtk_widget_destroy), GTK_OBJECT(w));
					gtk_widget_show_all (w);

					if (is_nonempty_string(replyto))
						pan_gtk_entry_set_text (mw->users, replyto);
					else {
						char author[512];
						article_get_author_str (a, author, sizeof(author));
						pan_gtk_entry_set_text (mw->users, author);
					}
				}
				/* Does Followup-To: contain a valid email address?
				   Son-of-1036 says this is invalid, but what the hell. */
				else if (gnksa_check_from (followup, FALSE) == GNKSA_OK)
				{
					GtkWidget * w;

					w = gtk_message_dialog_new (GTK_WINDOW(mw->window),
					                            GTK_DIALOG_DESTROY_WITH_PARENT,
					                            GTK_MESSAGE_INFO,
					                            GTK_BUTTONS_CLOSE,
					                            _("``Followup-To:'' contains an email address: sending email to author."));
					g_signal_connect_swapped (GTK_OBJECT(w), "response",
					                          G_CALLBACK (gtk_widget_destroy), GTK_OBJECT(w));
					gtk_widget_show_all (w);

					pan_gtk_entry_set_text (mw->users, followup);
				}
				else /* explicit followup groups */
				{
					pan_gtk_entry_set_text (mw->newsgroups, followup);
				}
			}
			else if (is_nonempty_string(newsgroups)) /* explicit groups */
			{
				pan_gtk_entry_set_text (mw->newsgroups, newsgroups);
				default_newsgroups = FALSE;
			}
		}

		/* are we sending mail? */
		if (mw->type==EMAIL_REPLY || mw->type==EMAIL_AND_POST_REPLY)
		{
			/* figure out who to send mail to by checking reply-to: and from: */
			if (a!=NULL)
			{
				const char * replyto = article_get_extra_header (a, HEADER_REPLY_TO);
				if (is_nonempty_string(replyto))
					pan_gtk_entry_set_text (mw->users, replyto);
				else {
					char author[512];
					article_get_author_str (a, author, sizeof(author));
					pan_gtk_entry_set_text (mw->users, author);
				}
			}
		}

		/* no explicit newsgroup specified for this post,
		 * so let's guess where the user might want to post */
		if (default_newsgroups && is_posting(mw->type))
		{
			GString * str = g_string_new (NULL);

			if (a!=NULL && a->group!=NULL)
				g_string_sprintfa (str, "%s,", a->group->name);
			else {
				gint i;
				Group * thread_group = articlelist_get_group ();
				GPtrArray * ggroups = grouplist_get_selected_groups ();
				if (ggroups->len<2 && thread_group!=NULL)
					g_string_sprintfa (str, "%s,", thread_group->name);
				else for (i=0; i<ggroups->len; ++i) {
					Group * g = GROUP(g_ptr_array_index(ggroups,i));
					if (g!=NULL && !group_is_folder(g))
						g_string_sprintfa (str, "%s,", g->name);
				}
				g_ptr_array_free (ggroups, TRUE);
			}

			if (str->len != 0) {
				g_string_truncate (str, str->len-1); /* zotz last comma */
				pan_gtk_entry_set_text (mw->newsgroups, str->str);
			}

			g_string_free (str, TRUE);
		}
	}

	/* populate the identity menu, setting a default id */
	default_id_name = determine_default_identity (mw);
	populate_identity_menu (mw, default_id_name);
	g_free (default_id_name);

	if (is_nonempty_string (mw->identity_name))
	{
		Identity * id = identity_manager_get_identity (mw->identity_name);

		if (id)
		{
			apply_identity (mw, id);
			pan_object_unref (PAN_OBJECT (id));
		}
	}
}


static GtkWidget*
create_body_pane_nolock (Compose * mw)
{
	GtkTextBuffer * buf;
	GtkWidget * scrolled_window;
	GtkStyle * style;

	g_return_val_if_fail (mw!=NULL, NULL);

	mw->body_view = gtk_text_view_new ();
	mw->body_buffer = buf = gtk_text_view_get_buffer (GTK_TEXT_VIEW(mw->body_view));
	gtk_text_buffer_create_tag (buf, "url", "underline", PANGO_UNDERLINE_SINGLE, "foreground_gdk", &text_url_color, NULL);
	gtk_text_buffer_create_tag (buf, "quote_0", "foreground_gdk", &text_fg_color, NULL);
	gtk_text_buffer_create_tag (buf, "quote_1", "foreground_gdk", &text_fg_color, NULL);
	gtk_text_buffer_create_tag (buf, "quote_2", "foreground_gdk", &text_fg_color, NULL);
	gtk_text_buffer_create_tag (buf, "quote_3", "foreground_gdk", &text_fg_color, NULL);

	/* make the compose window look like the text window, except force a monospace font */
	style = gtk_style_copy (gtk_widget_get_style (Pan.text));
	pango_font_description_free (style->font_desc);
	style->font_desc = pango_font_description_from_string (message_body_font_fixed);
	gtk_widget_set_style (GTK_WIDGET(mw->body_view), style);
	g_object_unref (G_OBJECT(style));

	gtk_text_view_set_wrap_mode (GTK_TEXT_VIEW(mw->body_view), GTK_WRAP_NONE);
	gtk_text_view_set_editable (GTK_TEXT_VIEW(mw->body_view), TRUE);
	scrolled_window = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW(scrolled_window), GTK_SHADOW_IN);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_window),
					GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_container_add (GTK_CONTAINER (scrolled_window), mw->body_view);

	return scrolled_window;
}


void
message_post_window (void)
{
	message_window_new (NULL, NNTP_POST);
}

static void
message_post_window_create (Compose * compose)
{
	GtkWidget * w;
	GtkWidget * v;

	pan_lock();
	compose->window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
	gtk_window_set_title (GTK_WINDOW(compose->window), _("New Message"));
	v = gtk_vbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER(compose->window), v);

	gtk_box_pack_start (GTK_BOX(v), create_main_menu(compose), FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX(v), create_toolbar(compose), FALSE, FALSE, 0);
	pan_unlock();

	w = create_post_info_pane (compose);
	populate_post_info_pane (compose);

	experimental_wrap_handler (compose->body_buffer,
	                           GTK_TOGGLE_BUTTON(compose->do_wrap_togglebutton));

	pan_lock ();
	gtk_box_pack_start (GTK_BOX(v), w, TRUE, TRUE, 0);
	pan_unlock();
}

static void
make_reply_window (Article * article, ComposeType type)
{
	char       * body;
	const char * charset;

	g_return_if_fail (article_is_valid(article));
	g_return_if_fail (type==EMAIL_REPLY
		|| type==NNTP_REPLY
		|| type==EMAIL_AND_POST_REPLY
		|| type==EMAIL_FORWARD);

	/* if this isn't the same article in the article reading window,
	 * then fire up a task to retreive the body from the local cache
	 * or on the net.  FIXME task-body really needs to be decoupled
	 * from viewing.
	 */
	if (article != get_current_article()) {
		text_set_from_article (article);
		return;
	}

	/* open up a populated reply window. */
	body = text_get_message_to_reply_to ();
	article_set_header (article, PAN_REPLY_PORTION, body, DO_CHUNK);
	charset = group_get_default_charset (article->group);
	article_set_header (article, PAN_CHARSET, charset, DO_CHUNK);
	message_window_new (article, type);
	g_free (body);
}

void
message_forward_window (void)
{
	Article * a = get_current_article();
	if (a != NULL)
		make_reply_window (a, EMAIL_FORWARD);
}

void
message_edit_window (Article * a)
{
	if (a == NULL)
		a = get_current_article();
	if (a != NULL)
		message_window_new (a, EDIT_ORIGINAL);
}

void
message_reply_window (void)
{
	Article * a = get_current_article();
	if (a != NULL)
		make_reply_window (a, EMAIL_REPLY);
}

void
message_followup_reply_window (void)
{
	Article * a = get_current_article();
	if (a != NULL)
		make_reply_window (a, EMAIL_AND_POST_REPLY);
}


void
message_followup_window (void)
{
	Article * a = get_current_article();
	if (a != NULL)
		make_reply_window (a, NNTP_REPLY);
}

static void
message_edit_window_create (Compose * compose)
{
	GtkWidget * w;
	GtkWidget * v;

	pan_lock ();
	compose->window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
	gtk_window_set_title (GTK_WINDOW(compose->window), article_get_subject(compose->article));
	v = gtk_vbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER(compose->window), v);

	gtk_box_pack_start (GTK_BOX(v), create_main_menu(compose), FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX(v), create_toolbar(compose), FALSE, FALSE, 0);
	pan_unlock ();

	w = create_post_info_pane (compose);
	populate_post_info_pane (compose);

	experimental_wrap_handler (compose->body_buffer,
	                           GTK_TOGGLE_BUTTON(compose->do_wrap_togglebutton));

	pan_lock ();
	gtk_box_pack_start (GTK_BOX(v), w, TRUE, TRUE, 0);
	gtk_text_view_set_editable (GTK_TEXT_VIEW(compose->body_view), TRUE);
	pan_unlock();
}
static void
message_reply_window_create (Compose * compose)
{
	GtkWidget * w;
	GtkWidget * v;
	char * title = make_reply_string (article_get_subject(compose->article));

	pan_lock();
	compose->window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
	gtk_window_set_title (GTK_WINDOW(compose->window), title);
	v = gtk_vbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER(compose->window), v);
	gtk_box_pack_start (GTK_BOX(v), create_main_menu(compose), FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX(v), create_toolbar(compose), FALSE, FALSE, 0);
	pan_unlock();

	w = create_post_info_pane (compose);
	populate_post_info_pane (compose);

	experimental_wrap_handler (compose->body_buffer,
	                           GTK_TOGGLE_BUTTON(compose->do_wrap_togglebutton));

	pan_lock ();
	gtk_box_pack_start (GTK_BOX(v), w, TRUE, TRUE, 0);
	pan_unlock();

	/* cleanup */
	g_free (title);
}

void
message_window_new (Article * article, ComposeType type)
{
	Compose *mw;

	if (article != NULL)
		group_ref_articles (article->group, NULL);

	mw = g_new0 (Compose, 1);
	mw->article = article;
	mw->type = type;

	switch (type)
	{
		case NNTP_POST:
			message_post_window_create (mw);
			break;

		case NNTP_REPLY:
		case EMAIL_REPLY:
		case EMAIL_AND_POST_REPLY:
		case EMAIL_FORWARD:
			message_reply_window_create (mw);
			break;

		case EDIT_ORIGINAL:
			message_edit_window_create (mw);
			break;

		default:
			pan_warn_if_reached ();
			break;
	}

	if (mw->window != NULL)
	{
		pan_lock ();
		g_signal_connect (mw->window, "delete_event",
		                  G_CALLBACK(window_delete_event_cb), mw);
		g_signal_connect (mw->window, "destroy",
		                  G_CALLBACK(window_destroy_cb), mw);
		gtk_window_set_policy (GTK_WINDOW(mw->window), TRUE, TRUE, TRUE);
		gtk_widget_grab_focus (mw->subject);
		pan_unlock ();

		if (!gui_restore_window_size (mw->window, "compose"))
			gtk_window_set_default_size (GTK_WINDOW(mw->window), 650, 575);

		pan_lock ();
		gtk_widget_show_all (mw->window);
		pan_unlock ();
	}
}

/***
****
****  SENDING THE MESSAGE
****
***/

static char*
get_text_from_editable (GtkWidget* w)
{
	char * pch;
	g_return_val_if_fail (GTK_IS_EDITABLE(w), NULL);
	pch = gtk_editable_get_chars (GTK_EDITABLE(w), 0, -1);
	g_strstrip (pch);
	return pch;
}

static char*
get_text_from_maybe_editable (GtkWidget * w)
{
	char * pch = NULL;

	if (w!=NULL && GTK_IS_EDITABLE(w))
	{
		pch = gtk_editable_get_chars (GTK_EDITABLE(w), 0, -1);
		g_strstrip (pch);
	}

	return pch;
}

static char*
message_window_get_reply_to (const Compose* mw)
{
	char * pch;

	g_return_val_if_fail (mw!=NULL, NULL);

	pch = get_text_from_maybe_editable (mw->reply_to);
	if (pch == NULL)
		pch = g_strdup ("");

	return pch;
}

static char*
message_window_get_group_str (GtkWidget * e)
{
	char * pch;

	g_return_val_if_fail (e!=NULL, NULL);

	pch = get_text_from_maybe_editable (e);
	if (pch == NULL)
	{
		pch = g_strdup ("");
	}
	else
	{
		char * tmp;

		/* ensure all the delimiters are commas */
		g_strdelimit (pch, ":; ", ',');

		/* remove leading/trailing commas */
		for (tmp=pch; *tmp==','; ++tmp);
		g_memmove (pch, tmp, strlen(tmp)+1);
		for (tmp=pch+strlen(pch)-1; tmp>pch && *tmp==','; --tmp);
		tmp[1] = '\0';

		/* remove empty entries */
		while ((tmp=pan_strstr(pch,",,")) != NULL)
			g_memmove (tmp, tmp+1, strlen(tmp+1)+1);
	}

	return pch;
}

static char*
message_window_get_organization (const Compose* mw)
{
	char * pch = NULL;
	g_return_val_if_fail (mw!=NULL, NULL);
	pch = get_text_from_maybe_editable (mw->organization);
	if (pch == NULL)
		pch = g_strdup ("");
	if (pch != NULL)
		g_strstrip (pch);
	return pch;
}

static char*
message_window_get_reverse_path (const Compose* mw)
{
	Identity * id;
	char    *  from;
	char    *  retval = NULL;
	int        status;
	char       address[512];
	char       realname[512];

	g_return_val_if_fail (mw!=NULL, NULL);
	g_return_val_if_fail (is_nonempty_string (mw->identity_name), NULL);

	id = identity_manager_get_identity (mw->identity_name);
	from = g_strdup_printf ("\"%s\" <%s>", id->author_real, id->author_addr);
	pan_object_unref (PAN_OBJECT(id));

	status = gnksa_do_check_from (from,
	                              address, sizeof(address),
	                              realname, sizeof(realname),
				      FALSE);

	if (status == GNKSA_OK)
		retval = g_strdup (address);

	g_free (from);

	return retval;
}


static void
populate_article_from_mw (Article * article, const Compose * mw)
{
	gint i;
	char ** sd;
	char * pch;
	Identity * id;

	g_return_if_fail (mw!=NULL);
	g_return_if_fail (is_nonempty_string (mw->identity_name));

	/* identity */
	id = identity_manager_get_identity (mw->identity_name);

	/* date */
	article->date = time(0);

	/* group's article number */
	article->number = 1;

	/* author */
	article->author_addr = is_nonempty_string (id->author_addr)
		? group_chunk_string (article->group, id->author_addr, TRUE)
		: NULL;
	article->author_real = is_nonempty_string (id->author_real)
		? group_chunk_string (article->group, id->author_real, TRUE)
		: NULL;

	/* message-id */
	if (is_nonempty_string (id->msg_id_fqdn)) 
		pch = gnksa_generate_message_id (id->msg_id_fqdn);
	else
		pch = gnksa_generate_message_id_from_email_addr (article->author_addr);
	article_init_header (article, HEADER_MESSAGE_ID, pch, DO_CHUNK);
	g_free (pch);

	/* subject */
	pch = get_text_from_editable (mw->subject);
	article_init_header (article, HEADER_SUBJECT, pch, DO_CHUNK_SHARE);
	g_free (pch);

	/* references (for posting articles) */
	if ((mw->type==NNTP_REPLY || mw->type==EMAIL_REPLY || mw->type==EMAIL_AND_POST_REPLY)
	    && (mw->article != NULL))
	{
		const char * refs = mw->article->references;
		const char * msg_id = article_get_message_id (mw->article);
		pch = gnksa_generate_references (refs, msg_id);
		if (pch != NULL) {
			article_init_header (article, HEADER_REFERENCES, pch, DO_CHUNK_SHARE);
			g_free (pch);
		}
	}

	/* organization */
	pch = message_window_get_organization(mw);
	article_init_header (article, HEADER_ORGANIZATION, pch, DO_CHUNK_SHARE);
	g_free (pch);

	/* charset */
	if (1) {
		const char * charset;
	       
		charset = pan_charset_picker_get_charset (mw->charset_om);
		article_init_header (article, PAN_CHARSET, charset, DO_CHUNK_SHARE);
	}

	/* followup-to */
	pch = message_window_get_group_str(mw->followup_to);
	article_init_header (article, HEADER_FOLLOWUP_TO, pch, DO_CHUNK_SHARE);
	g_free (pch);

	/* reply-to */
	pch = message_window_get_reply_to (mw);
	article_init_header (article, HEADER_REPLY_TO, pch, DO_CHUNK_SHARE);
	g_free (pch);

	/* pan-body */
	if (1) {
		GtkTextIter start;
		GtkTextIter end;
		gtk_text_buffer_get_bounds (mw->body_buffer, &start, &end);
		pch = gtk_text_buffer_get_text (mw->body_buffer, &start, &end, FALSE);
		article_init_header (article, PAN_BODY, pch, DO_CHUNK);
		g_free (pch);
	}

	/* newsgroups (for posting articles) */
	pch = message_window_get_group_str (mw->newsgroups);
	article_init_header (article, HEADER_NEWSGROUPS, pch, DO_CHUNK_SHARE);
	g_free (pch);

	/* pan-mail-to (for sending mail) */
	pch = get_text_from_editable (mw->users);
	article_init_header (article, PAN_MAIL_TO, pch, DO_CHUNK_SHARE);
	g_free (pch);

	/* pan-reverse-path (for sending mail) */
	pch = message_window_get_reverse_path (mw);
	if (is_nonempty_string(pch))
		article_init_header (article, PAN_REVERSE_PATH, pch, DO_CHUNK_SHARE);
	g_free (pch);

	/* attribution header (for message-check) */
	if (mw->article!=NULL && mw->identity_name!=NULL) {
		char * attribution = article_gen_attribution_string (mw->article, id->attribution);
		if (is_nonempty_string(attribution))
			article_init_header (article, PAN_ATTRIBUTION, attribution, DO_CHUNK);
		g_free (attribution);
	}

	/* custom headers */
	if (1) {
		GtkTextView * view = GTK_TEXT_VIEW(mw->custom_headers_text);
		GtkTextBuffer * buffer = gtk_text_view_get_buffer (view);
		GtkTextIter start, end;
		gtk_text_buffer_get_bounds (buffer, &start, &end);
		pch = gtk_text_iter_get_text (&start,  &end);
	}
	sd = g_strsplit (pch, "\n", -1);
	for (i=0; sd!=NULL && sd[i]!=NULL; ++i) {
		char * delimit;
		g_strstrip (sd[i]);
		delimit = strchr (sd[i], ':');
		if (delimit != NULL) {
			char * key = g_strndup (sd[i], delimit-sd[i]);
			char * val = g_strdup (delimit + 1);
			g_strstrip (key);
			g_strstrip (val);
			article_init_header (article, key, val, DO_CHUNK_SHARE);
			g_free (key);
			g_free (val);
		}
	}
	g_strfreev (sd);
	g_free (pch);

	/* cleanup */
	pan_object_unref(PAN_OBJECT(id));
}

static Article*
build_article_from_window (const Compose * mw)
{
	Group * folder;
	Article * article;
	debug_enter ("build_article_from_window");

	folder  = serverlist_get_named_folder (PAN_SENDLATER);
	article = article_new (folder);
	populate_article_from_mw (article, mw);

	debug_exit ("build_article_from_window");
	return article;
}

static void
save_cb (gpointer user_data, int action, GtkWidget * w)
{
	Compose * mw = (Compose*) user_data;
	Article * a = mw->article;
	Article * tmp_article;
	guint i;
	GPtrArray * headers;

	/* update the article fields from changes made in the gui */
	tmp_article = article_new (a->group);
	populate_article_from_mw (tmp_article, mw);
	a->author_addr = tmp_article->author_addr;
	a->author_real = tmp_article->author_real;
	a->subject = tmp_article->subject;

	/* replace old headers with new headers */
	headers = article_get_all_headers (a);
	for (i=0; i!=headers->len; i+=2) {
		char * key = (char*) g_ptr_array_index (headers, i);
		article_remove_header (a, key);
	}
	g_ptr_array_free (headers, TRUE);
	headers = article_get_all_headers (tmp_article);
	for (i=0; i!=headers->len; i+=2) {
		char * key = (char*) g_ptr_array_index (headers, i);
		char * val = (char*) g_ptr_array_index (headers, i+1);
		article_set_header (a, key, val, 0);
	}
	g_ptr_array_free (headers, TRUE);

	/* cleanup */
	message_window_destroy (mw);
}

typedef enum
{
	SEND_NOW,
	SEND_LATER
}
SendMode;

static void
post_checked_cb (Server        * server,
                 Article       * article,
                 GoodnessLevel   goodness,
                 gpointer        user_data)
{
	ArgSet * argset = (ArgSet*)user_data;
	Compose * mw = (Compose*) argset_get (argset, 0);
	SendMode mode = GPOINTER_TO_INT(argset_get(argset, 1));

	if (goodness == OKAY)
	{
		const char * pch;
		char * group_name;
		Group * sendlater = serverlist_get_named_folder (PAN_SENDLATER);

		/* gotta ref the articles to make sure that, after sendlater
		 * takes over memory management of article, it stays alive
		 * until after we've called queue_article_for_posting. */
		group_ref_articles (sendlater, NULL);

		/* store which server to post through, in case the article
		 * winds up waiting in pan.sendlater */
		if (!is_nonempty_string(article_get_extra_header (article, PAN_SERVER)))
			article_set_header (article, PAN_SERVER, server->name, DO_CHUNK_SHARE);

		/* add this article to the sendlater group */
		group_add_article (sendlater, article);

		/* try to post right now, if desired... */
		if (mode == SEND_NOW)
			queue_article_for_posting (server, article, FALSE);

		/* remember which identity we used */
		pch = article_get_extra_header (article, HEADER_NEWSGROUPS);
		while ((group_name = get_next_token_str (pch, ',', &pch))!=NULL)
		{
			Group * group;

			g_strstrip (group_name);
			group = server_get_named_group (server, group_name);
			if (group!=NULL && !group_is_folder (group))
				group_set_identity (group, mw->identity_name);

			g_free (group_name);
		}

		/* close this window */
		message_window_destroy (mw);

		group_unref_articles (sendlater, NULL);
	}

	/* cleanup */
	if (goodness!=OKAY && mw->type!=EDIT_ORIGINAL)
		group_remove_article (serverlist_get_named_folder(PAN_SENDLATER), article);

	argset_free (argset);
}

static void
post (Compose * mw, SendMode mode)
{
	Article * article;
	Server * server;

	debug_enter ("post");

       	server = serverlist_get_active_server ();

	/* Get the article pointer, either through repopulating the
	 * current article (if we're editing an existing message) or
	 * by creating a new article (if this is a new message).
	 */
	if (mw->type == EDIT_ORIGINAL) {
		article = mw->article;
		populate_article_from_mw (article, mw);
	} else {
		article = build_article_from_window (mw);
	}

	/* if the article's okay, then send or queue */
	check_article_and_prompt (mw->window,
	                          server,
	                          article,
	                          post_checked_cb,
	                          argset_new2 (mw, GINT_TO_POINTER(mode)));

	debug_exit ("post");
}

static void
send_now_cb (gpointer user_data, int action, GtkWidget * w)
{
	Compose * compose = (Compose*) user_data;
	post (compose, SEND_NOW);
}
static void
send_now_cb2 (GtkWidget * w, gpointer user_data)
{
	send_now_cb (user_data, 0, w);
}
static void
send_later_cb (gpointer user_data, int action, GtkWidget * w)
{
	Compose * compose = (Compose*) user_data;
	post (compose, SEND_LATER);
}
static void
send_later_cb2 (GtkWidget * w, gpointer user_data)
{
	send_later_cb (user_data, 0, w);
}

/**
***  Save as File
**/

static void
message_window_save_ok_clicked (GtkWidget * widget, GtkWidget * file_entry)
{
	const char * filename;
	char * writeme = NULL;
	const Compose * mw;

	/* get information from the widget */
	pan_lock();
	mw = (const Compose*) gtk_object_get_data (GTK_OBJECT(file_entry), "mw");
	filename = gtk_file_selection_get_filename (GTK_FILE_SELECTION (file_entry));
	pan_unlock();

	/* get the article text to save... */
	if (1) {
		Article * a = build_article_from_window (mw);
		writeme = article_get_message (a);
	}

	/* write the text */
	if (is_nonempty_string(writeme))
	{
		FILE * msg_dump = fopen (filename, "w");
		if (msg_dump != NULL)
		{
			fprintf (msg_dump, "%s\n", writeme);
			fclose (msg_dump);

			pan_lock();
			gtk_widget_destroy (file_entry);
			pan_unlock();
		}
		else
		{
		}
	}

	/* cleanup */
	g_free (writeme);
}

static void
message_window_save_cb (gpointer user_data, int action, GtkWidget * w)
{
	Compose * compose = (Compose*) user_data;
	GtkWidget *file_entry = NULL;
	
	pan_lock();
	
	file_entry = gtk_file_selection_new (_("Save message to file"));

	gtk_file_selection_hide_fileop_buttons (GTK_FILE_SELECTION (file_entry));

	g_signal_connect_swapped (GTK_FILE_SELECTION(file_entry)->cancel_button, "clicked",
	                          G_CALLBACK(gtk_widget_destroy), G_OBJECT(file_entry));

	/* message_id is const, but we don't alter it here */
	gtk_object_set_data (GTK_OBJECT (file_entry), "mw", compose);

	g_signal_connect (GTK_FILE_SELECTION(file_entry)->ok_button, "clicked",
	                  G_CALLBACK(message_window_save_ok_clicked), file_entry);

	gtk_widget_show_all (file_entry);
	pan_unlock();
}

/***
****
****  External Editor (mostly from Sylpheed)
****
***/

static void*
run_external_editor (void * compose_voidp)
{
	FILE * fp;
	char * fname = NULL;
	char ** argv = NULL;
	gboolean ok = TRUE;
	gboolean need_thaw = FALSE;
	Compose * compose = (Compose*) compose_voidp;

	/* open a new tmp file */
	if (ok) {
		GError * err = NULL;
		const int fd = g_file_open_tmp ("pan_edit_XXXXXX", &fname, &err);
		if (err == NULL)
			fp = fdopen (fd, "w");
		else {
			ok = FALSE;
			log_add_va (LOG_ERROR, _("Error opening temporary file: %s"), err->message);
			g_error_free (err);
		}
	}

	/* write the article contents to the tmp file */
	if (ok) {
		size_t len;
		char * chars;
		GtkTextIter start, end;

		pan_lock ();
		need_thaw = TRUE;
		gtk_widget_set_sensitive (compose->window, FALSE);
		gtk_text_buffer_get_bounds (compose->body_buffer, &start, &end);
		chars = gtk_text_buffer_get_text (compose->body_buffer, &start, &end, FALSE);
		pan_unlock ();

		len = strlen (chars);
		if (fwrite (chars, sizeof(char), len, fp) != len) {
			ok = FALSE;
			log_add_va (LOG_ERROR, _("Error writing article to temporary file: %s"), g_strerror(errno));
		}

		fclose (fp);
		fp = NULL;
		g_free (chars);

	}

	/* parse the command line */
	if (ok) {
		int argc;
		GError * err = NULL;
		char * cmd = pan_substitute (external_editor, "%t", fname);
		g_shell_parse_argv (cmd, &argc, &argv, &err);
		if (err != NULL) {
			log_add_va (LOG_ERROR, _("Error parsing \"external editor\" command line: %s"), err->message);
			log_add_va (LOG_ERROR, _("The command line was: %s"), cmd);
			g_error_free (err);
			ok = FALSE;
		}
		g_free (cmd);
	}

	/* spawn off the external editor */
	if (ok) {
		GError * err = NULL;
		g_spawn_sync (NULL, argv, NULL, G_SPAWN_SEARCH_PATH, NULL, NULL, NULL, NULL, NULL, &err);
		if (err != NULL) {
			log_add_va (LOG_ERROR, _("Error starting external editor: %s"), err->message);
			g_error_free (err);
			ok = FALSE;
		}
	}


	/* read the file contents back in */
	if (ok) {
		GError * err = NULL;
		char * body = NULL;
		gsize body_len = 0;

		g_file_get_contents (fname, &body, &body_len, &err);
		if (err != NULL) {
			log_add_va (LOG_ERROR, _("Error reading edited file: %s"), err->message);
			g_error_free (err);
			ok = FALSE;
		}
		else {
			GtkTextIter start, end;
			GtkTextBuffer * buf = compose->body_buffer;

			pan_lock ();
			gtk_text_buffer_get_bounds (buf, &start, &end);
			gtk_text_buffer_delete (buf, &start, &end);
			gtk_text_buffer_insert (buf, &start, body, body_len);
			pan_unlock ();
		}

		/* cleanup */
		g_free (body);
	}

	/* cleanup */
	remove (fname);
	g_free (fname);
	g_strfreev (argv);
	if (need_thaw) {
		pan_lock ();
		gtk_widget_set_sensitive (compose->window, TRUE);
		pan_unlock ();
	}
	return NULL;
}

static void
compose_ext_editor_cb2 (GtkWidget * w, gpointer user_data)
{
	run_in_worker_thread (run_external_editor, user_data);
}
static void
compose_ext_editor_cb (gpointer user_data, int action, GtkWidget * w)
{
	run_in_worker_thread (run_external_editor, user_data);
}
