#include <pan/group-store.h>
#include <pan/base/pan-glib-extensions.h>

/***
****  TreeModel Interface
***/

static GType column_gtypes[] = {
	G_TYPE_POINTER,
	G_TYPE_BOOLEAN,
	G_TYPE_STRING,
	G_TYPE_INT,
	G_TYPE_INT,
	G_TYPE_STRING
};

static guint
group_store_get_flags (GtkTreeModel * tree_model)
{
	g_return_val_if_fail (PAN_IS_GROUP_STORE(tree_model), 0);
	    
	return GTK_TREE_MODEL_ITERS_PERSIST | GTK_TREE_MODEL_LIST_ONLY;
}

static int
group_store_get_n_columns (GtkTreeModel *tree_model)
{
	return GROUP_STORE_COLUMN_QTY;
}

static GType
group_store_get_column_type (GtkTreeModel *tree_model, int column_index)
{
	GType retval = G_TYPE_INVALID;

	g_return_val_if_fail (PAN_IS_GROUP_STORE(tree_model), retval);
	g_return_val_if_fail (0<=column_index && column_index<GROUP_STORE_COLUMN_QTY, retval);

	return column_gtypes[column_index];
}

static gboolean
group_store_get_iter (GtkTreeModel * tree_model,
                      GtkTreeIter  * iter,
                      GtkTreePath  * path)
{
	int i;
	GroupStore * group_store = (GroupStore*) tree_model;

	g_return_val_if_fail (PAN_IS_GROUP_STORE(tree_model), FALSE);
	g_return_val_if_fail (0<gtk_tree_path_get_depth(path), FALSE);

	i = gtk_tree_path_get_indices (path)[0];
	if (i>=group_store->groups->len)
		return FALSE;

	iter->stamp = group_store->stamp;
	iter->user_data = GINT_TO_POINTER(i);

	return TRUE;
}

static GtkTreePath *
group_store_get_path (GtkTreeModel * tree_model,
                      GtkTreeIter  * iter)
{
	GtkTreePath * retval;
	GroupStore * group_store = (GroupStore*) tree_model;

	g_return_val_if_fail (PAN_IS_GROUP_STORE(tree_model), FALSE);
	g_return_val_if_fail (iter->stamp == group_store->stamp, NULL);

	retval = gtk_tree_path_new ();
	gtk_tree_path_append_index (retval, GPOINTER_TO_INT(iter->user_data));
	return retval;
}

static void
group_store_get_value (GtkTreeModel  * tree_model,
                       GtkTreeIter   * iter,
                       int             column,
                       GValue        * value)
{
	int row;
	Group * group;
	GroupStore * group_store = (GroupStore*) tree_model;

	g_return_if_fail (PAN_IS_GROUP_STORE(tree_model));
	g_return_if_fail (0<=column && column<GROUP_STORE_COLUMN_QTY);
	g_return_if_fail (iter->stamp == group_store->stamp);

	row = GPOINTER_TO_INT(iter->user_data);
	group = GROUP(g_ptr_array_index(group_store->groups,row));

	g_value_init (value, column_gtypes[column]);
	switch (column)
	{
		case GROUP_STORE_COLUMN_GROUP_POINTER:
			g_value_set_pointer (value, group);
			break;
		case GROUP_STORE_COLUMN_SUBSCRIBED:
			g_value_set_boolean (value, group_is_subscribed(group));
			break;
		case GROUP_STORE_COLUMN_NAME:
			g_value_set_string (value, group_get_readable_name(group));
			break;
		case GROUP_STORE_COLUMN_UNREAD:
			g_value_set_int (value, group->article_qty - group->article_read_qty);
			break;
		case GROUP_STORE_COLUMN_TOTAL:
			g_value_set_int (value, group->article_qty);
			break;
		case GROUP_STORE_COLUMN_DESCRIPTION:
			g_value_set_string (value, group->description);
			break;
	}
}

static gboolean
group_store_iter_next (GtkTreeModel  * tree_model,
                       GtkTreeIter   * iter)
{
	int row;
	gboolean retval;
	GroupStore * group_store = (GroupStore*) tree_model;

	g_return_val_if_fail (PAN_IS_GROUP_STORE(tree_model), FALSE);
	g_return_val_if_fail (iter!=NULL, FALSE);
	g_return_val_if_fail (iter->stamp == group_store->stamp, FALSE);

	row = GPOINTER_TO_INT(iter->user_data);
	retval = 0<=row && row<group_store->groups->len-1;

	if (retval)
		iter->user_data = GINT_TO_POINTER(row+1);

	return retval;
}

static gboolean
group_store_iter_children (GtkTreeModel * tree_model,
                           GtkTreeIter  * iter,
                           GtkTreeIter  * parent)
{
	GroupStore * group_store = (GroupStore*) tree_model;

	g_return_val_if_fail (PAN_IS_GROUP_STORE(tree_model), FALSE);
	g_return_val_if_fail (iter!=NULL, FALSE);
	g_return_val_if_fail (iter->stamp == group_store->stamp, FALSE);

	/* this is a list, nodes have no children */
	if (parent != NULL)
		return FALSE;

	/* but if parent == NULL we return the list itself as
	 * children of the "root" */
	if (group_store->groups->len)
	{
		iter->stamp = group_store->stamp;
		iter->user_data = GINT_TO_POINTER(0);
		return TRUE;
	}
	else
		return FALSE;
}

static gboolean
group_store_iter_has_child (GtkTreeModel *tree_model,
                            GtkTreeIter  *iter)
{
	  return FALSE;
}

static int
group_store_iter_n_children (GtkTreeModel *tree_model,
                             GtkTreeIter  *iter)
{
	int retval = 0;
	GroupStore * group_store = (GroupStore*) tree_model;

	g_return_val_if_fail (PAN_IS_GROUP_STORE(tree_model), -1);
	g_return_val_if_fail (iter==NULL || iter->stamp==group_store->stamp, -1);

	if (iter == NULL)
		retval = group_store->groups->len;

	return retval;
}

static gboolean
group_store_iter_nth_child (GtkTreeModel * tree_model,
                            GtkTreeIter  * iter,
                            GtkTreeIter  * parent,
                            int            n)
{
	GroupStore * group_store = (GroupStore*) tree_model;

	g_return_val_if_fail (PAN_IS_GROUP_STORE(tree_model), FALSE);
	g_return_val_if_fail (iter!=NULL, FALSE);

	if (parent==NULL && 0<=n && n<group_store->groups->len)
	{
		iter->stamp = group_store->stamp;
		iter->user_data = GINT_TO_POINTER(n);
		return TRUE;
	}
	else
		return FALSE;
}

static gboolean
group_store_iter_parent (GtkTreeModel *tree_model,
                         GtkTreeIter  *iter,
                         GtkTreeIter  *child)
{
	  return FALSE;
}

static void
group_store_tree_model_init (GtkTreeModelIface *iface)
{       
	iface->get_flags = group_store_get_flags;
	iface->get_n_columns = group_store_get_n_columns;
	iface->get_column_type = group_store_get_column_type;
	iface->get_iter = group_store_get_iter;
	iface->get_path = group_store_get_path;
	iface->get_value = group_store_get_value;
	iface->iter_next = group_store_iter_next;
	iface->iter_children = group_store_iter_children;
	iface->iter_has_child = group_store_iter_has_child;
	iface->iter_n_children = group_store_iter_n_children;
	iface->iter_nth_child = group_store_iter_nth_child;
	iface->iter_parent = group_store_iter_parent;
}

/***
****
***/

static void
group_store_class_init (GroupStoreClass *class)
{
}

static void
group_store_init (GroupStore * store)
{
	store->groups = g_ptr_array_new ();
	store->stamp = g_random_int ();
}

GtkType
group_store_get_type (void)
{
	static GType group_store_type = 0;

	if (!group_store_type)
	{
		static const GTypeInfo group_store_info =
		{
			sizeof(GroupStoreClass),
			NULL, /* base init */
			NULL, /* base finalize */
			(GClassInitFunc) group_store_class_init,
			NULL, /* class finalize */ 
			NULL, /* class data */
			sizeof(GroupStore),
			0,
			(GInstanceInitFunc) group_store_init
		};

		static const GInterfaceInfo tree_model_info =
		{
			(GInterfaceInitFunc) group_store_tree_model_init,
			NULL,
			NULL
		};

		group_store_type = g_type_register_static (G_TYPE_OBJECT,
		                                           "GroupStore",
		                                           &group_store_info,
		                                           0);

		g_type_add_interface_static (group_store_type,
		                             GTK_TYPE_TREE_MODEL,
		                             &tree_model_info);
	}

	return group_store_type;
}

GroupStore*
group_store_new (void)
{
	return GROUP_STORE (g_object_new (group_store_get_type(), NULL));
}

void
group_store_set_groups (GroupStore   * group_store,
                        Group       ** groups,
                        int            group_qty)
{
	int i;

	g_return_if_fail (group_store!=NULL);
	g_return_if_fail (PAN_IS_GROUP_STORE(group_store));
	g_return_if_fail (group_qty>=0);
	g_return_if_fail (groups!=NULL || group_qty==0);

	for (i=group_store->groups->len-1; i>=0; --i)
	{
		GtkTreePath * path = gtk_tree_path_new ();
		gtk_tree_path_append_index (path, i);
//if (!(i % 1000)) g_message ("sending row %d removed", i);
		gtk_tree_model_row_deleted (GTK_TREE_MODEL(group_store), path);
		gtk_tree_path_free (path);
	}

	pan_g_ptr_array_assign (group_store->groups, (gpointer*)groups, group_qty);

	for (i=0; i<group_store->groups->len; ++i)
	{
		GtkTreeIter iter;
		GtkTreePath * path;

		iter.stamp = group_store->stamp;
		iter.user_data = GINT_TO_POINTER(i);
		path = gtk_tree_path_new ();
		gtk_tree_path_append_index (path, i);
		gtk_tree_model_row_inserted (GTK_TREE_MODEL(group_store), path, &iter);
//if (!(i % 1000)) g_message ("sending row %d inserted", i);

		gtk_tree_path_free (path);
	}
}


static int
find_index_of_group (GroupStore  * group_store,
                     const Group * group)
{
	int i;
	int size;
	int retval;

	g_return_val_if_fail (PAN_IS_GROUP_STORE(group_store), -1);
	g_return_val_if_fail (group!=NULL, -1);

	for (i=0, retval=-1, size=group_store->groups->len; retval==-1 && i<size; ++i)
		if (g_ptr_array_index (group_store->groups, i) == group)
			retval = i;

	return retval;
}

gboolean
group_store_get_iter_to_group (GroupStore   * group_store,
                               const Group  * group,
                               GtkTreeIter  * iter)
{
	int i;

	g_return_val_if_fail (group!=NULL, FALSE);
	g_return_val_if_fail (iter!=NULL, FALSE);
	g_return_val_if_fail (PAN_IS_GROUP_STORE(group_store), FALSE);

	i = find_index_of_group (group_store, group);
	if (i == -1)
		return FALSE;
	else {
		iter->stamp = group_store->stamp;
		iter->user_data = GINT_TO_POINTER(i);
		return TRUE;
	}
}

void
group_store_remove_group        (GroupStore   * group_store,
                                 const Group  * group)
{
	int i;

	g_return_if_fail (PAN_IS_GROUP_STORE(group_store));
	g_return_if_fail (group!=NULL);

	i = find_index_of_group (group_store, group);
	if (i != -1)
	{
		GtkTreePath * path;
	       
		g_ptr_array_remove_index (group_store->groups, i);

		path = gtk_tree_path_new ();
		gtk_tree_path_append_index (path, i);
		gtk_tree_model_row_deleted (GTK_TREE_MODEL(group_store), path);

		gtk_tree_path_free (path);
	}
}

gboolean
group_store_fire_group_changed  (GroupStore   * group_store,
                                 const Group  * group)
{
	int i;

	g_return_val_if_fail (group!=NULL, FALSE);
	g_return_val_if_fail (PAN_IS_GROUP_STORE(group_store), FALSE);

	i = find_index_of_group (group_store, group);
	if (i == -1)
		return FALSE;
	else
	{
		GtkTreeIter iter;
		GtkTreePath * path;
	       
		iter.stamp = group_store->stamp;
		iter.user_data = GINT_TO_POINTER(i);
		path = gtk_tree_path_new ();
		gtk_tree_path_append_index (path, i);
		gtk_tree_model_row_changed (GTK_TREE_MODEL(group_store), path, &iter);

		gtk_tree_path_free (path);

		return TRUE;
	}
}
