/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Author: Charles Kerr <charles@rebelbase.com>
 *
 * Pan - A Newsreader for X
 * Copyright (C) 2002  Charles Kerr <charles@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <config.h>

#include <glib.h>

#include <pan/base/pan-i18n.h>
#include <pan/base/pan-glib-extensions.h>

#include <pan/filters/filter-thread.h>

#include <pan/thread-watch.h>

const gchar * FILTER_THREAD_CLASS_ID = "PanObject::Filter::FilterThread";

/************
*************  PROTECTED
************/

static Filter*
filter_thread_dup (const Filter * f_old)
{
	Filter * f_new = filter_thread_new ();
	FilterThread * ft_old = FILTER_THREAD(f_old);
	FilterThread * ft_new = FILTER_THREAD(f_new);
	filter_class_dup (f_old, f_new);
	ft_new->state = ft_old->state;
	return f_new;
}

static gchar*
filter_thread_to_string (const Filter * filter)
{
	gchar * pch;
	const FilterThread * f = FILTER_THREAD(filter);

	if (f->state==FILTER_THREAD_WATCHED) {
		if (filter->negate)
			pch = g_strdup (_("Thread isn't watched"));
		else
			pch = g_strdup (_("Thread is watched"));
	}
	else if (f->state==FILTER_THREAD_IGNORED) {
		if (filter->negate)
			pch = g_strdup (_("Thread isn't ignored"));
		else
			pch = g_strdup (_("Thread is ignored"));
	}
	else if (f->state==FILTER_THREAD_NOTHING) {
		if (filter->negate)
			pch = g_strdup (_("Thread is watched or ignored"));
		else
			pch = g_strdup (_("Thread is neither watched nor ignored"));
	}
	else pch = g_strdup (_("Error"));

	return pch;
}

static void
filter_thread_test_articles (Filter          * filter,
                             const Article  ** articles,
                             gint              article_qty,
                             gboolean        * passfail)
{
	gint i;
	const FilterThreadState state = FILTER_THREAD(filter)->state;

	for (i=0; i<article_qty; ++i)
	{
		const gint a_state = pan_thread_get_state (articles[i]);
		gboolean b;
		switch (state)
		{
			case FILTER_THREAD_WATCHED:
				b = a_state == THREAD_WATCHED;
				break;
			case FILTER_THREAD_IGNORED:
				b = a_state == THREAD_IGNORED;
				break;
			case FILTER_THREAD_NOTHING:
				b = a_state == THREAD_NOTHING;
				break;
			default:
				pan_warn_if_reached ();
				b = FALSE;
				break;
		}
		passfail[i] = b;
	}
}

static void
filter_thread_destructor (PanObject * o)
{
	filter_destructor (o);
}

static void
filter_thread_constructor (FilterThread * f)
{
	filter_constructor ((Filter*)f,
	                    filter_thread_destructor,
	                    filter_thread_test_articles,
	                    filter_thread_to_string,
	                    filter_thread_dup,
	                    FILTER_THREAD_CLASS_ID);
	f->state = FILTER_THREAD_WATCHED;
}

/************
*************  PUBLIC
************/

Filter*
filter_thread_new (void)
{
	FilterThread * f = g_new0 (FilterThread, 1);
	filter_thread_constructor (f);
	return FILTER(f);
}

void
filter_thread_set_state (FilterThread        * filter_thread,
                         FilterThreadState     state)
{
	g_return_if_fail (filter_thread!=NULL);
	g_return_if_fail (state==FILTER_THREAD_WATCHED || state==FILTER_THREAD_IGNORED || state==FILTER_THREAD_NOTHING);

	filter_thread->state = state;
}
