/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Author: Charles Kerr <charles@rebelbase.com>
 *
 * Pan - A Newsreader for X
 * Copyright (C) 2002  Charles Kerr <charles@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <config.h>

#include <glib.h>

#include <pan/base/base-prefs.h>
#include <pan/base/pan-callback.h>
#include <pan/base/pan-glib-extensions.h>
#include <pan/base/pan-i18n.h>

#include <pan/filters/filter-phrase.h>
#include <pan/filters/filter-manager.h>
#include <pan/filters/filter-xml.h>

static GPtrArray * _filters = NULL;

const gchar * bozo_filter_name = NULL;

const gchar * spam_filter_name = NULL;

/************
*************  PRIVATE
************/

static gchar*
get_filter_filename (void)
{
	return g_strdup_printf ("%s%cfilters.xml", get_data_dir(), G_DIR_SEPARATOR);
}

static gboolean
filter_manager_has_named_filter (const gchar * name)
{
	guint i;

	for (i=0; i!=_filters->len; ++i)
		if (!pan_strcmp (FILTER_TOP(g_ptr_array_index(_filters,i))->name, name))
			return TRUE;

	return FALSE;
}

static void
add_any_needed_canned_filters (GPtrArray * filters)
{
	if (!filter_manager_has_named_filter (bozo_filter_name))
	{
		Filter * f;
		Filter * bozo = filter_top_new ();
		filter_top_set_visible (FILTER_TOP(bozo), TRUE);
		filter_top_set_name (FILTER_TOP(bozo), bozo_filter_name);
		filter_aggregate_set_type (FILTER_AGGREGATE(bozo), AGGREGATE_TYPE_OR);
		f = filter_phrase_new ();
		filter_phrase_set (FILTER_PHRASE(f), FILTER_PHRASE_AUTHOR, "Dave Rhodes", FALSE);
		filter_aggregate_add (FILTER_AGGREGATE(bozo), &f, 1);
		pan_object_unref (PAN_OBJECT(f));
		g_ptr_array_add (filters, bozo);

g_warning ("Added canned filter %s", bozo_filter_name);
	}

	if (!filter_manager_has_named_filter (spam_filter_name))
	{
		Filter * f;
		Filter * spam = filter_top_new ();
		filter_top_set_visible (FILTER_TOP(spam), TRUE);
		filter_top_set_name (FILTER_TOP(spam), spam_filter_name);
		filter_aggregate_set_type (FILTER_AGGREGATE(spam), AGGREGATE_TYPE_OR);
		f = filter_phrase_new ();
		filter_phrase_set (FILTER_PHRASE(f), FILTER_PHRASE_SUBJECT, "MAKE MONEY FAST", FALSE);
		filter_aggregate_add (FILTER_AGGREGATE(spam), &f, 1);
		pan_object_unref (PAN_OBJECT(f));
		g_ptr_array_add (filters, spam);

g_warning ("Added canned filter %s", spam_filter_name);
	}
}


static GStaticMutex _fm_lock = G_STATIC_MUTEX_INIT;
static gboolean     _fm_inited = FALSE;

static gboolean
is_inited (void)
{
	gboolean inited = FALSE;
	g_static_mutex_lock (&_fm_lock);
	inited = _fm_inited;
	g_static_mutex_unlock (&_fm_lock);

	return inited;
}

static void
filter_manager_init (void)
{
	static GStaticMutex fm_lock = G_STATIC_MUTEX_INIT;
	char * fname;

	g_return_if_fail (!_fm_inited);

	g_static_mutex_lock (&fm_lock);

	bozo_filter_name = _("Bozos");
	spam_filter_name = _("Spam");

	/* populate _filters from the array */
	fname = get_filter_filename ();
	_filters = g_ptr_array_new ();
	filter_xml_read (fname, _filters);
	g_free (fname);

	add_any_needed_canned_filters (_filters);

	_fm_inited = TRUE;

	g_static_mutex_unlock (&fm_lock);
}

/************
*************  PUBLIC
************/

void
filter_manager_get_filters (GPtrArray * fillme)
{
	guint i;

	/* init */
	if (!is_inited())
		filter_manager_init ();

	/* sanity clause */
	g_return_if_fail (fillme!=NULL);
	g_return_if_fail (_filters!=NULL);

	for (i=0; i<_filters->len; ++i)
	{
		Filter * old = FILTER(g_ptr_array_index(_filters,i));
		Filter * dup = filter_dup (old);
		g_ptr_array_add (fillme, dup);
	}
}


Filter*
filter_manager_get_named_filter (const gchar * name)
{
	guint i;
	Filter * retval = NULL;

	/* init */
	if (!is_inited())
		filter_manager_init ();

	g_return_val_if_fail (is_nonempty_string(name), NULL);

	for (i=0; retval==NULL && i<_filters->len; ++i)
	{
		FilterTop * f = FILTER_TOP(g_ptr_array_index(_filters,i));
		if (!pan_strcmp (f->name, name))
			retval = FILTER(f);
	}

	if (retval != NULL)
		retval = filter_dup (retval);

	return retval;
}

void
filter_manager_set_filters (GPtrArray * filters)
{
	guint i;
	gchar * fname;
	GPtrArray * old;
	GPtrArray * tmp;

	/* init */
	if (!is_inited())
		filter_manager_init ();

	/* make copies of the new */
	tmp = g_ptr_array_new ();
	for (i=0; i<filters->len; ++i) {
		Filter * f = FILTER(g_ptr_array_index(filters,i));
		Filter * dup = filter_dup (f);
		g_ptr_array_add (tmp, dup);
	}

	/* empty out the old */
	old = _filters;
	_filters = tmp;
	if (old != NULL) {
		pan_g_ptr_array_foreach (old, (GFunc)pan_object_unref, NULL);
		g_ptr_array_free (old, TRUE);
	}

	/* save them to disk */
	fname = get_filter_filename ();
	filter_xml_write (fname, (FilterTop**)_filters->pdata, _filters->len);
	g_free (fname);

	/* fire a notice to listeners */
	pan_callback_call (filter_manager_get_filters_changed_callback(),
	                   _filters, NULL);
}

PanCallback*
filter_manager_get_filters_changed_callback (void)
{
	static PanCallback * cb = NULL;
	if (cb==NULL) cb = pan_callback_new ();
	return cb;
}
