/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999, 2000  Pan Development Team <pan@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include <glib.h>
#include <gtk/gtk.h>

#include <ctype.h>

#include <unistd.h>
#include <netinet/in.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <netdb.h>

#include <pan/base/pan-config.h>
#include <pan/base/pan-i18n.h>
#include <pan/base/serverlist.h>

#include <pan/globals.h>
#include <pan/pan.h>
#include <pan/gui.h>
#include <pan/prefs.h>
#include <pan/util.h>

#include <pan/identities/identity.h>
#include <pan/identities/identity-manager.h>

#include <pan/dialogs/dialogs.h>
#include <pan/xpm/pan-pixbufs.h>

enum
{
	PAN_RESPONSE_GO_BACK,
	PAN_RESPONSE_GO_FORWARD,
	PAN_RESPONSE_QUIT,
	PAN_RESPONSE_FINISHED
};

typedef struct
{
	GPtrArray * pages;
	GtkWidget * notebook;

	GtkWidget * full_name;
	GtkWidget * organization;

	GtkWidget * email;
	GtkWidget * smtp_server;
	GtkWidget * smtp_port;
	
	GtkWidget * server_address;
	GtkWidget * server_port;
	GtkWidget * auth_toggle;

	GtkWidget * server_username_label;
	GtkWidget * server_username;
	GtkWidget * server_password_label;
	GtkWidget * server_password;

	GtkWidget * profile_name;
}
Druid;

/***
****
****  HOSTNAME
****
***/


/* Returns the fully qualified domain name */
static char*
get_fqdn (void)
{
	char name[512] = { '\0' };
	char line[1025];
	char fqdn[1024] = { '\0' };
	char * cp;
	char * domain = NULL;
	FILE * inf;
	struct hostent * hp;
	struct in_addr in;

	if (gethostname(name, MAXHOSTNAMELEN))
		return NULL;

	if (isdigit((guchar)*name)) {
		in.s_addr = inet_addr (name);
		if ((hp = gethostbyaddr((char *) &in.s_addr, 4, AF_INET)))
			in.s_addr = (*hp->h_addr);
		return g_strdup((hp && strchr(hp->h_name, '.') ? hp->h_name : (char *) inet_ntoa(in)));
	}
	if ((hp = gethostbyname(name)) && !strchr(hp->h_name, '.'))
		if ((hp = gethostbyaddr(hp->h_addr, hp->h_length, hp->h_addrtype)))
			in.s_addr = (*hp->h_addr);

	sprintf(fqdn, "%s", hp ? strchr(hp->h_name, '.')
		? hp->h_name : (char *) inet_ntoa(in)
		: "");
	if (!*fqdn || (fqdn[strlen(fqdn) - 1] <= '9')) {
		*fqdn = '\0';
		inf = fopen("/etc/resolv.conf", "r");
		if (inf) {
			while (fgets(line, 1024, inf)) {
				line[1024] = '\0';
				g_strchug(line);
				g_strchomp(line);
				if (strncmp(line, "domain ", 7) == 0) {
					domain = line + 7;
					break;
				}
				if (strncmp(line, "search ", 7) == 0) {
					domain = line + 7;
					cp = strchr(domain, ' ');
					if (cp)
						*cp = '\0';
					break;
				}
			}
			if (domain)
				sprintf(fqdn, "%s.%s", name, domain);
			fclose(inf);
		}
	}
	return g_strdup (fqdn);
}

static char*
get_default_author_address (void)
{
	char * email = pan_config_get_string ("/Pan/User/Email");

	if (email != NULL)
		g_strstrip (email);

	/* if no email in prefs, do the best we can */
	if (!is_nonempty_string(email)) {
		char * fqdn = get_fqdn ();
		replace_gstr (&email,
			g_strdup_printf ("%s@%s", g_get_user_name(), fqdn));
		g_free (fqdn);
	}

	return email;
}

/***
***/

static void
druid_finish (Druid * druid)
{
	Server * server = NULL;
	GString * buf = g_string_new (NULL);
	int server_port = atoi(gtk_entry_get_text (GTK_ENTRY(druid->server_port)));
	Identity * id = NULL;

	int auth_toggle = GTK_TOGGLE_BUTTON (druid->auth_toggle)->active;
	const char * profile_name = gtk_entry_get_text (GTK_ENTRY(druid->profile_name));
	const char * server_address = gtk_entry_get_text (GTK_ENTRY(druid->server_address));
	const char * server_username = gtk_entry_get_text(GTK_ENTRY(druid->server_username));
	const char * server_password = gtk_entry_get_text(GTK_ENTRY(druid->server_password));
	const char * full_name = gtk_entry_get_text (GTK_ENTRY(druid->full_name));
	const char * email = gtk_entry_get_text (GTK_ENTRY(druid->email));
	const char * organization = gtk_entry_get_text (GTK_ENTRY(druid->organization));
	const char * smtp_server = gtk_entry_get_text (GTK_ENTRY(druid->smtp_server));
	const char * smtp_port = gtk_entry_get_text (GTK_ENTRY(druid->smtp_port));

	if (profile_name && server_address)
	{
		server = server_new ();
                server->name = g_strdup (profile_name);
		server->address = g_strdup (server_address);
                server->port = server_port;
                server->need_auth = auth_toggle;
                server->username = auth_toggle ? g_strdup (server_username) : NULL;
		server->password = auth_toggle ? g_strdup (server_password) : NULL;
                server->gen_msgid = TRUE;
		server->max_connections = 4;
		server->idle_secs_before_timeout = 180;
		serverlist_add_server (server);
		pan_config_set_string ("/Pan/State/Server", profile_name);
	}

	/* save user stats */
	pan_config_set_string ("/Pan/Mail/smtp_port", smtp_port);
	pan_config_set_string ("/Pan/Mail/smtp_address", smtp_server);
	mail_server_address = g_strdup (smtp_server);

	id = identity_new ();
	if (id != NULL)
	{
		char * str = g_strdup_printf ("%s <%s>", full_name, email);

		identity_set_name (id, str);
		identity_set_author_real (id, full_name);
		identity_set_author_addr (id, email);
		identity_set_organization (id, organization);

		identity_manager_add_identity (id);

		pan_object_unref (PAN_OBJECT (id));
		g_free (str);
	}

	/* start Pan proper */
	pan_config_sync ();
	gui_construct (NULL);
	gtk_widget_show_all (Pan.window);
	serverlist_set_active_server (server);

	/* cleanup */
	g_free (druid);
	g_string_free (buf, TRUE);
}

static void
dialog_response_cb (GtkDialog * dialog, int response, gpointer data)
{
	Druid * druid = (Druid*) data;
	const int page_qty = druid->pages->len;
	int current_page = gtk_notebook_get_current_page (GTK_NOTEBOOK(druid->notebook));

	if (response == PAN_RESPONSE_QUIT)
	{
		pan_shutdown ();
		gtk_widget_destroy (GTK_WIDGET(dialog));
	}
	else if (response == PAN_RESPONSE_FINISHED)
	{
		druid_finish (druid);
		gtk_widget_destroy (GTK_WIDGET(dialog));
	}
	else
	{
		if (response == PAN_RESPONSE_GO_BACK)
		{
			if (current_page > 0)
				gtk_notebook_set_current_page (GTK_NOTEBOOK(druid->notebook), --current_page);
		}
		else if (response == PAN_RESPONSE_GO_FORWARD)
		{
			if (current_page < (page_qty-1))
				gtk_notebook_set_current_page (GTK_NOTEBOOK(druid->notebook), ++current_page);
		}

		gtk_dialog_set_response_sensitive (dialog, PAN_RESPONSE_GO_BACK, current_page>0);
		gtk_dialog_set_response_sensitive (dialog, PAN_RESPONSE_GO_FORWARD, current_page<(page_qty-1));
		gtk_dialog_set_response_sensitive (dialog, PAN_RESPONSE_FINISHED, current_page==(page_qty-1));
	}
}

/***
****
***/

static GtkWidget*
druid_name_page (Druid * druid)
{
	GtkWidget * w;
	GtkWidget * table;

	table = gtk_table_new (5, 2, FALSE);
	gtk_table_set_row_spacings (GTK_TABLE(table), GUI_PAD_SMALL);
	gtk_table_set_col_spacings (GTK_TABLE(table), GUI_PAD_SMALL);

	w = gtk_label_new (_("When you post a message to a newsgroup, your display name will appear in the From field.  Type your name in as you would like it to appear."));
	gtk_misc_set_alignment (GTK_MISC(w), 0.0, 0.0);
	gtk_label_set_line_wrap (GTK_LABEL(w), TRUE);
	gtk_label_set_justify (GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	gtk_table_attach (GTK_TABLE(table), w, 0, 2, 0, 1, GTK_FILL, GTK_FILL, GUI_PAD, GUI_PAD_SMALL);

	w = gtk_label_new (_("Display Name:"));
	gtk_misc_set_alignment (GTK_MISC(w), 1.0, 0.5);
	gtk_table_attach (GTK_TABLE(table), w, 0, 1, 1, 2, GTK_FILL, GTK_FILL, GUI_PAD, GUI_PAD_SMALL);

	w = druid->full_name = gtk_entry_new ();
	gtk_table_attach (GTK_TABLE(table), w, 1, 2, 1, 2, GTK_FILL, GTK_FILL, GUI_PAD, GUI_PAD_SMALL);
	pan_gtk_entry_set_text (w, g_get_real_name());

	w = gtk_label_new (_("For example: John Doe"));
	gtk_misc_set_alignment (GTK_MISC(w), 0.0, 0.5);
	gtk_table_attach (GTK_TABLE(table), w, 1, 2, 2, 3, GTK_FILL, GTK_FILL, GUI_PAD, GUI_PAD_SMALL);

	w = gtk_label_new (_("If you belong to an organization and would like for your organization's name to appear with your posts, type in your organization name below.  This information is optional, you may leave it blank."));
	gtk_misc_set_alignment (GTK_MISC(w), 0.0, 0.0);
	gtk_label_set_line_wrap (GTK_LABEL(w), TRUE);
	gtk_label_set_justify (GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	gtk_table_attach (GTK_TABLE(table), w, 0, 2, 3, 4, GTK_FILL, GTK_FILL, GUI_PAD, GUI_PAD_SMALL);

	w = gtk_label_new (_("Organization:"));
	gtk_misc_set_alignment (GTK_MISC(w), 1.0, 0.5);
	gtk_table_attach (GTK_TABLE (table), w, 0, 1, 4, 5, GTK_FILL, GTK_FILL, GUI_PAD, GUI_PAD_SMALL);

	w = druid->organization = gtk_entry_new ();
	gtk_table_attach (GTK_TABLE(table), w, 1, 2, 4, 5, GTK_FILL, GTK_FILL, GUI_PAD, GUI_PAD_SMALL);

	return table;
}


/*
 * Page #2:
 *  "Internet News E-mail Address"
 */
static GtkWidget*
druid_email_page (Druid *druid)
{
	GtkWidget * table;
	GtkWidget * w;
	char * email;
	char * smtp_server;
	char * smtp_port;
	char * fqdn;

	table = gtk_table_new (6, 2, FALSE);
	gtk_table_set_row_spacings (GTK_TABLE(table), GUI_PAD_SMALL);
	gtk_table_set_col_spacings (GTK_TABLE(table), GUI_PAD_SMALL);

	w = gtk_label_new (_("People can reply to your posts by sending you an e-mail message at the following address.") );
	gtk_misc_set_alignment (GTK_MISC(w), 0.0, 0.0);
	gtk_label_set_line_wrap (GTK_LABEL(w), TRUE);
	gtk_label_set_justify (GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	gtk_table_attach (GTK_TABLE(table), w, 0, 2, 0, 1, GTK_FILL, GTK_FILL, GUI_PAD, GUI_PAD_SMALL);

	w = gtk_label_new (_("E-mail address:"));
	gtk_misc_set_alignment (GTK_MISC(w), 1.0, 0.5);
	gtk_table_attach (GTK_TABLE (table), w, 0, 1, 1, 2, GTK_FILL, GTK_FILL, GUI_PAD, GUI_PAD_SMALL);

	w = druid->email = gtk_entry_new ();
	email = get_default_author_address ();
	pan_gtk_entry_set_text (w, email);
	gtk_table_attach (GTK_TABLE(table), w, 1, 2, 1, 2, GTK_FILL, GTK_FILL, GUI_PAD, GUI_PAD_SMALL);

	w = gtk_label_new (_("For example: user@host.com"));
	gtk_misc_set_alignment (GTK_MISC(w), 0.0, 0.5);
	gtk_table_attach (GTK_TABLE(table), w, 1, 2, 2, 3, GTK_FILL, GTK_FILL, GUI_PAD, GUI_PAD_SMALL);

	w = gtk_label_new (_("Please enter the name of the mail server to use when sending e-mail messages."));
	gtk_misc_set_alignment (GTK_MISC(w), 0.0, 0.0);
	gtk_label_set_line_wrap (GTK_LABEL(w), TRUE);
	gtk_label_set_justify (GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	gtk_table_attach (GTK_TABLE(table), w, 0, 2, 3, 4, GTK_FILL, GTK_FILL, GUI_PAD, GUI_PAD_SMALL);

	w = gtk_label_new (_("SMTP Server:") );
	gtk_misc_set_alignment (GTK_MISC(w), 1.0, 0.5);
	gtk_label_set_line_wrap (GTK_LABEL(w), TRUE);
	gtk_label_set_justify (GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	gtk_table_attach (GTK_TABLE(table), w, 0, 1, 4, 5, GTK_FILL, GTK_FILL, GUI_PAD, GUI_PAD_SMALL);

	w = druid->smtp_server = gtk_entry_new ();
	fqdn = get_fqdn ();
	smtp_server = g_strdup_printf ("%s", fqdn);
	pan_gtk_entry_set_text (w, smtp_server);
	gtk_table_attach (GTK_TABLE(table), w, 1, 2, 4, 5, GTK_FILL, GTK_FILL, GUI_PAD, GUI_PAD_SMALL);
	
	w = gtk_label_new (_("SMTP Port:") );
	gtk_misc_set_alignment (GTK_MISC(w), 1.0, 0.5);
	gtk_label_set_line_wrap (GTK_LABEL(w), TRUE);
	gtk_label_set_justify (GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	gtk_table_attach (GTK_TABLE(table), w, 0, 1, 5, 6, GTK_FILL, GTK_FILL, GUI_PAD, GUI_PAD_SMALL);

	w = druid->smtp_port = gtk_entry_new ();
	smtp_port = g_strdup ("25");
	pan_gtk_entry_set_text (w, smtp_port);
	gtk_table_attach (GTK_TABLE(table), w, 1, 2, 5, 6, GTK_FILL, GTK_FILL, GUI_PAD, GUI_PAD_SMALL);

	/* cleanup */
	g_free (fqdn);
	g_free (email);
	g_free (smtp_port);
	g_free (smtp_server);
	return table;
}

static void
password_needed_cb (GtkToggleButton * tb, Druid * druid)
{
	const gboolean b = tb->active;
	gtk_widget_set_sensitive (druid->server_username, b);
	gtk_widget_set_sensitive (druid->server_password, b);
	gtk_widget_set_sensitive (druid->server_username_label, b);
	gtk_widget_set_sensitive (druid->server_password_label, b);
}

static char*
get_default_nntp_server (void)
{
	char * retval = NULL;

	if (!retval) {
		const char * cpch = g_getenv ("NNTPSERVER");
		if (is_nonempty_string (cpch))
			retval = g_strdup (cpch);
	}

	if (!retval) {
		char * fqdn = get_fqdn ();
		if (is_nonempty_string (fqdn)) {
			const char * dot = strchr (fqdn, '.');
			if (dot != NULL) {
				retval = g_strdup_printf ("news.%s", dot+1);
			}
		}
		g_free (fqdn);
	}

	return retval;
}

static char*
get_default_profile_name (void)
{
	char * retval = NULL;

	if (!retval) {
		char * pch = get_default_nntp_server ();
		if (is_nonempty_string (pch)) {
			char * lastdot = strrchr (pch, '.');
			if (lastdot != NULL)
				*lastdot = '\0';
			lastdot = strrchr (pch, '.');
			if (lastdot != NULL)
				retval = g_strdup (lastdot+1);
		}
		g_free (pch);
	}

	if (!retval) {
		retval = g_strdup (_("default"));
	}

	return retval;
}

/*
 * Page #3:
 *  "Internet News Server"
 */
static GtkWidget*
druid_server_page (Druid *druid)
{
	GtkWidget * w;
	GtkWidget * table;
	char * default_nntp_server;

	table = gtk_table_new (7, 2, FALSE);
	gtk_table_set_row_spacings (GTK_TABLE(table), GUI_PAD_SMALL);
	gtk_table_set_col_spacings (GTK_TABLE(table), GUI_PAD_SMALL);

	w = gtk_label_new(_("Type the name of your Internet news server."));
	gtk_misc_set_alignment (GTK_MISC(w), 0.0, 0.0);
	gtk_label_set_line_wrap (GTK_LABEL(w), TRUE);
	gtk_label_set_justify (GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	gtk_table_attach (GTK_TABLE(table), w, 0, 2, 0, 1, GTK_FILL, GTK_FILL, GUI_PAD, GUI_PAD_SMALL);

	w = gtk_label_new (_("News (NNTP) server:"));
	gtk_misc_set_alignment (GTK_MISC(w), 1.0, 0.5);
	gtk_table_attach (GTK_TABLE(table), w, 0, 1, 1, 2, GTK_FILL, GTK_FILL, GUI_PAD, GUI_PAD_SMALL);

	w = druid->server_address = gtk_entry_new();
	default_nntp_server = get_default_nntp_server ();
	if (is_nonempty_string (default_nntp_server))
		pan_gtk_entry_set_text (w, default_nntp_server);
	else
		pan_gtk_entry_set_text (w, "news");
	g_free (default_nntp_server);
	gtk_table_attach (GTK_TABLE (table), w, 1, 2, 1, 2, GTK_FILL, GTK_FILL, GUI_PAD, GUI_PAD_SMALL);

	w = gtk_label_new (_("Port:"));
	gtk_misc_set_alignment (GTK_MISC(w), 1.0, 0.5);
	gtk_table_attach (GTK_TABLE(table), w, 0, 1, 2, 3, GTK_FILL, GTK_FILL, GUI_PAD, GUI_PAD_SMALL);

	w = druid->server_port = gtk_entry_new();
	pan_gtk_entry_set_text (w, "119");
	gtk_table_attach (GTK_TABLE(table), w, 1, 2, 2, 3, GTK_FILL, GTK_FILL, GUI_PAD, GUI_PAD_SMALL);

	w = gtk_label_new (_("If you must log on to your news (NNTP) server, then select the check box below."));
	gtk_misc_set_alignment (GTK_MISC(w), 0.0, 0.5);
	gtk_label_set_line_wrap (GTK_LABEL(w), TRUE);
	gtk_label_set_justify (GTK_LABEL(w), GTK_JUSTIFY_LEFT);
 	gtk_table_attach (GTK_TABLE (table), w, 0, 2, 3, 4, GTK_FILL, GTK_FILL, GUI_PAD, GUI_PAD_SMALL);

	w = druid->auth_toggle = gtk_check_button_new_with_label (_("My server requires a password."));
	gtk_signal_connect (GTK_OBJECT(w), "toggled", GTK_SIGNAL_FUNC(password_needed_cb), druid);
 	gtk_table_attach (GTK_TABLE(table), w, 0, 2, 4, 5, GTK_FILL, GTK_FILL, GUI_PAD, GUI_PAD_SMALL);

	druid->server_username_label = w = gtk_label_new (_("Account name:"));
	gtk_misc_set_alignment (GTK_MISC(w), 1.0, 0.5);
 	gtk_table_attach (GTK_TABLE(table), w, 0, 1, 5, 6, GTK_FILL, GTK_FILL, GUI_PAD, GUI_PAD_SMALL);

	druid->server_username = w = gtk_entry_new ();
	gtk_table_attach (GTK_TABLE (table), w, 1, 2, 5, 6, GTK_FILL, GTK_FILL, GUI_PAD, GUI_PAD_SMALL);

	druid->server_password_label = w = gtk_label_new (_("Password:"));
	gtk_misc_set_alignment (GTK_MISC(w), 1.0, 0.5);
 	gtk_table_attach (GTK_TABLE(table), w, 0, 1, 6, 7, GTK_FILL, GTK_FILL, GUI_PAD, GUI_PAD_SMALL);

	druid->server_password = w = gtk_entry_new ();
	gtk_entry_set_visibility(GTK_ENTRY(w), FALSE);
	gtk_table_attach (GTK_TABLE(table), w, 1, 2, 6, 7, GTK_FILL, GTK_FILL, GUI_PAD, GUI_PAD_SMALL);

	password_needed_cb (GTK_TOGGLE_BUTTON(druid->auth_toggle), druid);

	return table;
}


/*
 * Page #4:
 *  "Save Profile"
 */
static GtkWidget *
druid_profile_page (Druid *druid)
{
	char * pch;
	GtkWidget * w = NULL;
	GtkWidget * table = NULL;

	table = gtk_table_new (4, 2, FALSE);
	gtk_table_set_row_spacings (GTK_TABLE(table), GUI_PAD_SMALL);
	gtk_table_set_col_spacings (GTK_TABLE(table), GUI_PAD_SMALL);

	w = gtk_label_new (_("Type in the name you would like Pan to show when referring to this profile you have just setup.  You can name the profile anything you want to."));
	gtk_misc_set_alignment (GTK_MISC(w), 0.0, 0.0);
	gtk_label_set_line_wrap (GTK_LABEL(w), TRUE);
	gtk_label_set_justify (GTK_LABEL(w), GTK_JUSTIFY_LEFT);
 	gtk_table_attach (GTK_TABLE(table), w, 0, 2, 0, 1, GTK_FILL, GTK_FILL, GUI_PAD, GUI_PAD_SMALL);

	w = gtk_label_new (_("Profile Name:"));
	gtk_misc_set_alignment (GTK_MISC (w), 1.0, 0.5);
	gtk_table_attach (GTK_TABLE(table), w, 0, 1, 1, 2, GTK_FILL, GTK_FILL, GUI_PAD, GUI_PAD_SMALL);

	w = druid->profile_name = gtk_entry_new();
	pch = get_default_profile_name ();
	pan_gtk_entry_set_text (w, pch);
	g_free (pch);
	gtk_table_attach (GTK_TABLE(table), w, 1, 2, 1, 2, GTK_FILL, GTK_FILL, GUI_PAD, GUI_PAD_SMALL);

	w = gtk_label_new (_("For example: UUNET Server"));
	gtk_misc_set_alignment (GTK_MISC(w), 0.0, 0.5);
	gtk_table_attach (GTK_TABLE(table), w, 1, 2, 2, 3, GTK_FILL, GTK_FILL, GUI_PAD, GUI_PAD_SMALL);

	return table;
}

static GtkWidget*
create_page (const char * title)
{
	GtkWidget * label;
	GtkWidget * vbox;
	GtkWidget * hbox;
	GtkWidget * image;
	GdkPixbuf * pixbuf;
	char * title_string;

	/* create vbox */
	vbox = gtk_vbox_new (FALSE, GUI_PAD);

	/* create the titlebar */
	hbox = gtk_hbox_new (FALSE, GUI_PAD_BIG);
	pixbuf = gdk_pixbuf_new_from_inline (-1, icon_pan, FALSE, NULL);
	image = gtk_image_new_from_pixbuf (pixbuf);
	g_object_unref (G_OBJECT(pixbuf));
	gtk_box_pack_start (GTK_BOX(hbox), image, FALSE, FALSE, 0);
     	title_string = g_strconcat ("<span size=\"xx-large\" weight=\"ultrabold\">", title ? title : "", "</span>", NULL);
	label = gtk_label_new (title_string);
	gtk_label_set_use_markup (GTK_LABEL(label), TRUE);
	g_free (title_string);
	gtk_box_pack_start (GTK_BOX(hbox), label, FALSE, FALSE, 0);

	/* pack the titlebar */
	gtk_box_pack_start (GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

	/* pack the separator */
	gtk_box_pack_start (GTK_BOX(vbox), gtk_hseparator_new(), FALSE, FALSE, 0);

	return vbox;
}

static GtkWidget*
create_page_with_text (const char * title, const char * text)
{
	GtkWidget * label;
	GtkWidget * page;
       
	page = create_page (title);
	label = gtk_label_new (text);
	gtk_box_pack_start (GTK_BOX(page), label, TRUE, TRUE, 0);

	return page;
}



/*---[ druid_spawn ]--------------------------------------------------
 * spawn the first-run wizard
 *--------------------------------------------------------------------*/
void
dialog_newuser (void)
{
	int i;
	GtkWidget * dialog;
	GtkWidget * notebook;
	GtkWidget * w;
	GtkWidget * alignment;
	Druid * druid;

	/* create the dialog */
	druid = g_new (Druid, 1);
	dialog = gtk_dialog_new_with_buttons (_("Pan Setup Wizard"),
	                                      NULL, 0,
	                                      GTK_STOCK_GO_BACK, PAN_RESPONSE_GO_BACK,
	                                      GTK_STOCK_GO_FORWARD, PAN_RESPONSE_GO_FORWARD,
	                                      GTK_STOCK_SAVE, PAN_RESPONSE_FINISHED,
	                                      GTK_STOCK_QUIT, PAN_RESPONSE_QUIT,
	                                      NULL);
	g_signal_connect (dialog, "response", G_CALLBACK(dialog_response_cb), druid);
	gtk_dialog_set_default_response (GTK_DIALOG(dialog), PAN_RESPONSE_GO_FORWARD);
	gtk_dialog_set_response_sensitive (GTK_DIALOG(dialog), PAN_RESPONSE_GO_BACK, FALSE);
	gtk_dialog_set_response_sensitive (GTK_DIALOG(dialog), PAN_RESPONSE_GO_FORWARD, TRUE);
	gtk_dialog_set_response_sensitive (GTK_DIALOG(dialog), PAN_RESPONSE_FINISHED, FALSE);

	/* create the notebook */
	notebook = druid->notebook = gtk_notebook_new ();
	gtk_notebook_set_show_tabs (GTK_NOTEBOOK(notebook), FALSE);
	gtk_notebook_set_show_border (GTK_NOTEBOOK(notebook), FALSE);
	gtk_box_pack_start (GTK_BOX(GTK_DIALOG(dialog)->vbox), notebook, TRUE, TRUE, 0);

	/**
	*** Initialize the druid pages
	**/

	druid->pages = g_ptr_array_new ();

	/* add welcome page */
	w = create_page (_("Thank you for trying Pan!"));
	g_ptr_array_add (druid->pages, w);
	if (1) {
		GdkPixbuf * pixbuf = gdk_pixbuf_new_from_inline (-1, icon_pan_druid, FALSE, NULL);
		GtkWidget * child = gtk_image_new_from_pixbuf (pixbuf);
		g_object_unref (G_OBJECT(pixbuf));
		gtk_box_pack_start (GTK_BOX(w), child, FALSE, FALSE, 0);
		child = gtk_label_new (_("Thank you for trying Pan!\n \nIf you are new to Usenet,\nhttp://pan.rebelbase.com/intro.php\nis a starting point to see what Usenet is,\nwhere to find a free news server,\nand how to use Pan.\n \nSince this is your first time running Pan,\n we'll need to gather some information.\n"));
		gtk_box_pack_start (GTK_BOX(w), child, FALSE, FALSE, 0);
	}

	/* add user name */
	w = create_page (_("Your Name"));
	alignment = gtk_alignment_new (0.5, 0.5, 0.5, 0.5);
	gtk_box_pack_start (GTK_BOX(w), alignment, TRUE, TRUE, 0);
	gtk_container_add (GTK_CONTAINER(alignment), druid_name_page(druid));
	g_ptr_array_add (druid->pages, w);

	/* add email */
	w = create_page (_("E-Mail"));
	alignment = gtk_alignment_new (0.5, 0.5, 0.5, 0.5);
	gtk_box_pack_start (GTK_BOX(w), alignment, TRUE, TRUE, 0);
	gtk_container_add (GTK_CONTAINER(alignment), druid_email_page(druid));
	g_ptr_array_add (druid->pages, w);

	/* news server */
	w = create_page (_("News Server"));
	alignment = gtk_alignment_new (0.5, 0.5, 0.5, 0.5);
	gtk_box_pack_start (GTK_BOX(w), alignment, TRUE, TRUE, 0);
	gtk_container_add (GTK_CONTAINER(alignment), druid_server_page(druid));
	g_ptr_array_add (druid->pages, w);

	/* save profile */
	w = create_page (_("Profile Name"));
	alignment = gtk_alignment_new (0.5, 0.5, 0.5, 0.5);
	gtk_box_pack_start (GTK_BOX(w), alignment, TRUE, TRUE, 0);
	gtk_container_add (GTK_CONTAINER(alignment), druid_profile_page(druid));
	g_ptr_array_add (druid->pages, w);

	/* finished */
	w = create_page_with_text (_("Finished!"), _("You have successfully entered all of\nthe information required to set up Pan.\n\nTo save these settings, click Save."));
	g_ptr_array_add (druid->pages, w);

	/* populate notebook */
	for (i=0; i<druid->pages->len; ++i)
		gtk_notebook_append_page (GTK_NOTEBOOK(notebook), GTK_WIDGET(g_ptr_array_index(druid->pages,i)), NULL);

	gtk_widget_show_all (dialog);
}
