/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999, 2000, 2001  Pan Development Team <pan@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>

#include <ctype.h>
#include <errno.h>
#include <limits.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <dirent.h>

#include <gmime/gmime.h>
#include <gmime/memchunk.h>

#include <pan/base/acache.h>
#include <pan/base/base-prefs.h>
#include <pan/base/debug.h>
#include <pan/base/pan-i18n.h>
#include <pan/base/pan-glib-extensions.h>
#include <pan/base/util-file.h>
#include <pan/base/log.h>

PanCallback* acache_bodies_added_callback = NULL;
PanCallback* acache_bodies_removed_callback = NULL;

static size_t acache_size = 0;
static GStaticMutex entries_mutex = G_STATIC_MUTEX_INIT;
static MemChunk * acache_entry_chunk = NULL;
static GHashTable * messageid_to_entry = NULL;

typedef struct
{
	gint refcount;
	char * message_id;
	size_t size;
	time_t date;
}
AcacheEntry;

/***
****
***/

static gboolean
get_file_message_id (const char * filename, char * msgid, gint msgid_len)
{
	char buf [2048];
	char * retval;
	FILE * fp;
	debug_enter ("get_file_message_id");

	g_return_val_if_fail (is_nonempty_string(filename), FALSE);
	g_return_val_if_fail (msgid!=NULL, FALSE);
	g_return_val_if_fail (msgid_len!=0, FALSE);

	/* try to find the line */
	*buf = '\0';
	fp = fopen (filename, "r");
	if (fp != NULL) {
		const char * key = "Message-I";
		const size_t key_len = 9;
		gboolean found;
		for (found=FALSE; !found && fgets(buf,sizeof(buf),fp); ) {
			if (buf[0]=='\0' || buf[1]=='\0') /* watch for header/body delimiter */
				break;
			if (!memcmp (key, buf, key_len))
				found = TRUE;
		}
		if (!found)
			*buf = '\0';
		fclose (fp);
	}

	/* parse the line */
	retval = NULL;
	if (*buf != '\0') {
		char * pch = strchr (buf, ':');
		if (pch != NULL) {
			++pch;
			while (*pch && isspace((guchar)*pch))
				++pch;
			retval = pch;
			while (*pch && !isspace((guchar)*pch))
				++pch;
			*pch = '\0';
		}
	}

	if (retval != NULL) 
		g_snprintf (msgid, msgid_len, "%s", retval);

	debug_exit ("get_file_message_id");
	return retval != NULL;
}

/***
****
****  PATHS & FILENAMES
****
***/

static char*
acache_get_path (void)
{
	char * pch = g_strdup_printf ("%s%ccache", get_data_dir(), G_DIR_SEPARATOR);
	pan_normalize_filename_inplace (pch);
	return pch;

}

static char*
acache_get_filename (const char * message_id, char * buf, gint len)
{
	register char * pch;
	const register char * cpch;
	const char * data_dir = get_data_dir ();
	const int data_dir_len = strlen (data_dir);
	debug_enter ("acache_get_filename");

	/* sanity clause */
	g_return_val_if_fail (buf!=NULL, NULL);
	g_return_val_if_fail (is_nonempty_string(message_id), NULL);
	g_return_val_if_fail (len >= strlen(message_id)+data_dir_len+13, NULL);

	/* build acache string.  data_dir/cache/msg-id.msg */
	pch = buf;
	memcpy (pch, data_dir, data_dir_len); pch+=data_dir_len;
	*pch++ = G_DIR_SEPARATOR;
	memcpy (pch, "cache", 5); pch+=5;
	*pch++ = G_DIR_SEPARATOR;
	for (cpch=message_id; *cpch!='\0'; ++cpch)
		if (*cpch!='%' && *cpch!='$' && *cpch!='<' && *cpch!='>' && *cpch!='/' && *cpch!='\\')
			*pch++ = *cpch;
	strcpy (pch, ".msg");
	pan_normalize_filename_inplace (buf);

	debug_exit ("acache_get_filename");
	return is_nonempty_string(buf) ? buf : NULL;
}

/***
****
****  INIT / SHUTDOWN
****
***/

void
acache_init (void)
{
	char * pch;
	struct dirent * dirent_p;
	char * dirpath;
	DIR * dir_p;
	char message_id[1024];
	char path [PATH_MAX];
	debug_enter ("acache_init");

	/* we don't need to call this multiple times */
	if (acache_entry_chunk != NULL)
		return;

	/* ensure the cache directory exists */
	pch = acache_get_path ();
	directory_check (pch);
	g_free (pch);

	/* init the callbacks */
	acache_bodies_added_callback = pan_callback_new ();
	acache_bodies_removed_callback = pan_callback_new ();

	/* init the message id hash  */
	acache_entry_chunk = memchunk_new (sizeof(AcacheEntry), 256, FALSE);
	messageid_to_entry = g_hash_table_new (g_str_hash, g_str_equal);
	acache_size = 0;
	dirpath = acache_get_path ();
	dir_p = opendir (dirpath);
	if (dir_p == NULL) {
		if (errno)
			log_add_va (LOG_ERROR, _("Error reading \"%s\": %s."), dirpath, g_strerror(errno));
		else
			log_add_va (LOG_ERROR, _("Error reading \"%s\"."), dirpath);
	}

	while (dir_p!=NULL && (dirent_p = readdir(dir_p)))
	{
		struct stat stat_p;

		if (!string_ends_with(dirent_p->d_name, ".msg"))
			continue;

		g_snprintf (path, sizeof(path), "%s%c%s", dirpath, G_DIR_SEPARATOR, dirent_p->d_name);
		if (stat(path, &stat_p) == 0)
		{
			if (get_file_message_id (path, message_id, sizeof(message_id)))
			{
				AcacheEntry * entry = (AcacheEntry*) memchunk_alloc (acache_entry_chunk);
				entry->message_id = g_strdup (message_id);
				entry->size = stat_p.st_size;
				entry->date = stat_p.st_mtime;
				entry->refcount = 0;
				g_hash_table_insert (messageid_to_entry, entry->message_id, entry);
				acache_size += entry->size;
			}
		}
	}

	if (dir_p!=NULL)
		closedir (dir_p);

	g_free (dirpath);

	log_add_va (LOG_INFO, _("Article cache contains %.1f MB in %d files"),
		(double)acache_size/(1024*1024),
		(int)g_hash_table_size(messageid_to_entry));

	debug_exit ("acache_init");
}

static void
acache_shutdown_ghfunc (gpointer key, gpointer val, gpointer data)
{
	AcacheEntry * entry = (AcacheEntry*)val;
	g_free (entry->message_id);
}
void
acache_shutdown (void)
{
	if (acache_flush_on_exit)
		acache_expire_all ();

	g_hash_table_foreach (messageid_to_entry, acache_shutdown_ghfunc, NULL);
	g_hash_table_destroy (messageid_to_entry);
	messageid_to_entry = NULL;

	memchunk_destroy (acache_entry_chunk);
	acache_entry_chunk = NULL;
}

/***
****
****  CHECKIN / CHECKOUT
****
***/

/* this must be called inside an entries_mutex lock */
static void
acache_file_update_refcount_nolock (const char ** message_ids,
                                    int           message_id_qty,
                                    int           inc)
{
	gint i;

	g_return_if_fail (message_ids != NULL);
	g_return_if_fail (message_id_qty >= 1);

	for (i=0; i<message_id_qty; ++i)
	{
		const char * message_id = message_ids[i];
		AcacheEntry * entry;

		entry = g_hash_table_lookup (messageid_to_entry, message_id);
		if (entry == NULL)
		{
			entry = (AcacheEntry*) memchunk_alloc (acache_entry_chunk);
			entry->refcount = 0;
			entry->message_id = g_strdup (message_id);
			entry->size = 0;
			entry->date = time (NULL);
			g_hash_table_insert (messageid_to_entry, entry->message_id, entry);
			debug1 (DEBUG_ACACHE, "Added new entry %d", entry->message_id);
		}

		/* If we're checking it out, then move it to the safe end of the
		   least-recently-used kill heuristic */
		if (inc > 0)
			entry->date = time (NULL);

		entry->refcount += inc;
		debug3 (DEBUG_ACACHE, "%s refcount - inc by %d to %d", entry->message_id, inc, entry->refcount);
	}
}

void
acache_file_checkout (const char ** message_ids,
                      int           message_id_qty)
{
	g_return_if_fail (message_ids!=NULL);
	g_return_if_fail (message_id_qty >= 1);

	g_static_mutex_lock (&entries_mutex);
	acache_file_update_refcount_nolock (message_ids, message_id_qty, 1);
	g_static_mutex_unlock (&entries_mutex);
}
void
acache_file_checkin (const char ** message_ids,
                     int           message_id_qty)
{
	g_return_if_fail (message_ids != NULL);
	g_return_if_fail (message_id_qty >= 1);

	g_static_mutex_lock (&entries_mutex);
	acache_file_update_refcount_nolock (message_ids, message_id_qty, -1);
	g_static_mutex_unlock (&entries_mutex);

	acache_expire ();
}


void
acache_file_checkout_fps (const char   ** message_ids,
                          int             message_id_qty,
                          FILE        *** fps_allocme)
{
	gint i;
	FILE ** fps = NULL;

	/* sanity clause */
	g_return_if_fail (message_ids!=NULL);
	g_return_if_fail (message_id_qty >= 1);

	/* ref them all */
	g_static_mutex_lock (&entries_mutex);
	acache_file_update_refcount_nolock (message_ids, message_id_qty, 1);
	g_static_mutex_unlock (&entries_mutex);

	/* get the FPs */
	fps = g_new0 (FILE*, message_id_qty);
	for (i=0; i<message_id_qty; ++i) {
		const char * message_id = message_ids[i];
		if (acache_has_message (message_id)) {
			char filename[PATH_MAX];
			if (acache_get_filename (message_id, filename, PATH_MAX)) {
				errno = 0;
				fps[i] = fopen (filename, "r");
				if (fps[i] == NULL)
					log_add_va (LOG_ERROR, _("Error opening decode file `%s': %s"), filename, g_strerror(errno));
			}
		}
	}

	*fps_allocme = fps;
}

/***
****  ADD / REMOVE FILES
***/

#if 0
static void
utf8ize_message_parts (GMimePart     * part,
                       gpointer        data)
{
	const char * charset;
	const char * default_charset = data;
	const GMimeDataWrapper * wrapper;
	GMimeFilter * filter;


	/* build a stream filter around the orignal stream */
       	wrapper = g_mime_part_get_content_object (part);
	original_stream = g_mime_data_wrapper_get_stream (wrapper);
	stream_filter = g_mime_stream_filter_new_with_stream (original_stream);

	/* make a utf-8 charset filter for the stream filter */
       	charset = g_mime_object_get_content_type_parameter (GMIME_OBJECT(part), "charset");
	if (charset == NULL)
		charset = default_charset;
       	filter = g_mime_filter_charset_new (charset, "UTF-8");
	if (filter != NULL) {
		g_mime_stream_filter_add (GMIME_STREAM_FILTER(stream_filter), filter);
		g_mime_data_wrapper_set_stream (wrapper, stream_filter);
	}

	g_object_unref (stream_filter);
}
#endif

void
acache_set_message (const char    * message_id,
                    const char    * message,
                    guint           message_len,
                    gboolean        checkout,
                    const char    * default_charset)
{
       	FILE * fp = NULL;
	char filename[PATH_MAX];
	AcacheEntry * entry;

	g_return_if_fail (is_nonempty_string(message));
	g_return_if_fail (is_nonempty_string(message_id));

	/* find out where to write the message */
	if (acache_get_filename (message_id, filename, PATH_MAX))
		fp = fopen (filename, "w+");

	if (fp != NULL)
	{
#if 0
		GMimeParser * parser;
		GMimeStream * stream;
		GMimeMessage * gmessage;
		char * gmessage_txt;
		size_t gmessage_txt_len;
		size_t bytes_written;

		stream = g_mime_stream_mem_new_with_buffer (message, message_len);
		parser = g_mime_parser_new ();
		g_mime_parser_init_with_stream (parser, stream);
		g_object_unref (stream);
		gmessage = g_mime_parser_construct_message (parser);
		g_object_unref (parser);

		if (GMIME_IS_MULTIPART (gmessage->mime_part))
			g_mime_multipart_foreach (GMIME_MULTIPART(gmessage->mime_part), utf8ize_message_parts, default_charset);
		else
			utf8ize_message_parts (GMIME_PART(gmessage->mime_part), default_charset);

		gmessage_txt = g_mime_message_to_string (gmessage);
		gmessage_txt_len = strlen (gmessage_txt);
		bytes_written = fwrite (gmessage_txt, sizeof(char), gmessage_txt_len, fp);
#else
		const size_t bytes_written = fwrite (message, sizeof(char), message_len, fp);
		fclose (fp);
#endif

		if (bytes_written < message_len)
		{
			/* couldn't save the whole message */
			char * path = acache_get_path ();
			log_add_va (LOG_ERROR, _("Error saving article \"%s\" (is %s full?)"), message_id, path);
			g_free (path);
		}
		else
		{
			/* update our tables */
			g_static_mutex_lock (&entries_mutex);
			acache_file_update_refcount_nolock (&message_id, 1, checkout?1:0);
			entry = g_hash_table_lookup (messageid_to_entry, message_id);
			g_assert (entry!=NULL);
			entry->size = message_len;
			entry->date = time (NULL);
			g_static_mutex_unlock (&entries_mutex);

			/* notify everyone that this message has been added */
			pan_callback_call (acache_bodies_added_callback, (gpointer)message_id, NULL);

			/* if the acache is too big, purge the old stuff */
			acache_size += entry->size;
			acache_expire ();
		}
	}
}

/**
 * This must be called from inside an entries_mutex lock.
 */
static void
acache_remove_entry (AcacheEntry * entry)
{
	char filename[PATH_MAX] = { '\0' };

	/* sanity clause */
	g_return_if_fail (entry!=NULL);

	/* let everyone know */
	debug2 (DEBUG_ACACHE, "Removing %s from cache with refcount %d", entry->message_id, entry->refcount);
	pan_callback_call (acache_bodies_removed_callback, (gpointer)entry->message_id, NULL);

	/* update the acache size */
	acache_size -= entry->size;

	/* remove the file */
	acache_get_filename (entry->message_id, filename, PATH_MAX);
	unlink (filename);

	/* update acache tables */
	g_hash_table_remove (messageid_to_entry, entry->message_id);
	g_free (entry->message_id);
	memchunk_free (acache_entry_chunk, entry);
}

char*
acache_get_message (const char * message_id, gboolean need_zero_termination)
{
	char * retval = NULL;
	g_return_val_if_fail (is_nonempty_string(message_id), NULL);

	if (acache_has_message (message_id)) {
		char filename[PATH_MAX];
		if (acache_get_filename (message_id, filename, PATH_MAX)) {
			gsize len = 0;
			g_file_get_contents (filename, &retval, &len, NULL);
			if (need_zero_termination && retval[len-1]!='\0') {
				retval = g_realloc (retval, len+1);
				retval[len] = '\0';
			}
		}
	}
	return retval;
}



gboolean
acache_has_message (const char * message_id)
{
	gboolean retval = FALSE;
	AcacheEntry * entry = NULL;

	g_return_val_if_fail (is_nonempty_string(message_id), FALSE);

	entry = g_hash_table_lookup (messageid_to_entry, message_id);
	if (entry != NULL)
		retval = entry->size != 0;

	return retval;
}

/***
****
****  EXPIRE
****
***/

static int
compare_ppEntry_ppEntry_by_youth (gconstpointer a, gconstpointer b, gpointer unused)
{
	const time_t aval = (**(AcacheEntry**)a).date;
	const time_t bval = (**(AcacheEntry**)b).date;
	return (gint) -difftime (aval, bval);
}


void
acache_expire_messages (const char   ** message_ids,
	                int             message_id_qty)
{
	gint i;

	g_return_if_fail (message_ids!=NULL);
	g_return_if_fail (message_id_qty>0);

	g_static_mutex_lock (&entries_mutex);

	for (i=0; i<message_id_qty; ++i)
	{
		AcacheEntry * entry = NULL;
		const char * message_id = NULL;

		/* find the entry */
		message_id = message_ids[i];
		entry = (AcacheEntry*) g_hash_table_lookup (messageid_to_entry, message_id);

		/* if we have such an entry, remove it */
		if (entry != NULL)
			acache_remove_entry (entry);
	}

	memchunk_clean (acache_entry_chunk);

	g_static_mutex_unlock (&entries_mutex);
}


static gint
acache_expire_to_size (gdouble max_megs)
{
	gint files_removed = 0;
	const size_t cache_max = (size_t)max_megs * (1024 * 1024);
	debug_enter ("acache_expire_to_size");

	/* lock the entries mutex so that our array doesn't wind up holding
	   pointers to free'd entries */
	g_static_mutex_lock (&entries_mutex);

	debug2 (DEBUG_ACACHE, "expiring to %lu; current size is %lu", cache_max, acache_size);

	if (cache_max < acache_size)
	{
		gint i;
		GPtrArray * entries;

		/* get an array of files sorted by age */
		entries = g_ptr_array_new ();
		pan_hash_to_ptr_array (messageid_to_entry, entries);
		g_ptr_array_sort_with_data (entries, compare_ppEntry_ppEntry_by_youth, NULL);

		/* Start blowing away files */
		for (i=entries->len; i>0 && cache_max<acache_size; --i) {
			AcacheEntry * entry = (AcacheEntry*) g_ptr_array_index (entries, i-1);
			if (entry->refcount <= 0) {
				acache_remove_entry (entry);
				entry = NULL;
				++files_removed;
			}
		}

		/*  cleanup */
		g_ptr_array_free (entries, TRUE);
		memchunk_clean (acache_entry_chunk);
	}

	/* log */
	if (files_removed != 0) {
		log_add_va (LOG_INFO, _("Removed %d articles from local cache"), files_removed);
		log_add_va (LOG_INFO, _("Article cache now has %.1f MB in %d articles"),
			(double)(acache_size/(1024.0*1024.0)),
			 (gint)g_hash_table_size(messageid_to_entry));
	}

	/* entries mutex lock */
	g_static_mutex_unlock (&entries_mutex);

	/* done */
	debug_exit ("acache_expire_to_size");
	return files_removed;
}

gint 
acache_expire (void)
{
	return acache_expire_to_size (acache_max_megs * 0.8);
}

gint
acache_expire_all (void)
{
	gint retval;
	printf ("Pan is flushing article cache... ");
	retval = acache_expire_to_size(0);
	printf ("%d files erased.\n", retval);
	fflush (NULL);
	return retval;
}
