/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Author: Charles Kerr <charles@rebelbase.com>
 *
 * Copyright (C) 2000, 2001  Pan Development Team <pan@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>
#include <string.h>

#include <glib.h>
#include <gtk/gtk.h>

#include <pan/base/article.h>
#include <pan/base/fnmatch.h>
#include <pan/base/pan-i18n.h>
#include <pan/base/pan-glib-extensions.h>

#include <pan/article-find.h>
#include <pan/articlelist.h>
#include <pan/globals.h>
#include <pan/gui.h>
#include <pan/util.h>

enum
{
	PAN_RESPONSE_FIND_NEXT = 1,
	PAN_RESPONSE_FIND_PREV = 2
};

static GtkWidget *find_dialog;

static void
find_dialog_destroy_cb (void)
{
	pan_lock();
	gtk_widget_destroy (find_dialog);
	pan_unlock();

	find_dialog = NULL;
}

typedef struct
{
	gboolean unread_only;
	gboolean case_sensitive;

	const gchar *author;
	const gchar *subject;
	const gchar *message_id;
} 
header_compare_data;

/**
 * Return 0 if equal
 */
static gboolean
compare_article (const Article* article,
                 const header_compare_data* header )
{
	int flags = header->case_sensitive ? 0 : PAN_CASEFOLD;

	/* if it ain't an article, it ain't a match */
	if (!article)
		return -1;

	/* make sure the unread state (if user specified) matches... */
	if (header->unread_only && article_is_read (article))
		return -1;

	/* make sure the author (if user specified one) matches.. */
	if (header->author && (!article->author_real || fnmatch(header->author,article->author_real,flags))
	                   && (!article->author_addr || fnmatch(header->author,article->author_addr,flags)) )
		return -1;

	/* make sure the subject (if user specified one) matches.. */
	if (header->subject && fnmatch(header->subject,article_get_subject(article),flags))
		return -1;

	/* make sure the message-id (if user specified one) matches.. */
	if (header->message_id && fnmatch(header->message_id,article_get_message_id(article),flags))
		return -1;

	/* hooray, a match */
	return 0;
}

static void
find_article (const gchar* author,
              const gchar* subject,
              const gchar* message_id,
              gboolean case_sensitive,
              gboolean unread_only,
              gboolean reverse)
{
	gchar * author_tmp;
	gchar * subject_tmp;
	gchar * message_id_tmp;
	GtkCTreeNode * sel;
	GtkCTreeNode * tmp;
	GtkCTree* tree = GTK_CTREE(Pan.article_ctree);
	header_compare_data header_data;

	/* wrap the key in wildcards to make it a substring search...
	   unless there are already wildcards present, indicating the
	   user already knows what he wants */
	subject_tmp = NULL;
	if (subject != NULL) {
		if (strchr(subject,'*'))
			subject_tmp = g_strdup (subject);
		else
			subject_tmp = g_strdup_printf ("*%s*", subject);
	}

	author_tmp = NULL;
	if (author != NULL) {
		if (strchr(author,'*'))
			author_tmp = g_strdup (author);
		else
			author_tmp = g_strdup_printf ("*%s*", author);
	}

	message_id_tmp = NULL;
	if (message_id != NULL) {
		if (strchr(message_id,'*'))
			message_id_tmp = g_strdup (message_id);
		else
			message_id_tmp = g_strdup_printf ("*%s*", message_id);
	}

	/* sanity checks */
	g_return_if_fail (articlelist_get_group()!=NULL);

	/* set up search function struct */
	header_data.author = author_tmp;
	header_data.subject = subject_tmp;
	header_data.message_id = message_id_tmp;
	header_data.unread_only = unread_only;
	header_data.case_sensitive = case_sensitive;

	/* search */
	sel = articlelist_get_selected_node();
	if (!sel)
		sel = gtk_ctree_node_nth (tree, 0);
	tmp = sel;
	for ( ;; )
	{
		const Article* article = NULL;

		/* next node */
		tmp = reverse
			? articlelist_node_prev (tmp, FALSE)
			: articlelist_node_next (tmp);
		if (!tmp) { /* wrap */
			int n = reverse
				? GTK_CLIST(tree)->rows-1
				: 0;
			tmp = gtk_ctree_node_nth (tree, n);
			if (tmp==sel)
				break;
			tmp = reverse
				? articlelist_node_prev (tmp, FALSE)
				: articlelist_node_next (tmp);
		}
		if (tmp==sel)
			break;

		/* compare */
		article = (const Article*)
			gtk_ctree_node_get_row_data (tree, tmp);
		if (!compare_article (article, &header_data))
			break;
	}

	/* select the node.. */
	if (tmp!=sel)
		articlelist_set_selected_nodes_nolock (&tmp, 1);

	/* cleanup */
	g_free (author_tmp);
	g_free (subject_tmp);
}

static gchar * find_article__author = NULL;
static gchar * find_article__subject = NULL;
static gchar * find_article__message_id = NULL;
static gboolean find_article__case_sensitive = FALSE;
static gboolean find_article__unread_only = FALSE;
static gboolean find_article__reverse = FALSE;


static void
find_dialog_response_cb (GtkDialog     * dialog,
                         int             response,
                         GtkWidget     * table)
{
	g_return_if_fail (table!=NULL);

	if (response==PAN_RESPONSE_FIND_PREV || response==PAN_RESPONSE_FIND_NEXT)
	{
		GtkWidget * author_entry;
		GtkWidget * subject_entry;
		GtkWidget * message_id_entry;
		GtkWidget * case_sensitive_checkbox;
		GtkWidget * unread_only_checkbox;
		gchar * author;
		gchar * subject;
		gchar * message_id;
		gboolean case_sensitive;
		gboolean unread_only;

		/* get author to search for, or NULL if none... */
		author_entry = (GtkWidget*) gtk_object_get_data ( GTK_OBJECT(find_dialog), "author" );
		g_return_if_fail ( author_entry!=NULL );
		author = g_strdup ( gtk_entry_get_text ( GTK_ENTRY(author_entry) ) );
		g_strstrip (author);
		if (!*author)
			replace_gstr (&author, NULL);

		/* get subject to search for, or NULL if none... */
		subject_entry = (GtkWidget*) gtk_object_get_data ( GTK_OBJECT(find_dialog), "subject" );
		g_return_if_fail ( subject_entry!=NULL );
		subject = g_strdup ( gtk_entry_get_text ( GTK_ENTRY(subject_entry) ) );
		g_strstrip (subject);
		if (!*subject)
			replace_gstr (&subject, NULL);

		/* get message-id to search for, or NULL if none...*/
		message_id_entry = (GtkWidget*) gtk_object_get_data ( GTK_OBJECT(find_dialog), "message-id" );
		g_return_if_fail ( message_id_entry != NULL );
		message_id = g_strdup ( gtk_entry_get_text ( GTK_ENTRY(message_id_entry) ) );
		g_strstrip (message_id);
		if (!*message_id)
			replace_gstr (&message_id, NULL);

		/* get case sensitivity... */
		case_sensitive_checkbox = (GtkWidget*) gtk_object_get_data ( GTK_OBJECT(find_dialog), "case" );
		g_return_if_fail ( case_sensitive_checkbox!=NULL );
		case_sensitive = gtk_toggle_button_get_active ( GTK_TOGGLE_BUTTON(case_sensitive_checkbox) );

		/* get case sensitivity... */
		unread_only_checkbox = (GtkWidget*) gtk_object_get_data ( GTK_OBJECT(find_dialog), "unread" );
		g_return_if_fail ( unread_only_checkbox!=NULL );
		unread_only = gtk_toggle_button_get_active ( GTK_TOGGLE_BUTTON(unread_only_checkbox) );

		/* update the find_article fields with these widget values */
		replace_gstr (&find_article__author, author);
		replace_gstr (&find_article__subject, subject);
		replace_gstr (&find_article__message_id, message_id);
		find_article__case_sensitive = case_sensitive;
		find_article__unread_only = unread_only;

		find_article__reverse = response == PAN_RESPONSE_FIND_PREV;
		articlelist_find_next_cb ( );
	}
	else
	{
		gtk_widget_unmap (GTK_WIDGET(find_dialog));
	}
}

void
articlelist_find_next_cb (void)
{
	/* user may have hit "find again" without loading a group first */
	if (!articlelist_get_group())
		return;

	/* user may have hit "find again" without hitting "find" first */
	if (!find_article__subject && !find_article__author && !find_article__message_id) {
		articlelist_find_text_cb();
		return;
	}

	/* got a subject and/or an author.. let's go find 'em */
	find_article (find_article__author,
	              find_article__subject,
	              find_article__message_id,
	              find_article__case_sensitive,
	              find_article__unread_only,
	              find_article__reverse);

	find_article__reverse = FALSE;
}


void
articlelist_find_text_cb (void)
{
	GtkWidget * table;
	GtkWidget * author_label;
	GtkWidget * author_entry;
	GtkWidget * subject_label;
	GtkWidget * subject_entry;
	GtkWidget * message_id_label;
	GtkWidget * message_id_entry;
	GtkWidget * case_sensitive_checkbutton;
	GtkWidget * unread_only_checkbutton;
	guint       row;

	/* There can be only one! (find window) */
	if (find_dialog != NULL)
	{
		gtk_widget_show_all (find_dialog);
		gtk_widget_map (find_dialog);
		return;
	}

	find_dialog = gtk_dialog_new_with_buttons (_("Pan: Find Message"),
	                                           GTK_WINDOW(Pan.window),
	                                           GTK_DIALOG_DESTROY_WITH_PARENT,
	                                           GTK_STOCK_GO_BACK, PAN_RESPONSE_FIND_PREV,
	                                           GTK_STOCK_GO_FORWARD, PAN_RESPONSE_FIND_NEXT,
	                                           GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE,
	                                           NULL);

	table = gtk_table_new (5, 2, FALSE);

	row = 0;
	subject_label = gtk_label_new ( _("Subject:") );
	gtk_table_attach ( GTK_TABLE(table),
			   subject_label,
			   0, 1,
			   row, row+1,
			   0,
			   0,
			   GUI_PAD_SMALL, GUI_PAD_SMALL );

	subject_entry = gtk_entry_new ( );
	gtk_object_set_data (GTK_OBJECT(find_dialog), "subject", subject_entry);
	gtk_table_attach (GTK_TABLE(table),
			  subject_entry,
			  1, 2,
			  row, row+1,
			  GTK_EXPAND|GTK_FILL|GTK_SHRINK,
			  0,
			  GUI_PAD_SMALL, GUI_PAD_SMALL );

	++row;
	author_label = gtk_label_new (_("Author:"));
	gtk_table_attach (GTK_TABLE(table),
			  author_label,
			  0, 1,
			  row, row+1,
			  0,
			  0,
			  GUI_PAD_SMALL, GUI_PAD_SMALL );

	author_entry = gtk_entry_new ( );
	gtk_object_set_data ( GTK_OBJECT(find_dialog), "author", author_entry );
	gtk_table_attach (GTK_TABLE(table),
			  author_entry,
			  1, 2,
			  row, row+1,
			  GTK_EXPAND|GTK_FILL|GTK_SHRINK,
			  0,
			  GUI_PAD_SMALL, GUI_PAD_SMALL );

	++row;
	message_id_label = gtk_label_new (_("Message-ID:"));
	gtk_table_attach (GTK_TABLE(table),
			message_id_label,
			0, 1,
			row, row+1,
			0,
			0,
			GUI_PAD_SMALL, GUI_PAD_SMALL );

	message_id_entry = gtk_entry_new ();
	gtk_object_set_data ( GTK_OBJECT(find_dialog), "message-id", message_id_entry );
	gtk_table_attach (GTK_TABLE(table),
			message_id_entry,
			1, 2,
			row, row+1,
			GTK_EXPAND|GTK_FILL|GTK_SHRINK,
			0,
			GUI_PAD_SMALL, GUI_PAD_SMALL );

	++row;
	unread_only_checkbutton =
		gtk_check_button_new_with_label (_("Unread Only"));
	gtk_toggle_button_set_active (
		GTK_TOGGLE_BUTTON(unread_only_checkbutton), FALSE);
	gtk_object_set_data (GTK_OBJECT(find_dialog),
		"unread", unread_only_checkbutton );
	gtk_table_attach ( GTK_TABLE(table),
			   unread_only_checkbutton,
			   0, 2,
			   row, row+1,
			   GTK_EXPAND | GTK_FILL | GTK_SHRINK,
			   GTK_EXPAND | GTK_FILL,
			   GUI_PAD_SMALL, GUI_PAD_SMALL );

	++row;
	case_sensitive_checkbutton = gtk_check_button_new_with_label ( _("Case Sensitive") );
	gtk_toggle_button_set_active ( GTK_TOGGLE_BUTTON(case_sensitive_checkbutton), FALSE );
	gtk_object_set_data ( GTK_OBJECT(find_dialog), "case", case_sensitive_checkbutton );
	gtk_table_attach ( GTK_TABLE(table),
			   case_sensitive_checkbutton,
			   0, 2,
			   row, row+1,
			   GTK_EXPAND | GTK_FILL | GTK_SHRINK,
			   GTK_EXPAND | GTK_FILL,
			   GUI_PAD_SMALL, GUI_PAD_SMALL );
	
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG(find_dialog)->vbox), table, FALSE, FALSE, 0);
	
	g_signal_connect (find_dialog, "response",
	                  G_CALLBACK(find_dialog_response_cb), table);
	g_signal_connect (find_dialog, "destroy",
	                  G_CALLBACK(find_dialog_destroy_cb), NULL);

	gtk_widget_grab_focus (subject_entry);

	gtk_widget_show_all (find_dialog);
}
