/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999, 2000, 2001  Pan Development Team <pan@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

/*********************
**********************  Includes
*********************/

#include <config.h>

#include <errno.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <time.h>

#include <glib.h>

#include <pan/base/acache.h>
#include <pan/base/article.h>
#include <pan/base/base-prefs.h>
#include <pan/base/debug.h>
#include <pan/base/decode.h>
#include <pan/base/log.h>
#include <pan/base/pan-i18n.h>
#include <pan/base/pan-glib-extensions.h>
#include <pan/base/util-file.h>

#include <pan/globals.h>
#include <pan/nntp.h>
#include <pan/queue.h>
#include <pan/task-save.h>

/*********************
**********************  Defines / Enumerated types
*********************/

/*********************
**********************  Macros
*********************/

/*********************
**********************  Structures / Typedefs
*********************/

/*********************
**********************  Private Function Prototypes
*********************/

static gint task_save_run (Task* task);

static char* task_save_describe (const StatusItem* item);

static void task_save_prune (TaskSave  * task);

/*********************
**********************  Variables
*********************/

/***********
************  Extern
***********/

/***********
************  Public
***********/

/***********
************  Private
***********/

/*********************
**********************  BEGINNING OF SOURCE
*********************/

static char*
task_save_describe (const StatusItem* item)
{
	TaskSave *save = TASK_SAVE(item);

	/* sanity checks */
	g_return_val_if_fail (item!=NULL, NULL);
	g_return_val_if_fail (save->readable_name!=NULL, NULL);

        return g_strdup_printf (_("Saving \"%s\""), save->readable_name);
}

/************
*************  PUBLIC ROUTINES
************/

/*****
******
*****/

static void
task_save_destructor (PanObject* object)
{
	TaskSave * task = TASK_SAVE(object);

	/* destruct the TaskSave parts */
        debug1 (DEBUG_PAN_OBJECT, "task_save destructor: %p", object);
	replace_gstr (&task->readable_name, NULL);
	replace_gstr (&task->path_attachments, NULL);
	replace_gstr (&task->filename_attachments, NULL);
	replace_gstr (&task->path_bodies, NULL);
	replace_gstr (&task->filename_bodies, NULL);

	/* deflag */
	if (1) {
		GSList * l;
		GPtrArray * a = g_ptr_array_new ();
		for (l=task->articles; l!=NULL; l=l->next)
			g_ptr_array_add (a, l->data);
		articles_remove_flag ((Article**)a->pdata, a->len, STATE_SAVE_QUEUED);
		g_ptr_array_free (a, TRUE);
	}

	/* checkin the article bodies */
	if (1) {
		GSList * l;
		GPtrArray * a = g_ptr_array_new ();
		for (l=task->articles; l!=NULL; l=l->next)
			g_ptr_array_add (a, (gpointer)article_get_message_id(ARTICLE(l->data)));
		acache_file_checkin ((const gchar**)a->pdata, a->len);
		if (a->len>1 && task->runval==TASK_SUCCESS)
			acache_expire_messages ((const gchar**)a->pdata, a->len);
		g_ptr_array_free (a, TRUE);
	}

	/* unref articles */
	g_slist_free (task->articles);
	group_unref_articles (task->group, NULL);

	/* destruct the superclass */
	task_destructor (PAN_OBJECT(object));
}

PanObject*
task_save_new (Group          * group,
               const char     * readable_name,
               const GSList   * articles)
{
	GSList * l;
	GPtrArray * a;
	TaskSave * task;
	gboolean need_socket;

	g_return_val_if_fail (group_is_valid(group), NULL);
	g_return_val_if_fail (articles!=NULL, NULL);
	g_return_val_if_fail (readable_name!=NULL, NULL);
	g_return_val_if_fail (articles->data!=NULL, NULL);

	/* ref the group's articles */
	group_ref_articles (group, NULL);

	/* checkout the article bodies */
	if (1) {
		const GSList * l;
		GPtrArray * a = g_ptr_array_new ();
		for (l=articles; l!=NULL; l=l->next)
			g_ptr_array_add (a, (gpointer)article_get_message_id(ARTICLE(l->data)));
		acache_file_checkout ((const gchar**)a->pdata, a->len);
		g_ptr_array_free (a, TRUE);
	}

	if (1) {
		const GSList * l;
		need_socket = FALSE;
		for (l=articles; l!=NULL; l=l->next) {
			Article * a = ARTICLE(l->data);
			if (!acache_has_message (article_get_message_id(a))) {
				need_socket = TRUE;
				break;
			}
		}
	}

	/* create the task */
       	task = g_new0 (TaskSave, 1);
        debug1 (DEBUG_PAN_OBJECT, "task_save_new: %p", task);
	task_constructor (TASK(task),
			  task_save_destructor,
			  task_save_describe,
			  task_save_run,
			  group->server, FALSE, need_socket);

	/* create the task-save */
	TASK(task)->type = TASK_TYPE_SAVE;
	task->runval = TASK_FAIL;
	task->group = group;
	task->readable_name = g_strdup(readable_name);
	task->articles = g_slist_copy ((GSList*)articles);
	task->path_attachments = NULL;
	task->filename_attachments = NULL;
	task->path_bodies = NULL;
	task->filename_bodies = NULL;

	/* flag the articles */
	a = g_ptr_array_new ();
	for (l=task->articles; l!=NULL; l=l->next)
		g_ptr_array_add (a, l->data);
	articles_add_flag ((Article**)a->pdata, a->len, STATE_SAVE_QUEUED);
	g_ptr_array_free (a, TRUE);

	/* prune the articles */
	task_save_prune (task);

	return PAN_OBJECT(task);
}


extern char*
expand_download_dir (const char * dir, const Group * group);

void
task_save_set_attachments (TaskSave      * task,
                           const gchar   * path,
                           const gchar   * filename)
{
	debug_enter ("task_save_set_attachments");
	g_return_if_fail (task!=NULL);

	task->save_attachments = TRUE;
	replace_gstr (&task->path_attachments, g_strdup(path));
	replace_gstr (&task->filename_attachments, g_strdup(filename));

	/* if no path specified, fall back on Pan group and global defaults */
	if (!task->path_attachments)
		task->path_attachments = g_strdup (TASK_SAVE(task)->group->download_dir);
	if (!task->path_attachments)
		task->path_attachments = g_strdup (download_dir);

	/* expand */
	replace_gstr (&task->path_attachments,
	              expand_download_dir (task->path_attachments,  task->group));

	debug_exit ("task_save_set_attachments");
}
 
void
task_save_set_bodies (TaskSave      * task,
                      const gchar   * path,
                      const gchar   * filename)
{
	debug_enter ("task_save_set_bodies");
	g_return_if_fail (task!=NULL);

	task->save_bodies = TRUE;
	replace_gstr (&task->path_bodies, g_strdup(path));
	replace_gstr (&task->filename_bodies, g_strdup(filename));

	/* if no path specified, fall back on Pan group and global defaults */
	if (!task->path_bodies)
		task->path_bodies = g_strdup (TASK_SAVE(task)->group->download_dir);
	if (!task->path_bodies)
		task->path_bodies = g_strdup (download_dir);

	/* expand */
	replace_gstr (&task->path_bodies,
	              expand_download_dir (task->path_bodies,  task->group));

	debug_exit ("task_save_set_bodies");
}

/*****
******
*****/

/* FIXME: this has a lot in common with decode.c; maybe promote to util-file */
static gchar*
get_unique_fname (const gchar * path, const gchar * fname)
{
	gint i;
	gchar * retval;
	GString * filename;

	/* create the unchanged filename */
	if (!is_nonempty_string (fname))
		fname = _("UNKNOWN");
	filename = g_string_new (fname);

	/* filter out directory characters */
	if (1) {
		const gchar * in;
		gchar * buf = g_malloc (strlen(fname)*2);
		gchar * out = buf;
		for (in=fname; *in; ++in) {
			if (*in==G_DIR_SEPARATOR) {
				*out++ = '_';
			}
			else if (*in=='\\') {
				*out++ = '\\',
				*out++ = '\\';
			}
			else {
				*out++ = *in;
			}
		}
		*out = '\0';
		g_string_assign (filename, buf);
		g_free (buf);
	}

	
	for (i=1;; ++i)
	{
		const gchar * front = filename->str;
		const gchar * lastdot = strrchr (front, '.');
		gchar * unique;
		gchar * lead;
		gchar * tail;

		if (lastdot == NULL) {
			lead = g_strdup (front);
			tail = g_strdup ("");
		} else {
			lead = g_strndup (front, lastdot-front);
			tail = g_strdup (lastdot);
		}

		if (i==1 && is_nonempty_string(path))
		{
			unique = g_strdup_printf ("%s%c%s%s",
			                          path, G_DIR_SEPARATOR,
			                          lead, tail);
		}
		else if (i==1)
		{
			unique = g_strdup_printf ("%s%s", lead, tail);
		}
		else if (is_nonempty_string(path))
		{
			unique = g_strdup_printf ("%s%c%s_%d%s",
			                         path, G_DIR_SEPARATOR,
			                         lead, i, tail);
		}
		else
		{
			unique = g_strdup_printf ("%s_%d%s", lead, i, tail);
		}

		/* cleanup */
		g_free (lead);
		g_free (tail);

		if (!file_exists(unique)) {
			g_string_assign (filename, unique);
			break;
		}

		g_free (unique);
	}

	retval = filename->str;
	g_string_free (filename, FALSE);
	pan_normalize_filename_inplace (retval);
	return retval;
}

static gint
task_save_run (Task * task)
{
	gint result = TASK_SUCCESS;
	GSList * articles;
	Server * server;
	Group * group;
	Article * article;
	debug_enter ("task_save_run");

	/* sanity clause */
	g_return_val_if_fail (task!=NULL, TASK_SAVE(task)->runval=TASK_FAIL_HOPELESS);
	g_return_val_if_fail (TASK_SAVE(task)->articles!=NULL, TASK_SAVE(task)->runval=TASK_FAIL_HOPELESS);

	server = task->server;
	group = TASK_SAVE(task)->group;
	articles = TASK_SAVE(task)->articles;

	/**
	*** Get the article bodies that we don't already have
	*** (nntp_download_bodies will just skip over any
	*** that are already in acache)
	**/

	if (1)
	{
		GSList * l;
		gint index = 0;
		GPtrArray * article_array = g_ptr_array_new ();
		for (l=articles; l!=NULL; l=l->next)
			g_ptr_array_add (article_array, l->data);
		if (article_array->len)
			result = nntp_download_bodies (STATUS_ITEM(task),
						       TASK_SAVE(task)->group,
						       task->sock,
						       &task->hint_abort,
						       article_array,
						       &index,
						       FALSE, TRUE);
		g_ptr_array_free (article_array, TRUE);

		if (result != TASK_SUCCESS) {
			status_item_emit_error_va (STATUS_ITEM(task),
                           _("Unable to save `%s': the article(s) couldn't be downloaded."),
			   article_get_subject(ARTICLE(articles->data)));
			debug_exit ("task_save_run");
			return TASK_SAVE(task)->runval = result;
		}
	}

	/**
	***  Save Articles
	**/

       	article = ARTICLE(articles->data);
	articles_set_read (&article, 1, TRUE); 

	if (TASK_SAVE(task)->save_attachments)
	{
		gboolean decode_ok;

		decode_data * dd = g_new0 (decode_data, 1);
		dd->server = server;
		dd->group = group;
		dd->articles = articles;
		dd->item = STATUS_ITEM(task);
		dd->path = TASK_SAVE(task)->path_attachments;
		dd->filename = TASK_SAVE(task)->filename_attachments;
		decode_ok = decode_article (dd);
		g_free (dd);

		/* if we failed to decode something that we think we should've
		   been able to decode, then fail. */
		if (!decode_ok && ARTICLE(articles->data)->parts>0)
			return TASK_SAVE(task)->runval = TASK_FAIL;
	}

	if (TASK_SAVE(task)->save_bodies)
	{
		gboolean bodies_ok = TRUE;
		GSList * l;

		for (l=articles; l!=NULL; l=l->next)
		{
			gboolean body_ok = TRUE;
			const Article * a = ARTICLE(l->data);
			char * txt = article_get_message (a);
			char * fname;
			char * path;

			/* get the filename */
			fname = g_strdup (TASK_SAVE(task)->filename_bodies);
			if (!is_nonempty_string(fname))
				replace_gstr (&fname, g_strdup_printf ("%s.txt", article_get_subject(a)));
			replace_gstr (&fname, get_unique_fname (TASK_SAVE(task)->path_bodies, fname));

			/* get the directory */
			path = g_path_get_dirname (fname);
			if (!directory_check (path)) {
				log_add_va (LOG_ERROR, _("Save Article can't access path \"%s\""), path);
				body_ok = FALSE;
			}

			/* open the file for writing */
			if (body_ok) {
				FILE * fp;
				errno = 0;
				fp = fopen (fname, "w+");
				if (fp == NULL) {
					log_add_va (LOG_ERROR, _("Can't create file `%s' %s"), fname, g_strerror(errno));
					body_ok = FALSE;
				} else {
					fwrite (txt, sizeof(char), strlen(txt), fp);
					fclose (fp);
					log_add_va (LOG_INFO, _("Saved article body to `%s'"), fname);
				}
			}

			/* cleanup */
			g_free (txt);
			g_free (path);
			g_free (fname);

			if (!body_ok)
				bodies_ok = FALSE;
		}

		if (!bodies_ok)
			return TASK_SAVE(task)->runval = TASK_FAIL;
	}

	debug_exit ("task_save_run");
	return TASK_SAVE(task)->runval = TASK_SUCCESS;
}


/*****
******
******  VALIDATE-AND-QUEUE or SELF-DESTRUCT
******
*****/

static void
weed_duplicate_parts (TaskSave  * task)
{
	GSList* l;
	Article *last_article = NULL;
	
	l = task->articles;
	while (l != NULL)
	{
		Article *article = ARTICLE (l->data);

		/* increment l here so that removing article
		   from task->articles doesn't invalidate it */
		l = l->next;

		/* If this is a reply sitting in a multipart thread,
		 * don't use the reply. */
		if (article->part == 0)
		{
			task->articles = g_slist_remove (task->articles, article);
		}
		else if (last_article && (last_article->part == article->part))
		{
			/* Select the better of the two assuming
			 * that more lines is better */
			if (last_article->linecount > article->linecount)
			{
				task->articles = g_slist_remove (task->articles, article);
			}
			else if (last_article->linecount < article->linecount)
			{
				task->articles = g_slist_remove (task->articles, last_article);
				last_article = article;
			}
			else
			{
				/* Check the date next if lines match,
				 * last posted is preferred */
				if (last_article->date > article->date)
				{
					task->articles = g_slist_remove (task->articles, article);
				}
				else
				{ 
					/* if article is newer, or the same, kill the other one */
					task->articles = g_slist_remove (task->articles, last_article);
					last_article = article;
				}	
			}
		}
		else
		{
			last_article = article;
		}
	}
}

static void
task_save_prune (TaskSave  * task)
{
	const int length = g_slist_length(task->articles);

	/**
	 * (1) If we have a number of articles in the save list, and
	 * (2) the number of parts we expected doesn't match it, and
	 * (3) if we have a clue what the number of parts is supposed to be
	 *     (we may not, if user didn't post with a [1/24]-style header),
	 * then try to weed out the ones we don't want.
	 */
	if ((length>1)
		&& (length!=ARTICLE(task->articles->data)->parts)
		&& ARTICLE(task->articles->data)->parts)
	{
		weed_duplicate_parts (task);	
	}
}
