/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999, 2000, 2001  Pan Development Team <pan@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>

#include <glib.h>
#include <gtk/gtk.h>

#include <stdlib.h>
#include <string.h>

#include <pan/base/acache.h>
#include <pan/base/base-prefs.h>
#include <pan/base/debug.h>
#include <pan/base/pan-config.h>
#include <pan/base/pan-i18n.h>
#include <pan/base/pan-glib-extensions.h>
#include <pan/base/serverlist.h>
#include <pan/base/util-file.h>

#include <pan/articlelist.h>
#include <pan/globals.h>
#include <pan/gui.h>
#include <pan/gui-headers.h>
#include <pan/gui-paned.h>
#include <pan/nntp.h>
#include <pan/pan-color-picker.h>
#include <pan/pan-font-picker.h>
#include <pan/pan-file-entry.h>
#include <pan/prefs.h>
#include <pan/queue.h>
#include <pan/server-ui.h>
#include <pan/task-headers.h>
#include <pan/text.h>
#include <pan/util.h>

#include <pan/xpm/pan-pixbufs.h>

 
#define DISPLAY_PREFS_PREVIEW_FONT_SIZE 14

#define NORMAL_PANGO_DESCRIPTION "serif 11"
#define BOLD_PANGO_DESCRIPTION  "serif bold 11"
#define MONOSPACE_PANGO_DESCRIPTION  "courier 11"


/* server profile */
typedef struct
{
	GtkWidget * profile_name_entry;
	GtkWidget * server_address;
	GtkWidget * server_port;
	GtkWidget * auth_cbutton;
	GtkWidget * server_username_label;
	GtkWidget * server_username;
	GtkWidget * server_password_label;
	GtkWidget * server_password;
	GtkWidget * msgid_cbutton;
	GtkWidget * idle_secs_before_timeout;
	Server * server;
}
server_profile;

typedef struct
{
	const Server * server;
	GtkWidget * max_connections_sb;
	GtkWidget * reserve_tb;
}
server_connection;

typedef struct
{
	GtkWidget * dialog;
	GtkWidget * notebook;
	GtkWidget * server_clist;

	/* smtp server*/
	GtkWidget * smtp_address;
	GtkWidget * smtp_port;

	/* newsrc */
	GtkWidget * newsrc_useless_label;
	GtkWidget * newsrc_import_check;
	GtkWidget * newsrc_filename_entry;
	GtkWidget * newsrc_subscribed_only_check;

	/* cache */
	GtkWidget * cache_megs_sb;
	GtkWidget * flush_cache_on_exit_check;

	/* general */
	GtkWidget * single_click_selects_cbutton;
	GtkWidget * remove_failed_tasks_cbutton;
	GtkWidget * fetch_new_on_group_load_cbutton;
	GtkWidget * fetch_new_on_startup_cbutton;
	GtkWidget * fetch_new_and_bodies_on_startup_cbutton;
	GtkWidget * break_thread_when_subject_changes_cbutton;
	GtkWidget * expand_all_threads_by_default_cbutton;
	GtkWidget * hide_mpart_child_nodes_cbutton;
	GtkWidget * external_editor_combo;
	GtkWidget * external_browser_combo;

	/* wrapping */
	GtkWidget * wrap_column_tb;
	GtkWidget * wrap_column_sb;

	/* directories */
	GtkWidget * dir_download;
	GtkWidget * dir_temp;
	GtkWidget * dir_data;

	/* display */
	GtkWidget * grouplist_gfp;
	GtkWidget * thread_normal_gfp;
	GtkWidget * thread_new_replies_gfp;
	GtkWidget * message_gfp;
	GtkWidget * message_fixed_gfp;
	GtkWidget * thread_read_pcp;
	GtkWidget * thread_normal_pcp;
	GtkWidget * ignored_pcp;
	GtkWidget * watched_pcp;
	GtkWidget * use_system_bg_check;
	GtkWidget * use_system_fg_check;
	GtkWidget * text_fg_pcp;
	GtkWidget * text_bg_pcp;
	GtkWidget * text_url_pcp;
	GtkWidget * text_quoted_pcp[3];
	GtkWidget * text_quoted_chars_entry;
	GtkWidget * thread_date_entry;
	GtkWidget * body_date_entry;
	GtkWidget * smooth_scrolling_check;
	GtkWidget * smooth_scrolling_speed_sb;

	/* layout */
	GtkWidget * layout_page;

	/* article headers */
	GtkWidget * article_headers_clist;

	/* connections */
	GtkWidget  * max_connections;
	GtkWidget  * max_tries;
	GSList * server_connections;

}
PrefsWindow;

typedef struct
{
	Server *old;
	Server *new;
}
prefs_server;


static gchar* layout_get_new_string (GtkWidget * layout_page);
static void new_server_cb (void);
static void edit_server_cb (void);
static void delete_server_cb (void);
static void prefs_create_clist (void);
static void edit_profile_dialog_response (GtkDialog*, int button, prefs_server*);
static void edit_profile_dialog (prefs_server *data);
static gboolean server_clist_button_press (GtkWidget*, GdkEventButton*);
static void prefs_servers_changed (void);
static GtkWidget * prefs_general_page ( void );
static GtkWidget * prefs_smtp_page (void);
static GtkWidget * prefs_cache_page (void);
static GtkWidget * prefs_server_page (void);
static GtkWidget * prefs_connections_page (void);
static GtkWidget * prefs_grouplist_page (void);
static GtkWidget * prefs_layout_page (void);
static gulong get_header_flags (void);

static PrefsWindow * win;
static gboolean servers_changed = FALSE;
static GSList * server_deletes = NULL;
static GtkWidget *sections_clist;
static int nntp_clist_row = -1;
static int nntp_connections_row = -1;
static GSList * sprof_list = NULL; /* holds the profile editor GnomeDialogs */

extern GtkTooltips * ttips;

static char * text_quote_chars                = NULL;
char  * newsrc_filename                       = NULL;
gboolean collapse_group_names                 = FALSE;
gboolean use_system_bg                        = TRUE;
gboolean use_system_fg                        = TRUE;
gboolean newsrc_do_port                       = FALSE;
gboolean newsrc_port_subscribed_only          = FALSE;
gboolean text_window_smooth_scrolling         = TRUE;
int      text_window_smooth_scrolling_speed   = 10;

char * thread_date_format                     = NULL;
char * body_date_format                       = NULL;
char * layout_str                             = NULL;
int wrap_column                               = 74;
int mail_server_port                          = 0;
gboolean text_use_fixed_font                  = FALSE;
gboolean expand_all_threads_by_default        = FALSE;
gboolean hide_mpart_child_nodes               = TRUE;
gboolean pan_mute                             = FALSE;
gboolean fetch_new_on_group_load              = TRUE;
gboolean fetch_new_on_startup                 = FALSE;
gboolean fetch_new_and_bodies_on_startup      = FALSE;
gboolean header_pane_is_threaded              = TRUE;
gboolean remove_failed_tasks                  = TRUE;
gboolean single_click_selects                 = FALSE;
gboolean show_group_pane                      = TRUE;
gboolean show_header_pane                     = TRUE;
gboolean show_body_pane                       = TRUE;

char * grouplist_font = NULL;
char * thread_normal_font = NULL;
char * thread_new_replies_font = NULL;
char * message_body_font = NULL;
char * message_body_font_fixed = NULL;
char * attribution_line = NULL;
char * external_editor = NULL;
char * external_web_browser = NULL;
char * mail_server_address = NULL;


static void
set_color (GdkColor      * color,
           const gchar   * key,
           guint           r,
           guint           g,
           guint           b)
{
	char buf[1024];

	g_snprintf (buf, sizeof(buf), "/Pan/Display/%s_r=%u", key, r);
	color->red = pan_config_get_int (buf);

	g_snprintf (buf, sizeof(buf), "/Pan/Display/%s_g=%u", key, g);
	color->green = pan_config_get_int (buf);

	g_snprintf (buf, sizeof(buf), "/Pan/Display/%s_b=%u", key, b);
	color->blue = pan_config_get_int (buf);

	if (!gdk_color_alloc (cmap, color))
		g_error ("couldn't allocate `%s' color", key);
}

static void
date_help_clicked_cb (void)
{
	const gchar * str = _("%a - abbreviated weekday name\n"
	                      "%A - full weekday name\n"
	                      "%b - abbreviated month name\n"
	                      "%B - full month name\n"
	                      "%c - local date & time\n"
		              "%d - day of the month\n"
		              "%H - hour (24-hour clock)\n"
		              "%I - hour (12-hour clock)\n"
		              "%j - day of the year (001-366)\n"
		              "%m - month (01-12)\n"
		              "%M - minute (00-59)\n"
		              "%p - local equivalent of AM/PM\n"
		              "%S - second (00-61)\n"
		              "%x - local date\n"
		              "%X - local time\n"
		              "%y - two-digit year\n"
		              "%Y - four-digit year\n"
		              "%% - %");
	GtkWidget * w = gtk_message_dialog_new (GTK_WINDOW(win->dialog),
	                                        GTK_DIALOG_DESTROY_WITH_PARENT,
	                                        GTK_MESSAGE_INFO,
	                                        GTK_BUTTONS_CLOSE, "%s", str);
	g_signal_connect_swapped (GTK_OBJECT(w), "response",
	                          G_CALLBACK(gtk_widget_destroy), GTK_OBJECT (w));
	gtk_widget_show_all (w);
}

static void
pan_prefs_changed (GtkDialog * dialog)
{
        gtk_dialog_set_response_sensitive (GTK_DIALOG(win->dialog), GTK_RESPONSE_OK, TRUE);
        gtk_dialog_set_response_sensitive (GTK_DIALOG(win->dialog), GTK_RESPONSE_APPLY, TRUE);
}

#define connect_signal_to_prefs_changed(object,signal_name) \
	g_signal_connect_swapped (object, signal_name, G_CALLBACK(pan_prefs_changed), win->dialog)

#define connect_signal_to_profile_changed(object,signal_name,dialog) \
	g_signal_connect_swapped (object, signal_name, G_CALLBACK(edit_profile_changed), dialog)

/**
***  UPDATE UTILS
**/

static gboolean
update_entry_and_bool_from_toggle_button (gboolean * setme,
                                          const char * key,
                                          GtkWidget * toggle)
{
	gboolean changed = FALSE;
	const gboolean b = gtk_toggle_button_get_active (
		GTK_TOGGLE_BUTTON(toggle));

	if (b != *setme) {
		*setme = b;
		pan_config_set_bool (key, b);
		changed = TRUE;
	}

	return changed;
}

static void
set_config_from_editable (const char   * key,
                          GtkWidget    * entry)
{
	gchar * text = gtk_editable_get_chars (GTK_EDITABLE(entry), 0, -1);
	g_strstrip (text);
	if (is_nonempty_string(text))
		pan_config_set_string (key, text);
	else
		pan_config_clean_key (key);
	g_free (text);
}

static void
replace_string_and_config_from_editable (gchar       ** setme,
                                         const char   * key,
                                         GtkWidget    * e)
{
	gchar * s = gtk_editable_get_chars (GTK_EDITABLE(e), 0, -1);
	g_strstrip (s);
	if (!pan_strcmp(*setme, s))
		g_free (s);
	else {
		pan_config_set_string (key, s);
		replace_gstr (setme, s);
	}
}

static gboolean
handle_font_picker (GtkWidget * pfp, gchar ** p_old_font, const gchar * config_key)
{
	gboolean changed = FALSE;

	/* get new... */
	char * new_font = pan_font_picker_get_font (pfp);

	/* compare old with new... */
	if (pan_strcmp (*p_old_font, new_font)) {
		replace_gstr (p_old_font, g_strdup(new_font));
		pan_config_set_string (config_key, new_font);
		changed = TRUE;
	}

	g_free (new_font);
	return changed;
}

static gboolean
handle_color_picker (const gchar    * key,
                     GdkColor       * color,
                     GtkWidget      * w)
{
	GdkColor picker_color;
	gboolean color_changed = FALSE;

	pan_color_picker_get_color (w, &picker_color);

	color_changed = (color->red != picker_color.red)
		     || (color->green != picker_color.green)
		     || (color->blue != picker_color.blue);

	if (color_changed)
	{
		gchar buf[1024];

		color_changed = TRUE;

		/* update the color */
		gdk_colormap_free_colors (cmap, color, 1);
		color->red = picker_color.red;
		color->green = picker_color.green;
		color->blue = picker_color.blue;
		if (!gdk_color_alloc (cmap, color))
			g_error ("couldn't allocate `%s' color", key);

		/* update the config */
		g_snprintf (buf, sizeof(buf), "/Pan/Display/%s_r", key);
		pan_config_set_int (buf, picker_color.red);
		g_snprintf (buf, sizeof(buf), "/Pan/Display/%s_g", key);
		pan_config_set_int (buf, picker_color.green);
		g_snprintf (buf, sizeof(buf), "/Pan/Display/%s_b", key);
		pan_config_set_int (buf, picker_color.blue);
	}

	return color_changed;
}

/**
***  UPDATE EVERYTHING
**/



/* Ok or Apply pressed in the Preferences dialog, save all information */
static void
prefs_apply (void)
{
	gboolean articlelist_changed = FALSE;
	gboolean server_connections_changed = FALSE;
	gboolean color_changed;
	gboolean b;

	/* Commit all of the data to the config file */

	/* No entry for these yet
	 * pan_config_set_int("/Pan/Articles/Expiration", 30);
	 */

	/* thread date format */
	if (1) {
		gchar * tmp = g_strdup (thread_date_format);
		replace_string_and_config_from_editable (
			&thread_date_format,
			"/Pan/Display/Thread_Date_Format",
			win->thread_date_entry);
		if (pan_strcmp (tmp, thread_date_format))
			articlelist_changed = TRUE;
		g_free (tmp);
	}

	set_config_from_editable (
		"/Pan/Paths/download_dir",
		pan_file_entry_gtk_entry(win->dir_download));

	/**
	***  Newsrc
	**/

	update_entry_and_bool_from_toggle_button (&newsrc_do_port,
	                                          "/Pan/Newsrc/do_port",
	                                          win->newsrc_import_check);
	update_entry_and_bool_from_toggle_button (&newsrc_port_subscribed_only,
	                                          "/Pan/Newsrc/port_subscribed_only",
	                                          win->newsrc_subscribed_only_check);
	replace_string_and_config_from_editable (&newsrc_filename,
	                                         "/Pan/Newsrc/filename",
	                                         win->newsrc_filename_entry);

	/**
	***  Mail Server
	**/

	replace_string_and_config_from_editable (&mail_server_address,
	                                         "/Pan/Mail/smtp_address",
	                                         win->smtp_address);

	if (1) {
		int i = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON(win->smtp_port));
		if (i != mail_server_port) {
			pan_config_set_int ("/Pan/Mail/smtp_port", i);
			mail_server_port = i;
		}
	}

	/**
	***  Windows
	**/

	/* update the articlelist colors */
	articlelist_changed |= handle_color_picker ("killfile_color", &killfile_color, win->ignored_pcp);
	articlelist_changed |= handle_color_picker ("watched_color", &watched_color, win->watched_pcp);
	articlelist_changed |= handle_color_picker ("thread_normal_color", &thread_normal_color, win->thread_normal_pcp);
	articlelist_changed |= handle_color_picker ("thread_read_color", &thread_read_color, win->thread_read_pcp);

	/* text window colors */
	b = 0;
	b |= update_entry_and_bool_from_toggle_button (&use_system_bg, "/Pan/Display/use_system_bg", win->use_system_bg_check);
	b |= update_entry_and_bool_from_toggle_button (&use_system_fg, "/Pan/Display/use_system_fg", win->use_system_fg_check);
	b |= handle_color_picker ("text_fg_color", &text_fg_color, win->text_fg_pcp);
	b |= handle_color_picker ("text_bg_color", &text_bg_color, win->text_bg_pcp);
	b |= handle_color_picker ("text_url_color", &text_url_color, win->text_url_pcp);
	b |= handle_color_picker ("text_quoted_color_1", &text_quoted_color[0], win->text_quoted_pcp[0]);
	b |= handle_color_picker ("text_quoted_color_2", &text_quoted_color[1], win->text_quoted_pcp[1]);
	b |= handle_color_picker ("text_quoted_color_3", &text_quoted_color[2], win->text_quoted_pcp[2]);
	color_changed = b;

	/* text window date format */
	if (1) {
		gchar * tmp = g_strdup (body_date_format);
		replace_string_and_config_from_editable (
			&body_date_format,
			"/Pan/Display/Body_Date_Format",
			win->body_date_entry);
		if (pan_strcmp (tmp, body_date_format))
			color_changed = TRUE;
		g_free (tmp);
	}

	/* quote prefix characters */
	if (1) {
		char * tmp = g_strdup (text_quote_chars);
		replace_string_and_config_from_editable (
			&text_quote_chars,
			"/Pan/Display/text_quote_chars",
			win->text_quoted_chars_entry);
		if (pan_strcmp (tmp, text_quote_chars))
			color_changed = TRUE;
		g_free (tmp);
	}

	if (color_changed)
	{
		GtkStyle * style;

		style = gtk_rc_get_style (Pan.text);

		style = style != NULL ? gtk_style_copy (style) 
				      : gtk_style_new ();

		if (!use_system_bg)
			style->base[0] = text_bg_color;
		if (!use_system_fg)
			style->text[0] = text_fg_color;

		gtk_widget_set_style (Pan.text, style);

		pan_widget_set_font (Pan.text, 
			text_use_fixed_font ? message_body_font_fixed 
			                    : message_body_font);

		text_refresh ();
	}

	update_entry_and_bool_from_toggle_button (
		&remove_failed_tasks,
		"/Pan/General/remove_failed_tasks",
		win->remove_failed_tasks_cbutton);

	update_entry_and_bool_from_toggle_button (
		&fetch_new_on_group_load,
		"/Pan/General/fetch_new_on_group_load",
		win->fetch_new_on_group_load_cbutton);

	update_entry_and_bool_from_toggle_button (
		&fetch_new_on_startup,
		"/Pan/General/fetch_new_from_subscribed_on_startup",
		win->fetch_new_on_startup_cbutton);

	update_entry_and_bool_from_toggle_button (
		&single_click_selects,
		"/Pan/General/single_click_selects",
		win->single_click_selects_cbutton);

	update_entry_and_bool_from_toggle_button (
		&fetch_new_and_bodies_on_startup,
		"/Pan/General/fetch_new_and_bodies_from_subscribed_on_startup",
		win->fetch_new_and_bodies_on_startup_cbutton);

	/**
	***  Font Pickers
	**/

	if (handle_font_picker (win->thread_normal_gfp,
	                        &thread_normal_font,
	                        "/Pan/Display/thread_normal_font"))
		articlelist_changed = TRUE;

	if (handle_font_picker (win->thread_new_replies_gfp,
	                        &thread_new_replies_font,
	                        "/Pan/Display/thread_new_replies_font"))
		articlelist_changed = TRUE;

	if (handle_font_picker (win->grouplist_gfp,
	                        &grouplist_font,
	                        "/Pan/Display/grouplist_font"))
		pan_widget_set_font (GTK_WIDGET(Pan.group_tree), grouplist_font);

	b = handle_font_picker (win->message_gfp,
	                        &message_body_font,
	                        "/Pan/Display/message_body_font_norm");

	b |= handle_font_picker (win->message_fixed_gfp,
	                        &message_body_font_fixed,
	                        "/Pan/Display/message_body_font_fixed");

	if (b)
		text_set_font ();


	/* cache */
	if (1) {
		acache_max_megs = gtk_spin_button_get_value_as_int (
			GTK_SPIN_BUTTON(win->cache_megs_sb));
		pan_config_set_int ("/Pan/Cache/MaxMegs", acache_max_megs);
		acache_expire ();
	}
	pan_config_set_bool ("/Pan/Cache/FlushOnExit",
			gtk_toggle_button_get_active (
				GTK_TOGGLE_BUTTON( win->flush_cache_on_exit_check)));

	/* article headers */
	if (1) {
		gulong flags = get_header_flags ( );
		if (header_flags != flags) {
			pan_config_set_int ("/Pan/State/Headers", flags);
			header_flags = flags;
			text_refresh ();
		}
	}
	
	if (servers_changed) {
		/* Handle the server mess */
		prefs_servers_changed ();
	}

	/**
	***  Filling
	**/

	if (1)
	{
		gboolean tmp = text_get_wrap ();

		update_entry_and_bool_from_toggle_button (
			&tmp,
			"/Pan/Display/do_wrap",
			win->wrap_column_tb);

		text_set_wrap (tmp);
	}


	if (update_entry_and_bool_from_toggle_button (
		&break_thread_when_subject_changes,
		"/Pan/General/break_thread_when_subject_changes",
		win->break_thread_when_subject_changes_cbutton))
			articlelist_changed = TRUE;

	if (update_entry_and_bool_from_toggle_button (
		&expand_all_threads_by_default,
		"/Pan/General/expand_all_threads_by_default",
		win->expand_all_threads_by_default_cbutton))
			articlelist_changed = TRUE;

	if (update_entry_and_bool_from_toggle_button (
		&hide_mpart_child_nodes,
		"/Pan/General/hide_mpart_child_nodes",
		win->hide_mpart_child_nodes_cbutton))
			articlelist_changed = TRUE;


	if (1) {
		GtkSpinButton* w = GTK_SPIN_BUTTON(win->wrap_column_sb);
		int i = gtk_spin_button_get_value_as_int (w);
		if (i != wrap_column) {
			wrap_column = i;
			pan_config_set_int ("/Pan/General/wrap_column", i);
			text_refresh ();

		}
	}


	update_entry_and_bool_from_toggle_button (&text_window_smooth_scrolling,
	                                          "/Pan/Display/smooth_scrolling",
	                                          win->smooth_scrolling_check);

	if (1) {
		GtkSpinButton* w = GTK_SPIN_BUTTON(
			win->smooth_scrolling_speed_sb);
		const gint i = gtk_spin_button_get_value_as_int (w);
		if (i != text_window_smooth_scrolling_speed) {
			text_window_smooth_scrolling_speed = i;
			pan_config_set_int (
				"/Pan/Display/smooth_scrolling_speed_2", i);

		}
	}

	replace_string_and_config_from_editable (
		&external_editor,
		"/Pan/User/external_editor_2",
		GTK_COMBO(win->external_editor_combo)->entry);

	replace_string_and_config_from_editable (
		&external_web_browser,
		"/Pan/User/external_web_browser",
		GTK_COMBO(win->external_browser_combo)->entry);

	/* server connections ui */
	if (1)
	{
		GSList * l;
		for (l=win->server_connections; l!=NULL; l=l->next)
		{
			server_connection * sc = (server_connection*) l->data;
			Server * server = SERVER(sc->server);
			int i;

			/* max connections for this server */
			i = gtk_spin_button_get_value_as_int (
				GTK_SPIN_BUTTON(sc->max_connections_sb));
			if (i != server->max_connections)
			{
				server->max_connections = i;
				server_connections_changed = TRUE;
			}

			/* reserve for bodies */
			i = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(sc->reserve_tb));
			if ((i!=0) != (server->reserve_connection_for_bodies!=0))
			{
				server->reserve_connection_for_bodies = i;
				server_connections_changed = TRUE;
			}
		}
	}

	/* maximum number of connections */
	if (1) {
		GtkSpinButton * sb = GTK_SPIN_BUTTON (win->max_connections);
		const int i = gtk_spin_button_get_value_as_int (sb);
		if (i != queue_get_max_sockets()) {
			queue_set_max_sockets (i);
			server_connections_changed = TRUE;
			pan_config_set_int ("/Pan/General/Max_Connections_Total", i);
		}
	}

	/* maximum number of tries */
	if (1) {
		GtkSpinButton * sb = GTK_SPIN_BUTTON (win->max_tries);
		const int i = gtk_spin_button_get_value_as_int (sb);
		if (i != queue_get_max_tries()) {
			queue_set_max_tries (i);
			server_connections_changed = TRUE;
			pan_config_set_int ("/Pan/General/Max_Task_Tries_Total", i);
		}
	}

	/* layout */
	if (1) {
		gchar * str = layout_get_new_string (win->layout_page);
		if (pan_strcmp (str, layout_str)) {
			replace_gstr (&layout_str, g_strdup(str));
			pan_config_set_string ("/Pan/Display/Layout", layout_str);
			gui_layout_set_mode (GUI_PANED);
		}
		g_free (str);
	}

	/**
	***
	***
	**/

	if (articlelist_changed)
		articlelist_reset_style_nolock ();

	if (server_connections_changed) {
		queue_wakeup ();
		server_menu_update ();
	}

	pan_config_sync ();

	prefs_init ();
	gtk_dialog_set_response_sensitive (GTK_DIALOG(win->dialog), GTK_RESPONSE_CANCEL, FALSE);
	gtk_dialog_set_response_sensitive (GTK_DIALOG(win->dialog), GTK_RESPONSE_APPLY, FALSE);
}

static void
prefs_servers_changed (void)
{
	gint row = 0;
	prefs_server * p_server;

	while ((p_server = gtk_clist_get_row_data (GTK_CLIST(win->server_clist), row)))
	{
		if (p_server->new)
		{
			if (p_server->old)
			{
				/* Modified */
				Server * new = p_server->new;
				Server * old = p_server->old;

				replace_gstr (&old->name, g_strdup(new->name));
				replace_gstr (&old->address, g_strdup(new->address));
				replace_gstr (&old->username, g_strdup(new->username));
				replace_gstr (&old->password, g_strdup(new->password));
				old->port = new->port;
				old->need_auth = new->need_auth;
				old->max_connections = new->max_connections;
				old->idle_secs_before_timeout = new->idle_secs_before_timeout;
				old->gen_msgid = new->gen_msgid;
			}
			else
			{
				/* New addition */
				Server * new = p_server->new;
				char path[PATH_MAX];
				g_snprintf (path, sizeof(path), "%s%c%s", get_data_dir(), G_DIR_SEPARATOR, new->name);
				directory_check (path);
				serverlist_add_server (new);
				p_server->new = NULL;
			}
		}

		++row;
	}

	if (server_deletes != NULL)
	{
		GSList * l;

		/* remove from server list */
		for (l=server_deletes; l!=NULL; l=l->next)
		{
			prefs_server * p_server = l->data;
			Server * server= p_server->old;
			GSList * m;
			GSList ** pm;

			/* remove it from the pan server list */
			serverlist_remove_server (server);

			/* remove from the prefs connections tab */
			pm = &win->server_connections;
			for (m=*pm; m!=NULL; m=m->next)
			{
				server_connection * sc = (server_connection*) m->data;
				if (sc->server == server)
				{
					*pm = g_slist_remove (*pm, sc);
					g_free (sc);
					break;
				}
			}
		}

		/* cleanup */
		g_slist_foreach (server_deletes, (GFunc)g_free, NULL);
		g_slist_free (server_deletes);
		server_deletes = NULL;
	}

	server_menu_update ();
	serverlist_save ();
}

static void
ensure_trailing_slash (gchar ** pch)
{
	gchar * m = strrchr (*pch, G_DIR_SEPARATOR);
	if (!m || m[1]!='\0')
		replace_gstr (pch, g_strdup_printf ("%s%c", *pch, G_DIR_SEPARATOR));
}

/**
 * This is where all the user preferences get loaded in.
 */
void
prefs_init (void)
{
	debug_enter ("prefs_init");

	/* get the layout string */
	replace_gstr (&layout_str, pan_config_get_string ("/Pan/Display/Layout"));
	if (!is_nonempty_string (layout_str))
		replace_gstr (&layout_str, g_strdup ("4gta"));

	/* get download directory */
	replace_gstr (&download_dir, pan_config_get_string ("/Pan/Paths/download_dir"));
	if (is_nonempty_string (download_dir))
		ensure_trailing_slash (&download_dir);
	else
		download_dir = g_strdup_printf ("%s/News/Pan/", g_get_home_dir());


	acache_max_megs = pan_config_get_int ("/Pan/Cache/MaxMegs=10");

	/* general preferences */

	/* display preferences */
	pan_config_push_prefix ("/Pan/Display/");
	header_pane_is_threaded = pan_config_get_bool("header_pane_is_threaded=TRUE");
	show_group_pane = pan_config_get_bool("show_group_pane=TRUE");
	show_header_pane = pan_config_get_bool("show_header_pane=TRUE");
	show_body_pane = pan_config_get_bool("show_body_pane=TRUE");
	text_set_wrap (pan_config_get_bool("do_wrap=FALSE"));
	collapse_group_names = pan_config_get_bool("collapse_group_names=FALSE");
	text_window_smooth_scrolling = pan_config_get_bool("smooth_scrolling=TRUE");
	text_window_smooth_scrolling_speed = pan_config_get_int("smooth_scrolling_speed_2=10");
	text_use_fixed_font = pan_config_get_bool("use_fixed_font=FALSE");
	replace_gstr (&text_quote_chars, pan_config_get_string ("text_quote_chars"));
	if (!is_nonempty_string (text_quote_chars))
		replace_gstr (&text_quote_chars, g_strdup (">:}|"));
	pan_config_pop_prefix ();

	/* general preferences */
	pan_config_push_prefix ("/Pan/General/");
	single_click_selects = pan_config_get_bool ("single_click_selects=FALSE");
	remove_failed_tasks = pan_config_get_bool ("remove_failed_tasks=TRUE");
	wrap_column = pan_config_get_int ("wrap_column=74");
	fetch_new_on_group_load = pan_config_get_bool ("fetch_new_on_group_load=TRUE");
	fetch_new_and_bodies_on_startup = pan_config_get_bool ("fetch_new_and_bodies_from_subscribed_on_startup=FALSE");
	hide_mpart_child_nodes = pan_config_get_bool ("hide_mpart_child_nodes=TRUE");
	break_thread_when_subject_changes = pan_config_get_bool (
			"break_thread_when_subject_changes=FALSE"); 
	expand_all_threads_by_default = pan_config_get_bool (
			"expand_all_threads_by_default=FALSE");
	pan_config_pop_prefix ();


	header_flags = (gulong) pan_config_get_int ("/Pan/State/Headers=243");

	/**
	***  Newsrc
	**/

	newsrc_do_port =
		pan_config_get_bool ("/Pan/Newsrc/do_port=FALSE");
	newsrc_port_subscribed_only =
		pan_config_get_bool ("/Pan/Newsrc/port_subscribed_only=FALSE");
	replace_gstr (&newsrc_filename, pan_config_get_string ("/Pan/Newsrc/filename"));
	if (!is_nonempty_string(newsrc_filename))
		replace_gstr (&newsrc_filename, g_strdup ("~/.newsrc"));

	/**
	***  Unsorted
	**/

	replace_gstr (&thread_date_format, pan_config_get_string ("/Pan/Display/Thread_Date_Format"));
	if (!is_nonempty_string (thread_date_format))
		replace_gstr (&thread_date_format, g_strdup ("%m.%d %H:%M"));

	replace_gstr (&body_date_format, pan_config_get_string ("/Pan/Display/Body_Date_Format"));
	if (!is_nonempty_string (body_date_format))
		replace_gstr (&body_date_format, g_strdup("%a, %d %b %Y %H:%M:%S"));

	replace_gstr (&external_editor, pan_config_get_string ("/Pan/User/external_editor_2"));
	if (!external_editor)
		external_editor = g_strdup ("xterm -e vi %t");

	replace_gstr (&external_web_browser, pan_config_get_string ("/Pan/User/external_web_browser"));
	if (!external_web_browser) {
		gchar * browser = getenv ("BROWSER");
		if (!is_nonempty_string(browser))
			browser = "netscape %s";
		external_web_browser = g_strdup (browser);
	}

	/* mail server preferences */
	replace_gstr (&mail_server_address, pan_config_get_string ("/Pan/Mail/smtp_address"));
	mail_server_port = pan_config_get_int ("/Pan/Mail/smtp_port=25");

	/**
	***  Fonts
	**/

	replace_gstr (&grouplist_font, pan_config_get_string ("/Pan/Display/grouplist_font"));
	if (!is_nonempty_string (grouplist_font))
		replace_gstr (&grouplist_font, g_strdup(NORMAL_PANGO_DESCRIPTION));

	replace_gstr (&thread_normal_font, pan_config_get_string ("/Pan/Display/thread_normal_font"));
	if (!is_nonempty_string (thread_normal_font))
		replace_gstr (&thread_normal_font, g_strdup(NORMAL_PANGO_DESCRIPTION));

	replace_gstr (&thread_new_replies_font, pan_config_get_string ("/Pan/Display/thread_new_replies_font"));
	if (!is_nonempty_string (thread_new_replies_font))
		replace_gstr (&thread_new_replies_font, g_strdup(BOLD_PANGO_DESCRIPTION));

	replace_gstr (&message_body_font, pan_config_get_string ("/Pan/Display/message_body_font_norm"));
	if (!is_nonempty_string(message_body_font))
		replace_gstr (&message_body_font, g_strdup(NORMAL_PANGO_DESCRIPTION));

	replace_gstr (&message_body_font_fixed, pan_config_get_string ("/Pan/Display/message_body_font_fixed"));
	if (!is_nonempty_string(message_body_font_fixed))
		replace_gstr (&message_body_font_fixed, g_strdup(MONOSPACE_PANGO_DESCRIPTION));

	/**
	***  Colors
	**/

	use_system_bg = pan_config_get_bool ("/Pan/Display/use_system_bg=true");
	use_system_fg = pan_config_get_bool ("/Pan/Display/use_system_fg=true");
	set_color (&thread_normal_color,   "thread_normal_color",       0,     0,     0);
	set_color (&thread_read_color,     "thread_read_color",     40000, 40000, 40000);
	set_color (&killfile_color,        "killfile_color",        35723, 17733,  4883);
	set_color (&watched_color,         "watched_color",             0, 52428,     0);
	set_color (&text_fg_color,         "text_fg_color",             0,     0,     0);
	set_color (&text_bg_color,         "text_bg_color",         65335, 65335, 65335);
	set_color (&text_url_color,        "text_url_color",            0,     0, 40000);
	set_color (&text_quoted_color[0],  "text_quoted_color_1",   40000, 10000, 40000);
	set_color (&text_quoted_color[1],  "text_quoted_color_2",   50000,     0,  2000);
	set_color (&text_quoted_color[2],  "text_quoted_color_3",   45000, 28000,     0);

	base_prefs_init (download_dir,
	                 text_quote_chars,
	                 acache_max_megs,
			 pan_config_get_bool("/Pan/Cache/FlushOnExit=false"),
	                 break_thread_when_subject_changes);

	debug_exit ("prefs_init");
}

static void
prefs_select_row_cb (GtkCList *clist,
		     gint row,
		     gint column,
		     GdkEventButton *event,
		     gpointer user_data)
{
	gtk_notebook_set_page (GTK_NOTEBOOK (win->notebook), row);
}

static int
prefs_add_section_page (gchar     *name,
			GtkWidget *widget)
{
	gchar * text[2];

	text[0] = name;
	text[1] = NULL;

	gtk_notebook_append_page (GTK_NOTEBOOK(win->notebook),
	                          widget,
				  gtk_label_new (name));
	return gtk_clist_append (GTK_CLIST (sections_clist), text);

}

static void
prefs_create_clist (void)
{
	gint i;
	GPtrArray * servers;
	GtkCList * list = GTK_CLIST(win->server_clist);

	/* get the servers */
	servers = g_ptr_array_new ();
	serverlist_get_servers (servers);

	/* build the clist */
	gtk_clist_freeze (list);
	gtk_clist_clear (list);
	for (i=0; i<servers->len; ++i)
	{
		Server * server = SERVER(g_ptr_array_index(servers,i));
		prefs_server * p_server;
		gchar * text[2];
		gchar * server_name = (gchar *) server_get_name (server);
		int row;

		/* skip the internal folder... */
		if (!pan_strcmp (server_name, INTERNAL_SERVER_NAME))
			continue;

		p_server = g_new0 (prefs_server, 1);
		p_server->old = server;
		text[0] = server_name ? server_name : _("Not Named");
		text[1] = server->address ? server->address : _("No Address");
		row = gtk_clist_prepend (list, text);
		gtk_clist_set_row_data (list, row, p_server);
	}
	gtk_clist_thaw (list);

	/* cleanup */
	g_ptr_array_free (servers, TRUE);
}

static void
edit_profile_dialog_response (GtkDialog     * dialog,
                              int             response,
                              prefs_server  * p_server)
{
	server_profile *sprof = gtk_object_get_data(GTK_OBJECT(dialog),"sprof");

	g_assert (sprof);

	if (response == GTK_RESPONSE_OK)
	{
		gint row;
		Server *server;

		/* OK button */
		server = server_new ();
		server->name = gtk_editable_get_chars (GTK_EDITABLE (sprof->profile_name_entry), 0, -1);
		server->address = gtk_editable_get_chars (GTK_EDITABLE (sprof->server_address), 0, -1);
		server->port = atoi((gtk_entry_get_text (GTK_ENTRY(sprof->server_port))));
		server->need_auth = (!GTK_TOGGLE_BUTTON (sprof->auth_cbutton)->active) ? 0 : 1;
		server->gen_msgid = (!GTK_TOGGLE_BUTTON (sprof->msgid_cbutton)->active) ? 0 : 1;
		server->username = gtk_editable_get_chars (GTK_EDITABLE (sprof->server_username), 0, -1);
		server->password = gtk_editable_get_chars (GTK_EDITABLE (sprof->server_password), 0, -1);
		server->idle_secs_before_timeout = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (sprof->idle_secs_before_timeout));

		if (p_server == NULL)
		{
			gchar *text[2];
			GtkCList * list = GTK_CLIST(win->server_clist);
			gchar * server_name = (gchar *) server_get_name (server);

			p_server = g_new0 (prefs_server, 1);
			p_server->new = server;
			text[0] = server_name ? server_name : _("Not Named");
			text[1] = server->address ? server->address : _("No Address");
			row = gtk_clist_append (list, text);
			gtk_clist_set_row_data (list, row, p_server);
		}
		else
		{
			GtkCList * list = GTK_CLIST(win->server_clist);
			if (p_server->new)
				pan_object_unref (PAN_OBJECT(p_server->new));
			p_server->new = server;
			row = gtk_clist_find_row_from_data (list, p_server);
			gtk_clist_set_text (list, row, 1, server->address);
			gtk_clist_set_text (list, row, 0, server_get_name (server));
		}
	}

	gtk_widget_destroy (GTK_WIDGET (dialog));
	sprof_list = g_slist_remove (sprof_list, dialog);
}



static void
auth_cbutton_set (GtkWidget* widget, server_profile *sprof)
{
	const gboolean b = GTK_TOGGLE_BUTTON(widget)->active;
	servers_changed = TRUE;
	gtk_widget_set_sensitive (sprof->server_username_label, b);
	gtk_widget_set_sensitive (sprof->server_username, b);
	gtk_widget_set_sensitive (sprof->server_password_label, b);
	gtk_widget_set_sensitive (sprof->server_password, b);
}

static void
edit_profile_changed (GtkDialog *d)
{
        servers_changed = TRUE;
        gtk_dialog_set_response_sensitive (d, GTK_RESPONSE_OK, TRUE);
        pan_prefs_changed (GTK_DIALOG(win->dialog));
}

/*---[ edit_profile_dialog ]------------------------------------------
 * The dialog for editing a server profile.
 *--------------------------------------------------------------------*/
static void
edit_profile_dialog (prefs_server *p_server)
{
	GtkWidget * w;
	GtkWidget * dialog;
	GtkWidget * table;
	GtkWidget * frame;
	GtkWidget * vbox;
	GtkWidget * hbox;
	GtkAdjustment * adj;
	Server * server;
	server_profile * sprof;

	if (p_server)
		if (p_server->new)
			server = p_server->new;
		else
			server = p_server->old;
	else 
		server = NULL;

	sprof = g_new0 (server_profile, 1);
	
	dialog = gtk_dialog_new_with_buttons (_("Pan: New/Edit Server"),
	                                      GTK_WINDOW(win->dialog),
	                                      GTK_DIALOG_DESTROY_WITH_PARENT,
	                                      GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
	                                      GTK_STOCK_OK, GTK_RESPONSE_OK,
	                                      NULL);
	g_object_set_data_full (G_OBJECT(dialog), "sprof", sprof, g_free);
	gtk_dialog_set_response_sensitive (GTK_DIALOG(dialog), GTK_RESPONSE_OK, FALSE);

	g_signal_connect (GTK_OBJECT(dialog), "response",
	                  G_CALLBACK(edit_profile_dialog_response), p_server);

	table = gtk_table_new (4, 3, FALSE);
	gtk_container_set_border_width (GTK_CONTAINER (table), GUI_PAD_SMALL);
	gtk_table_set_row_spacings (GTK_TABLE (table), GUI_PAD_SMALL);
	gtk_table_set_col_spacings (GTK_TABLE (table), GUI_PAD_SMALL);

	gtk_container_add (GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), table);

	/**
	***  FRAME:  server information
	**/

	frame = gtk_frame_new (_("Server Information"));
	vbox = gtk_vbox_new (FALSE, GUI_PAD_SMALL);
	gtk_container_set_border_width (GTK_CONTAINER (vbox), GUI_PAD_SMALL);
	gtk_container_add (GTK_CONTAINER (frame), vbox);
	hbox = gtk_hbox_new (FALSE, GUI_PAD_SMALL);
	gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);

	/* profile name label */
	w = gtk_label_new (_("Profile Name"));
	gtk_misc_set_alignment (GTK_MISC(w), 1.0, 0.5);
	gtk_box_pack_start (GTK_BOX(hbox), w, TRUE, TRUE, 0);

	/* profile name entry field */
	w = gtk_entry_new ();
	if (server && server_get_name (server))
	{
		pan_gtk_entry_set_text (w, server_get_name (server));
		gtk_editable_set_editable (GTK_EDITABLE(w), FALSE);
	}
	connect_signal_to_prefs_changed (w, "changed");
	connect_signal_to_profile_changed (w, "changed", dialog);
	gtk_box_pack_start (GTK_BOX(hbox), w, FALSE, FALSE, 0);
	sprof->profile_name_entry = w;

	hbox = gtk_hbox_new (FALSE, GUI_PAD_SMALL);
	gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);

	/* server address label */
	w = gtk_label_new (_("Server Address"));
	gtk_misc_set_alignment (GTK_MISC (w), 1.0, 0.5);
	gtk_box_pack_start (GTK_BOX (hbox), w, TRUE, TRUE, 0);

	/* server address */
	w = gtk_entry_new ();
	if (server && server->address)
		pan_gtk_entry_set_text (w, server->address);
	connect_signal_to_prefs_changed (w, "changed");
	connect_signal_to_profile_changed (w, "changed", dialog);
	gtk_box_pack_start (GTK_BOX (hbox), w, FALSE, FALSE, 0);
	sprof->server_address = w;
	hbox = gtk_hbox_new (FALSE, GUI_PAD_SMALL);
	gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);

	/* server port label */
	w = gtk_label_new (_("Server Port"));
	gtk_misc_set_alignment (GTK_MISC(w), 1.0, 0.5);
	gtk_box_pack_start (GTK_BOX (hbox), w, TRUE, TRUE, 0);

	/* server port */
	w = gtk_entry_new ();
	if (1) {
		int num = server && server->port ? server->port : 119;
		char buf[32];
		g_snprintf (buf, sizeof(buf), "%d", num);
		pan_gtk_entry_set_text (w, buf);
	}
	connect_signal_to_prefs_changed (w, "changed");
	connect_signal_to_profile_changed (w, "changed", dialog);
	gtk_box_pack_start (GTK_BOX (hbox), w, FALSE, FALSE, 0);
	sprof->server_port = w;

	gtk_table_attach (GTK_TABLE(table), frame, 
			  0, 2, 0, 3,
			  GTK_EXPAND | GTK_FILL,
			  GTK_EXPAND | GTK_FILL,
			  0, 0);

/* FRAME: Authorization */
	frame = gtk_frame_new (_("Authorization"));
	vbox = gtk_vbox_new (FALSE, GUI_PAD_SMALL);
	gtk_container_set_border_width (GTK_CONTAINER (vbox), GUI_PAD_SMALL);
	gtk_container_add (GTK_CONTAINER (frame), vbox);
	
  /* needs autorization information checkbox */


	w = gtk_check_button_new_with_label (_("My server requires my username and password"));
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), server && server->need_auth);
	g_signal_connect (w, "toggled", G_CALLBACK(auth_cbutton_set), sprof);
	connect_signal_to_prefs_changed (w, "toggled");
	connect_signal_to_profile_changed (w, "toggled", dialog);
	gtk_box_pack_start (GTK_BOX (vbox), w, FALSE, FALSE, 0);
	sprof->auth_cbutton = w;

	hbox = gtk_hbox_new (FALSE, GUI_PAD_SMALL);
	gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);

	/* server username */
	w = gtk_label_new (_("Username"));
	gtk_box_pack_start (GTK_BOX(hbox), w, TRUE, TRUE, 0);
	gtk_misc_set_alignment (GTK_MISC(w), 1.0, 0.5);
	gtk_widget_set_sensitive (w, server && server->need_auth);
	sprof->server_username_label = w;

	w = gtk_entry_new ();
	gtk_widget_set_sensitive (w, server && server->need_auth);
	if (server && server->username)
		pan_gtk_entry_set_text (w, server->username);
	connect_signal_to_prefs_changed (w, "changed");
	connect_signal_to_profile_changed (w, "changed", dialog);
	gtk_box_pack_start (GTK_BOX (hbox), w, FALSE, FALSE, 0);
	sprof->server_username = w;

	/* server password */
	hbox = gtk_hbox_new (FALSE, GUI_PAD_SMALL);
	gtk_box_pack_start (GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
	w = gtk_label_new (_("Password"));
	gtk_widget_set_sensitive (w, server && server->need_auth);
	gtk_box_pack_start (GTK_BOX (hbox), w, TRUE, TRUE, 0);
	gtk_misc_set_alignment (GTK_MISC(w), 1.0, 0.5);
	sprof->server_password_label = w;

	w = gtk_entry_new ();
	gtk_widget_set_sensitive (w, server && server->need_auth);
	gtk_entry_set_visibility (GTK_ENTRY(w), FALSE);
	if (server && server->password)
		pan_gtk_entry_set_text (w, server->password);
	connect_signal_to_prefs_changed (w, "changed");
	connect_signal_to_profile_changed (w, "changed", dialog);
	gtk_box_pack_start (GTK_BOX (hbox), w, FALSE, FALSE, 0);
	sprof->server_password = w;

	gtk_table_attach (GTK_TABLE(table), frame, 
			  0, 2, 4, 6,
			  GTK_EXPAND | GTK_FILL,
			  GTK_EXPAND | GTK_FILL,
			  0, 0);
	
	/* FRAME: Misc */
        frame = gtk_frame_new (_("Misc"));
        vbox = gtk_vbox_new (FALSE, GUI_PAD_SMALL);
        gtk_container_set_border_width (GTK_CONTAINER (vbox), GUI_PAD_SMALL);
        gtk_container_add (GTK_CONTAINER (frame), vbox);

	/* SPIN: Idle Time */
        hbox = gtk_hbox_new (FALSE, GUI_PAD_SMALL);
	w = gtk_label_new (_("Idle Seconds Before Timeout:"));
	gtk_box_pack_start (GTK_BOX(hbox), w, FALSE, FALSE, 0);

	adj = (GtkAdjustment *) gtk_adjustment_new (180.0, 1.0, 360.0, 10.0, 10.0, 0.0);
	w = sprof->idle_secs_before_timeout = gtk_spin_button_new (adj, 0, 0);

	if (server)  {
		gtk_spin_button_set_value (
			GTK_SPIN_BUTTON (sprof->idle_secs_before_timeout),
			server->idle_secs_before_timeout);
        }

	connect_signal_to_prefs_changed (w, "changed");
	connect_signal_to_profile_changed (w, "changed", dialog);
	gtk_box_pack_start (GTK_BOX (hbox), sprof->idle_secs_before_timeout, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);

        /* TOGGLE: Generate Message-ID */
        w = gtk_check_button_new_with_label (_("Generate Message-ID"));
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), server && server->gen_msgid);
	connect_signal_to_prefs_changed (w, "toggled");
	connect_signal_to_profile_changed (w, "toggled", dialog);
        gtk_box_pack_start (GTK_BOX(vbox), w, FALSE, FALSE, 0);
        sprof->msgid_cbutton = w;

        gtk_table_attach (GTK_TABLE(table), frame,
                          0, 2, 7, 9,
                          GTK_EXPAND | GTK_FILL,
                          GTK_EXPAND | GTK_FILL,
                          0, 0);

	sprof_list = g_slist_prepend (sprof_list, dialog);

	gtk_widget_show_all (dialog);
}


static gboolean
server_clist_button_press (GtkWidget *widget, GdkEventButton *bevent)
{
	int row;
	int column;
	prefs_server *p_server = NULL;

	if (bevent->button != 1)
		gtk_signal_emit_stop_by_name (GTK_OBJECT(widget),
					      "button_press_event");

	if ((bevent->button) == 1 && (bevent->type == GDK_2BUTTON_PRESS))
	{
		gtk_signal_emit_stop_by_name (GTK_OBJECT(widget),
					      "button_press_event");
		gtk_clist_get_selection_info (GTK_CLIST(widget), bevent->x,
					      bevent->y, &row, &column);
		p_server = gtk_clist_get_row_data (GTK_CLIST(widget), row);
		edit_profile_dialog (p_server);
	}

	return FALSE;
}


static void
new_server_cb (void)
{
	edit_profile_dialog (NULL);
}


static void
edit_server_cb (void)
{
	GList * list = GTK_CLIST(win->server_clist)->selection;
	if (list != NULL)
	{
		prefs_server * p_server = gtk_clist_get_row_data (
			GTK_CLIST (win->server_clist),
			GPOINTER_TO_INT (list->data));
		edit_profile_dialog (p_server);
	}
}


static void
delete_server_cb (void)
{
	GtkCList * clist = GTK_CLIST(win->server_clist);
	GList * l = clist->selection;
	if (l != NULL)
	{
		const int row = GPOINTER_TO_INT(l->data);
		prefs_server * p_server = gtk_clist_get_row_data (clist, row);

		server_deletes = g_slist_prepend (server_deletes, p_server);
		gtk_clist_remove (clist, row);
		pan_prefs_changed (GTK_DIALOG(win->dialog));
		servers_changed = TRUE;
	}
}


static void
save_dir_help_clicked_cb (void)
{
	const gchar * str = _("%g - group as one directory (alt.binaries.pictures.trains)\n"
	                      "%G - group as nested directory (/alt/binaries/pictures/trains)\n"
	                      " \n"
	                      "\"/home/user/News/Pan/%g\" becomes\n"
	                      "\"/home/user/News/Pan/alt.binaries.pictures.trains\", and\n"
	                      "\"/home/user/News/Pan/%G\" becomes\n"
	                      "\"/home/user/News/Pan/alt/binaries/pictures/trains\",");
	GtkWidget * w = gtk_message_dialog_new (GTK_WINDOW(win->dialog),
	                                        GTK_DIALOG_DESTROY_WITH_PARENT,
	                                        GTK_MESSAGE_INFO,
	                                        GTK_BUTTONS_CLOSE, "%s", str);
	g_signal_connect_swapped (GTK_OBJECT(w), "response",
	                          G_CALLBACK(gtk_widget_destroy), GTK_OBJECT (w));
	gtk_widget_show_all (w);
}


static GtkWidget*
prefs_general_page (void)
{
	char * p;
	GList * l;
	GtkWidget * w;
	GtkWidget * h;
	GtkWidget * v;
	GtkWidget * vbox;
	GtkWidget * frame;

	vbox = gtk_vbox_new (FALSE, GUI_PAD);
	gtk_container_set_border_width (GTK_CONTAINER(vbox), GUI_PAD_BIG);

	/**
	***  External Applications
	**/

	frame = gtk_frame_new (_("External Applications"));
	v = gtk_vbox_new (FALSE, GUI_PAD);
	gtk_container_set_border_width (GTK_CONTAINER(v), GUI_PAD);
	gtk_container_add (GTK_CONTAINER (frame), v);
	gtk_box_pack_start (GTK_BOX(vbox), frame, FALSE, FALSE, 0);

	w = gtk_label_new(_("Editor (%t will be replaced with filename)"));
	gtk_misc_set_alignment (GTK_MISC(w), 0.0, 0.5);
	gtk_box_pack_start (GTK_BOX(v), w, FALSE, FALSE, 0);

	/* external editor */
	win->external_editor_combo = w = gtk_combo_new ();
	l = NULL;
	l = g_list_append (l, "gedit %t");
	l = g_list_append (l, "mgedit --no-fork %t");
	l = g_list_append (l, "emacs %t");
	l = g_list_append (l, "xemacs %t");
	l = g_list_append (l, "xterm -e \"jed %t\"");
	l = g_list_append (l, "xterm -e \"vi %t\"");
	l = g_list_append (l, "konsole -e \"jed %t\"");
	l = g_list_append (l, "konsole -e \"vi %t\"");
	l = g_list_append (l, "gnome-terminal -e \"jed %t\"");
	l = g_list_append (l, "gnome-terminal -e \"vi %t\"");
	gtk_combo_set_popdown_strings (GTK_COMBO(w), l);
	g_list_free (l);
	pan_gtk_entry_set_text (GTK_COMBO(w)->entry, external_editor);
	connect_signal_to_prefs_changed (GTK_COMBO(w)->entry, "changed");
	gtk_box_pack_start (GTK_BOX(v), w, FALSE, FALSE, 0);

	w = gtk_label_new(_("Web Browser (%s will be replaced with URL)"));
	gtk_misc_set_alignment (GTK_MISC(w), 0.0, 0.5);
	gtk_box_pack_start (GTK_BOX(v), w, FALSE, FALSE, 0);

	win->external_browser_combo = w = gtk_combo_new ();
	l = NULL;
	l = g_list_append (l, "gnome-moz-remote --raise --newwin '%s'");
	l = g_list_append (l, "galeon --existing '%s'");
	l = g_list_append (l, "netscape '%s'");
	l = g_list_append (l, "netscape -remote 'openURL(%s,raise)'");
	l = g_list_append (l, "mozilla -remote 'openURL(%s,raise)'");
	l = g_list_append (l, "konqueror '%s'");
	l = g_list_append (l, "xterm -e \"w3m '%s'\"");
	l = g_list_append (l, "xterm -e \"lynx '%s'\"");
	l = g_list_append (l, "xterm -e \"links '%s'\"");
	l = g_list_append (l, "konsole -e \"w3m '%s'\"");
	l = g_list_append (l, "konsole -e \"lynx '%s'\"");
	l = g_list_append (l, "konsole -e \"links '%s'\"");
	l = g_list_append (l, "gnome-terminal -e \"w3m '%s'\"");
	l = g_list_append (l, "gnome-terminal -e \"lynx '%s'\"");
	l = g_list_append (l, "gnome-terminal -e \"links '%s'\"");
	gtk_combo_set_popdown_strings (GTK_COMBO(w), l);
	gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), w,
	                      _("If this field contains the string \"%s\", the URL to be viewed is substituted there.\n \nIf this field is left blank, then your $BROWSER environmental variable will be used (see http://www.tuxedo.org/~esr/BROWSER/).  If that isn't set either, then the default Gnome setting will be used."), "");
	pan_gtk_entry_set_text (GTK_COMBO(w)->entry,
	                        external_web_browser ? external_web_browser : "");
	connect_signal_to_prefs_changed (GTK_COMBO(w)->entry, "changed");
	gtk_box_pack_start (GTK_BOX(v), w, FALSE, FALSE, 0);

	/**
	***  Directories
	**/

	frame = gtk_frame_new (_("Directories"));
	v = gtk_vbox_new (FALSE, GUI_PAD);
	gtk_container_set_border_width (GTK_CONTAINER(v), GUI_PAD);
	gtk_container_add (GTK_CONTAINER (frame), v);
	gtk_box_pack_start (GTK_BOX(vbox), frame, FALSE, FALSE, 0);

	h = gtk_hbox_new (FALSE, GUI_PAD);
	w = gtk_label_new (_("Download Directory:"));
	gtk_box_pack_start (GTK_BOX(h), w, FALSE, FALSE, 0);
	w = win->dir_download = pan_file_entry_new (_("Download Directory"));
	if ((p = pan_config_get_string ("/Pan/Paths/download_dir")) == NULL)
		p = g_strdup_printf ("%s/News/Pan/", g_get_home_dir());
	pan_file_entry_set (win->dir_download, p);
	g_free (p);
	connect_signal_to_prefs_changed (pan_file_entry_gtk_entry(w), "changed");
	gtk_box_pack_start (GTK_BOX(h), w, TRUE, TRUE, 0);
	w = gtk_button_new_with_label (_("Help"));
	g_signal_connect (w, "clicked", G_CALLBACK (save_dir_help_clicked_cb), NULL);
	gtk_box_pack_start (GTK_BOX(h), w, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX(v), h, FALSE, FALSE, 0);


	/**
	***  Tweaks
	**/

	frame = gtk_frame_new (_("Other Options"));
	v = gtk_vbox_new (FALSE, GUI_PAD);
	gtk_container_set_border_width (GTK_CONTAINER(v), GUI_PAD);
	gtk_container_add (GTK_CONTAINER (frame), v);
	gtk_box_pack_start (GTK_BOX(vbox), frame, FALSE, FALSE, 0);

	/* download new headers */
	w = gtk_check_button_new_with_label (_("Single-click selects, rather than activates, groups and articles"));
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), single_click_selects);
	connect_signal_to_prefs_changed (w, "toggled");
	win->single_click_selects_cbutton = w;
	gtk_box_pack_start (GTK_BOX(v), w, FALSE, FALSE, 0);

	/* download new headers */
	w = gtk_check_button_new_with_label (_("Download new headers from subscribed groups when starting Pan"));
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), fetch_new_on_startup);
	connect_signal_to_prefs_changed (w, "toggled");
	win->fetch_new_on_startup_cbutton = w;
	gtk_box_pack_start (GTK_BOX(v), w, FALSE, FALSE, 0);

	/* download new headers and bodies */
	w = gtk_check_button_new_with_label (_("Download new headers and bodies from subscribed groups when starting Pan"));
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), fetch_new_and_bodies_on_startup);
	connect_signal_to_prefs_changed (w, "toggled");
	win->fetch_new_and_bodies_on_startup_cbutton = w;
	gtk_box_pack_start (GTK_BOX(v), w, FALSE, FALSE, 0);

	/* remove failed tasks */
	w = gtk_check_button_new_with_label (_("Automatically remove failed tasks from the task manager"));
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), remove_failed_tasks);
	connect_signal_to_prefs_changed (w, "toggled");
	win->remove_failed_tasks_cbutton = w;
	gtk_box_pack_start (GTK_BOX(v), w, FALSE, FALSE, 0);

	return vbox;
}




static GtkWidget*
create_font_picker (const gchar * name, const char * font, GtkWidget ** setme_picker)
{
	GtkWidget * picker;
	GtkWidget * label;
	GtkWidget * hbox;

	/* sanity clause */
	g_return_val_if_fail (is_nonempty_string(name), NULL);
	g_return_val_if_fail (is_nonempty_string(font), NULL);

	/* create the label */
	label = gtk_label_new (name);

	/* create the picker */
	*setme_picker = picker = pan_font_picker_new ();
	if (is_nonempty_string (font))
		pan_font_picker_set_font (picker, font);
	connect_signal_to_prefs_changed (picker, "clicked");

	/* create the container */
	hbox = gtk_hbox_new (FALSE, GUI_PAD_SMALL);
	gtk_box_pack_start (GTK_BOX(hbox), picker, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX(hbox), label, FALSE, FALSE, 0);

	return hbox;
}


static GtkWidget*
create_color_picker (const gchar * name, const GdkColor * c, GtkWidget ** setme_picker)
{
	GtkWidget * picker;
	GtkWidget * label;
	GtkWidget * hbox;

	/* create the label */
	label = gtk_label_new (name);

	/* create the picker */
	*setme_picker = picker = pan_color_picker_new ();
	pan_color_picker_set_color (picker, c);
	connect_signal_to_prefs_changed (picker, "clicked");

	/* create the container */
	hbox = gtk_hbox_new (FALSE, GUI_PAD_SMALL);
	gtk_box_pack_start (GTK_BOX(hbox), picker, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX(hbox), label, FALSE, FALSE, 0);

	return hbox;
}

/**
***  prefs_headers_page
**/

typedef struct
{
	gchar* text;
	gulong flag;
}
MessageListThrowawayStruct;


static GtkWidget *
prefs_headers_page (void)
{
	GtkWidget * frame;
	GtkWidget * v;
	GtkCList  * list = GTK_CLIST (gtk_clist_new(1));
	gint      i;
	gulong    flags;
	MessageListThrowawayStruct rows [] = {
		{NULL, UI_HEADER_AUTHOR},
		{NULL, UI_HEADER_DATE},
		{NULL, UI_HEADER_FOLLOWUP_TO},
		{NULL, UI_HEADER_MESSAGE_ID},
		{NULL, UI_HEADER_NEWSGROUPS},
		{NULL, UI_HEADER_REFERENCES},
		{NULL, UI_HEADER_REPLY_TO},
		{NULL, UI_HEADER_NEWSREADER},
		{NULL, UI_HEADER_SUBJECT}
	};
	const int row_qty = G_N_ELEMENTS (rows);
	flags = (gulong) pan_config_get_int ("/Pan/State/Headers=243");
	rows[0].text = _("Author");
	rows[1].text = _("Date");
	rows[2].text = _("Followup-To");
	rows[3].text = _("Message-Id");
	rows[4].text = _("Newsgroups (if more than one group)");
	rows[5].text = _("References (if article is a reply)");
	rows[6].text = _("Reply-To (if different from Author)");
	rows[7].text = _("Newsreader");
	rows[8].text = _("Subject");
	win->article_headers_clist = GTK_WIDGET(list);
	gtk_clist_set_selection_mode (list, GTK_SELECTION_MULTIPLE);

	for (i=0; i!=row_qty; ++i)
	{
		gint row = gtk_clist_insert (list, -1, &rows[i].text);

		gulong* plong = g_new0 (gulong, 1);
		*plong = rows[i].flag;
		gtk_clist_set_row_data_full (list, row, plong, g_free);

		if (rows[i].flag & flags)
			gtk_clist_select_row ( list, row, -1 );
	}

	connect_signal_to_prefs_changed (list, "select-row");
	connect_signal_to_prefs_changed (list, "unselect-row");
	v = gtk_vbox_new (FALSE, 0);
	gtk_container_set_border_width (GTK_CONTAINER(v), GUI_PAD);
	gtk_box_pack_start (GTK_BOX(v), GTK_WIDGET(list), FALSE, FALSE, 0);
	frame = gtk_frame_new (_("Headers to Show"));
	gtk_container_add (GTK_CONTAINER(frame), v);

	return frame;
}


/**
***  prefs_text_page
**/

static gulong
get_header_flags (void)
{
	GtkCList *clist = GTK_CLIST(win->article_headers_clist);
	GList* sel = clist->selection;
	gulong flags = 0;

	while ( sel != NULL ) {
		int row_index = GPOINTER_TO_INT ( sel->data );
		gulong* pulong = gtk_clist_get_row_data ( clist, row_index );
		flags |= *pulong;
		sel = sel->next;
	}

	return flags;
}

static GtkWidget*
prefs_text_page (void)
{
	GtkAdjustment * adj;
	GtkWidget * frame;
	GtkWidget * vbox;
	GtkWidget * w;
	GtkWidget * v;
	GtkWidget * h;

	vbox = gtk_vbox_new (FALSE, 0);
	gtk_container_set_border_width (GTK_CONTAINER(vbox), GUI_PAD_SMALL);

	/**
	***  Fonts
	**/

	v = gtk_vbox_new (FALSE, 0);
	w = create_font_picker (_("Message Body"), message_body_font, &win->message_gfp);
	gtk_box_pack_start (GTK_BOX(v), w, 0, 0, GUI_PAD_SMALL);
	w = create_font_picker (_("Message Body (Monospace)"), message_body_font_fixed, &win->message_fixed_gfp);
	gtk_box_pack_start (GTK_BOX(v), w, 0, 0, GUI_PAD_SMALL);
	frame = gtk_frame_new (_("Fonts"));
	gtk_container_add (GTK_CONTAINER(frame), v);
	gtk_container_set_border_width (GTK_CONTAINER(v), GUI_PAD);
	gtk_box_pack_start (GTK_BOX(vbox), frame, FALSE, FALSE, GUI_PAD_SMALL);

	/**
	*** Colors
	**/

	v = gtk_vbox_new (FALSE, 0);
	h = gtk_hbox_new (FALSE, 0);
	w = create_color_picker (_("Body Background"), &text_bg_color, &win->text_bg_pcp);
	gtk_box_pack_start (GTK_BOX(h), w, 0, 0, 0);
	w = win->use_system_bg_check = gtk_check_button_new_with_label (_("Use System Background"));
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), use_system_bg);
	connect_signal_to_prefs_changed (w, "toggled");
	gtk_box_pack_end (GTK_BOX(h), w, 0, 0, 0);
	gtk_box_pack_start (GTK_BOX(v), h, 0, 0, 0);
	h = gtk_hbox_new (FALSE, 0);
	w = create_color_picker (_("Body Foreground"), &text_fg_color, &win->text_fg_pcp);
	gtk_box_pack_start (GTK_BOX(h), w, 0, 0, 0);
	w = win->use_system_fg_check = gtk_check_button_new_with_label (_("Use System Foreground"));
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), use_system_fg);
	connect_signal_to_prefs_changed (w, "toggled");
	gtk_box_pack_end (GTK_BOX(h), w, 0, 0, 0);
	gtk_box_pack_start (GTK_BOX(v), h, 0, 0, 0);
	w = create_color_picker (_("Quoted Text 1"), &text_quoted_color[0], &win->text_quoted_pcp[0]);
	gtk_box_pack_start (GTK_BOX(v), w, 0, 0, 0);
	w = create_color_picker (_("Quoted Text 2"), &text_quoted_color[1], &win->text_quoted_pcp[1]);
	gtk_box_pack_start (GTK_BOX(v), w, 0, 0, 0);
	w = create_color_picker (_("Quoted Text 3"), &text_quoted_color[2], &win->text_quoted_pcp[2]);
	gtk_box_pack_start (GTK_BOX(v), w, 0, 0, 0);
	w = create_color_picker (_("URLs"), &text_url_color, &win->text_url_pcp);
	gtk_box_pack_start (GTK_BOX(v), w, 0, 0, 0);
	frame = gtk_frame_new (_("Colors"));
	gtk_container_set_border_width (GTK_CONTAINER(v), GUI_PAD);
	gtk_container_add (GTK_CONTAINER(frame), v);
	gtk_box_pack_start (GTK_BOX(vbox), frame, FALSE, FALSE, GUI_PAD_SMALL);

	/**
	***  Options
	**/

	v = gtk_vbox_new (FALSE, GUI_PAD);
	gtk_container_set_border_width (GTK_CONTAINER(v), GUI_PAD_SMALL);

	h = gtk_hbox_new (FALSE, GUI_PAD_SMALL);
	w = gtk_label_new (_("Characters which denote quoted text:"));
	gtk_box_pack_start (GTK_BOX(h), w, FALSE, FALSE, 0);
	w = gtk_entry_new_with_max_length (20);
	gtk_entry_set_text (GTK_ENTRY(w), text_quote_chars);
	connect_signal_to_prefs_changed (w, "changed");
	gtk_box_pack_start (GTK_BOX(h), w, TRUE, TRUE, 0);
	win->text_quoted_chars_entry = w;
	gtk_box_pack_start (GTK_BOX(v), h, FALSE, FALSE, 0);

	h = gtk_hbox_new (FALSE, GUI_PAD_SMALL);
	w = gtk_label_new (_("Date format:"));
	gtk_box_pack_start (GTK_BOX(h), w, FALSE, FALSE, 0);
	w = gtk_entry_new_with_max_length (128);
	pan_gtk_entry_set_text (w, body_date_format);
	connect_signal_to_prefs_changed (w, "changed");
	gtk_box_pack_start (GTK_BOX(h), w, TRUE, TRUE, 0);
	win->body_date_entry = w;
	w = gtk_button_new_with_label (_("Help"));
	g_signal_connect (w, "clicked",
			  G_CALLBACK (date_help_clicked_cb), NULL);
	gtk_box_pack_start (GTK_BOX(h), w, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX(v), h, FALSE, FALSE, 0);

	h = gtk_hbox_new (FALSE, GUI_PAD_SMALL);
	w = gtk_check_button_new_with_label (_("Smooth scrolling at speed:"));
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), text_window_smooth_scrolling);
	connect_signal_to_prefs_changed (w, "toggled");
	gtk_box_pack_start (GTK_BOX(h), w, FALSE, FALSE, 0);
	win->smooth_scrolling_check = w;
	adj = (GtkAdjustment *) gtk_adjustment_new (1.0, 1.0, 10.0, 1.0, 1.0, 0.0);
	w = gtk_spin_button_new (adj, 0, 0);
	gtk_spin_button_set_value (GTK_SPIN_BUTTON(w), text_window_smooth_scrolling_speed);
	connect_signal_to_prefs_changed (w, "changed");
	gtk_box_pack_start (GTK_BOX(h), w, FALSE, FALSE, 0);
	win->smooth_scrolling_speed_sb = w;
	gtk_box_pack_start (GTK_BOX(v), h, FALSE, FALSE, 0);

	h = gtk_hbox_new (FALSE, GUI_PAD_SMALL);
	w = gtk_check_button_new_with_label (_("Wrap text at column"));
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), text_get_wrap());
	connect_signal_to_prefs_changed (w, "toggled");
	gtk_box_pack_start (GTK_BOX(h), w, FALSE, FALSE, 0);
	win->wrap_column_tb = w;
	adj = (GtkAdjustment *) gtk_adjustment_new (256.0, 1.0, 256.0, 1.0, 1.0, 0.0);
	w = gtk_spin_button_new (adj, 0, 0);
	gtk_spin_button_set_value (GTK_SPIN_BUTTON(w), wrap_column);
	connect_signal_to_prefs_changed (w, "changed");
	gtk_box_pack_start (GTK_BOX(h), w, FALSE, FALSE, 0);
	win->wrap_column_sb = w;
	gtk_box_pack_start (GTK_BOX(v), h, FALSE, FALSE, 0);

	frame = gtk_frame_new (_("Options"));
	gtk_container_set_border_width (GTK_CONTAINER(v), GUI_PAD);
	gtk_container_add (GTK_CONTAINER(frame), v);
	gtk_box_pack_start (GTK_BOX(vbox), frame, FALSE, FALSE, 0);

	return vbox;
}


static GtkWidget*
prefs_articlelist_page (void)
{
	GtkWidget * frame;
	GtkWidget * vbox;
	GtkWidget * w;
	GtkWidget * h;
	GtkWidget * v;

	vbox = gtk_vbox_new (FALSE, 0);
	gtk_container_set_border_width (GTK_CONTAINER(vbox), GUI_PAD_SMALL);

	/**
	***  Fonts
	**/

	v = gtk_vbox_new (FALSE, 0);
	w = create_font_picker (_("Old Articles with no New Replies"), thread_normal_font, &win->thread_normal_gfp);
	gtk_box_pack_start (GTK_BOX(v), w, 0, 0, GUI_PAD_SMALL);
	w = create_font_picker (_("New Articles / Articles with New Replies"), thread_new_replies_font, &win->thread_new_replies_gfp);
	gtk_box_pack_start (GTK_BOX(v), w, 0, 0, GUI_PAD_SMALL);
	frame = gtk_frame_new (_("Fonts"));
	gtk_container_add (GTK_CONTAINER(frame), v);
	gtk_container_set_border_width (GTK_CONTAINER(v), GUI_PAD);
	gtk_box_pack_start (GTK_BOX(vbox), frame, FALSE, FALSE, GUI_PAD_SMALL);

	/**
	***  Colors
	**/

	v = gtk_vbox_new (FALSE, 0);
	w = create_color_picker (_("Unread Articles / Articles with Unread Replies"), &thread_normal_color, &win->thread_normal_pcp);
	gtk_box_pack_start (GTK_BOX(v), w, 0, 0, 0);
	w = create_color_picker (_("Read Articles with no Unread Replies"), &thread_read_color, &win->thread_read_pcp);
	gtk_box_pack_start (GTK_BOX(v), w, 0, 0, 0);
	w = create_color_picker (_("Ignored Threads"), &killfile_color, &win->ignored_pcp);
	gtk_box_pack_start (GTK_BOX(v), w, 0, 0, 0);
	w = create_color_picker (_("Watched Threads"), &watched_color, &win->watched_pcp);
	gtk_box_pack_start (GTK_BOX(v), w, 0, 0, 0);
	frame = gtk_frame_new (_("Colors"));
	gtk_container_add (GTK_CONTAINER(frame), v);
	gtk_container_set_border_width (GTK_CONTAINER(v), GUI_PAD);
	gtk_box_pack_start (GTK_BOX(vbox), frame, FALSE, FALSE, GUI_PAD_SMALL);

	/**
	***  General
	**/

	v = gtk_vbox_new (FALSE, GUI_PAD_SMALL);
	gtk_container_set_border_width (GTK_CONTAINER(v), GUI_PAD);

	/* break threads when subject changes? */
	w = gtk_check_button_new_with_label (_("When a Follow-Up subject header changes, show as new thread"));
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), break_thread_when_subject_changes);
	connect_signal_to_prefs_changed (w, "toggled");
	gtk_box_pack_start (GTK_BOX(v), w, FALSE, FALSE, 0);
	win->break_thread_when_subject_changes_cbutton = w;

	/* expand threads by default? */
	w = gtk_check_button_new_with_label (_("Expand all threads by default"));
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), expand_all_threads_by_default);
	connect_signal_to_prefs_changed (w, "toggled");
	gtk_box_pack_start (GTK_BOX(v), w, FALSE, FALSE, 0);
	win->expand_all_threads_by_default_cbutton = w;

	/* multipart: one node or all */
	w = gtk_check_button_new_with_label (_("Show complete multipart posts as a single article"));
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), hide_mpart_child_nodes);
	connect_signal_to_prefs_changed (w, "toggled");
	gtk_box_pack_start (GTK_BOX(v), w, FALSE, FALSE, 0);
	win->hide_mpart_child_nodes_cbutton = w;

	/* date format */
	h = gtk_hbox_new (FALSE, GUI_PAD_SMALL);
	w = gtk_label_new (_("Date format:"));
	gtk_box_pack_start (GTK_BOX(h), w, FALSE, FALSE, 0);
	w = gtk_entry_new_with_max_length (128);
	pan_gtk_entry_set_text (w, thread_date_format);
	connect_signal_to_prefs_changed (w, "changed");
	gtk_box_pack_start (GTK_BOX(h), w, TRUE, TRUE, 0);
	win->thread_date_entry = w;
	w = gtk_button_new_with_label (_("Help"));
	g_signal_connect (w, "clicked",
	                  G_CALLBACK (date_help_clicked_cb), NULL);
	gtk_box_pack_start (GTK_BOX(h), w, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX(v), h, FALSE, FALSE, 0);

	/* frame */
	frame = gtk_frame_new (_("Options"));
	gtk_container_add (GTK_CONTAINER(frame), v);
	gtk_box_pack_start (GTK_BOX(vbox), frame, FALSE, FALSE, 0);

	return vbox;
}



/**
***
***  LAYOUT
***
**/

static gchar*
layout_get_new_string (GtkWidget * page)
{
	int i;
	int layout = -1;
	char str[16];
	GtkCList * clist = GTK_CLIST(gtk_object_get_data (GTK_OBJECT(page), "order_clist"));
	GtkWidget ** layout_buttons = (GtkWidget**) gtk_object_get_data (GTK_OBJECT(page), "layout_buttons");

	for (i=0; i<5; ++i) {
		GtkToggleButton * tb = GTK_TOGGLE_BUTTON(layout_buttons[i]);
		if (gtk_toggle_button_get_active(tb))
			layout = i + 1;
	}

	g_snprintf (str, sizeof(str), "%d%c%c%c", layout,
		(gchar) GPOINTER_TO_INT (gtk_clist_get_row_data (clist, 0)),
		(gchar) GPOINTER_TO_INT (gtk_clist_get_row_data (clist, 1)),
		(gchar) GPOINTER_TO_INT (gtk_clist_get_row_data (clist, 2)));

	return g_strdup (str);
}

static void
layout_changed_cb (GtkToggleButton * togglebutton, gpointer user_data)
{
	GtkWidget ** layout_buttons = (GtkWidget**) user_data;
	static gboolean dampen_feedback_loop = FALSE;

	if (!dampen_feedback_loop) {
		int i;
		dampen_feedback_loop = TRUE;
		for (i=0; i<5; ++i) {
			GtkToggleButton * tb = GTK_TOGGLE_BUTTON(layout_buttons[i]);
			gboolean active = togglebutton==tb;
			if (gtk_toggle_button_get_active(tb) != active)
				gtk_toggle_button_set_active (tb, active);
		}
		dampen_feedback_loop = FALSE;
	}
}

static const gchar*
layout_get_pane_name (char ch)
{
	switch (ch) {
		case 'g': case 'G': return _("Grouplist Pane");
		case 'a': case 'A': return _("Body Pane");
		case 't': case 'T': return _("Header Pane");
		default: return _("Error");
	}
}

static void
layout_update_clist_row (GtkCList * clist, int row)
{
	char ch = (char)GPOINTER_TO_INT(gtk_clist_get_row_data(clist,row));
	const char * pane_name = layout_get_pane_name (ch);
	char buf[512];
	g_snprintf (buf, sizeof(buf), "%d. %s", (row+1), pane_name);
	gtk_clist_set_text (clist, row, 0, buf);
}

static void
layout_arrow_up_cb (GtkButton * button, gpointer user_data)
{
	GtkCList * clist = GTK_CLIST (user_data);
	GList * sel = clist->selection;
	int row = -1;
	if (sel != NULL)
		row = GPOINTER_TO_INT (sel->data);
	if (row > 0) {
		gtk_clist_row_move (clist, row, row-1);
		layout_update_clist_row (clist, row-1);
		layout_update_clist_row (clist, row);
	}
}

static void
layout_arrow_down_cb (GtkButton * button, gpointer user_data)
{
	GtkCList * clist = GTK_CLIST (user_data);
	GList * sel = clist->selection;
	int row = -1;
	if (sel != NULL)
		row = GPOINTER_TO_INT (sel->data);
	if (row>-1 && row<clist->rows) {
		gtk_clist_row_move (clist, row, row+1);
		layout_update_clist_row (clist, row);
		layout_update_clist_row (clist, row+1);
	}
}

static GtkWidget*
prefs_layout_page (void)
{
	int i;
	GtkWidget * frame;
	GtkWidget * vbox;
	GtkWidget * hbox;
	GtkWidget * bbox;
	GtkWidget * w;
	GtkWidget * clist;
	GtkWidget * l;
	const char lch = *layout_str;
	const char * pch;
	int row;
	gchar buf [64];
	gchar * text [1];
	const guint8 * inline_txt [] = { icon_layout_1, icon_layout_2, icon_layout_3, icon_layout_4, icon_layout_5};
	GtkWidget ** layout_buttons = g_malloc (sizeof(GtkWidget*) * 5);

       	text[0]= buf;

	/* Root container */
	vbox = gtk_vbox_new (FALSE, GUI_PAD_SMALL);

	/* Label for the layout buttons */
	l = gtk_label_new (_("Selected layout: "));
	gtk_misc_set_alignment (GTK_MISC(l), 0.0, 0.5);
	gtk_box_pack_start (GTK_BOX(vbox), l, FALSE, FALSE, GUI_PAD_SMALL);

	/* Layout buttons */
	hbox = gtk_hbox_new (FALSE, GUI_PAD_SMALL);

	for (i=0; i<5; ++i)
	{
		char buf[64];
		GdkPixbuf * pixbuf;
		GdkPixmap * pixmap;
		GdkBitmap * bitmap;

		g_snprintf (buf, sizeof(buf), "%d", i+1);
		w = gtk_toggle_button_new ();
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), *buf==lch);

		pixbuf = gdk_pixbuf_new_from_inline (-1, inline_txt[i], FALSE, NULL);
		gdk_pixbuf_render_pixmap_and_mask_for_colormap (pixbuf, gtk_widget_get_colormap(w), &pixmap, &bitmap, 128);
		gtk_container_add (GTK_CONTAINER(w), gtk_image_new_from_pixmap (pixmap, bitmap));

		g_object_unref (pixbuf);
		g_object_unref (pixmap);
		g_object_unref (bitmap);

		gtk_box_pack_start (GTK_BOX(hbox), w, FALSE, FALSE, 0);
		g_signal_connect (w, "toggled", G_CALLBACK(layout_changed_cb), layout_buttons);
		connect_signal_to_prefs_changed (w, "toggled");
		layout_buttons[i] = w;
	}

	gtk_box_pack_start (GTK_BOX(vbox), hbox, FALSE, FALSE, GUI_PAD_SMALL);

	/* Label for layout clist */
	l = gtk_label_new (_("Assigned panes: "));
	gtk_misc_set_alignment (GTK_MISC(l), 0.0, 0.5);
	gtk_box_pack_start (GTK_BOX(vbox), l, FALSE, FALSE, GUI_PAD_SMALL);

	/* Panes clist */
	hbox = gtk_hbox_new (FALSE, GUI_PAD_SMALL);

	clist = gtk_clist_new (1);
	gtk_clist_set_column_width (GTK_CLIST(clist), 0, 200);

	pch = layout_str + 1;
	g_snprintf (buf, sizeof(buf), "1. %s", layout_get_pane_name(*pch));
	row = gtk_clist_insert (GTK_CLIST(clist), -1, text);
	gtk_clist_set_row_data (GTK_CLIST(clist), row, GINT_TO_POINTER((int)*pch));

	++pch;
	g_snprintf (buf, sizeof(buf), "2. %s", layout_get_pane_name(*pch));
	row = gtk_clist_insert (GTK_CLIST(clist), -1, text);
	gtk_clist_set_row_data (GTK_CLIST(clist), row, GINT_TO_POINTER((int)*pch));

	++pch;
	g_snprintf (buf, sizeof(buf), "3. %s", layout_get_pane_name(*pch));
	row = gtk_clist_insert (GTK_CLIST(clist), -1, text);
	gtk_clist_set_row_data (GTK_CLIST(clist), row, GINT_TO_POINTER((int)*pch));

	/* panes arrowbuttons */
	bbox = gtk_vbox_new (FALSE, 0);

	w = gtk_button_new ();
	gtk_container_add (GTK_CONTAINER(w), gtk_arrow_new (GTK_ARROW_UP, GTK_SHADOW_OUT));
	gtk_box_pack_start (GTK_BOX(bbox), w, FALSE, TRUE, 0);
	g_signal_connect (w, "clicked", G_CALLBACK(layout_arrow_up_cb), clist);
	connect_signal_to_prefs_changed (w, "clicked");

	w = gtk_button_new ();
	gtk_container_add (GTK_CONTAINER(w), gtk_arrow_new (GTK_ARROW_DOWN, GTK_SHADOW_OUT));
	gtk_box_pack_end (GTK_BOX(bbox), w, FALSE, FALSE, 0);
	g_signal_connect (w, "clicked", G_CALLBACK(layout_arrow_down_cb), clist);
	connect_signal_to_prefs_changed (w, "clicked");

	gtk_box_pack_start (GTK_BOX(hbox), clist, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX(hbox), bbox, FALSE, FALSE, 0);

	gtk_box_pack_start (GTK_BOX(vbox), hbox, FALSE, FALSE, 0);


	/* Frame it */
        frame = gtk_frame_new (_("Layout"));
        gtk_container_add (GTK_CONTAINER(frame), vbox);
        gtk_container_set_border_width (GTK_CONTAINER(vbox), GUI_PAD_SMALL);

	/* Data assigned to this page */
	gtk_object_set_data (GTK_OBJECT(frame), "order_clist", clist);
	gtk_object_set_data_full (GTK_OBJECT(frame), "layout_buttons", layout_buttons, g_free);

	return frame;
}

/**
***
***  GROUPLIST
***
**/

static GtkWidget*
prefs_grouplist_page (void)
{
	GtkWidget * frame;
	GtkWidget * vbox;
	GtkWidget * w;
	GtkWidget * v;

	vbox = gtk_vbox_new (FALSE, 0);
	gtk_container_set_border_width (GTK_CONTAINER(vbox), GUI_PAD_SMALL);

	/**
	***  Fonts
	**/

	v = gtk_vbox_new (FALSE, 0);
	w = create_font_picker (_("Group list font"), grouplist_font, &win->grouplist_gfp);
	gtk_box_pack_start (GTK_BOX(v), w, 0, 0, GUI_PAD_SMALL);
	frame = gtk_frame_new (_("Fonts"));
	gtk_container_add (GTK_CONTAINER(frame), v);
	gtk_container_set_border_width (GTK_CONTAINER(v), GUI_PAD);
	gtk_box_pack_start (GTK_BOX(vbox), frame, FALSE, FALSE, GUI_PAD_SMALL);

	/**
	***  Options
	**/

	v = gtk_vbox_new (FALSE, 0);

	/* download new headers */
	w = gtk_check_button_new_with_label (_("Download New Headers when Loading a Group"));
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), fetch_new_on_group_load);
	connect_signal_to_prefs_changed (w, "toggled");
	gtk_box_pack_start (GTK_BOX(v), w, FALSE, FALSE, 0);
	win->fetch_new_on_group_load_cbutton = w;

	frame = gtk_frame_new (_("Options"));
	gtk_container_add (GTK_CONTAINER(frame), v);
	gtk_container_set_border_width (GTK_CONTAINER(v), GUI_PAD);
	gtk_box_pack_start (GTK_BOX(vbox), frame, FALSE, FALSE, GUI_PAD_SMALL);

	/**
	***
	**/

	return vbox;
}


/*---[ prefs_connections_page ]---------------------------------------
 * Connections preferences page for setting the data, temp, and download
 * dirs in Pan.
 *--------------------------------------------------------------------*/

static GtkWidget*
inset (GtkWidget* w)
{
	GtkWidget* outer;
	GtkWidget* inner;

	inner = gtk_frame_new (NULL);
	gtk_frame_set_shadow_type (GTK_FRAME(inner), GTK_SHADOW_NONE);
	gtk_container_set_border_width (GTK_CONTAINER(inner), GUI_PAD_SMALL);
	gtk_container_add (GTK_CONTAINER(inner), w);

	outer = gtk_frame_new (NULL);
	gtk_frame_set_shadow_type (GTK_FRAME(outer), GTK_SHADOW_IN);
	gtk_container_add (GTK_CONTAINER(outer), inner);
	gtk_container_set_border_width (GTK_CONTAINER(inner), 0);

	return outer;
}

static GtkWidget*
prefs_connections_page (void)
{
	GtkAdjustment *adj;
	GtkWidget *vbox;
	GtkWidget *hbox;
	GtkWidget *table;
	GtkWidget *frame;
	GtkWidget *w;
	gint i;
	GPtrArray * servers;
	const int max_per_server =
		pan_config_get_int ("/Pan/General/Max_Connections_Per_Server=4");

	vbox = gtk_vbox_new (FALSE, GUI_PAD_SMALL);

	/* max connections total */
	hbox = gtk_hbox_new (FALSE, GUI_PAD_SMALL);
	w = gtk_label_new (_("Maximum Total Number of Connections"));
	gtk_box_pack_start (GTK_BOX(hbox), w, FALSE, FALSE, 0);
	adj = (GtkAdjustment *) gtk_adjustment_new (128.0, 1.0, 128.0, 1.0, 1.0, 0.0);
	win->max_connections = w = gtk_spin_button_new (adj, 0, 0);
	gtk_spin_button_set_value (GTK_SPIN_BUTTON (win->max_connections), queue_get_max_sockets());
	connect_signal_to_prefs_changed (w, "changed");
	gtk_box_pack_start (GTK_BOX(hbox), win->max_connections, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

	/* task retries */
	hbox = gtk_hbox_new (FALSE, GUI_PAD_SMALL);
	w = gtk_label_new (_("Maximum Number of Task Retries before Giving Up"));
	gtk_box_pack_start (GTK_BOX(hbox), w, FALSE, FALSE, 0);
	adj = (GtkAdjustment *) gtk_adjustment_new (4.0, 1.0, 999.0, 1.0, 1.0, 0.0);
	win->max_tries = w = gtk_spin_button_new (adj, 0, 0);
	gtk_spin_button_set_value (GTK_SPIN_BUTTON (win->max_tries), queue_get_max_tries());
	connect_signal_to_prefs_changed (w, "changed");
	gtk_box_pack_start (GTK_BOX(hbox), win->max_tries, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

	/* pack the vbox into a frame, which is itself the first element in a new vbox */
	frame = gtk_frame_new (_("Session Preferences"));
	gtk_container_set_border_width (GTK_CONTAINER(vbox), GUI_PAD_SMALL);
	gtk_container_add (GTK_CONTAINER(frame), vbox);
	vbox = gtk_vbox_new (FALSE, GUI_PAD_BIG);
	gtk_box_pack_start (GTK_BOX(vbox), frame, FALSE, FALSE, 0);

	servers = g_ptr_array_new ();
	serverlist_get_servers (servers);

	/* per-server preferences */
	table = gtk_table_new (servers->len+1, 3, FALSE);
	w = gtk_label_new (_("News Server"));
	gtk_misc_set_padding (GTK_MISC(w), GUI_PAD_SMALL, GUI_PAD_SMALL);
	gtk_table_attach (GTK_TABLE (table), w, 0, 1, 0, 1,  GTK_FILL, GTK_FILL, 0, 0);
	w = gtk_label_new (_("Maximum\nConnections"));
	gtk_misc_set_padding (GTK_MISC(w), GUI_PAD_SMALL, GUI_PAD_SMALL);
	gtk_table_attach (GTK_TABLE (table), w, 1, 2, 0, 1,  GTK_FILL, GTK_FILL, 0, 0);
	w = gtk_label_new (_("Reserve a connection\nfor reading articles"));
	gtk_misc_set_padding (GTK_MISC(w), GUI_PAD_SMALL, GUI_PAD_SMALL);
	gtk_table_attach (GTK_TABLE (table), w, 2, 3, 0, 1,  GTK_FILL, GTK_FILL, 0, 0);
	for (i=0; i<servers->len; ++i)
	{
		server_connection * sc;
		const gint row = i + 1;
		const Server * server = SERVER(g_ptr_array_index(servers,i));

		/* skip the internal server */
		if (!pan_strcmp(server_get_name (server), INTERNAL_SERVER_NAME))
			continue;

		sc = g_new0 (server_connection, 1);
		sc->server = server;

		/* server name */
		w = gtk_label_new (server_get_name (server));
		gtk_table_attach (GTK_TABLE(table), inset(w), 0, 1, row, row+1, GTK_FILL, GTK_FILL, 0, 0);

		/* max connections per server */
		adj = (GtkAdjustment *) gtk_adjustment_new (max_per_server, 1.0, max_per_server, 1.0, 1.0, 0.0);
		w = gtk_spin_button_new (adj, 0, 0);
		gtk_spin_button_set_value (GTK_SPIN_BUTTON(w), server->max_connections);
		connect_signal_to_prefs_changed (w, "changed");
		gtk_table_attach (GTK_TABLE(table), inset(w), 1, 2, row, row+1, GTK_FILL, GTK_FILL, 0, 0);
		sc->max_connections_sb = w;

		/* reserve */
		w = gtk_check_button_new_with_label (_("Reserve"));
		sc->reserve_tb = w;
		gtk_table_attach (GTK_TABLE(table), inset(w), 2, 3, row, row+1, GTK_FILL, GTK_FILL, 0, 0);
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), server->reserve_connection_for_bodies);
		connect_signal_to_prefs_changed (w, "toggled");

		/* remember this server */
		win->server_connections = g_slist_append (win->server_connections, sc);
	}

	frame = gtk_frame_new (_("Per-Server Preferences"));
	gtk_container_set_border_width (GTK_CONTAINER(table), GUI_PAD_SMALL);
	gtk_container_add (GTK_CONTAINER(frame), table);
	gtk_box_pack_start (GTK_BOX(vbox), frame, FALSE, FALSE, 0);

	g_ptr_array_free (servers, TRUE);
	return vbox;
}


/**
***
***
**/


static void
clear_acache_clicked_cb (GtkButton *button, gpointer data)
{
	const int i = acache_expire_all ();
	if (i != 0)
	{
		GtkWidget * w  = gtk_message_dialog_new (GTK_WINDOW(win->dialog),
		                                         GTK_DIALOG_DESTROY_WITH_PARENT,
		                                         GTK_MESSAGE_INFO,
		                                         GTK_BUTTONS_CLOSE,
		                                         _("%d files deleted."), i);
		g_signal_connect_swapped (GTK_OBJECT(w), "response",
		                          G_CALLBACK(gtk_widget_destroy), GTK_OBJECT(w));
		gtk_widget_show_all (w);
	}
}

/* Build and return the 'Cache' page */
static GtkWidget*
prefs_cache_page (void)
{
	const gboolean flush_on_exit = pan_config_get_bool("/Pan/Cache/FlushOnExit=false");
	GtkObject * adj;
	GtkWidget * frame;
	GtkWidget * label;
	GtkWidget * vbox = gtk_vbox_new (FALSE, GUI_PAD);
	GtkWidget * hbox = gtk_hbox_new (FALSE, GUI_PAD);
	GtkWidget * button;
	GtkWidget * tb;
	GtkWidget * w;

	gtk_container_set_border_width (GTK_CONTAINER (vbox), GUI_PAD_BIG);

	/* explanation */
	label = gtk_label_new (_("The cache stores articles on your computer\nto reduce network connection time."));
	gtk_box_pack_start (GTK_BOX(vbox), label, FALSE, FALSE, 0);

	/* disc cache label */
	hbox = gtk_hbox_new (FALSE, GUI_PAD_SMALL);		
	label = gtk_label_new (_("Maximum Cache Size (Megs):"));
	gtk_label_set_justify (GTK_LABEL(label), GTK_JUSTIFY_LEFT);
	gtk_box_pack_start (GTK_BOX(hbox), label, FALSE, FALSE, 0);

	/* disc cache adjustment */
	adj = gtk_adjustment_new  (acache_max_megs, 1.0, 1000, 1.0, 1.0, 1.0);
	win->cache_megs_sb = w = gtk_spin_button_new (GTK_ADJUSTMENT(adj), 1.0, 0);
	connect_signal_to_prefs_changed (w, "changed");
	gtk_widget_set_usize (GTK_WIDGET (w), 80, 0);  /*Fixed width because of GTK 1.2 limitations*/
	gtk_box_pack_start (GTK_BOX(hbox), w, FALSE, TRUE, 0);
	
	/* clear cache */
	button = gtk_button_new_with_label (_("Clear Cache Now"));
	g_signal_connect (button, "clicked",
	                  G_CALLBACK (clear_acache_clicked_cb), NULL);
	gtk_box_pack_start (GTK_BOX(hbox), button, FALSE, FALSE, GUI_PAD);
	gtk_box_pack_start (GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

	
	
	/*delete cache on exit? */
	tb = win->flush_cache_on_exit_check =
		gtk_check_button_new_with_label (_("Clear Cache when exiting Pan"));
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(tb), flush_on_exit);
	connect_signal_to_prefs_changed (tb, "toggled");
	gtk_box_pack_start (GTK_BOX(vbox), tb, FALSE, FALSE, GUI_PAD_SMALL);

	/* frame */
	frame = gtk_frame_new (_("Article Cache"));
	gtk_container_add (GTK_CONTAINER(frame), vbox);

	return frame;
}

/* Build and return the 'SMTP Server' page */
static GtkWidget *
prefs_smtp_page (void)
{
	GtkWidget * frame;
	GtkWidget * table;
	GtkWidget * w;

	table = gtk_table_new (2, 2, FALSE);
        gtk_table_set_row_spacings (GTK_TABLE (table), GUI_PAD);
        gtk_table_set_col_spacings (GTK_TABLE (table), GUI_PAD_SMALL);
	gtk_container_set_border_width (GTK_CONTAINER (table), GUI_PAD_SMALL);

	/* address */
	w = gtk_label_new (_("Address:"));
	gtk_misc_set_alignment (GTK_MISC(w), 1.0, 0.5);
	gtk_table_attach (GTK_TABLE(table), w, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 4, 0);
	win->smtp_address = w = gtk_entry_new ();
	if (is_nonempty_string(mail_server_address))
		pan_gtk_entry_set_text (w, mail_server_address);
	gtk_table_attach (GTK_TABLE (table), w, 1, 2, 0, 1, GTK_FILL|GTK_EXPAND, GTK_FILL, 4, 0);
	connect_signal_to_prefs_changed (w, "changed");

	/* port */
	w = gtk_label_new (_("Port:"));
	gtk_misc_set_alignment (GTK_MISC(w), 1.0, 0.5);
	gtk_table_attach (GTK_TABLE(table), w, 0, 1, 1, 2, GTK_FILL, GTK_FILL, 4, 0);
	win->smtp_port = w = gtk_spin_button_new (
		GTK_ADJUSTMENT(gtk_adjustment_new (mail_server_port, 0, 65536, 1, 1, 1)), 1, 0);
	gtk_table_attach (GTK_TABLE(table), w, 1, 2, 1, 2,
			  GTK_FILL|GTK_EXPAND, GTK_FILL, 4, 0);
	connect_signal_to_prefs_changed (w, "changed");

	/* frame */
	frame = gtk_frame_new (_("Mail Server"));
	gtk_container_add (GTK_CONTAINER(frame), table);

	return frame;
}

/**
***  Newsrc
**/

static void
do_import_toggled_cb (GtkWidget* widget, server_profile *sprof)
{
	const gboolean b = GTK_TOGGLE_BUTTON(widget)->active;
	gtk_widget_set_sensitive (win->newsrc_useless_label, b);
	gtk_widget_set_sensitive (win->newsrc_filename_entry, b);
	gtk_widget_set_sensitive (win->newsrc_subscribed_only_check, b);
}

static GtkWidget *
prefs_newsrc_page (void)
{
	GtkWidget * frame;
	GtkWidget * v;
	GtkWidget * w;
	GtkWidget * h;

	/* vbox */
	v = gtk_vbox_new (FALSE, GUI_PAD);
	gtk_container_set_border_width (GTK_CONTAINER(v), GUI_PAD_SMALL);

	/* explanation */
	h = gtk_hbox_new (FALSE, GUI_PAD);
	w = gtk_label_new (_(".newsrc files are used by many newsreaders to remember\nwhich articles you've read and which groups you're subscribed to.\nIf you plan on using other newsreaders along with Pan,\nyou can set Pan to automatically import them on startup and\nexport them on shutdown."));
	gtk_label_set_justify (GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	gtk_misc_set_alignment (GTK_MISC(w), 0.0, 0.5);
	gtk_box_pack_start (GTK_BOX(h), w, TRUE, FALSE, GUI_PAD_SMALL);
	gtk_box_pack_start (GTK_BOX(v), h, FALSE, FALSE, GUI_PAD_SMALL);

	/* import/export togglebutton */
	w = gtk_check_button_new_with_label (_("Import/Export .newsrc files automatically"));
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), newsrc_do_port);
	g_signal_connect (w, "toggled",
	                  G_CALLBACK(do_import_toggled_cb), NULL);
	connect_signal_to_prefs_changed (w, "toggled");
	win->newsrc_import_check = w;
	gtk_box_pack_start (GTK_BOX(v), w, FALSE, FALSE, 0);

	/* .newsrc filename */
	h = gtk_hbox_new (FALSE, GUI_PAD);
	w = gtk_label_new (_(".newsrc filename:"));
	gtk_box_pack_start (GTK_BOX(h), w, FALSE, FALSE, 0);
	gtk_widget_set_sensitive (w, newsrc_do_port);
	win->newsrc_useless_label = w;
	w = gtk_entry_new ();
	if (is_nonempty_string(newsrc_filename))
		pan_gtk_entry_set_text (w, newsrc_filename);
	gtk_box_pack_start (GTK_BOX(h), w, TRUE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX(v), h, FALSE, FALSE, 0);
	gtk_widget_set_sensitive (w, newsrc_do_port);
	win->newsrc_filename_entry = w;

	/* use subscribed only? */
	w = gtk_check_button_new_with_label (_("read/write subscribed groups only"));
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), newsrc_port_subscribed_only);
	connect_signal_to_prefs_changed (w, "toggled");
	gtk_box_pack_start (GTK_BOX(v), w, FALSE, FALSE, 0);
	gtk_widget_set_sensitive (w, newsrc_do_port);
	gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), w, _("Many users prefer to save memory and disk space by only keeping subscribed groups in their .newsrc file."), NULL);
	win->newsrc_subscribed_only_check = w;

	/* frame */
	frame = gtk_frame_new (_(".Newsrc Files"));
	gtk_container_add (GTK_CONTAINER(frame), v);
	return frame;
}



/*---[ prefs_server_page ]--------------------------------------------
 * This is the Server tab of the preferences.  It has a list of the
 * server's and buttons that allow users to make a new server entry,
 * edit a previous one, or delete one.  This doesn't actually let
 * them switch servers, that's done from the main menu of Pan.
 *--------------------------------------------------------------------*/
static GtkWidget *
prefs_server_page (void)
{
	GtkWidget *vbox, *hbox, *button;

	char *titles[2];
	titles[0] = _("Name");
	titles[1] = _("Address");

	hbox = gtk_hbox_new (FALSE, 0);

	win->server_clist = gtk_clist_new_with_titles (2, titles);
	gtk_clist_column_titles_passive (GTK_CLIST(win->server_clist));
	gtk_clist_set_selection_mode (GTK_CLIST(win->server_clist),
				      GTK_SELECTION_SINGLE);

	gtk_clist_set_column_width (GTK_CLIST(win->server_clist), 0, 100);
	gtk_clist_set_column_auto_resize (GTK_CLIST(win->server_clist), 1, TRUE);

	gtk_widget_set_usize (win->server_clist, 280, 200);
	gtk_box_pack_start (GTK_BOX(hbox), win->server_clist, TRUE, TRUE, 5);
	g_signal_connect (win->server_clist, "button_press_event",
	                  G_CALLBACK(server_clist_button_press), NULL);

	prefs_create_clist ();

	/* one button vbox */
	vbox = gtk_vbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX(hbox), vbox, FALSE, FALSE, 5);

	/* new server button */
	button = gtk_button_new_with_label (_("New..."));
	gtk_box_pack_start (GTK_BOX(vbox), button, FALSE, FALSE, GUI_PAD);
	g_signal_connect (button, "clicked",
	                  G_CALLBACK (new_server_cb), NULL);

	/* edit server button */
	button = gtk_button_new_with_label (_("Edit..."));
	gtk_box_pack_start (GTK_BOX(vbox), button, FALSE, FALSE, GUI_PAD);
	g_signal_connect (button, "clicked",
	                  G_CALLBACK (edit_server_cb), NULL);

	/* delete server button */
	button = gtk_button_new_with_label(_("Delete"));
	gtk_box_pack_start (GTK_BOX(vbox), button, FALSE, FALSE, GUI_PAD);
	g_signal_connect (button, "clicked",
	                  G_CALLBACK (delete_server_cb), NULL);

	return hbox;
}

void
prefs_spawn_to_news_connections (void)
{
	prefs_spawn ();

	pan_lock ();
	gtk_clist_select_row (GTK_CLIST(sections_clist), nntp_connections_row, 0);
	pan_unlock ();
}

void
prefs_spawn_to_news_servers (void)
{
	prefs_spawn ();

	pan_lock ();
	gtk_clist_select_row (GTK_CLIST(sections_clist), nntp_clist_row, 0);
	pan_unlock ();
}

static void
prefs_destroy_cb (GtkObject * object, gpointer user_data)
{
	int r = 0;
	prefs_server * p_server;

	while ((p_server = gtk_clist_get_row_data (GTK_CLIST(win->server_clist), r))) {
		if (p_server->new)
			pan_object_unref (PAN_OBJECT(p_server->new));
		g_free (p_server);
		++r;
	}

	/* cleanup widgets */
	g_slist_foreach (sprof_list, (GFunc)gtk_widget_destroy, NULL);
	g_slist_free (sprof_list);
	sprof_list = NULL;

	/* cleanup main window */
	g_free (win);
	win = NULL;
}

static void
prefs_response_cb (GtkDialog * dialog, int response, gpointer user_data)
{
	if (response == GTK_RESPONSE_OK || response == GTK_RESPONSE_APPLY)
		prefs_apply ();

	if (response == GTK_RESPONSE_OK || response == GTK_RESPONSE_CANCEL)
		gtk_widget_destroy (GTK_WIDGET(dialog));
}

void
prefs_spawn (void)
{
	GtkWidget * w;
	GtkWidget * hbox;

	/* There can be only one! (properties window) */
	if (win) {
		gdk_window_raise (GTK_WIDGET(win->dialog)->window);
		return;
	}

	win = g_new0 (PrefsWindow, 1);

	win->dialog = w = gtk_dialog_new_with_buttons (_("Pan: Preferences"),
	                                               GTK_WINDOW(Pan.window),
	                                               GTK_DIALOG_DESTROY_WITH_PARENT,
	                                               GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
	                                               GTK_STOCK_APPLY, GTK_RESPONSE_APPLY,
	                                               GTK_STOCK_OK, GTK_RESPONSE_OK,
	                                               NULL);
	gtk_dialog_set_response_sensitive (GTK_DIALOG(w), GTK_RESPONSE_OK, FALSE);
	gtk_dialog_set_response_sensitive (GTK_DIALOG(w), GTK_RESPONSE_APPLY, FALSE);

	/* create the workarea hbox */
	hbox = gtk_hbox_new (FALSE, GUI_PAD);
	gtk_container_set_border_width (GTK_CONTAINER (hbox), GUI_PAD_BIG);
	gtk_box_pack_start (GTK_BOX(GTK_DIALOG(win->dialog)->vbox), hbox, TRUE, TRUE, 0);

	/* the selection clist */
	sections_clist = gtk_clist_new (1);
	gtk_widget_set_usize (GTK_WIDGET (sections_clist), 150, 0);
	gtk_clist_set_selection_mode (GTK_CLIST (sections_clist),
				      GTK_SELECTION_BROWSE);
	g_signal_connect (G_OBJECT (sections_clist), "select_row",
			  G_CALLBACK(prefs_select_row_cb), NULL);
	gtk_box_pack_start (GTK_BOX (hbox), sections_clist, TRUE, TRUE, 0);

	/* create the notebook and its pages... */
	win->notebook = gtk_notebook_new ();
	gtk_notebook_set_show_tabs (GTK_NOTEBOOK(win->notebook), FALSE);
	gtk_notebook_set_show_border (GTK_NOTEBOOK(win->notebook), FALSE);
	gtk_box_pack_start (GTK_BOX (hbox), win->notebook, TRUE, TRUE, 0);

	/* add the tabs */
	prefs_add_section_page (_("Cache"), prefs_cache_page());
	prefs_add_section_page (_("Display - Pane Layout"), win->layout_page=prefs_layout_page());
	prefs_add_section_page (_("Display - Group Pane"), prefs_grouplist_page());
	prefs_add_section_page (_("Display - Header Pane"), prefs_articlelist_page());
	prefs_add_section_page (_("Display - Body Pane"), prefs_text_page());
	prefs_add_section_page (_("Display - Headers"), prefs_headers_page());
	prefs_add_section_page (_("General"), prefs_general_page());
	prefs_add_section_page (_(".Newsrc Files"), prefs_newsrc_page());
	nntp_connections_row =prefs_add_section_page (_("Online Preferences"), prefs_connections_page());
	nntp_clist_row = prefs_add_section_page (_("Servers - News"), prefs_server_page());
	prefs_add_section_page (_("Servers - Mail"), prefs_smtp_page());

	/* callbacks */
	g_signal_connect (win->dialog, "response",
	                  G_CALLBACK(prefs_response_cb), NULL);
	g_signal_connect (win->dialog, "destroy",
	                  G_CALLBACK(prefs_destroy_cb), NULL);

	gtk_widget_show_all (win->dialog);
}
