/*
 *                 Author:  Christopher G. Phillips
 *              Copyright (C) 1994 All Rights Reserved
 *
 *                              NOTICE
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted
 * provided that the above copyright notice appear in all copies and
 * that both the copyright notice and this permission notice appear in
 * supporting documentation.
 *
 * The author makes no representations about the suitability of this
 * software for any purpose.  This software is provided ``as is''
 * without express or implied warranty.
 */

/************************************************************************/
/*	Copyright for PILOTDIS - modified from original m68k program	*/
/*	Copyright (C) 1998 David Griffiths				*/
/*									*/
/*	Copyright for PILOTDIS - modified from original m68k program	*/
/*	Copyright (C) 1999 Nick Spence					*/
/*									*/
/*	Copyright for PILOTDIS - modified from original m68k program	*/
/*	Copyright (C) 2000 Ali Akcaagac					*/
/************************************************************************/

/************************************************************************/
/*	normal includes							*/
/************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include "dis.h"

/************************************************************************/
/*	normal defines							*/
/************************************************************************/
#define ISASCII(c)		(0x20 <= (c) && (c) <= 0x7e)

#ifdef DEBUG
#define INITIAL			0x01
#define DELETIONS		0x02
#define TRY			0x04
#define LASTTRY			0x08
#define MAKEGOOD		0x10
#define OVERLAPS		0x20
#define LABELS			0x40
int	debug = 0;
#endif

/************************************************************************/
/*	eatype								*/
/************************************************************************/
char *eatype[] = {
	"?",
	"B",
	"S",
	"L",
	"R",
	"W"
};

/************************************************************************/
/*	normal structure						*/
/************************************************************************/
struct queue {
	m68kaddr address;
	struct queue *next;
};

static struct queue head = { 0, NULL };

/************************************************************************/
/*	these macros are used for convenience. they do the following:	*/
/*									*/
/*	NREQD(o)	returns the number of instructions instruction	*/
/*			"o" depends on					*/
/*	JUMP(o)		returns whether instruction "o" changes pc	*/
/*			unconditionally	(modulo "jfile" instructions)	*/
/*	FETCH(o,s)	returns whether instruction "o" is valid and	*/
/*			has size "s" (for "s" positive)			*/
/*	DELETE(o)	renders instruction "o" invalid			*/
/*	SETLABEL(o)	marks instruction "o" as requiring a label in	*/
/*			the output					*/
/************************************************************************/
#define NREQD(o)	(insts[o].flags & 3)
#define JUMP(o)		(insts[o].flags & (ISBRA | ISRTS | ISJMP | ISJSR))
#define FETCH(o,s)	(/* (o) >= 0 && */ (o) <= maxoffset && ((s) == -1 && insts[o].size || insts[o].size == (int)(s)))
#define DELETE(o)	do { \
				free(insts[o].required); \
				insts[o].required = NULL; \
				insts[o].size = 0; \
				insts[o].flags = 0; \
			} \
			while(0);
#define SETLABEL(o)	if((o) >= initialpc && (o) <= initialpc + maxoffset) \
				insts[(o) - initialpc].flags |= ISLABEL

/************************************************************************/
/*	normal variables						*/
/************************************************************************/
static size_t longestinstsize;		/* max possible instruction length    */
static long gfsize = 0;			/* file size                          */
static m68kaddr *notinsts = NULL;	/* array of values in "nfile"         */
static size_t nnotinsts = 0;
static m68kaddr *breaks = NULL;		/* array of values in "bfile"         */
static size_t nbreaks = 0;
m68kaddr curoffset;			/* current offset into file           */
short flags;				/* flags for current instruction      */
m68kaddr required[3];			/* instructions that must be valid    */
					/* for current instruction to also be */
					/* valid                              */
int pcrelative = 0;			/* used to signal that pc-relative    */
					/* addresses referenced by jmp and    */
					/* jsr instructions should be stored  */
					/* in "required"                      */
extern char instbuf[];			/* used to store the nibbles of an    */
					/* instruction in hexadecimal         */
extern size_t leninstbuf;		/* current length of string in        */
					/* instbuf                            */
struct inst *insts;			/* instruction database               */
m68kaddr maxoffset;			/* last offset that may hold an       */
					/* instruction                        */

#ifndef NOBAD
static m68kaddr	*bad;			/* addresses that cause instructions  */
					/* to be deleted                      */
static m68kaddr	*good;			/* addresses that cause instructions  */
					/* to be accepted in the final output */
#endif

/************************************************************************/
/*	variables for rotating dash					*/
/************************************************************************/
static int counter = 0;
static int dash = 0;

/************************************************************************/
/*	free dynamically allocated memory and longjmp back to process	*/
/*	the next object file.						*/
/************************************************************************/
void jumpfree(void)
{
	struct global *gptr;

#ifndef NOBAD

	m68kaddr offset;

	for(offset = 0; offset <= maxoffset; offset += (odd ? 1 : WORDSIZE)) {
		if(dobad && insts[offset].size == 0) {
			fprintf(stderr, "%s: Instruction %lx: deleted because of %lx\n", sfile, (long)(offset + initialpc), (long)bad[offset]);
		}

		if(insts[offset].flags & ISGOOD && bad[offset]) {
			fprintf(stderr, "%s: Instruction %lx: assumed good because of %lx, deleted because of %lx\n", sfile, (long)(offset + initialpc), (long)good[offset], (long)bad[offset]);
		}
	}

	free(bad);
	bad = NULL;
	free(good);
	good = NULL;

#endif

	while(gfsize--) {
		free(insts[gfsize].required);

		if(insts[gfsize].labelname) {
			free(insts[gfsize].labelname);
		}
	}

	free(insts);
	insts = NULL;
	gfsize = 0;

	for(gptr = globals; gptr; gptr = globals) {
		globals = gptr->next;

		if(gptr->name) {
			free(gptr->name);
		}

		if(gptr->used) {
			free(gptr->used);
		}

		free(gptr);
	}

	longjmp(jmp, 1);
}

/************************************************************************/
/*	search "p" for a string of at least "minlen" consecutive	*/
/*	printable characters. return the index of the start of the	*/
/*	string (or -1).							*/
/************************************************************************/
static int findstring(size_t stored, const unsigned char *p)
{
	int i;
	int inarow;

	for(inarow = i = 0; i < (int) stored; i++, p++) {
		if(use_isprint && isprint(*p) || !use_isprint && ISASCII(*p)) {
			if(++inarow >= minlen) {
				return i - inarow + 1;
			}
		}
		else {
			inarow = 0;
		}
	}

	return -1;
}

/************************************************************************/
/*	"p" contains the nibbles of a floating-point constant in	*/
/*	hexadecimal. these nibbles are converted into longword values	*/
/*	and passed to "fpoint" which formats the floating-point value	*/
/*	in "s" (in printable form).					*/
/************************************************************************/
static int sfpoint(int type, const unsigned char *p, char *s)
{
	u32bit_t longwords[3];
	size_t nlongwords;
	size_t i, j;

	switch(type) {

	case SINGLE:
		nlongwords = 1;
		break;
	case DOUBLE:
		nlongwords = 2;
		break;
	case EXTENDED:
	case PACKED:
		nlongwords = 3;
		break;
	default:
		break;
	}

/************************************************************************/
/*	convert string to longs.					*/
/************************************************************************/
	for(i = 0; i < nlongwords; i++) {
		longwords[i] = 0;

		for(j = 0; j < 2 * WORDSIZE; j++) {
			longwords[i] += (*p++) << (CHAR_BIT * (2 * WORDSIZE - 1 - j));
		}
	}

	return fpoint(longwords, type, s);
}

/************************************************************************/
/*	output constants. return how many floating-point constants were	*/
/*	output.								*/
/*									*/
/*	todo		number of input bytes to output.		*/
/*	p		nibbles of the constant(s) in hexadecimal.	*/
/*	floattype	floating-point type expected (or 0).		*/
/*	numforce	specifies how many floating-point constants	*/
/*			should be output regardless of value if		*/
/*			"floattype" is nonzero, else specifies that	*/
/*			output must be forced if nonzero.		*/
/*									*/
/*	returns the number of bytes output.				*/
/************************************************************************/
static size_t dcflush(size_t todo, const unsigned char *p, int floattype, int numforce)
{
	char format[BUFSIZ];
	int i;
	m68kaddr value;
	size_t n;
	size_t j;
	size_t length;
	int lflags;
	int first = 1;
	int dofp;
	size_t total = todo;

	while(todo) {
		lflags = ops2f(1);

		if((length = fsizeof(floattype)) == 0) {
			floattype = 0;
		}

/************************************************************************/
/*	check that there are no labels within data			*/
/************************************************************************/
		for(i = 1; i < (int)length; i++) {
			if(insts[i + ppc - initialpc].labelname) {
				length = i;
			}
		}

/************************************************************************/
/*	determine if a floating-point constant should be output.	*/
/*	after forced constants are output, we stay with the		*/
/*	same type until we get nan or denormalized.			*/
/************************************************************************/
		dofp = 0;

		if(floattype && length <= todo) {
			if(sfpoint(floattype, p, format) != -1) {
				dofp = 1;
			}
		}

		if(floattype && --numforce >= 0 || dofp) {
			dofp = 1;
			lflags |= size2f(floattype);
		}
		else if(floattype) {
			return total - todo;
		}

/************************************************************************/
/*	for integral constants, we do 2 words at a time.		*/
/*	if the address is odd, we do the first byte by itself.		*/
/************************************************************************/
		if(!dofp) {
			length = 2 * WORDSIZE;

/************************************************************************/
/*	check that there are no labels within data			*/
/************************************************************************/
			for(i = 1; i < (int)length; i++) {
				if(insts[i + ppc - initialpc].labelname) {
					length = i;
				}
			}

			if(todo < length) {
				if(numforce) {
					length = todo;
				}
				else {
					return total - todo;
				}
			}

			if(first && (ppc & 1) && length != todo) {
				length = 1;
			}

			if((length % WORDSIZE) == 0) {
				lflags |= size2f(WORD);

				for(n = 0, i = 0; i < (int) length / 2; i++) {
					for(value = 0, j = 0; j < WORDSIZE; j++) {
						value += p[i * WORDSIZE + j] << (CHAR_BIT * (WORDSIZE - 1 - j));
						value = signextend(value, 16);
					}

					if(i) {
						n += sprintf(format + n, ",");
					}

					n += sprintf(format + n, "#%d", value);
				}
			}
			else {
				lflags |= size2f(BYTE);

				for(n = 0, i = 0; i < (int) length; i++) {
					if(i) {
						n += sprintf(format + n, ",");
					}

					n += sprintf(format + n, "#%u", p[i]);
				}
			}
		}

		leninstbuf = 0;

		for(i = 0; i < (int) length; i++) {
			leninstbuf += sprintf(instbuf + leninstbuf, "%02x", p[i]);
		}

		pc += length;
		instprint(lflags, "DC", format);
		todo -= length;
		p += length;
		first = 0;
	}

	return total - todo;
}

/************************************************************************/
/*	output a string. return number of characters output.		*/
/************************************************************************/
static size_t ascflush(size_t stored, const unsigned char *p, int force)
{
	char format[BUFSIZ];
	size_t length;
	size_t i;
	size_t n;
	size_t left;
	size_t nbytes;

	for(length = 0; length < stored; length++) {
		if(use_isprint && !isprint(p[length]) || !use_isprint && !ISASCII(p[length])) {
			break;
		}
	}

	if(length == stored && !force) {
		return 0;
	}

	left = length;

	while(left) {
		nbytes = (left > slenprint) ? slenprint : left;

		n = 1;
		format[0] = '\'';

		for(i = length - left; i < length - left + nbytes; i++) {
			format[n++] = p[i];

/************************************************************************/
/*	double single quotes in strings.				*/
/************************************************************************/
			if(p[i] == '\'') {
				format[n++] = '\'';
			}
		}

		format[n++] = '\'';
		format[n] = '\0';
		leninstbuf = 0;

		for(i = length - left; i < length - left + nbytes; i++) {
			leninstbuf += sprintf(instbuf + leninstbuf, "%02x", p[i]);

			if(leninstbuf == 20) {
				pc += leninstbuf / 2;

				if(format[0]) {
					instprint(ops2f(1) | size2f(BYTE), "DC", format);
				}
				else {
					if(!codeonly) {
						instprint(ops2f(0), "");
					}
				}

				format[0] = '\0';
				leninstbuf = 0;
			}
		}

		if(leninstbuf) {
			pc += leninstbuf / 2;

			if(format[0]) {
				instprint(ops2f(1) | size2f(BYTE), "DC", format);
			}
			else {
				if(!codeonly) {
					instprint(ops2f(0), "");
				}
			}
		}

		left -= nbytes;
	}

	return length;
}

/************************************************************************/
/*	convert a floating-point-label type to a floating-point type.	*/
/************************************************************************/
static int fl2ftype(int lflags)
{
	if(lflags & ISLABEL) {
		if(lflags & L_ISSINGLE) {
			return SINGLE;
		}

		if(lflags & L_ISDOUBLE) {
			return DOUBLE;
		}

		if(lflags & L_ISEXTENDED) {
			return EXTENDED;
		}

		if(lflags & L_ISPACKED) {
			return PACKED;
		}
	}

	return 0;
}

/************************************************************************/
/*	output the "stored" input bytes contained in "consts".		*/
/*	if flags specifies a floating-point type, output		*/
/*	floating-point constants of that type as long as the input	*/
/*	looks like them. output strings as appropriate. otherwise,	*/
/*	output integral constants. return number of input bytes *not*	*/
/*	output.								*/
/************************************************************************/
static size_t flush(size_t stored, const unsigned char *consts, int lflags, int force)
{
	size_t length;
	int spos;
	int labelfptype = fl2ftype(lflags);
	int labelfpsize = fsizeof(labelfptype);
	int first = 1;

	while(stored) {
		spos = findstring(stored, consts);

		if(first && labelfpsize && stored >= (size_t) labelfpsize) {
			if(spos == -1) {
				length = stored / labelfpsize * labelfpsize;
			}
			else if(spos > labelfpsize) {
				length = spos / labelfpsize * labelfpsize;
			}
			else {
				length = labelfpsize;
			}

/************************************************************************/
/*	force a floating-point constant.				*/
/************************************************************************/
			length = dcflush(length, consts, labelfptype, 1);
			stored -= length;
			consts += length;
			first = 0;
			continue;
		}

		if(spos) {
			int lforce = 1;

/************************************************************************/
/*	output integral constant(s).
/************************************************************************/
			if(spos < 0) {
				if(force) {
					length = stored;
				}
				else {
					if(stored < (size_t) minlen) {
						return stored;
					}

					length = stored - minlen + 1;
					lforce = 0;
				}
			}
			else {
				length = spos;
			}

			if(length > 0) {
				if((length = dcflush(length, consts, 0, lforce)) == 0) {
					return stored;
				}

				stored -= length;
				consts += length;
			}
			else {
				return stored;
			}
		}

		if(spos >= 0) {

/************************************************************************/
/*	output string.							*/
/************************************************************************/
			if((length = ascflush(stored, consts, force)) == 0) {
				return stored;
			}

			stored -= length;
			consts += length;
		}

		first = 0;
	}

	return 0;
}

/************************************************************************/
/*	read a word (and extension words as necessary) from the input	*/
/*	file and determine if it is a possible instruction.		*/
/*									*/
/*	"valid" is set to signal this.					*/
/************************************************************************/
static int validinst(void)
{
	m68kword inst;

	valid = 0;
	rtsflag = 0;

	if(nextword(&inst) == 0) {

		switch(inst >> 12) {

		case 0:
			bit_movep_immediate(inst);
			break;
		case 1:
			movebyte(inst);
			break;
		case 2:
			movelong(inst);
			break;
		case 3:
			moveword(inst);
			break;
		case 4:
			misc(inst);
			break;
		case 5:
			addq_subq_scc_dbcc_trapcc(inst);
			break;
		case 6:
			valid = bcc_bsr(inst);
			break;
		case 7:
			moveq(inst);
			break;
		case 8:
			or_div_sbcd(inst);
			break;
		case 9:
			sub_subx(inst);
			break;
		case 10:
			aline(inst);
			break;
		case 11:
			cmp_eor(inst);
			break;
		case 12:
			and_mul_abcd_exg(inst);
			break;
		case 13:
			add_addx(inst);
			break;
		case 14:
			shift_rotate_bitfield(inst);
			break;
		case 15:
			coprocessor(inst);

			if(!valid) {
				fline(inst);
			}

			break;
		default:
			break;
		}
	}

	return valid;
}

/************************************************************************/
/*	now that we know where the constants are, make another pass	*/
/*	to determine which of them are referenced using the pc-relative	*/
/*	addressing mode.						*/
/************************************************************************/
static void dcpass(void)
{
	m68kaddr offset;

	pass = DCLABELSPASS;

	if(fseek(infp, 0, SEEK_SET) == -1) {
		perror("fseek");
		jumpfree();
	}

	if(verbose) {
		fprintf(stderr, "Determing PC-relative addressing ..: [ ]\033[2D");
	}

	for(curoffset = offset = 0; offset <= maxoffset; ) {
		if(insts[offset].size) {
			if(curoffset != offset && fseek(infp, curoffset = offset, SEEK_SET) == -1) {
				perror("fseek");
				jumpfree();
			}

			flags = 0;
			pc = ppc = offset + initialpc;
			leninstbuf = 0;
			validinst();
			offset += insts[offset].size;

		}
		else if(odd) {
			offset++;
		}
		else {
			offset += WORDSIZE;
		}

		if(verbose) {
			rotdash(5);
		}
	}

	if(verbose) {
		fprintf(stderr, "x\n");
	}
}

/************************************************************************/
/*	make a pass over the input, outputting things as we currently	*/
/*	see them.							*/
/************************************************************************/
static void printall(void)
{
	m68kaddr offset;
	unsigned char consts[BUFSIZ];
	size_t stored = 0;

	if(fseek(infp, 0, SEEK_SET) == -1) {
		perror("fseek");
		jumpfree();
	}

	pc = ppc = initialpc;
	leninstbuf = 0;
	pass = DEBUGPASS;

	for(curoffset = offset = 0; offset < (m68kaddr) gfsize; offset += (odd ? 1 : WORDSIZE)) {

/************************************************************************/
/*	determine if there might be a valid instruction which has the	*/
/*	bytes at "offset" as operands.					*/
/************************************************************************/
		if(insts[offset].size == 0) {
			int i = 0;
			size_t size;

			for(size = odd ? 1 : WORDSIZE; size <= longestinstsize && size <= offset; size += (odd ? 1 : WORDSIZE)) {
				if(size < (size_t) insts[offset - size].size) {
					i = 1;
					break;
				}
			}

			if(i) {
				continue;
			}
		}

		if(curoffset != offset && fseek(infp, curoffset = offset, SEEK_SET) == -1) {
			perror("fseek");
			jumpfree();
		}

		flags = 0;

		if(insts[offset].size) {
			if(stored) {
				flush(stored, consts, 0, 1);
				stored = 0;
			}

			pc = ppc = initialpc + offset;
			leninstbuf = 0;
			validinst();
		}
		else {
			size_t i;

			if(stored == 0) {
				pc = ppc = initialpc + offset;
			}

			for(i = 0; i < WORDSIZE; i++) {
				if(fread(consts + stored, 1, 1, infp) == 1) {
					if(++stored >= sizeof consts) {
						stored = flush(stored, consts, 0, 0);
						memmove(consts, consts + sizeof consts - stored, stored);
					}
				}
			}
		}
	}

	if(stored) {
		flush(stored, consts, 0, 1);
	}

	fflush(NULL);
}

/************************************************************************/
/*	make the first pass over the input.				*/
/*	gather references to other addresses.				*/
/************************************************************************/
void pass1(void)
{
	m68kaddr offset;

	if(verbose) {
		fprintf(stderr, "Gathering References ..............: [ ]\033[2D");
	}

	pass = FIRSTPASS;

	for(curoffset = offset = 0; offset <= maxoffset; offset += (odd ? 1 : WORDSIZE)) {
		if(curoffset != offset && fseek(infp, curoffset = offset, SEEK_SET) == -1) {
			perror("fseek");
			jumpfree();
		}

		flags = 0;
		pc = ppc = offset + initialpc;
		leninstbuf = 0;

		if(validinst()) {

/************************************************************************/
/* it's a potential instruction.					*/
/************************************************************************/
			insts[offset].size = curoffset - offset;
			insts[offset].flags = flags;

			if(flags & 3) {
				m68kaddr *reqd;

				if((reqd = malloc((flags & 3) * sizeof(*reqd))) == NULL) {
					perror("malloc");
					jumpfree();
				}

				switch(flags & 3) {

				case 3:
					reqd[2] = required[2];
				case 2:
					reqd[1] = required[1];
				case 1:
					reqd[0] = required[0];
				}

				insts[offset].required = reqd;
			}
			else {
				insts[offset].required = NULL;
			}

#ifdef DEBUG

			if(debug & INITIAL) {
				int i;

				fprintf(outfp, "Writing offset = %lx, size = %d, flags = %d", (long)offset, insts[offset].size, insts[offset].flags);

				for(i = 0; i < (insts[offset].flags & 3); i++) {
					fprintf(outfp, ", reqd = %lx", (long)insts[offset].required[i]);
				}

				fprintf(outfp, "\n");
			}

#endif

		}
		else {

#ifndef NOBAD

			bad[offset] = offset + initialpc;

#endif

#ifdef DEBUG

			if(debug & DELETIONS) {
				fprintf(outfp, "0. Deleting offset %lx\n", (long)offset);
			}
#endif

		}

		if(verbose) {
			rotdash(1000);
		}
	}

	if(verbose) {
		fprintf(stderr, "x\n");
	}

#if !defined(NOBAD) || defined(DEBUG)

	for(offset = maxoffset + (odd ? 1 : WORDSIZE); offset <= maxoffset + longestinstsize; offset += (odd ? 1 : WORDSIZE)) {

#ifndef NOBAD

		if(offset <= maxoffset) {
			bad[offset] = offset + initialpc;
		}

#endif

#ifdef DEBUG

		if(debug & DELETIONS) {
			fprintf(outfp, "0. Deleting offset %lx\n", (long)offset);
		}

#endif

	}

#endif

}

/************************************************************************/
/*	make a pass over the instruction database, checking for		*/
/*	consistency.							*/
/************************************************************************/
void findbadrefs(void)
{
	long offset;
	long offset2;
	int i;
	size_t size;
	int changes;

#if 0

	int totalchanges = 0;

#endif

#ifdef DEBUG

	int try = 0;

	if(debug & TRY) {
		printall();
		fprintf(outfp, "\n\n\n");
	}

#endif

/************************************************************************/
/*	instructions that don't set pc must be followed by a valid	*/
/*	instruction.							*/
/************************************************************************/

	do {
		changes = 0;

#ifdef DEBUG

		try++;

#endif

		for(offset = maxoffset + longestinstsize; offset >= 0; offset -= (odd ? 1 : WORDSIZE)) {

/************************************************************************/
/*	back up to a possible instruction.				*/
/*	we do this to jump over a large data section.			*/
/************************************************************************/
			for(offset2 = offset; offset2 >= 0 && insts[offset2].size == 0; offset2 -= (odd ? 1 : WORDSIZE))
				;

			if(offset2 < 0) {
				break;
			}

			if(offset2 + (long)longestinstsize < offset) {
				offset = offset2 + longestinstsize;
			}

			if(!linkfallthrough && (insts[offset].flags & ISLINK) || !FETCH((m68kaddr)offset, -1)) {

/************************************************************************/
/*	we've found an invalid instruction.				*/
/*	see if any instructions advance pc here based on the size of	*/
/*	the instruction and its operands.				*/
/************************************************************************/
				for(size = odd ? 1 : WORDSIZE; size <= longestinstsize && size <= (size_t) offset; size += (odd ? 1 : WORDSIZE)) {
					if(FETCH((m68kaddr)(offset - size), size) && !JUMP(offset - size)) {

#ifndef NOBAD

						if(!linkfallthrough && (insts[offset].flags & ISLINK)) {
							bad[offset - size] = offset + initialpc;
						}
						else {
							bad[offset - size] = bad[offset];
						}

#endif

#ifdef DEBUG

						if(debug & DELETIONS) {
							fprintf(outfp, "1. Deleting offset %lx, size %d, flags = %d\n", (long)(offset - size), size, insts[offset - size].flags);
						}

#endif

						DELETE(offset - size);
						changes++;
					}
				}
			}
		}

/************************************************************************/
/*	see if any instructions require an invalid instruction to be	*/
/*	valid.								*/
/************************************************************************/
		for(offset2 = 0; (m68kaddr) offset2 <= maxoffset; offset2 += (odd ? 1 : WORDSIZE)) {
			if(insts[offset2].size == 0) {
				continue;
			}

			for(i = 0; i < NREQD(offset2); i++) {
				if(insts[offset2].required[i] >= initialpc && insts[offset2].required[i] <= initialpc + maxoffset && !FETCH(insts[offset2].required[i] - initialpc, -1)) {

#ifndef NOBAD

					bad[offset2] = bad[insts[offset2].required[i] - initialpc];

#endif

#ifdef DEBUG

					if(debug & DELETIONS) {
						fprintf(outfp, "2. Deleting offset %lx, size %d because %lx is not valid\n", (long)offset2, insts[offset2].size, (long)insts[offset2].required[i]);
					}

#endif

					DELETE(offset2);
					changes++;
					break;
				}
			}
		}

#ifdef DEBUG

		if(debug & TRY) {
			fprintf(outfp, "TRY %d ###############################\n", try);
			printall();
			fprintf(outfp, "\n\n\n");
		}

#endif

#if 0

		totalchanges += changes;

#endif

	}
	while(changes);

#if 0

	printf("Tries = %d\n", (int)try);
	printf("Changes = %d\n", (int)totalchanges);

#endif

#ifdef DEBUG

	if(debug & LASTTRY) {
		fprintf(outfp, "TRY %d ###############################\n", try - 1);
		printall();
		fprintf(outfp, "\n\n\n");
	}

#endif

}

/************************************************************************/
/*	add to the queue.						*/
/************************************************************************/
static void writeq(m68kaddr address)
{
	struct queue *tail;
	struct queue *newq;

	if((newq = malloc(sizeof(*newq))) == NULL) {
		perror("malloc");
		jumpfree();
	}

	newq->address = address;

#ifdef DEBUG

	if(debug & MAKEGOOD) {
		fprintf(outfp, "Wrote offset = %lx\n", (long)address);
	}

#endif

	newq->next = NULL;

	for(tail = &head; tail->next; tail = tail->next)
		;

	tail->next = newq;
}

/************************************************************************/
/*	read (and delete) from the queue.				*/
/************************************************************************/
static struct queue *readq(void)
{
	struct queue *result;

	if(head.next) {
		result = head.next;
		head.next = head.next->next;

#ifdef DEBUG

		if(debug & MAKEGOOD) {
			fprintf(outfp, "Read offset = %lx\n", (long)result->address);
		}

#endif

	}
	else {
		result = NULL;
	}

	return result;
}

/************************************************************************/
/*	mark "offset" as an instruction to be included in the final	*/
/*	output. recursively mark as good all instructions that		*/
/*	reference it. delete instructions that contradict those marked	*/
/*	good. return the number off offsets deleted.			*/
/************************************************************************/
static unsigned makegood(m68kaddr offset)
{
	size_t size;
	struct queue *qptr = NULL;
	m68kaddr origoffset = offset;
	unsigned deletions = 0;

	if(insts[offset].flags & ISGOOD) {
		return deletions;
	}

#ifdef DEBUG

	if(debug & MAKEGOOD) {
		fprintf(outfp, "makegood(%lx)\n", (long)offset);
	}

#endif

	do {
		if(qptr) {
			offset = qptr->address;
			free(qptr);
		}

#ifdef DEBUG

		if(debug & MAKEGOOD) {
			fprintf(outfp, "Going with offset = %lx\n", (long)offset);
		}

#endif

		while(1) {

#ifdef DEBUG

			if(debug & MAKEGOOD) {
				fprintf(outfp, "Offset = %lx\n", (long)offset);
			}

#endif

			if(insts[offset].size == 0 || insts[offset].flags & ISGOOD) {
				break;
			}

/************************************************************************/
/*	we have a "good" instruction. instructions that overlap it	*/
/*	should be deleted.						*/
/************************************************************************/
			for(size = odd ? 1 : WORDSIZE; size < longestinstsize && size + maxoffset >= offset && size <= offset; size += (odd ? 1 : WORDSIZE)) {
				if(insts[offset - size].size > (int) size) {

#ifndef NOBAD

					bad[offset - size] = origoffset + initialpc;

#endif

#ifdef DEBUG

					if(debug & DELETIONS) {
						fprintf(outfp, "3. Deleting offset %lx, size %d, flags = %d because of %lx\n", (long)(offset - size), insts[offset - size].size, insts[offset - size].flags, (long)offset);
					}

#endif

					DELETE(offset - size);
					deletions++;
				}
			}

			for(size = odd ? 1 : WORDSIZE; (int) size < insts[offset].size && maxoffset >= offset + size; size += (odd ? 1 : WORDSIZE)) {
				if(insts[offset + size].size) {

#ifndef NOBAD

					bad[offset + size] = origoffset + initialpc;

#endif

#ifdef DEBUG

					if(debug & DELETIONS) {
						fprintf(outfp, "4. Deleting offset %lx, size %d, flags = %d because of %lx\n", (long)(offset + size), insts[offset + size].size, insts[offset + size].flags, (long)offset);
					}

#endif

					DELETE(offset + size);
					deletions++;
				}
			}

			insts[offset].flags |= ISGOOD;

#ifndef NOBAD

			good[offset] = origoffset + initialpc;

#endif

			if((insts[offset].flags & ISBRA) || ((insts[offset].flags & ISJMP) && insts[offset].flags & 3)) {
				if(insts[offset].required[0] >= initialpc && insts[offset].required[0] <= initialpc + maxoffset) {
					offset = insts[offset].required[0] - initialpc;
				}
				else {
					break;
				}
			}
			else if((insts[offset].flags & (ISBSR | ISJSR | ISBRA | ISBRcc | ISDBcc | ISJMP)) && (insts[offset].flags & 3) && insts[offset].required[0] >= initialpc && insts[offset].required[0] <= initialpc + maxoffset) {
				writeq(insts[offset].required[0] - initialpc);
				offset += insts[offset].size;
			}
			else if(insts[offset].flags & (ISRTS | ISJMP)) {
				break;
			}
			else {
				offset += insts[offset].size;
			}
		}

		qptr = readq();

	}
	while(qptr);

	return deletions;
}

/************************************************************************/
/*	determine the number of remaining instances of still-valid	*/
/*	instructions overlapping each other.				*/
/************************************************************************/
static long overlaps(int list)
{
	long num = 0;
	m68kaddr offset;
	size_t size;

	for(offset = 0; offset <= maxoffset; offset += (odd ? 1 : WORDSIZE)) {
		if(insts[offset].size == 0) {
			continue;
		}

		for(size = odd ? 1 : WORDSIZE; size < longestinstsize && size <= offset; size += (odd ? 1 : WORDSIZE)) {
			if(insts[offset - size].size > (int) size) {
				num++;

				if(list) {
					fprintf(stderr, "%s: Overlap at %lx\n", sfile, offset);
				}
			}
		}
	}

	return num;
}

/************************************************************************/
/*	starting from "offset", return the next highest offset		*/
/*	containing an instruction marked good (or -1).			*/
/************************************************************************/
static long nextgood(m68kaddr offset)
{
	m68kaddr next;

	for(next = offset + (odd ? 1 : WORDSIZE); next <= maxoffset && (insts[next].size == 0 || (insts[next].flags & ISGOOD) == 0); next += (odd ? 1 : WORDSIZE))
		;

	return (next > maxoffset) ? -1 : next;
}

/************************************************************************/
/*	return whether program flow reaches address "target" starting	*/
/*	from address "prevonly".					*/
/************************************************************************/
static int reach(m68kaddr prevonly, m68kaddr target)
{
	m68kaddr offset;

	for(offset = prevonly; insts[offset].size && offset < target; offset += insts[offset].size)
		;

	return offset == target;
}

/************************************************************************/
/*	fix the overlap situation. do this by minimizing the amount	*/
/*	of data (constants) in the input.				*/
/************************************************************************/
static unsigned fixoverlaps(void)
{
	m68kaddr offset;
	m68kaddr goodone = nextgood(0);
	unsigned deletions = 0;

	for(offset = 0; offset <= maxoffset; ) {
		if(insts[offset].size == 0) {
			if(odd) {
				offset++;
			}
			else {
				offset += WORDSIZE;
			}

			continue;
		}

		if(insts[offset].flags & ISGOOD) {
			offset += insts[offset].size;
			continue;
		}

		if(goodone <= offset) {
			goodone = nextgood(offset);
		}

		if(reach(offset, goodone)) {
			deletions += makegood(offset);
		}
		else if(odd) {
			offset++;
		}
		else {
			offset += WORDSIZE;
		}
	}

	return deletions;
}

/************************************************************************/
/*	addrcmp								*/
/************************************************************************/
static int addrcmp(const void *p1, const void *p2)
{
	m68kaddr a1 = *(m68kaddr *)p1;
	m68kaddr a2 = *(m68kaddr *)p2;

	if(a1 < a2) {
		return -1;
	}
	else if(a1 > a2) {
		return 1;
	}
	else {
		return 0;
	}
}

/************************************************************************/
/*	return whether a "math" fpu instruction has the word at		*/
/*	"offset" as an extension word.					*/
/************************************************************************/
static int fpuoverlap(m68kaddr offset)
{
	size_t size;

	if(!FPU(chip) || (insts[offset].flags & (ISBSR | ISBRA | ISBRcc)) == 0) {
		return 0;
	}

	for(size = 1; size < longestinstsize && size <= offset; size++) {
		if(insts[offset - size].size > (int) size && insts[offset - size].flags & ISFPU) {
			return 1;
		}
	}

	return 0;
}

/************************************************************************/
/*	read label name and flags if present format = label:lines	*/
/*									*/
/*	if there is a label then a label will be created at this	*/
/*	offset and it will be given the supplied label name. However	*/
/*	if the supplied name was "L" or "l" then a label will be	*/
/*	generated but it will be a regular numeric label.		*/
/*									*/
/*	if the label was followed by a colon, then the decimal number	*/
/*	indicates the number of blank lines to be inserted before the	*/
/*	label line. if the number was zero then a form feed is		*/
/*	inserted.							*/
/************************************************************************/
static void readlabel(char *cp, m68kaddr offset)
{
	char name[80];
	char *cp1;
	unsigned long ul;

	while(cp && isspace(*cp))
		cp++;

	if(cp && ISASCII(*cp)) {
		strncpy(name, cp, 79);
		name[79] = 0;
		cp = name;

		while(*cp && *cp != ':' && *cp != '\n')
			cp++;

		if(*cp == ':') {
			ul = strtoul(cp+1, &cp1, 10);

			if(ul > 128L) {
				ul = 128;
			}

			if(cp+1 != cp1) {
				insts[offset].labellines = (ul ? ul : -1);
			}
		}

		*cp = 0;

		if(strlen(name)) {
			SETLABEL(offset);

			if(strcmp(name, "L") && strcmp(name, "l")) {
				insts[offset].labelname = (char*) malloc(strlen(name)+1);

				if(insts[offset].labelname == NULL) {
					perror("malloc");
					jumpfree();
				}
				else {
					strcpy(insts[offset].labelname,name);
				}
			}
		}
	}
}

/************************************************************************/
/*	disassemble!							*/
/************************************************************************/
void disassemble(void)
{
	m68kaddr offset;
	long fsize;
	int changes;
	int total_changes;
	short nlabels;
	int i;
	unsigned char consts[BUFSIZ];
	size_t stored;
	size_t curnbreak;
	static int breakssaved = 0;

/************************************************************************/
/*	determine the longest possible length of an instruction in	*/
/*	words.								*/
/************************************************************************/
	switch(CPU(chip)) {

	case MC68000:
	case MC68010:
		longestinstsize = 10;
		break;
	case MC68020:
	case MC68030:
	default:
		if(FPU(chip)) {
			longestinstsize = 20;
		}
		else {
			longestinstsize = 20;
		}

		break;
	}

/************************************************************************/
/*	determine the size of the input file.				*/
/************************************************************************/
	if(fseek(infp, 0, SEEK_END) == -1) {
		perror("fseek");
		jumpfree();
	}

	if((gfsize = fsize = ftell(infp)) == -1) {
		perror("ftell");
		jumpfree();
	}
	else if(fsize < WORDSIZE) {
		jumpfree();
	}

	if(fseek(infp, 0, SEEK_SET) == -1) {
		perror("fseek");
		jumpfree();
	}

	maxoffset = (fsize - (WORDSIZE - 1)) & ~(WORDSIZE - 1);

/************************************************************************/
/*	malloc and initialize instruction structures.			*/
/************************************************************************/
	if((insts = malloc((fsize + maxoffset) * sizeof(*insts))) == NULL) {
		perror("malloc");
		jumpfree();
	}

#ifndef NOBAD

	if((bad = malloc(fsize * sizeof(*bad))) == NULL || (good = malloc(fsize * sizeof(*good))) == NULL) {
		perror("malloc");
		jumpfree();
	}

	memset(bad, '\0', fsize * sizeof(*bad));
	memset(good, '\0', fsize * sizeof(*good));

	if(!odd) {
		size_t	i;

		for(i = 1; i < (size_t) fsize; i += 2) {
			bad[i] = i + initialpc;
		}
	}

#endif

	memset(insts, '\0', (fsize + maxoffset) * sizeof(*insts));

	while(fsize--) {
		insts[fsize].required = NULL;
	}

	if(verbose) {
		fprintf(stderr, "\nInput Filename ....................: [%s]", vfile);
		fprintf(stderr, "\nInput Filesize ....................: [%d bytes]", (fsize + maxoffset));
		fprintf(stderr, "\nOutput Filename ...................: [%s]", sfile);
		fprintf(stderr, "\nOutput Filesize (aprox) ...........: [%d bytes]\n", ((fsize + maxoffset) * sizeof(*insts))/2);
	}

/************************************************************************/
/*	pass 1:								*/
/*									*/
/*	determine where the code is and where the data is.		*/
/************************************************************************/
	pass1();

	if(onepass == INCONSISTENT) {
		printall();
		jumpfree();
	}

	if(verbose) {
		fprintf(stderr, "Processing user specified offsets .: [ ]\033[2D");
	}

/************************************************************************/
/*	process offsets specified by the user as places in data to	*/
/*	start a new line of output.					*/
/************************************************************************/
	if(bfile && !breakssaved) {
		FILE *bfp;

		bfp = fopen(bfile, "r");

		if(bfp) {
			char bbuf[80];
			unsigned long ul;
			char *cp;
			m68kaddr *tmp;

			while(fgets(bbuf, sizeof(bbuf), bfp)) {
				ul = strtoul(bbuf, &cp, 0);

				if(cp != bbuf && (odd || (ul & 1) == 0)) {
					if(ul >= initialpc && ul <= initialpc + maxoffset) {
						offset = ul - initialpc;
						tmp = realloc(breaks, (nbreaks + 1) * sizeof(*breaks));

						if(tmp) {
							breaks = tmp;
							breaks[nbreaks++] = offset;
						}
					}
					else {
						fprintf(outfp, "File %s: bad pc: %s\n", bfile, bbuf);
					}
				}
				else {
					fprintf(outfp, "File %s: bad pc: %s\n", bfile, bbuf);
				}

				if(verbose) {
					rotdash(1000);
				}
			}

			(void)fclose(bfp);
			qsort(breaks, nbreaks, sizeof(*breaks), addrcmp);
		}
		else {
			perror(bfile);
		}

		breakssaved = 1;
	}

/************************************************************************/
/*	process offsets specified by the user as being data.		*/
/************************************************************************/
	if(nfile) {
		FILE *nfp;
		static int notinstssaved = 0;

		nfp = fopen(nfile, "r");

		if(nfp) {
			char nbuf[80];
			unsigned long ul;
			char *cp;
			m68kaddr *nottmp;
			size_t size;

			while(fgets(nbuf, sizeof(nbuf), nfp)) {
				ul = strtoul(nbuf, &cp, 0);

				if(cp != nbuf && (odd || (ul & 1) == 0)) {
					if(ul >= initialpc && ul <= initialpc + maxoffset) {
						offset = ul - initialpc;
						nottmp = realloc(notinsts, (nnotinsts + 1) * sizeof(*notinsts));

						if(nottmp) {
							notinsts = nottmp;
							notinsts[nnotinsts++] = offset;
							readlabel(cp, offset);
						}

						for(size = 1; size < longestinstsize && size <= offset; size++) {
							if(insts[offset - size].size > (int) size) {

#ifndef NOBAD

								bad[offset - size] = (insts[offset].size) ? offset + initialpc : bad[offset];

#endif

#ifdef DEBUG

								if(debug & DELETIONS) {
									fprintf(outfp, "5. Deleting offset %lx, size %d, flags = %d because %lx specified\n", (long)(offset - size), insts[offset - size].size, insts[offset - size].flags, (long)offset);
								}

#endif

								DELETE(offset - size);
							}
						}

						if(insts[offset].size) {

#ifndef NOBAD

							bad[offset] = offset + initialpc;

#endif

#ifdef DEBUG

							if(debug & DELETIONS) {
								fprintf(outfp, "6. Deleting offset %lx, size %d, flags = %d because specified\n", (long)offset, insts[offset].size, insts[offset].flags);
							}

#endif

							DELETE(offset);
						}
					}
					else {
						fprintf(outfp, "File %s: bad pc: %s\n", nfile, nbuf);
					}
				}
				else {
					fprintf(outfp, "File %s: bad pc: %s\n", nfile, nbuf);
				}

				if(verbose) {
					rotdash(1000);
				}
			}

			(void)fclose(nfp);

			if(!notinstssaved) {
				qsort(notinsts, nnotinsts, sizeof(*notinsts), addrcmp);
				notinstssaved = 1;
			}
		}
		else {
			perror(nfile);
		}
	}

/************************************************************************/
/*	process offsets specified by the user as not being the start	*/
/*	of valid instructions.						*/
/************************************************************************/
	if(nsfile) {
		FILE *nfp;
		static int notinstssaved = 0;

		nfp = fopen(nsfile, "r");

		if(nfp) {
			char nbuf[80];
			unsigned long ul;
			char *cp;
			m68kaddr *nottmp;

			while(fgets(nbuf, sizeof(nbuf), nfp)) {
				ul = strtoul(nbuf, &cp, 0);

				if(cp != nbuf && (odd || (ul & 1) == 0)) {
					if(ul >= initialpc && ul <= initialpc + maxoffset) {
						offset = ul - initialpc;
						nottmp = realloc(notinsts, (nnotinsts + 1) * sizeof(*notinsts));

						if(nottmp) {
							notinsts = nottmp;
							notinsts[nnotinsts++] = offset;
						}

						if(insts[offset].size) {

#ifndef NOBAD

							bad[offset] = offset + initialpc;

#endif

#ifdef DEBUG

							if(debug & DELETIONS) {
								fprintf(outfp, "6. Deleting offset %lx, size %d, flags = %d because specified\n", (long)offset, insts[offset].size, insts[offset].flags);
							}

#endif

							DELETE(offset);
						}
					}
					else {
						fprintf(outfp, "File %s: bad pc: %s\n", nsfile, nbuf);
					}
				}
				else {
					fprintf(outfp, "File %s: bad pc: %s\n", nsfile, nbuf);
				}

				if(verbose) {
					rotdash(1000);
				}
			}

			(void)fclose(nfp);

			if(!notinstssaved) {
				qsort(notinsts, nnotinsts, sizeof(*notinsts), addrcmp);
				notinstssaved = 1;
			}
		}
		else {
			perror(nsfile);
		}
	}

/************************************************************************/
/*	process offsets specified by the user as being instructions.	*/
/*	note that does *not* make an invalid instruction suddenly	*/
/*	valid.								*/
/************************************************************************/
	if(ifile) {
		FILE *ifp;

		ifp = fopen(ifile, "r");

		if(ifp) {
			char ibuf[80];
			unsigned long ul;
			char *cp;

			while(fgets(ibuf, sizeof(ibuf), ifp)) {
				ul = strtoul(ibuf, &cp, 0);

				if(cp != ibuf && (odd || (ul & 1) == 0)) {
					if(ul >= initialpc && ul <= initialpc + maxoffset) {
						makegood(ul - initialpc);
						readlabel(cp, ul - initialpc);
					}
					else {
						fprintf(outfp, "File %s: bad pc: %s\n", ifile, ibuf);
					}
				}
				else {
					fprintf(outfp, "File %s: bad pc: %s\n", ifile, ibuf);
				}

				if(verbose) {
					rotdash(1000);
				}
			}

			(void)fclose(ifp);
		}
		else {
			perror(ifile);
		}
	}

	if(verbose) {
		fprintf(stderr, "x\n");
	}

/************************************************************************/
/*	load global addresses and labels				*/
/************************************************************************/
	if(gfile) {
		FILE *gfp;

		gfp = fopen(gfile, "r");

		if(gfp) {
			char gbuf[80];
			long gl;
			char *cp,*cp1;

			while(fgets(gbuf, sizeof(gbuf), gfp)) {
				gl = strtol(gbuf, &cp, 0);

				if(cp != gbuf) {
					while(isspace(*cp)) {
						cp++;
					}

					cp1 = cp;

					while(*cp1 && !isspace(*cp1)) {
						cp1++;
					}

					*cp1 = '\0';
					addglobal(gl, cp);
				}
				else {
					fprintf(outfp, "File %s: bad pc: %s\n", gfile, gbuf);
				}
			}

			(void)fclose(gfp);
		}
		else {
			perror(gfile);
		}
	}

/************************************************************************/
/*	instructions that don't set pc must be followed by a valid	*/
/*	instruction.							*/
/*									*/
/*	instructions must reference valid instructions.			*/
/************************************************************************/
	findbadrefs();

	if(onepass == CONSISTENT) {
		printall();
		jumpfree();
	}

	if(verbose) {
		fprintf(stderr, "Finding bad References ............: [ ]\033[2D");
	}

/************************************************************************/
/*	assume that all link instructions that are referenced by bsr	*/
/*	or jsr instructions are valid.					*/
/************************************************************************/
	for(offset = 0; offset <= maxoffset; offset += (odd ? 1 : WORDSIZE)) {
		if(insts[offset].size && (insts[offset].flags & (ISBSR | ISJSR)) && (insts[offset].flags & 3) && insts[offset].required[0] >= initialpc && insts[offset].required[0] <= initialpc + maxoffset && (insts[insts[offset].required[0] - initialpc].flags & ISLINK)) {

#ifdef DEBUG

			if(debug & OVERLAPS) {
				fprintf(outfp, "Number of overlaps = %ld\n", overlaps(0));
			}

#endif

			makegood(insts[offset].required[0] - initialpc);
		}

		if(verbose) {
			rotdash(1000);
		}
	}

	findbadrefs();

#ifdef DEBUG

	if(debug & OVERLAPS) {
		fprintf(outfp, "Number of overlaps after LINKs with labels = %ld\n", overlaps(0));
	}

#endif

/************************************************************************/
/*	assume that all remaining link instructions are valid.		*/
/************************************************************************/
	total_changes = 0;

	for(offset = 0; offset <= maxoffset; offset += (odd ? 1 : WORDSIZE)) {
		if(insts[offset].size && (insts[offset].flags & ISGOOD) == 0 && (insts[offset].flags & ISLINK)) {

#ifdef DEBUG

			if(debug & OVERLAPS) {
				fprintf(stderr, "Number of overlaps = %ld\n", overlaps(0));
			}

#endif

			total_changes += makegood(offset);
		}

		if(verbose) {
			rotdash(1000);
		}
	}

	if(total_changes) {
		findbadrefs();
	}

#ifdef DEBUG

	if(debug & OVERLAPS) {
		fprintf(stderr, "Number of overlaps after all LINKs = %ld\n", overlaps(0));
	}

#endif

/************************************************************************/
/*	assume that branch instructions that jump to valid instructions	*/
/*	and that cannot be extension words of math fpu instructions	*/
/*	are valid.							*/
/************************************************************************/
	total_changes = 0;

	do {
		changes = 0;

		for(offset = 0; offset <= maxoffset; offset += (odd ? 1 : WORDSIZE)) {
			if(insts[offset].size && (insts[offset].flags & ISGOOD) == 0 && (insts[offset].flags & (ISBSR | ISJSR | ISBRA | ISBRcc | ISDBcc | ISJMP)) && (insts[offset].flags & 3) && insts[offset].required[0] >= initialpc && insts[offset].required[0] <= initialpc + maxoffset && (insts[insts[offset].required[0] - initialpc].flags & ISGOOD) && !fpuoverlap(offset)) {

#ifdef DEBUG

				if(debug & OVERLAPS) {
					fprintf(stderr, "Number of overlaps = %ld\n", overlaps(0));
				}

#endif

				makegood(offset);
				changes++;
			}
		}

		total_changes += changes;

		if(verbose) {
			rotdash(1000);
		}
	}
	while(changes);

	if(total_changes) {
		findbadrefs();
	}

#ifdef DEBUG

	if(debug & OVERLAPS) {
		fprintf(stderr, "Number of overlaps after all LINKs and jumps to good addresses = %ld\n", overlaps(0));
	}

#endif

/************************************************************************/
/*	assume that branch instructions that jump outside the current	*/
/*	object file and that cannot be extension words of math fpu	*/
/*	instructions are valid.						*/
/************************************************************************/
	total_changes = 0;

	do {
		changes = 0;

		for(offset = 0; offset <= maxoffset; offset += (odd ? 1 : WORDSIZE)) {
			if(insts[offset].size && (insts[offset].flags & ISGOOD) == 0 && (insts[offset].flags & (ISBSR | ISJSR | ISBRA | ISBRcc | ISDBcc | ISJMP | ISRTS)) && !fpuoverlap(offset)) {

#ifdef DEBUG

				if(debug & OVERLAPS) {
					fprintf(stderr, "Number of overlaps = %ld\n", overlaps(0));

#endif

				makegood(offset);
				changes++;
			}
		}

		total_changes += changes;

		if(verbose) {
			rotdash(1000);
		}
	}
	while(changes);

	if(total_changes) {
		findbadrefs();
	}

	if(verbose) {
		fprintf(stderr, "x\n");
	}

#ifdef DEBUG

	if(debug & OVERLAPS) {
		fprintf(stderr, "Number of overlaps after all LINKs and branches = %ld\n", overlaps(0));
	}

#endif

	if(fixoverlaps()) {
		findbadrefs();
	}

#if 1

	{
		long noverlaps = overlaps(1);

		if(noverlaps

#ifdef DEBUG

		|| debug & OVERLAPS

#endif

		)

			fprintf(stderr, "%s: Number of overlaps = %ld\n", sfile, noverlaps);
	}

#endif

/************************************************************************/
/*	get the labels.							*/
/************************************************************************/
	dcpass();

	if(verbose) {
		fprintf(stderr, "Getting Offsets ...................: [ ]\033[2D");
	}

	for(offset = 0; offset <= maxoffset; ) {
		if(insts[offset].size) {
			for(i = 0; i < NREQD(offset); i++) {
				SETLABEL(insts[offset].required[i]);
			}

			offset += insts[offset].size;
		}
		else if(odd) {
			offset++;
		}
		else {
			offset += WORDSIZE;
		}

		if(verbose) {
			rotdash(1000);
		}
	}

	if(verbose) {
		fprintf(stderr, "x\n");
	}

	if(verbose) {
		fprintf(stderr, "Getting Labels ....................: [ ]\033[2D");
	}

	for(nlabels = 0, offset = 0; offset <= maxoffset; offset++) {
		if(insts[offset].flags & ISLABEL) {

#ifdef DEBUG

			if(debug & LABELS) {
				fprintf(outfp, "Label %d at %lx\n", nlabels + 1, (long)offset);
			}

#endif

			insts[offset].labelnum = ++nlabels;
		}

		if(verbose) {
			rotdash(1000);
		}
	}

	if(verbose) {
		fprintf(stderr, "x\n");
	}

/************************************************************************/
/*	last pass: print!						*/
/************************************************************************/
	if(fseek(infp, 0, SEEK_SET) == -1) {
		perror("fseek");
		jumpfree();
	}

	pc = ppc = initialpc;
	leninstbuf = 0;
	curnbreak = 0;
	pass = LASTPASS;

/************************************************************************/
/*	check if we are in pilamode					*/
/************************************************************************/
	if(verbose) {
		fprintf(stderr, "Saving Source .....................: [ ]\033[2D");
	}

	if(pilamode) {
		fprintf(outfp, ";-----------------------------------------------------------------------\n");
		fprintf(outfp, ";-\tMotorola 68k Series Disassembler (modified for PalmOS computers)\n");
		fprintf(outfp, ";-\tLast modified version (rev %s) by Ali Akcaagac.\n", patchlevel);
		fprintf(outfp, ";-----------------------------------------------------------------------\n");
		fprintf(outfp, "\n\t\tAppl\t\"<AppName>\", '<AppID>'\n");
		fprintf(outfp, "\n\t\tinclude\t\"pilot3.inc\"\n\n");
		fprintf(outfp, ";-----------------------------------------------------------------------\n");
		fprintf(outfp, ";-\tCodeStart\n");
		fprintf(outfp, ";-----------------------------------------------------------------------\n");
	}

	for(stored = 0, curoffset = offset = 0; offset < (m68kaddr) gfsize; ) {
		if(curoffset != offset && fseek(infp, curoffset = offset, SEEK_SET) == -1) {
			perror("fseek");
			jumpfree();
		}

		flags = 0;

		if(insts[offset].size) {
			if(stored) {
				flush(stored, consts, insts[offset - stored].flags, 1);
				stored = 0;
			}

			pc = ppc = initialpc + offset;
			leninstbuf = 0;
			validinst();
			offset = curoffset;
		}
		else {
			int incrbreak;

			if(stored == 0) {
				pc = ppc = initialpc + offset;
			}

/************************************************************************/
/*	break the constant data up so that all labels are printed on a	*/
/*	new line.							*/
/************************************************************************/
			while(curnbreak < nbreaks && breaks[curnbreak] < offset) {
				curnbreak++;
			}

			incrbreak = curnbreak < nbreaks && offset == breaks[curnbreak];

			if(insts[offset].flags & ISLABEL || incrbreak) {
				flush(stored, consts, insts[offset - stored].flags, 1);
				stored = 0;

				if(incrbreak) {
					curnbreak++;
				}
			}

			if(fread(consts + stored, 1, 1, infp) == 1) {
				if(++stored >= sizeof consts) {
					stored = flush(stored, consts, insts[offset - stored].flags, 0);
					memmove(consts, consts + sizeof consts - stored, stored);
				}
			}

			offset++;
		}

		if(verbose) {
			rotdash(1000);
		}
	}

	if(stored) {
		flush(stored, consts, insts[offset - stored].flags, 1);
	}

/************************************************************************/
/*	check if we are in pilamode					*/
/************************************************************************/
	if(pilamode) {
		fprintf(outfp, "\n");
		fprintf(outfp, ";-----------------------------------------------------------------------\n");
		fprintf(outfp, ";-\tResourceStart\n");
		fprintf(outfp, ";-----------------------------------------------------------------------\n");
		fprintf(outfp, "\n\t\tres\t'<ResName>', $<ResAddr>, \"<ResFileName>\"\n\n");
		fprintf(outfp, ";-----------------------------------------------------------------------\n");
		fprintf(outfp, ";-\tMotorola 68k Series Disassembler (modified for PalmOS computers)\n");
		fprintf(outfp, ";-\tLast modified version (rev %s) by Ali Akcaagac.\n", patchlevel);
		fprintf(outfp, ";-----------------------------------------------------------------------\n");
	}

	if(verbose) {
		fprintf(stderr, "x\n");
	}

/************************************************************************/
/*	print label cross reference					*/
/************************************************************************/
	if(labelxref) {
		if(verbose) {
			fprintf(stderr, "Label Cross Reference .............: [ ]\033[2D");
		}

		fprintf(outfp, "\n-------------------------------------------------------------------------------");
		fprintf(outfp, "\nLabel Cross Reference");
		fprintf(outfp, "\n-------------------------------------------------------------------------------\n\n");

		fprintf(outfp, "%-*s Address  :  Used by (B=Branch, S=Call Subroutine, L=Load)\n", MAXLABELLEN, "Label");
		fprintf(outfp, "%-*s -------     ---------------------------------------------", MAXLABELLEN, "-----");

		for(curoffset = 0; curoffset <= maxoffset; ) {
			if(insts[curoffset].flags & ISLABEL || insts[curoffset].labelname) {
				fprintf(outfp, "\n%-*s %08lx :", MAXLABELLEN, labelname(curoffset), curoffset);
				leninstbuf = 0;

				for(offset = 0; offset <= maxoffset; ) {
					if(insts[offset].size) {
						if(insts[curoffset].labelnum && insts[curoffset].labelnum == insts[offset].ealabelnum) {
							if(leninstbuf >= 4) {
								fprintf(outfp, "\n           %*s", MAXLABELLEN, "");
								leninstbuf = 0;
							}

							fprintf(outfp, "  %08lx(%s)", offset, eatype[insts[offset].eatype]);
							leninstbuf++;
						}

						offset += insts[offset].size;
					}
					else if(odd) {
						offset++;
					}
					else {
						offset += WORDSIZE;
					}

					if(verbose) {
						rotdash(100000);
					}
				}
			}

			if(insts[curoffset].size) {
				curoffset += insts[curoffset].size;
			}
			else if(odd) {
				curoffset++;
			}
			else {
				curoffset += WORDSIZE;
			}
		}

		fprintf(outfp, "\n\n-------------------------------------------------------------------------------\n\n");

		if(verbose) {
			fprintf(stderr, "x\n");
		}
	}

/************************************************************************/
/*	print global cross reference					*/
/************************************************************************/
	if(globalxref && globals) {
		struct global *gptr;

		if(verbose) {
			fprintf(stderr, "Global Cross Reference ............: [ ]\033[2D");
		}

		fprintf(outfp, "\n-------------------------------------------------------------------------------");
		fprintf(outfp, "\nGlobal Cross Reference");
		fprintf(outfp, "\n-------------------------------------------------------------------------------\n\n");

		fprintf(outfp, "%-*s   Offset :  Used by (R=Read, W=Write, L=Load)\n", MAXLABELLEN, "Global");
		fprintf(outfp, "%-*s   ------    ---------------------------------", MAXLABELLEN, "------");

		for(gptr = globals; gptr; gptr = gptr->next) {
			fprintf(outfp, "\n%-*s %8ld :", MAXLABELLEN, globalname(gptr->offset), gptr->offset);
			leninstbuf = 0;

			for(offset = 0; offset < (m68kaddr) gptr->numused; offset++) {

				if(leninstbuf >= 4) {
					fprintf(outfp, "\n           %*s", MAXLABELLEN, "");
					leninstbuf = 0;
				}

				fprintf(outfp, "  %08lx(%s)", gptr->used[offset].addr, eatype[gptr->used[offset].eatype]);
				leninstbuf++;

				if(verbose) {
					rotdash(5);
				}
			}
		}

		fprintf(outfp, "\n\n-------------------------------------------------------------------------------\n\n");

		if(verbose) {
			fprintf(stderr, "x\n");
		}
	}

	if(verbose) {
		fprintf(stderr, "\n");
	}

	jumpfree();
}

/************************************************************************/
/*	rotating dash							*/
/************************************************************************/
void rotdash(int x)
{
	if(counter++ >= x) {
		if(dash == 0) {
			fprintf(stderr, "-\033[1D");
			dash++;
		}
		else if(dash == 1) {
			fprintf(stderr, "\\\033[1D");
			dash++;
		}
		else if(dash == 2) {
			fprintf(stderr, "|\033[1D");
			dash++;
		}
		else if(dash == 3) {
			fprintf(stderr, "/\033[1D");
			dash = 0;
		}

		counter = 0;
  	}
}
