package org.jayasoft.ivyde.eclipse.cpcontainer;


import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.net.URL;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.core.runtime.jobs.ISchedulingRule;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.jdt.core.IClasspathAttribute;
import org.eclipse.jdt.core.IClasspathContainer;
import org.eclipse.jdt.core.IClasspathEntry;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jdt.internal.core.ClasspathAttribute;
import org.eclipse.jdt.internal.core.ClasspathEntry;
import org.eclipse.jdt.internal.core.JavaModelManager;
import org.eclipse.swt.widgets.Display;
import org.jayasoft.ivyde.eclipse.IvyPlugin;

import fr.jayasoft.ivy.Artifact;
import fr.jayasoft.ivy.Ivy;
import fr.jayasoft.ivy.ModuleDescriptor;
import fr.jayasoft.ivy.ModuleId;
import fr.jayasoft.ivy.event.IvyEvent;
import fr.jayasoft.ivy.event.IvyListener;
import fr.jayasoft.ivy.event.download.EndArtifactDownloadEvent;
import fr.jayasoft.ivy.event.download.PrepareDownloadEvent;
import fr.jayasoft.ivy.event.download.StartArtifactDownloadEvent;
import fr.jayasoft.ivy.event.resolve.EndResolveDependencyEvent;
import fr.jayasoft.ivy.event.resolve.StartResolveDependencyEvent;
import fr.jayasoft.ivy.parser.ModuleDescriptorParserRegistry;
import fr.jayasoft.ivy.report.ResolveReport;
import fr.jayasoft.ivy.report.XmlReportOutputter;
import fr.jayasoft.ivy.repository.TransferEvent;
import fr.jayasoft.ivy.repository.TransferListener;
import fr.jayasoft.ivy.util.Message;
import fr.jayasoft.ivy.xml.XmlReportParser;

/**
 *   
 * 
 */
public class IvyClasspathContainer implements IClasspathContainer {

    private final class IvyResolveJob extends Job implements TransferListener, IvyListener {
        long _expectedTotalLength = 1;
        long _currentLength = 0;

        private IProgressMonitor _monitor;
        private IProgressMonitor _dlmonitor;
        private Ivy _ivy;
        private boolean _usePreviousResolveIfExist;
        private int _workPerArtifact = 100;
        private boolean _notify;

        public IvyResolveJob(Ivy ivy, boolean usePreviousResolveIfExist, boolean notify) {
        	super("Resolve "+_javaProject.getProject().getName()+"/"+_ivyXmlPath+" dependencies");
        	_ivy = ivy;
        	_usePreviousResolveIfExist = usePreviousResolveIfExist;
        	_notify = notify;
        }

        public void transferProgress(TransferEvent evt) {
        	switch (evt.getEventType()) {
        	case TransferEvent.TRANSFER_INITIATED:
        		_monitor.setTaskName("downloading "+evt.getResource());
        		break;
        	case TransferEvent.TRANSFER_STARTED:
        		_currentLength = 0;
        		if (evt.isTotalLengthSet()) {
        			_expectedTotalLength = evt.getTotalLength();
        			_dlmonitor.beginTask("downloading "+evt.getResource(), 100);
        		}
        		break;
        	case TransferEvent.TRANSFER_PROGRESS:
        		if (_expectedTotalLength > 1) {
        			_currentLength += evt.getLength();
        			_dlmonitor.worked((int)(_currentLength * 100 / _expectedTotalLength));
        			_monitor.subTask((_currentLength/1024)+" / "+(_expectedTotalLength/1024)+"kB");
        		}
        		break;
        	default:
        	}
        }

        public void progress(IvyEvent event) {
        	if (event instanceof TransferEvent) {
        		if (_dlmonitor != null) {
        			transferProgress((TransferEvent)event);
        		}
        	} else if (event instanceof PrepareDownloadEvent) {
        		PrepareDownloadEvent pde = (PrepareDownloadEvent)event;                    
        		Artifact[] artifacts = pde.getArtifacts();
        		if (artifacts.length > 0) {
        			_workPerArtifact = 1000 / artifacts.length;
        		}
        	} else if (event instanceof StartArtifactDownloadEvent) {
        		StartArtifactDownloadEvent evt = (StartArtifactDownloadEvent)event;
        		_monitor.setTaskName("downloading "+evt.getArtifact());
        		if (_dlmonitor != null) {
        			_dlmonitor.done();
        		}
        		_dlmonitor = new SubProgressMonitor(_monitor, _workPerArtifact);
        	} else if (event instanceof EndArtifactDownloadEvent) {
        		if(_dlmonitor != null) {
        			_dlmonitor.done();
        		}
        		_monitor.subTask(" ");
        		_dlmonitor = null;
        	} else if (event instanceof StartResolveDependencyEvent) {
        		StartResolveDependencyEvent ev = (StartResolveDependencyEvent) event;
        		_monitor.subTask("resolving "+ev.getDependencyDescriptor().getDependencyRevisionId());
        	} else if (event instanceof EndResolveDependencyEvent) {
        		_monitor.subTask(" ");
        	}
        }

        protected IStatus run(IProgressMonitor monitor) {
            Message.info("resolving dependencies of "+_ivyXmlFile); 
        	_monitor = monitor;
        	final IStatus[] status = new IStatus[1];
        	final File[][] jarFiles = new File[1][];

        	Thread resolver = new Thread() {
        		public void run() {
        			_ivy.addIvyListener(IvyResolveJob.this);

        			_monitor.beginTask("resolving dependencies", 1000);
					_monitor.setTaskName("resolving dependencies...");
        			//context Classloader hook for commonlogging used by httpclient  
        			ClassLoader old = Thread.currentThread().getContextClassLoader();
        			List problemMessages = Collections.EMPTY_LIST;
        			ModuleDescriptor md = null;
        			try {
        				Thread.currentThread().setContextClassLoader(IvyClasspathContainer.class.getClassLoader());
        				final URL ivyURL = _ivyXmlFile.toURL();
        				String[] confs;
        				boolean resolved = false;
        				try {

        					if (_usePreviousResolveIfExist) {
        						md = ModuleDescriptorParserRegistry.getInstance().parseDescriptor(_ivy, ivyURL, false);
        						if (_confs.length == 1 && "*".equals(_confs[0])) {
        							confs = md.getConfigurationsNames();
        						} else {
        							confs = _confs;
        						}

        						// we check if all required configurations have been resolved
        						for (int i = 0; i < confs.length; i++) {
        							File report = new File(_ivy.getDefaultCache(), XmlReportOutputter.getReportFileName(md.getModuleRevisionId().getModuleId(), confs[i]));
        							if (!report.exists()) {
        								// no resolve previously done for at least one conf... we do it now
        								Message.info("\n\nIVY DE: previous resolve of " + md.getModuleRevisionId().getModuleId() + " doesn't contain enough data: resolving again\n");
        								ResolveReport r = _ivy.resolve(ivyURL, null, _confs, _ivy.getDefaultCache(), null, true);
        								resolved = true;
        								confs = r.getConfigurations();
                						//eventually do a retrieve
                						if(IvyPlugin.shouldDoRetrieve(_javaProject)) {
                							_monitor.setTaskName("retrieving dependencies in "+IvyPlugin.getFullRetrievePatternHerited(_javaProject));
                							_ivy.retrieve(md.getModuleRevisionId().getModuleId(), confs, _ivy.getDefaultCache(), IvyPlugin.getFullRetrievePatternHerited(_javaProject));
                						}
        								break;
        							}
        						}
        					} else {
        						Message.info("\n\nIVYDE: calling resolve on " + ivyURL + "\n");
        						ResolveReport report = _ivy.resolve(ivyURL, null, _confs, _ivy.getDefaultCache(), null, true);
        						problemMessages = report.getAllProblemMessages();
        						confs = report.getConfigurations();
        						md = report.getModuleDescriptor();
        						resolved = true;

        						if (_monitor.isCanceled()) {
        							status[0] = Status.CANCEL_STATUS;
        							return;
        						}
        						//eventually do a retrieve
        						if(IvyPlugin.shouldDoRetrieve(_javaProject)) {
        							_monitor.setTaskName("retrieving dependencies in "+IvyPlugin.getFullRetrievePatternHerited(_javaProject));
        							_ivy.retrieve(md.getModuleRevisionId().getModuleId(), confs, _ivy.getDefaultCache(), IvyPlugin.getFullRetrievePatternHerited(_javaProject));
        						}
        					}
        				} catch (FileNotFoundException e) {
        					status[0] = new Status(Status.ERROR, IvyPlugin.ID, Status.ERROR, "ivy file not found: "+_ivyXmlFile+"\nPlease configure your IvyDE ClasspathContainer properly", e);
        					return;
        				} catch (ParseException e) {
        					status[0] = new Status(Status.ERROR, IvyPlugin.ID, Status.ERROR, "parse exception in: "+_ivyXmlFile+"\n"+e.getMessage(), e);
        					return;
        				} finally {
        					Thread.currentThread().setContextClassLoader(old);
        				}
        				ModuleId mid = md.getModuleRevisionId().getModuleId();

        				try {
        					if (!resolved) {
        						Message.info("\n\nIVY DE: using cached data of previous resolve of "+md.getModuleRevisionId().getModuleId()+"\n");
        					}
        					jarFiles[0] = parseResolvedConfs(confs, mid);
        				} catch (Exception ex) {
        					if (!resolved) {
        						//maybe this is a problem with the cache, we retry with an actual resolve
        						Message.info("\n\nIVYDE: tryed to build classpath from cache, but files seemed to be corrupted... trying with an actual resolve");
        						ResolveReport report = _ivy.resolve(ivyURL, null, _confs, _ivy.getDefaultCache(), null, true);
        						jarFiles[0] = parseResolvedConfs(report.getConfigurations(), mid);
        					}
        				}
        			} catch (Exception e) {
        				status[0] = new Status(Status.ERROR, IvyPlugin.ID, Status.ERROR, "An internal error occured while resolving dependencies of "+_ivyXmlFile+"\nPlease see eclipse error log and IvyConsole for details", e);
        				return;
        			} finally {
        				_monitor.done();
            			_ivy.removeIvyListener(IvyResolveJob.this);
        			}
        			
	    			if (!problemMessages.isEmpty()) {
	    				StringBuffer problems = new StringBuffer();
	    				for (Iterator iter = problemMessages.iterator(); iter.hasNext();) {
							String msg = (String) iter.next();
							problems.append(msg).append("\n");
						}
	    				status[0] = new Status(Status.ERROR, IvyPlugin.ID, Status.ERROR, "Impossible to resolve dependencies of "+md.getModuleRevisionId()+":\n"+problems+"\nSee IvyConsole for further details", null);
	    				return;
	    			} else {
	    				status[0] = Status.OK_STATUS;
	    				return;
	    			}
        		}
        	};
        	
        	try {
        		resolver.start();
        		while (true) {
        			try {
        				resolver.join(100);
        			} catch (InterruptedException e) {
        				_ivy.interrupt(resolver);
        				return Status.CANCEL_STATUS;
        			}
        			synchronized (status) { // ensure proper sharing of done var
        				if (status[0] != null || !resolver.isAlive()) {
        					break;
        				}
        			}
        			if (_monitor.isCanceled()) {
        				_ivy.interrupt(resolver);
        				return Status.CANCEL_STATUS;
        			}
        		}
        		if (status[0] == Status.OK_STATUS) {
	    			updateClasspathEntries(_usePreviousResolveIfExist, _notify, jarFiles[0]);
        		}
        		return status[0];
        	} finally {
        		synchronized (IvyClasspathContainer.this) {
        			_job = null;
				}
                IvyPlugin.log(IStatus.INFO, "resolved dependencies of "+_ivyXmlFile, null); 
        	}
        	

        }

		private File[] parseResolvedConfs(String[] confs, ModuleId mid) throws ParseException, IOException {
            File[] jarFiles;
            XmlReportParser parser = new XmlReportParser();
            Collection all = new LinkedHashSet();
            for (int i = 0; i < confs.length; i++) {
                Artifact[] artifacts = parser.getArtifacts(mid, confs[i], _ivy.getDefaultCache());
                all.addAll(Arrays.asList(artifacts));
            }
            Collection files = new LinkedHashSet();
            for (Iterator iter = all.iterator(); iter.hasNext();) {
                Artifact artifact = (Artifact)iter.next();
                if (IvyPlugin.accept(_javaProject, artifact)) {
                	files.add(_ivy.getArchiveFileInCache(_ivy.getDefaultCache(), artifact));
                }
            }
             jarFiles = (File[])files.toArray(new File[files.size()]);
            return jarFiles;
        }

    }

    public static final String IVY_CLASSPATH_CONTAINER_ID = "org.jayasoft.ivyde.eclipse.cpcontainer.IVYDE_CONTAINER";
    
	private IClasspathEntry[] _classpathEntries;
	private IJavaProject _javaProject;
	
	
    private File _ivyXmlFile;
    private String _ivyXmlPath;
	private String[] _confs = new String[] { "default" };

    private IPath _path;
    
	private IvyResolveJob _job;


	public IvyClasspathContainer(IJavaProject javaProject, IPath path, String ivyFile, String[] confs ) {
		_javaProject = javaProject;
        _path = path;
		
        _ivyXmlPath = ivyFile;
		_ivyXmlFile = resolveFile( ivyFile );
        _confs = confs;   
        //do  execute this job in current thread 
        computeClasspathEntries(true, false).run(new NullProgressMonitor());
        if (_classpathEntries == null) {
            _classpathEntries = new IClasspathEntry[0];
        }
        IvyPlugin.getDefault().register(this);
	}
	
    private File resolveFile( String path ) {
		IFile iFile = _javaProject.getProject().getFile( path );
		return new File( iFile.getLocation().toOSString() );
	}
	
	
	/* (non-Javadoc)
	 * @see org.eclipse.jdt.core.IClasspathContainer#getDescription()
	 */
	public String getDescription() {
		return _ivyXmlPath+" "+Arrays.asList(_confs);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jdt.core.IClasspathContainer#getKind()
	 */
	public int getKind() {
		return K_APPLICATION;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jdt.core.IClasspathContainer#getPath()
	 */
	public IPath getPath() {
		return _path;
	}
	
	
    /* (non-Javadoc)
     * @see org.eclipse.jdt.core.IClasspathContainer#getClasspathEntries()
     */
    public IClasspathEntry[] getClasspathEntries() {
        return _classpathEntries;
    }

	private final static ISchedulingRule RESOLVE_EVENT_RULE = new ISchedulingRule() {
		public boolean contains(ISchedulingRule rule) {
			return rule == this;
		}

		public boolean isConflicting(ISchedulingRule rule) {
			return rule == this;
		}
	};
    /* (non-Javadoc)
     * @see org.eclipse.jdt.core.IClasspathContainer#getClasspathEntries()
     */
    private IvyResolveJob computeClasspathEntries(final boolean usePreviousResolveIfExist, boolean notify) {
        try {
        	Ivy ivy = IvyPlugin.getIvy(_javaProject);

        	// resolve job already running
        	synchronized (this) {
	        	if (_job != null) {
	        		return _job;
	        	}
	        	_job = new IvyResolveJob(ivy, usePreviousResolveIfExist, notify);
	        	_job.setUser(true);
	        	_job.setRule(RESOLVE_EVENT_RULE);
	        	return _job;
        	}
        } catch (Exception e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
            return null;
        }        
    }

    public void resolve() {
        computeClasspathEntries(false, true).schedule();
    }
    public void refresh() {
        computeClasspathEntries(true, true).schedule();
    }


    private void updateClasspathEntries(final boolean usePreviousResolveIfExist, boolean notify, final File[] jarFiles) {
    	if(jarFiles != null) {
	        IClasspathEntry[] entries = new IClasspathEntry[ jarFiles.length ];
	        
	        for( int i=0; i<jarFiles.length; i++ ) {
	            Path path = new Path( jarFiles[ i ].getAbsolutePath() );
	            entries[ i ] = JavaCore.newLibraryEntry(path, 
	                                                    IvyPlugin.getDefault().getPackageFragmentExtraInfo().getSourceAttachment(path), 
	                                                    IvyPlugin.getDefault().getPackageFragmentExtraInfo().getSourceAttachmentRoot(path),
	                                                    ClasspathEntry.NO_ACCESS_RULES,
	                                                    getExtraAttribute(path),
	                                                    false);
	        }
	        setClasspathEntries(entries);
    	} else {
            setClasspathEntries(new IClasspathEntry[0]);
    	}
    	if (notify) {
    		notifyUpdateClasspathEntries();
    	}
    }

    private IClasspathAttribute[] getExtraAttribute(Path path) {
        List result  = new ArrayList();
        IPath p = IvyPlugin.getDefault().getPackageFragmentExtraInfo().getDocAttachment(path);
        if(p != null) {
            result.add(new ClasspathAttribute(IClasspathAttribute.JAVADOC_LOCATION_ATTRIBUTE_NAME, p.toPortableString()));
        }
        return (IClasspathAttribute[]) result.toArray(new IClasspathAttribute[result.size()]);
    }



    private void setClasspathEntries(final IClasspathEntry[] entries) {
        Display.getDefault().syncExec(new Runnable() {
            public void run() {
                _classpathEntries = entries;
            }
        });
    }

    private void notifyUpdateClasspathEntries() {
        Display.getDefault().asyncExec(new Runnable() {
            public void run() {
                try {
                    JavaModelManager manager = JavaModelManager.getJavaModelManager();
                    manager.containerPut(_javaProject, _path, null);
                    JavaCore.setClasspathContainer(
                            _path,
                            new IJavaProject[] {_javaProject},
                            new IClasspathContainer[] {IvyClasspathContainer.this},
                            null);
                } catch (JavaModelException e) {
                    // TODO Auto-generated catch block
                    e.printStackTrace();
                }                    
            }
        });
    }

    public static String[] getConfigurations(IPath containerPath) {
        return containerPath.lastSegment().split(",");
    }

    public static String getConfigurationsText(IPath containerPath) {
        return containerPath.lastSegment();
    }


    public static String getIvyFilePath(IPath containerPath) {
        return ((IPath)containerPath.clone()).removeFirstSegments(1).removeLastSegments(1).toString();
    }
    
    
    public static boolean isIvyClasspathContainer(IPath containerPath) {
        return containerPath.segmentCount() >= 3 && IvyClasspathContainer.IVY_CLASSPATH_CONTAINER_ID.equals(containerPath.segment(0)) ;
    }

    /**
     * Resolves the classpath container corresponding to the given ivy file, if any.
     * @param file
     */
    public static void resolveIfNeeded(IFile file) {
        IJavaProject javaProject = JavaCore.create(file.getProject());
        try {
            IClasspathEntry[] entries = javaProject.getRawClasspath();
            for (int i= 0; i < entries.length; i++) {
                IClasspathEntry entry= entries[i];
                if (entry != null && entry.getEntryKind() == IClasspathEntry.CPE_CONTAINER) { 
                    IPath path = entry.getPath();
                    if (isIvyClasspathContainer(path) && file.getProjectRelativePath().toString().equals(getIvyFilePath(path))) {
                        IClasspathContainer cp = JavaCore.getClasspathContainer(path, javaProject);
                        
                        if (cp instanceof IvyClasspathContainer) {
                            IvyClasspathContainer c = (IvyClasspathContainer)cp;
                            c.resolve();
                        }
                    }
                }
            }
        } catch (JavaModelException e) {
            e.printStackTrace();
        }
    }

    public static void resolve(IJavaProject javaProject) {
        try {
            IClasspathEntry[] entries = javaProject.getRawClasspath();
            for (int i= 0; i < entries.length; i++) {
                IClasspathEntry entry= entries[i];
                if (entry != null && entry.getEntryKind() == IClasspathEntry.CPE_CONTAINER) { 
                    IPath path = entry.getPath();
                    if (isIvyClasspathContainer(path)) {
                        IClasspathContainer cp = JavaCore.getClasspathContainer(path, javaProject);
                        
                        if (cp instanceof IvyClasspathContainer) {
                            IvyClasspathContainer c = (IvyClasspathContainer)cp;
                            c.resolve();
                        }
                    }
                }
            }
        } catch (JavaModelException e) {
            e.printStackTrace();
        }
    }

    public IFile getIvyFile() {
        return _javaProject.getProject().getFile(_ivyXmlPath);
    }

    public URL getReportUrl() {
    	try {
    		Ivy ivy = IvyPlugin.getIvy(_javaProject);
    		URL ivyURL = _ivyXmlFile.toURL();
    		ModuleDescriptor md = ModuleDescriptorParserRegistry.getInstance().parseDescriptor(ivy, ivyURL, false);
    		
    		return new File(ivy.getDefaultCache(), XmlReportOutputter.getReportFileName(md.getModuleRevisionId().getModuleId(), md.getConfigurationsNames()[0])).toURL();
    	} catch (Exception ex) {
    		return null;
    	}
    }

	public IJavaProject getProject() {
		return _javaProject;
	}
}