/*	$Id: poppass.c,v 1.5 2007-08-03 21:16:24 hngott Exp $ */

/*
 * Copyright (C) 2005-2007 Thomas Birnthaler
 *                         Hermann Gottschalk
 *                         <openpoppassd@ostc.de>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include <sys/param.h>
#include <errno.h>
#include <poll.h>
#include <pwd.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "poppassd.h"

#define MASTER 0
#define SLAVE  1

volatile sig_atomic_t poppass_quit = 0;

/* Prototypes */
static void poppass_sighdlr (int);
static int  setup_listeners (struct servent *se);

pid_t
fork_slave(int pipe_prnt[2])
{
	pid_t               pid;
	struct passwd*      pw;
	struct sockaddr_in  client_info;
	struct servent*     se;
	int                 server_socket;
	int                 talk_socket;
	int                 length;

	switch (pid = fork()) {
		case -1:
			fatal("cannot fork");
		case 0:
			log_debug("forked (slave)");
			break;
		default:
			log_debug("forked (master)");
			return (pid);
	}

	/* Chroot and bind unprivileged slave process */
	if ((se = getservbyname("poppassd", "tcp")) == NULL) {
		/*
		           man 3 getservent
		           struct  servent {
		           char    *s_name;        official name of service
		           char    **s_aliases;    alias list
		           int     s_port;         port service resides at
		           char    *s_proto;       protocol to use
		           };
		*/
		se = calloc(1, sizeof(struct servent));
		se->s_name    = strdup("poppassd");
		se->s_aliases = calloc(1, sizeof(char *));
		se->s_port    = ntohs(106);
		se->s_proto   = strdup("tcp");
		log_warn("poppassd not found in /etc/services - using port 106/tcp");
	};

	if ((pw = getpwnam(POPPASSD_USER)) == NULL)
		fatal("missing user account");

	if (chroot(pw->pw_dir) == -1)
		fatal("chroot");
	if (chdir("/") == -1)
		fatal("chdir(\"/\")");

	setproctitle("[slave]");

	/* Open TCP-socket */
	server_socket = setup_listeners(se);

	/* Set group access list and (effective) group/user id to "_poppass" */
	if (setgroups(1, &pw->pw_gid) ||
	    setegid(pw->pw_gid) || setgid(pw->pw_gid) ||
	    seteuid(pw->pw_uid) || setuid(pw->pw_uid))
		fatal("can't drop privileges");

	/* necessary ??? */
	endpwent();
	endservent();

	/* Catch some signals */
	signal(SIGTERM, poppass_sighdlr);
	signal(SIGINT,  poppass_sighdlr);
	signal(SIGPIPE, SIG_IGN);
	signal(SIGHUP,  SIG_IGN);

	close(pipe_prnt[MASTER]);

	log_debug("openpoppassd slave ready");

	/* loop: wait for connection, talk to client, close connection */
	while (poppass_quit == 0) {

		/* Accept connection to client */
		talk_socket = accept(server_socket,
			(struct sockaddr *)&client_info,
			&length);

		client_talk(pipe_prnt[SLAVE], talk_socket);

		log_debug("trying to close");
		close(talk_socket);
		log_debug("closed");
	}

	log_debug("openpoppassd slave terminating");
	_exit(0);
}

/* Binding openpoppassd to port */
static int
setup_listeners(struct servent *se)
{
	int                fd;
	struct sockaddr_in server_sock;

	log_debug("listening on localhost");

	if ((fd = socket(AF_INET, SOCK_STREAM, 0)) == -1)
		fatal("socket");

	server_sock.sin_family      = AF_INET;
	server_sock.sin_addr.s_addr = inet_addr(LISTEN_IP);
	server_sock.sin_port        = (short)se->s_port;
	log_debug("openpoppassd listening on: %s:%hd",
		LISTEN_IP,
		htons((short)se->s_port));

	if (bind(fd, (struct sockaddr *)&server_sock, sizeof(server_sock)) == -1)
		fatal("bind");
	// Not working: SA_LEN((struct sockaddr*)&server_sock)) == -1)

	listen(fd, 15);

	return (fd);
}

/* Some signals prepare slave to quit */
static void
poppass_sighdlr(int sig)
{
	switch (sig) {
		case SIGINT:
		case SIGTERM:
			exit(100);
			break;
	}
}
