/*	$Id: openpoppassd.c,v 1.5 2007-08-03 21:16:24 hngott Exp $ */

/*
 * Copyright (C) 2005-2007 Thomas Birnthaler
 *                         Hermann Gottschalk
 *                         <openpoppassd@ostc.de>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/wait.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <errno.h>
#include <fcntl.h>
#include <poll.h>
#include <pwd.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "poppassd.h"

#define MASTER 0
#define SLAVE  1

static void sighdlr     (int sig);
static void usage       (void);
static int  check_child (pid_t, const char *);

volatile sig_atomic_t quit    = 0;
volatile sig_atomic_t sigchld = 0;

int   debug = 0;

int
main(int argc, char *argv[])
{
	pid_t chld_pid = 0, pid;
	int   ch;
	int   pipe_chld[2];
	int   cnt;
	char  msg;
	char  buffer[BUFSIZE];
	char  errmsg[BUFSIZE];

	log_init(1);		/* log to stderr until daemonized */

	while ((ch = getopt(argc, argv, "d")) != -1) {
		switch (ch) {
		case 'd':
			debug = 1;
			break;
		default:
			usage();
			/* NOTREACHED */
		}
	}

	/* Am I not root (id=0)? -> break */
	if (geteuid()) {
		fprintf(stderr, "openpoppassd: need root privileges\n");
		exit(1);
	}

	if (getpwnam(POPPASSD_USER) == NULL) {
		fprintf(stderr, "openpoppassd: unknown user %s\n", POPPASSD_USER);
		exit(1);
	}
	endpwent(); /* necessary??? */

	log_init(debug);

	if (!debug)
		daemon(1, 0);

	if (socketpair(AF_UNIX, SOCK_STREAM, PF_UNSPEC, pipe_chld) == -1)
		fatal("socketpair");

	/* fork slave */
	setpgid(getpid(), getpid());
	chld_pid = fork_slave(pipe_chld);
	setpgid(chld_pid, getpid());

	setproctitle("[master]");

	/* Catch some signals*/
	signal(SIGTERM, sighdlr);
	signal(SIGINT,  sighdlr);
	signal(SIGCHLD, sighdlr);

	close(pipe_chld[SLAVE]);

	/* loop: receive and process messages, send answers */
	while (quit == 0) {
		errmsg[0] = '\0';
		if ((cnt = read(pipe_chld[MASTER], buffer, sizeof(buffer))) > 0) {
			msg = recv_change_pw(cnt, buffer, errmsg);	
			if (msg == MSG_PASSWD_ERROR)
			{
				log_debug("msg: %d %s", msg, errmsg);
				buffer[0] = msg;
				strncpy(buffer + 1, errmsg, BUFSIZE - 1);
				write(pipe_chld[MASTER], buffer, strlen(errmsg) + 1);
			}
			else
			{
				log_debug("msg: %d", msg);
				write(pipe_chld[MASTER], &msg, 1);
			}
		}

		if (sigchld) {
			if (!(chld_pid = check_child(chld_pid, "slave")))
				quit = 1;
			sigchld = 0;
		}
	}

	/* never talk to child again (e.g. if doing something wrong) */
	signal(SIGCHLD, SIG_IGN);

	/* kill child if CHPID != 0 */
	if (chld_pid)
		kill(chld_pid, SIGTERM);

	/* wait for any child until all dead */
	do {
		if ((pid = wait(NULL)) == -1 &&
		    errno != EINTR && errno != ECHILD)
			fatal("wait");
	} while (pid != -1 || (pid == -1 && errno == EINTR)); /* why ??? */

	log_warnx("openpoppassd master terminating");
	exit(0);
}

void
usage(void)
{
	extern char *__progname;

	fprintf(stderr, "usage: %s [-d]\n", __progname);
	exit(1);
}

/* If signal SIGCHLD arrived, this function is called to check
   whether child exists anymore. Returns:
     1 = child finished
     0 = child exists
*/
int
check_child(pid_t pid, const char *pname)
{
	int status;

	/* pid = stopped or terminated child -> its PID (again) */
	if (waitpid(pid, &status, WNOHANG) > 0) {
		if (WIFEXITED(status)) {
			log_warnx("lost child: %s exited", pname);
			return (0);
		}
		if (WIFSIGNALED(status)) {
			log_warnx("lost child: %s terminated; signal %d",
			    pname, WTERMSIG(status));
			return (0);
		}
	}

	return (pid);
}

enum msg_type
change_pw(char* user, char* oldpass, char* newpass, char* errmsg)
{
	struct passwd* pw;
	enum err_type  ret;

	log_debug("change_pw: '%s'", user);

	if ((pw = getpwnam(user)) == NULL)
	{
		return MSG_AUTH_WRONG;
	}

	if (strlen(newpass) < 5)
	{
		return MSG_PWD_TOO_SIMPLE;
	}

	if (strcmp(crypt(oldpass, pw->pw_passwd), pw->pw_passwd) != 0)
	{
		return MSG_AUTH_WRONG;
	}

	// TODO: Passwort wirklich ndern !!!
	if ((ret = change_passwd(user, oldpass, newpass, errmsg)) != 0)
	{
		switch (ret)
		{
			case ERR_UNKNOWN_USER:
				log_debug("unknown user, '%s'.", user);
				break;
			case ERR_NO_PTY_FOUND:
				log_debug("can't find pty");
				break;
			case ERR_CANT_PASSWD_FORK:
				log_debug("can't fork for passwd");
				break;
			case ERR_FAILED_ATTEMPT:
				log_debug("failed attempt by '%s' (%s)", user, errmsg);
				break;
			case ERR_WAIT_FAILED:
				log_debug("wait for /usr/bin/passwd child failed");
				break;
			case ERR_WRONG_CHILD:
				log_debug("wrong child (/usr/bin/passwd) waited for!");
				break;
			case ERR_KILLED_CHILD:
				log_debug("child (/usr/bin/passwd) killed?");
				break;
			case ERR_CHILD_EXITED:
				log_debug("child (/usr/bin/passwd) exited abnormally");
				break;
			default:
				log_debug("unknown /usr/bin/passwd error");
				break;
		}
		log_debug("password for '%s' couldn't be changed", user);
		return MSG_PASSWD_ERROR;
	}

	return MSG_CHG_OK;
}

static void
sighdlr(int sig)
{
	switch (sig) {
		case SIGTERM:
		case SIGINT:
			killpg(0, sig);
			exit(100);
			break;
		case SIGCHLD:
			sigchld = 1;
			break;
	}
}
