;;;
;;; GC Monitor
;;;
(import "java.awt.FlowLayout")
(import "java.awt.BorderLayout")
(import "java.awt.event.WindowEvent")
(import "java.lang.Double")
(import "java.lang.Runtime")
(import "java.lang.System")
(import "javax.swing.JFrame")
(import "javax.swing.JLabel")
(import "javax.swing.JProgressBar")
(import "javax.swing.Timer")

;;; Convert total bytes into one decimal place megabytes.
(define (mb_1 total) (/ (inexact->exact (* total 1.0e-5)) 10.0))

(define (f/ a b) (/ a (exact->inexact b)))

(define (millis) (System.currentTimeMillis))

(define gc-status
  (let* ((r (Runtime.getRuntime))
	 (interval 5000)		; 5 sec.
	 (start-time (System.currentTimeMillis))
	 (start-free (.freeMemory r))
	 (rate 0.0))
    (lambda (pb label)
      (let* ((total (.totalMemory r))
	     (free (.freeMemory r))
	     (used (- total free))
	     (now (System.currentTimeMillis))
	     (time (- now start-time)))
	(if (>  time interval)
	    (let ((new-rate 
		   (/ (inexact->exact (* (f/ (- start-free free) time) 10))
		      10.0)))
	      (set! rate (if (> new-rate 0.0) new-rate rate))
	      (set! start-time now)
	      (set! start-free free)))
	(.setValue pb (inexact->exact (* (f/ used total) 100)))
	(.setText label (string-append (mb_1 used) "/" (mb_1 total) " "
				       rate))))))

(define (borderPanel c n s e w)
  (define (maybe-add p x how) (if x (add p x how)))
  (let ((panel (JPanel. (BorderLayout.))))
    (maybe-add panel c BorderLayout.CENTER$)
    (maybe-add panel n BorderLayout.NORTH$)
    (maybe-add panel s BorderLayout.SOUTH$)
    (maybe-add panel e BorderLayout.EAST$)
    (maybe-add panel w BorderLayout.WEST$)
    panel))

(define (frame name panel window-listener)
  (let ((f (JFrame. name)))
    (.setContentPane f panel)
    (if (not (eq? window-listener #null)) (.addWindowListener window-listener))
    (.pack f)
    (.setVisible f #t)
    f))

(define (GCMonitor interval)
  ;; Create the GC Monitor.  Interval is sampling interval in millisec.
  (let* ((f (JFrame. "GCMonitor"))
	 (p (.getContentPane f))
	 (pb (JProgressBar.))
	 (label (JLabel. "000.0/000.0 000.0"))
	 (timer (javax.swing.Timer. interval 
			(Listener. (lambda (e) (gc-status pb label))))))
    (.setLayout p (BorderLayout.))
    (.setToolTipText pb "% of memory used")
    (.setStringPainted pb #t)
    (.add p pb BorderLayout.WEST$)
    (.setToolTipText label "Used MB / Total MB KB/sec")
    (.add p label BorderLayout.EAST$)
    (.addWindowListener 
     f (Listener.
	(lambda (e)
	  (if (= (.getID e) WindowEvent.WINDOW_CLOSING$)
	      (begin
		(.dispose f)
		(.stop timer)
		)))))
    (.pack f)
    (.start timer)
    (.setVisible f #t)
    f))

;;; Start GCMonitor sampling every 0.5 sec.
(define f (GCMonitor 500))

(define (GCLog file interval max)
  ;; Log GC information to a file.
  (let ((start (System.currentTimeMillis))
	(stream (java.io.PrintWriter. (java.io.FileOutputStream. file)))
	(r (Runtime.getRuntime)))
    (letrec ((gcLogger (let ((i 0))
			 (lambda (e)
			   (set! i (+ i 1))
			   (if (<= i max)
			       (begin
				 '(print
				  (string-append
				   "gc," (- (System.currentTimeMillis) start) ","
				   (.freeMemory r) ","
				   (.totalMemory r)))
				 (.println
				  stream
				  (string-append
				   (- (System.currentTimeMillis) start) ","
				   (.freeMemory r) ","
				   (.totalMemory r))))
			       (begin (.close stream)
				      (print (list 'close stream))
				      (.stop timer))))))
	     (timer (javax.swing.Timer. interval (Listener. gcLogger))))
      (.start timer)
      stream)))

'(define logStream (GCLog "d:/temp/gc.log" 500 1000))

(define (topThreadGroup)
  (define (topThreadGroup tg)
    (let ((it (.getParent tg)))
      (if (eq? it #null) tg (topThreadGroup it))))
  (topThreadGroup (.getThreadGroup (Thread.currentThread))))

(define (threads)
  (let* ((top (topThreadGroup))
	 (ts (make-array Thread.class (.activeCount top))))
    (.enumerate top ts)
    ts))
