;;;
;;; Jscheme tests
;;;

;;; This is the current test suite for Jscheme.  The original version is
;;; in demo/java10/Test.scm

;;; These tests roughly in the order they were used for debugging.
;;; So, it is fairly good history of Jscheme bugs.  The original version
;;; we got from Peter Norvig was 2500 lines of compiled code, but he
;;; couldn't test it.  So, the first test, typing 3 to the REPL, failed.


;;; (run-test debug?) will run the tests.  Printing all tests if debug is #t.
;;; (define-test) will define a test.
;;; (define-tests) defines tests in bulk:
;;; A test is either 
;;; - an expression, which is (eval)'ed
;;; - `(,eq ,expression ,expect) which is an expression and its expected 
;;;   value.  Where eq can be one of '(= eq? eqv? equal?)
;;; - `(true ,expression) which is an expression that must be true.
;;; - `(,let ,args .tests) A one level let, let* or letrec that can have
;;; test expressions and other expressions in its body, see examples.
;;; Side effects in let body are evaluated at load time, not at
;;; run-test time, which may cause tests to fail.

(load "build/bootstrap.scm")
(load "elf/mbe.scm")

(import "java.util.Hashtable")
(import "javax.swing.JFrame")

;; used in a test
(define-method (foo (x Integer)) (list 'int x))
(define-method (bar (x Symbol)) x)
(define-method (bar (x Number)) (* 2 x))
(define-method (bar (x Byte)) (* 8 x))

;;; Used in a test
(define-syntax casequal
  (syntax-rules (else)
    ((casequal val ((m1 m2 ...) e1 e2 ...) clauses ...)
     (let ((key val))
       (cond ((member key '(m1 m2 ...)) e1 e2 ...)
             (else (casequal key clauses ...)))))
    ((casequal val (else e1 e2 ...)) (begin e1 e2 ...))
    ((casequal val) (if #f #f))))

(set! debug #t)				; This turns on U.p().

(define-macro (define-tests . forms)
  `(begin ,@(map (lambda (f) `(define-test ,f)) forms)))

(begin					; Don't print this!
  (set! test-queue			; Norvig queue of tests.
	(let ((it (cons '() '())))
	  (set-car! it it)))
  #f)

(define (enqueue-test test)
  (set-cdr! (car test-queue) (cons test '()))
  (set-car! test-queue (cdar test-queue)))

(define (tests) (cdr test-queue))

(define-macro (define-test form)
  (cond ((test? form) (define-test-test form))
	((let? form) (define-test-let form))
	(else (error "Don't know how to compile " form))))

(define (test? exp)
    (and (pair? exp) (memq (car exp) '(= eq? eqv? equal? true))))

(define (let? exp) (and (pair? exp) (member (car exp) '(let let* letrec))))

(define (define-test-test form)
  (let ((text `(lambda () ',form)))
    `(make-test ,text (lambda () ,form))))

(define (equality-test? test)
  (and (pair? test) (member (car test) '(= eqv? equal? eq?))))

(define (make-test text test)
  (enqueue-test (cons text test)))
(define test-text car)
(define test-test cdr)
(define (run-test test debug)
  (let ((result ((test-test test))))
    (if (or debug (not result))
	(begin
	  (write ((test-text test)))
	  (display (string-append " " (if result ""
							 "failed!") "\n"))))
    result))

(define (true x) x)

(define (define-test-let form)
  ;; Allow one level of let.
  (define let-kind car)
  (define let-parameters cadr)
  (define let-body cddr)
  (define (define-test-let-form form)
    (if (test? form) `(define-test ,form)
	form))
  `(,(let-kind form)
    ,(let-parameters form)
    ,@(map define-test-let-form  (let-body form))))

(define-tests
  ;; KRA 02DEC99: Nothing works.
  (equal? (= 3 0) #f)
  (= 3 3)
  (= (sin 0.0) 0.0)
  (eq? 'x (quote x))
  (equal? (list 3) '(3))
  (equal? (list 3 4) '(3 4))
  (equal? (cons 3 4) '(3 . 4))
  (equal? (cons 3 (list 4)) '(3 4))
  (equal? #(1) (vector 1))
  (equal? #(2 3) (vector 2 3))
  (= (+) 0)
  (equal? '(1 2) (list 1 2))
  (= 2 2)
  (= (/ 3 4) 0)
  (= (+ 1 2 3) 6)
  (true (procedure? (lambda () 3)))
  (= ((lambda () 3)) 3)
  (let ((x 3))
    (= ((lambda (x) 3) 15) 3)
    (= ((lambda (x) x) 15) 15)
    (= ((lambda (x) (if (> x 3) 5 9)) 2) 9)
    (= ((lambda (x) (if (> x 3) 5 9)) 17) 5)
    (= ((lambda (x y) (+ x y)) 5 2) 7)
    )
  (equal? ((lambda z z) 1 2 3 4) '(1 2 3 4))
  (equal? (apply list '(1 2 3 4)) '(1 2 3 4))
  (= (apply + '(1 2 3)) 6)
  (= (apply + '(1 2 3)) (+ 1 2 3))

  (equal? '(a . b) (cons 'a 'b))
  (equal? ((lambda (x . b) b) 2 3) '(3))
					;    (= (/ 3) (/ 1 3))
  (equal? ((lambda z z) 1 2 3) '(1 2 3))
  (equal? (apply (lambda z z) '(1 2 3)) '(1 2 3))

  (= (eval ((macro (x) (list '+ x 3)) 3)) 6)

  (eqv? (let ()
	  (define (f x) (+ x 3))
	  (f 5))
	8)

  (equal? (reverse '(1 2 3)) '(3 2 1))
  (= ((lambda (x) (define (f x) (* x x)) (f (f x))) 3) 81)

  (equal? (let ()
	    (define (g x)
	      (define (sq y) (* (f y) y))
	      (define (f a) (+ a x))
	      (sq (sq x)))
	    (g 3))
	  378)

  (equal? (let ()
	    (define (g x) 
	      (letrec ((sq (lambda (y) (* (f y) y)))
		       (f  (lambda (a) (+ a x))))
		(sq (sq x))))
	    (g 3))
	  378)

  (equal? (let ((x 3) (y 4)) (* x y)) 12)
  (equal? (let* ((x 3) (y (* x 2))) (+ x y)) 9)

  (equal? (map (lambda (var val) `(set! ,var ,val)) '(a b c) '(1 2 3))
	  '((set! a 1) (set! b 2) (set! c 3)))

  (equal? (append) '())
  (equal? (append '()) '())
  (equal? (append () ()) '())
  (equal? (append '(1 2) ()) '(1 2))
  (equal? (append '(1 2) '(3 4)) '(1 2 3 4))
  (equal? (append 3) 3)
  ;; ??? (append 3 3)
  (let ((x 3)
	(y '(1 2)))
    (equal? `(,x ,y) '(3 (1 2)))
    (equal? `(,x ,@y) '(3 1 2)))

  (equal? (case 3 ((1 2) 'low) ((3) 'hi) ((4 5) 'very high)) 'hi)

  (equal? (.first$ (cons 1 2)) 1)
  (equal? (let ((it (cons 1 2)))
	    (.first$ it 3)
	    it)
	  '(3 . 2))

  (equal? (list->vector '()) #())
  (equal? (list->vector '(1 2 3)) (vector 1 2 3))
  (equal? (class 'Hashtable) (class 'java.util.Hashtable))
  (equal? (class 'JFrame) (class 'javax.swing.JFrame))
  (let ((h (new 'java.util.Hashtable)))
    (invoke h 'put 'ken 3)
    (invoke h 'put 'tim 3)
    (= (invoke h 'get 'ken) 3)
    (= (invoke h 'get 'ken) (invoke h 'get 'tim))
    (equal? (.get h 'ken) 3)
    (equal? (.get h 'tim) ((method "get" "Hashtable" "Object") h 'tim)))

  (true (not (null? (invoke-static 'System 'getProperty "user.dir"))))

  (let ((x (new 'jsint.Pair 1 2)))
    (equal? x '(1 . 2))
    (equal? (peek x "first") 1)
    (equal? (peek x "rest") 2)
    (equal? (invoke-static 'jsint.Symbol 'intern "x") 'x)
    ;; (equal? (.getGlobalValue (invoke-static 'jsint.Symbol 'intern "x")) x)
    )
  (let ((x (new 'jsint.Pair 1 2)))
    (poke x "rest" 4)
    (poke x "first" 7)
    (equal? x '(7 . 4)))
  
  (equal? (peek-static 'jsint.Symbol "SET") 'set!)                 
  (equal? (Hashtable. 10) ((constructor "java.util.Hashtable" "int") 10))


  ;; KRA 28DEC99: 
  (equal? (list->string (string->list "abc")) "abc")
  (equal? (list->string (string->list "")) "")
  (let ()
    (define (g x)
      (lambda (y) (+ x y)))
    (equal? (equal? (g 3) (g 3)) #f))
  (equal? (substring "012345" 4 6) "45")

  ;; KRA 29DEC99: 
					;    (equal? (/ 3) (/ 1 3))
  (equal? (- 3) -3)

  ;; KRA 30DEC99:
  (equal? (let ((v (vector 1 2 3))) (vector-set! v 2 4) v) #(1 2 4))

  ;; KRA 03JAN00: Did not compile.
  (let ()
    (define f (lambda () ()))
    (equal? (f) '()))

  ;; KRA 31JAN00: 
  (equal? (equal? (list #null) (list #null)) #t)
  (equal? (expt 10 2) 100.0)

  ;; KRA 14MAY00: correct behavior of U.toStr() and (string-append).
  (equal? (string-append #\a #\b #\c) "abc")
  (equal? (string-append "hi" 'buster 3) "hibuster3")
  (equal? (string->list "abc") '(#\a #\b #\c))

  ;; KRA 26JUL00: Test jsint.SI.class
  (equal? (SI.eval '(+ 2 3)) (+ 2 3))
  (equal? (SI.eval "(+ 2 3)") (+ 2 3))
  (equal? (SI.call "+" 2 3) (+ 2 3))
  (equal? (SI.call + 2 3) (+ 2 3))
  (equal? (SI.apply "+" (SI.list 2 3)) (+ 2 3))
  (let ()
    (SI.load "(define (si:f x) (+ x (si:g x))) (define (si:g x) (* x 3))")
    (equal? (si:f 3) 12))

  ;; KRA 07AUG00: Bug in Pair.equal.
  (equal? '(#(1). #(2)) '(#(1) . #(2)))
  (equal? '(1 2 "hi") '(1 2 "hi"))
  (equal? (.hashCode '(#(1). #(2))) (.hashCode '(#(1). #(2))))

  ;; KRA 29OCT00: Boolean eq? failed.
  (equal? (eq? (Boolean. #t) #t) #t)
  (equal? (eq? (Boolean. #f) #f) #t)
  (equal? (eq? #t (Boolean. #t)) #t)
  (equal? (eq? #f (Boolean. #f)) #t)
  (equal? (eq? #t #t) #t)
  (equal? (eq? #f #f) #t)

  ;; KRA 28NOV00: Truncate and several other procedures failed.
  (eqv? (max 3 4) 4)
  (eqv? (max 3.9 4) 4.0)

  (eqv? (+ 3 4) 7)
  (eqv? (+ 3) 3)
  (eqv? (+) 0)
  (eqv? (* 4) 4)
  (eqv? (*) 1)

  (eqv? (- 3 4) -1)
  ;; (eqv? (- 3 4 5) -6)		; KRA 28NOV00: - takes 1 or 2 args.
  (eqv? (- 3) -3)
  ;; (eqv? (/ 3 4 5) 0)		; KRA 28NOV00: / takes 2 args.
  ;; (/ 3)				
	  
  (eqv? (abs -7) 7)
  (eqv? (remainder  13  4)  1)
  (eqv? (modulo    -13  4)  3)
  (eqv? (remainder -13  4) -1)
  (eqv? (modulo     13 -4) -3)
  (eqv? (remainder  13 -4)  1)

  ;;    (eqv? (remainder -13 -4.0) -1.0) ;; TJH 8/31/2001  THIS TEST FAILS AS IT RETURNS -1 and not -1.0
  ;; BUG

  (eqv? (gcd 32 -36) 4)
  ;; (eqv? (gcd) 0)			; KRA 28NOV00: takes 1+ args.
  (eqv? (lcm 32 -36) 288)
  (= (lcm 32.0 -36) 288.0)  ;; TJH 31AUG01 standard only requires "=" equality for non-integer args
  (eqv? (lcm) 1)

  (= (floor -4.3) -5.0)
  (= (ceiling -4.3) -4.0)
  (= (truncate -4.3) -4.0)
  (= (round -4.3) -4.0)
    
  (= (floor 3.5) 3.0)
  (= (ceiling 3.5) 4.0)
  (= (truncate 3.5) 3.0)
  (= (round 3.5) 4.0)
  (eqv? (round 7) 7)

  ;; KRA 09JAN01: (integer->char 128) caused out of bounds exception.
  (eq? (integer->char 0) (integer->char 0)) ; Cached.
  (eq? (integer->char 127) (integer->char 127))
  ;;; These tests check if we can construct such characters:
  (eqv? (char->integer (integer->char -3)) 65533)
  (eqv? (char->integer (integer->char 128)) 128)
  (eqv? (char->integer (integer->char 12345)) 12345)

  ;; KRA 27JUN01: #\s #\S #\n and #\N were not read properly.
  (eqv? #\s #'s')
  (eqv? #\S #'S')
  (eqv? #\n #'n')
  (eqv? #\N #'N')
  (eqv? #\space #' ')
  (eqv? #\newline #'\n')
  (eqv? #\SPACE #' ')
  (eqv? #\NEWLINE #'\n')
  (eqv? #'&' #'\u0026')
  (eqv? #'\046' #'&')
  (eqv? #\' #'\'')
  (eqv? #\" #'\"')
  (eqv? #\\ #'\\')
  (eqv? #\# #'#')

  ;;  TJH 31AUG01: New Exception handling.
  (true (.isInstance java.lang.ArithmeticException.class (tryCatch (/ 1 0) (lambda(e) e))))
  (true (.isInstance jscheme.SchemeException.class (tryCatch (a b) (lambda(e) e))))
  (true (.isInstance java.lang.ArithmeticException.class 
		     (tryCatch 
		      (tryCatch (/ 1 0) (lambda(e) (throw e)))
		      (lambda(e) e))))

  ;; KRA 23NOV01: (list? 3) errored!
 (eq? (list? '(a b c)) #t)
 (eq? (list? '()) #t)  
 (eq? (list? '(a . b)) #f)

;;; BUG this cyclic structure kills the jscheme/BootstrapCore.scm version...
; (let ((x (list 'a)))
;   (set-cdr! x x)
;   (eq? (list? x) #f))

 (eq? (list? 3) #f)  

  ;; TJH 8DEC01
  (equal? (make-vector 3 2) #(2 2 2))
  (equal? (make-vector 3) #(#null #null #null))

  ;; TJH 9Mar02, tests of quasi-string notation
  ;; and exception of symbols containing array marker []
  (equal? {abc"[(+ 1 2)]"\{\}\[\]} "abc\"3\"{}[]")
  (equal? {a[(if (< 1 2) {bc} {de})]d} "abcd")
  (equal? 'Object[] (string->symbol "Object[]"))
  (equal? {static ['Object[]] x;} "static Object[] x;")
  (equal? Object[].class (Class.forName "[Ljava.lang.Object;"))
  (equal? {[String[].class]} "class [Ljava.lang.String;")

  ;; TJH 3/10/02 -- tests in response to Bill Hale's bug report
  (eqv? (modulo  123456789  100)  89)
  (eqv? (modulo  123456789 -100) -11)
  (eqv? (modulo -123456789  100)  11)
  (eqv? (modulo -123456789 -100) -89)

;;  (eqv? (modulo 123456789012 100) 12)  ;; BUG this currently Fails!!

  ;; TJH 3/10/02 -- tests in response to Bill Hale's list-tail bug report
  (equal? (list-tail '(1 2 3) 3) '())

  (equal? (tryCatch (list-tail '(1 2 3) 4) (lambda(e) 'error)) 'error)

  (equal? (list-tail '(1 2 3) 1) '(2 3))

  ;; KRA 11MAR02: Derek Upham <Derek.Upham@ontain.com> found a bug in cond
  ;; i introduced:
  (eq? (cond ((eq? 'a 'a) => (lambda (x) x))) #t)


  ;; TJH 3/11/02 -- check Richard Kelsey's narrowing in instance methods
  (equal? (tryCatch 
            (let ((x (java.lang.reflect.Array.newInstance byte.class 10)))
              (Array.setByte x 5 10)
              (Array.getByte x 5))
            (lambda(e) e))
         10)
  ;; TJH 3/11/02 -- check that Bytes and Shorts are handled properly by eqv? and equal?
  (eqv? 10 (java.lang.Byte. "10"))
  (eqv? 10L (java.lang.Byte. "10"))
  (eqv? 10 (java.lang.Short. "10"))
  (eqv? (java.lang.Byte. "10") (java.lang.Short. "10"))

  (equal? 10 (java.lang.Byte. "10"))
  (equal? 10 (java.lang.Short. "10"))
  (equal? 10 (java.lang.Short. "10"))
  (equal? (java.lang.Byte. "10") (java.lang.Short. "10"))

  ;; KRA 14MAR02: list->String[] was called list->StringArray temporarily.
  (equal? (list->String[] '(1 2 3)) (array String.class "1" "2" "3"))

  ;; Derek Upham 12MAR02: 
  ;; Dorai Sitaram's "Macros By Example" implementation uploaded
  ;; yesterday had a slight violation of R5RS semantics (it was
  ;; treating pattern keywords as literal identifiers, which messed up
  ;; recursive macro definitions).

  (equal? (casequal "foo" (("foo") 1) (("bar") 2)) 1)
  (equal? (casequal "bar" (("foo") 1) (("bar") 2)) 2)
  (equal? (casequal "baz" (("foo") 1) (("bar") 2)) #f)
  (equal? (casequal "baz" (("foo") 1) (("bar") 2) (else 3)) 3)

  ;; Derek Uphap 18MAR02
  ;; (define-method (foo (x Integer)) (list 'int x))
  ;; (define-method (bar (x Symbol)) x)
  ;; (define-method (bar (x Number)) (* 2 x))
  ;; (define-method (bar (x Byte)) (* 8 x))

  (equal? (foo 3) '(int 3))
  (equal? (tryCatch (foo 3L) (lambda(e) 'error)) 'error)
  (equal? (tryCatch (foo (Byte. "3")) (lambda(e) 'error)) 'error)
  (equal? (tryCatch (foo 'apple) (lambda(e) 'error)) 'error)
  (equal? (tryCatch (bar "apple") (lambda(e) 'error)) 'error)
  (equal? (bar (Byte. "3")) 24)
  (equal? (bar 3) 6)
  (equal? (bar 3L) 6L)
  (equal? (bar 'L) 'L)

  ;; enhanced javadot tests
  ;; private fields
  (equal? (tryCatch (.name$  'a) (lambda(e) 'error)) 'error)
  (equal? (tryCatch (.name$# 'a) (lambda(e) 'error)) "a")

  (equal? (tryCatch (.Symbol.name$  'a) (lambda(e) 'error)) 'error)
  (equal? (tryCatch (.Symbol.name$# 'a) (lambda(e) 'error)) "a")

  (equal? (tryCatch (.jsint.Symbol.name$  'a) (lambda(e) 'error)) 'error)
  (equal? (tryCatch (.jsint.Symbol.name$# 'a) (lambda(e) 'error)) "a")

  ;; private instance methods
  (equal? (tryCatch (.equalsFirst '(a b) '(a c)) (lambda(e) 'error)) 'error)
  (equal? (tryCatch (.equalsFirst# '(a b) '(a c)) (lambda(e) 'error)) #t)

  (equal? (tryCatch (.Pair.equalsFirst '(a b) '(a c)) (lambda(e) 'error)) 'error)
  (equal? (tryCatch (.Pair.equalsFirst# '(a b) '(a c)) (lambda(e) 'error)) #t)

  (equal? (tryCatch (.jsint.Pair.equalsFirst '(a b) '(a c)) (lambda(e) 'error)) 'error)
  (equal? (tryCatch (.jsint.Pair.equalsFirst# '(a b) '(a c)) (lambda(e) 'error)) #t)

  ;; private static methods
  (equal? (tryCatch (Pair.hashCode0  null) (lambda(e) 'error)) 'error)
  (equal? (tryCatch (Pair.hashCode0#  null) (lambda(e) 'error)) 17)

  (equal? (tryCatch (jsint.Pair.hashCode0  null) (lambda(e) 'error)) 'error)
  (equal? (tryCatch (jsint.Pair.hashCode0#  null) (lambda(e) 'error)) 17)

  ;; private constructors
  (equal? (tryCatch (Symbol. "abc") (lambda(e) 'error)) 'error)
  (equal? (tryCatch (.name$# (Symbol.#  "abc")) (lambda(e) 'error)) "abc")

  (equal? (tryCatch (jsint.Symbol. "abc") (lambda(e) 'error)) 'error)
  (equal? (tryCatch (.jsint.Symbol.name$# (jsint.Symbol.#  "abc")) (lambda(e) 'error)) "abc")

  )

(define (run-tests debug)
  (let ((total 0)
	(failures 0))
    (for-each
     (lambda (t)
       (let ((result (run-test t debug)))
	 (set! total (+ total 1))
	 (if (not result) (set! failures (+ failures 1)))))
     (tests))
    (display (string-append "Tests: " total " Failures: " failures "\n"))
    (display "\nAll tests have completed with any errors as shown above\nTry (run-tests #t) to see both failing and successful tests and their results\n")))

(run-tests #f)

