/*
 * Copyright (c) 2002, The EROS Group, LLC and Johns Hopkins
 * University. All rights reserved.
 * 
 * This software was developed to support the EROS secure operating
 * system project (http://www.eros-os.org). The latest version of
 * the OpenCM software can be found at http://www.opencm.org.
 * 
 * Redistribution and use in source and binary forms, with or
 * without modification, are permitted provided that the following
 * conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials
 *    provided with the distribution.
 * 
 * 3. Neither the name of the The EROS Group, LLC nor the name of
 *    Johns Hopkins University, nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <opencm.h>
#include <repos/opencmrepos.h>
#include <sys/stat.h>

#define XDCS_ARCHIVE_MAX (2 * 1024 * 1024)
#define XDCS_THRESH 400
#define SDR_XDCONTENT SDR_BINARY

// #define VERBOSE

static void
sxdfsrepository_ReviseEntity(Repository *r,
			     const char *mURI,
			     const char *prevTrueName, void *s)
     __attribute__ ((unused));

     static void
     sxdfsrepository_ReviseEntity(Repository *r,
				  const char *mURI,
				  const char *prevTrueName, void *s)
{
  SDR_stream *archive = NULL;
  const char *trueName = ser_getTrueName(s);
  const char *entpath = path_join(r->uri->path, R_ENTITIES);
  const char *tmppath = path_join(entpath, trueName);
  const char *xdfsarchive = path_join(entpath, xstrcat(trueName, ",x"));
  const char *prevpath = 0;

  OC_bool newFile = TRUE;
  SDR_stream *out = NULL;
  URI *uri = uri_create(mURI);

  tmppath = xstrcat(tmppath, ",x-tmp");
  onthrow_remove(tmppath);

  report(3, "XDFS Enter ReviseEntity %s", trueName);

  if (!path_exists(xdfsarchive)) {
    assert(prevTrueName == 0 || nmequal(prevTrueName, trueName) == 0);

    TRY {
      SDR_stream *strm = NULL;

      if (!nmequal(uri->netloc, r->repositoryID))
	THROW(ExNoAccess, 
	      format("Cannot revise %s on non-owning repository", mURI));

      prevpath = path_join(r->uri->path, R_ENTITIES);

      /* Prepare to write the entity by doing the in-memory serialization: */
      TRY {
	strm = stream_createBuffer(SDR_XDCONTENT);
	sdr_write("ENTITY", strm, s);
      }
      DEFAULT(ex) {
	if (strm) stream_close(strm);
	RETHROW(ex);
      }
      END_CATCH;

      stream_close(strm);
      stream_reread(strm);

      /* Figure out where it is going: */

#ifdef VERBOSE
      if (!prevTrueName)
	xprintf("*** NO prevTrueName for \"%s\"\n", trueName);
#endif

#ifdef VERBOSE
      if (prevTrueName && nmequal(prevTrueName, trueName))
	xprintf("*** prevTrueName \"%s\" matches trueName\n", prevTrueName);
#endif

      if (prevTrueName && !nmequal(prevTrueName, trueName)) {
	const char *path = path_join(prevpath, xstrcat(prevTrueName, ",x"));

	/* It might be a valid predecessor that we do not have locally: */
	if (path_exists(path)) {
	  newFile = FALSE;
	  prevpath = path;
#ifdef VERBOSE
	  xprintf("*** prevTrueName \"%s\" precedes \"%s\"\n", prevTrueName, trueName);
#endif
	}
      }

      if (newFile)
	prevpath = "/dev/null";
      
      archive = NULL;
      TRY {
	archive = stream_fromfile(prevpath, SDR_XDCS);
	out = stream_createfile(tmppath, SDR_XDCS);

#if 1
	/* At some threshold the size of the archive becomes a
	   processing bottleneck. We are still figuring out where that
	   threshold is. For the moment, if 

	   sizeof(archive) > THRESH * sizeof(strm)

	   then punt and create a new archive.

	   Also, create a new archive if the absolute size of the
	   archive file exceeds 2 Mbytes.
	*/

	if (stream_length(archive) > XDCS_ARCHIVE_MAX) {
	  stream_close(archive);
	  archive = stream_fromfile("/dev/null", SDR_XDCS);
	  newFile = TRUE;
	  xprintf("*** Archive max exceeded\n");
	}
	else if (stream_length(archive) > (stream_length(strm) * XDCS_THRESH)) {
	  stream_close(archive);
	  archive = stream_fromfile("/dev/null", SDR_XDCS);
	  newFile = TRUE;
	  xprintf("*** Archive ratio exceeded\n");
	}
#endif

	xdcs_insert(archive, strm, trueName, out);
	stream_close(archive);
	stream_close(out);

	if (path_file_length(tmppath) == 0) {
	  xprintf("Object (new: %c) file \"%s\" has zero size\n", 
		  newFile ? 'T' : 'F', tmppath);
	  THROW(ExNoObject, 
		format("Object write failed for \"%s\"", tmppath));
	}
      }
      DEFAULT(ex) {
	if (archive) stream_close(archive);
	if (out) stream_close(out);
	RETHROW(ex);
      }
      END_CATCH;

      path_rename(tmppath, xdfsarchive);

      if (newFile == FALSE) {
	const char *clobber = prevpath;

	if (path_issymlink(clobber))
	  clobber = path_resolvelink(clobber);

	path_rename(xdfsarchive, clobber);

	if (path_mksymlink(path_tail(clobber), xdfsarchive) < 0)
	  THROW(ExNoObject, 
		format("Could not create symlink for \"%s\"", xdfsarchive));
      }

      report(3, "Created \"%s\". newFile=%c\n", 
	     xdfsarchive, newFile ? 'T' : 'F');

      if (!path_exists(xdfsarchive))
	THROW(ExNoObject, 
	      format("No file \"%s\". newFile=%c prev=\"%s\"",
		     xdfsarchive, newFile ? 'T' : 'F', prevpath));

      if (path_file_length(xdfsarchive) == 0)
	THROW(ExNoObject,
	      format("Object length bad in \"%s\"!\n", xdfsarchive));
    }
    DEFAULT(ex) {
      RETHROW(ex);
    }
    END_CATCH;
  }
  report(3, "XDFS Exit ReviseEntity %s", trueName);
}

static void *
sxdfsrepository_GetEntity(Repository *r, const char *mURI /* unused */,
			  const char *trueName)
     __attribute__ ((unused));

     static void *
     sxdfsrepository_GetEntity(Repository *r, const char *mURI /* unused */,
			       const char *trueName)
{
  void *ent = 0;
  const char *entpath;
  const char *copath;
  const char *xdfspath;
  SDR_stream *out = NULL;
  SDR_stream *archive = NULL;

  report(3, "XDFS Enter GetEntity %s", trueName);

  if (strncmp(trueName, "sha1_", 5) != 0)
    THROW(ExMalformed, "Bad or unknown hash type");

  TRY {
    out = stream_createBuffer(SDR_RAW);
    entpath = path_join(r->uri->path, R_ENTITIES);
    copath = path_join(entpath, trueName);
    xdfspath = xstrcat(copath, ",x");

    if (!path_exists(xdfspath))
      THROW(ExNoObject, format("Path not found: %s", xdfspath));

    report(3, "Begin reconstruction of %s\n", trueName);
    archive = stream_fromfile(xdfspath, SDR_XDCS);
    xdcs_extract(archive, trueName, out);
    stream_close(archive);
    archive = NULL;

    report(3, "Extract object from buffer for %s\n", trueName);
    
    /* out now holds the serialized version of the object AS A RAW
       BYTE STREAM. In theory, we now need to open a typed byte stream
       out of this buffer so we can deserialize it. */
    stream_reread(out);

    {
      SDR_stream *reread =
	stream_fromBuffer(stream_asBuffer(out), SDR_XDCONTENT);
      ent = sdr_read("ENTITY", reread);
      stream_close(reread);
    }
    stream_close(out);
    out = 0;
    report(3, "Got extracted object %s\n", trueName);
  }
  DEFAULT(ex) {
    if (archive) stream_close(archive);
    if (out) stream_close(out);
    RETHROW(ex);
  }
  END_CATCH;

  report(3, "XDFS Exit GetEntity %s", trueName);

  report(3, "Getting %s got %s\n", trueName, ser_getTrueName(ent));

  if (!nmequal(trueName, ser_getTrueName(ent)))
    THROW(ExMalformed, "Bad response from GetEntity()!");
  return ent;
}

void
sxdfsrepository_init(Repository *r)
{
  fsrepos_init(r);

  r->ReviseEntity = sxdfsrepository_ReviseEntity;
  r->GetEntity    = sxdfsrepository_GetEntity;
}
