/*
 * Copyright (c) 2002, The EROS Group, LLC and Johns Hopkins
 * University. All rights reserved.
 * 
 * This software was developed to support the EROS secure operating
 * system project (http://www.eros-os.org). The latest version of
 * the OpenCM software can be found at http://www.opencm.org.
 * 
 * Redistribution and use in source and binary forms, with or
 * without modification, are permitted provided that the following
 * conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials
 *    provided with the distribution.
 * 
 * 3. Neither the name of the The EROS Group, LLC nor the name of
 *    Johns Hopkins University, nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/* True Names (and other lies)
 *
 * For a content entity, the "true name" of the entity is the concatenation of
 * "sha1_" with the string form of the encoding of its SHA1 cryptographic hash
 * (either hex or base64 encoding is used).
 *
 * For a sequence object, such as a branch, the "true name" is a randomly
 * generated set of bits. Good random number generation is one place where the
 * canonical Python libraries (along with the libraries for every *other*
 * language on the planet) are seriously deficient. For the moment, I am
 * ignoring this problem entirely.
 */

#include <opencm.h>
#include <openssl/rand.h>
#include <openssl/err.h>

const char *
truename_FromHash(const char *prefix, const char *hex)
{
#if 0
  char *s = xstrcat(prefix, xstrcat("_", xstrcat("sha1_", hex)));
#else
  char *s = xstrcat("sha1_", hex);
#endif

  return s;
}

const char *
truename_FromBytes(const char *prefix, unsigned len, const void *v)
{
  const char *hex;
  OPENCM_SHA *sha = sha_create();
  sha_append(sha, len, v);
  sha_finish(sha);
  
  hex = sha_hexdigest(sha);
  return truename_FromHash(prefix, hex);
}

const char *
truename_FromString(const char *prefix, const char *s)
{
  OPENCM_SHA *sha = sha_from_string(s);
  const char *hex = sha_hexdigest(sha);
  return truename_FromHash(prefix, hex);
}

const char *
truename_NewName()
{
  unsigned char rnd_buf[20];

  if (RAND_bytes(rnd_buf, 20) <= 0)
    THROW(ExPrngError, format("Problem with PRNG: %s",
                              ERR_reason_error_string(ERR_get_error())));

#ifdef USE_BASE16
  return hex_encode(rnd_buf, 20);
#else
  return b64_encode(rnd_buf, 20);
#endif
}

  /* Assumption: the hash bytes will always be at the end of the
     truename. Following quasi-hash is byteorder dependent, but it
     doesn't matter. */
tnhash_t
truename_hash(const char *tn)
{
  int i; 
  const char *p = tn;
  tnhash_t tnHash;
  char *tnHashP = (char *)&tnHash;

  while (*p)
    p++;

  for (i = 0; i < sizeof(tnHash); i ++)
    *tnHashP++ = *--p;

  return tnHash;
}
