/*
 * Copyright (c) 2002, The EROS Group, LLC and Johns Hopkins
 * University. All rights reserved.
 * 
 * This software was developed to support the EROS secure operating
 * system project (http://www.eros-os.org). The latest version of
 * the OpenCM software can be found at http://www.opencm.org.
 * 
 * Redistribution and use in source and binary forms, with or
 * without modification, are permitted provided that the following
 * conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials
 *    provided with the distribution.
 * 
 * 3. Neither the name of the The EROS Group, LLC nor the name of
 *    Johns Hopkins University, nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <opencm.h>
#include <stdlib.h>

/* We use the following ordering for seeding the RNG:

   /dev/urandom, if we've got it. If we do, ignore everything else.

   opt_RandomFile, it is is set

   The RANDFILE env variable, if that is set
      [this is assumed to point to a plain file, or a character device]

   The EGD_PATH env variable, if that is set
      [this is assumed to point to an EGD socket]

   The enviromental variable names were not at random; both are used fairly
   widely across OpenSSL-based programs (stunnel, openldap, etc)

   We limit the read size to 32K, which seems more than enough.
*/
void initialize_PRNG(void)
{
  /* RAND_status will return 1 without us doing a thing if /dev/urandom
     is around for OpenSSL to seed itself. In this case, don't bother with
     this other stuff.
  */
  if(RAND_status())
    return;

  #define MAX_READ 32*1024

  if(opt_RandomFile)
    RAND_load_file(opt_RandomFile, MAX_READ);

  if(getenv("RANDFILE"))
    RAND_load_file(getenv("RANDFILE"), MAX_READ);

  /* We limit gathering entropy from EGD to only 128 bytes. This is because
       a) EGD is slow
       b) EGD will sometimes crash if too many requests come in. I reported
          this to the author a few months ago, but never got a response.
       c) EGD gathers entropy at a not very fast rate; if we read much
          more than this, we'll probably start running out, which would
          be bad, especially for the client (several invocations might occur
          over the course of only a few minutes)

     However, EGD is by far the best choice for a RNG on systems that don't
     have /dev/random, so it seems wise to support it.
   */
  if(getenv("EGD_PATH"))
    RAND_egd_bytes(getenv("EGD_PATH"), 128);

  /* Check to see if we are sufficiently initialized */
  if(!RAND_status())
  {
    THROW(ExPrngError, "Insufficient randomness to seed the PRNG");
  }
}
