/*
 * Copyright (c) 2002, The EROS Group, LLC and Johns Hopkins
 * University. All rights reserved.
 * 
 * This software was developed to support the EROS secure operating
 * system project (http://www.eros-os.org). The latest version of
 * the OpenCM software can be found at http://www.opencm.org.
 * 
 * Redistribution and use in source and binary forms, with or
 * without modification, are permitted provided that the following
 * conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials
 *    provided with the distribution.
 * 
 * 3. Neither the name of the The EROS Group, LLC nor the name of
 *    Johns Hopkins University, nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <opencm.h>

void
entity_serialize(SDR_stream *strm, const void *ob)
{
  const Entity *e = ob;
  
  /* NOTE: IF YOU CHANGE THIS, CHANGE IT IN WSENTITY ALSO!!! */
  sdr_w_obname("bits", strm, e->contentTrueName);
  sdr_w_u32("len", strm, e->length);
  sdr_w_obname("cmtInfo", strm, e->commitInfoTrueName);
  sdr_w_obname("parent", strm, e->parent);
  sdr_w_obname("mrgParent", strm, e->mergeParent);
  sdr_w_obname("chgparent", strm, e->change_parent);
  sdr_w_obname("chgmrgParent", strm, e->change_mergeParent);
  sdr_w_obname("famNID", strm, e->familyNID);
  sdr_w_string("fsName", strm, e->fsName);
  sdr_w_u8("type", strm, e->entityType);
  sdr_w_u32("perms", strm, e->entityPerms);
}

void *
entity_deserialize(const DeserializeInfo *di, SDR_stream *strm)
{
  Entity *e = (Entity *) GC_MALLOC(sizeof(Entity));

  e->serType             = &Entity_SerType;
  SER_MODIFIED(e);

  /* NOTE: IF YOU CHANGE THIS, CHANGE IT IN WSENTITY ALSO!!! */
  e->contentTrueName     = sdr_r_obname("bits", strm);
  e->length              = sdr_r_u32("len", strm);
  e->commitInfoTrueName  = sdr_r_obname("cmtInfo", strm);
  e->parent              = sdr_r_obname("parent", strm);
  e->mergeParent         = sdr_r_obname("mrgParent", strm);
  e->change_parent       = sdr_r_obname("chgparent", strm);
  e->change_mergeParent  = sdr_r_obname("chgmrgParent", strm);
  e->familyNID           = sdr_r_obname("famNID", strm);
  e->fsName              = sdr_r_string("fsName", strm);
  e->entityType          = sdr_r_u8("type", strm);
  e->entityPerms         = sdr_r_u32("perms", strm);

  e->famNidHash = truename_hash(e->familyNID);

  return e;
}

void
entity_show(const void *v)
{
  const Entity *e = v;
  
  xprintf("Entity %s <content suppressed>\n", ser_getTrueName(e));
  xprintf("FamNID       %s\n", e->familyNID);
  xprintf("Type:        %c\n", e->entityType);
  xprintf("Length:      %d\n", e->length);
  xprintf("FS Name:     %s\n", e->fsName);
  xprintf("Permissions: 0x%x\n", e->entityPerms);
  xprintf("CommitInfo:  %s\n", e->commitInfoTrueName);
  xprintf("Bits:        %s\n", e->contentTrueName);
  xprintf("Parent:      %s\n", e->parent ? e->parent : "<none>");
  xprintf("MergeParent: %s\n", e->mergeParent ? e->mergeParent : "<none>");
  xprintf("Change Parent:      %s\n", 
	  e->change_parent ? e->change_parent : "<none>");
  xprintf("Change MergeParent: %s\n", 
	  e->change_mergeParent ? e->change_mergeParent : "<none>");
}

Entity *
entity_dup(const Entity *ent)
{
  Entity *e = (Entity *) GC_MALLOC(sizeof(Entity));

  e->serType = &Entity_SerType;
  SER_MODIFIED(e);

  e->familyNID = ent->familyNID;
  e->famNidHash = ent->famNidHash;
  e->contentTrueName = ent->contentTrueName;
  e->commitInfoTrueName = ent->commitInfoTrueName;
  e->fsName = ent->fsName;
  e->entityType = ent->entityType;
  e->entityPerms = ent->entityPerms;
  e->length = ent->length;

  e->parent = ent->parent;
  e->mergeParent = ent->mergeParent;

  e->change_parent = ent->change_parent;
  e->change_mergeParent = ent->change_mergeParent;

  return e;
}

const char *
entity_CreateFamNID(void)
{
  return xstrcat("nid_", truename_NewName());
}

Entity *
entity_create(char eType, const char *famNID)
{
  Entity *e = (Entity *) GC_MALLOC(sizeof(Entity));

  e->serType = &Entity_SerType;
  SER_MODIFIED(e);

  e->familyNID = famNID ? famNID : entity_CreateFamNID();
  e->famNidHash = truename_hash(e->familyNID);
  e->contentTrueName = 0;
  e->commitInfoTrueName = 0;
  e->fsName = 0;
  e->entityPerms = 0;
  e->entityType = eType;

  e->parent = 0;
  e->mergeParent = 0;

  e->change_parent = 0;
  e->change_mergeParent = 0;

  return e;
}
