/*
 * Copyright (c) 2002, The EROS Group, LLC and Johns Hopkins
 * University. All rights reserved.
 * 
 * This software was developed to support the EROS secure operating
 * system project (http://www.eros-os.org). The latest version of
 * the OpenCM software can be found at http://www.opencm.org.
 * 
 * Redistribution and use in source and binary forms, with or
 * without modification, are permitted provided that the following
 * conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials
 *    provided with the distribution.
 * 
 * 3. Neither the name of the The EROS Group, LLC nor the name of
 *    Johns Hopkins University, nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <opencm.h>

/* The CommitInfo object does NOT name the branch/version into which
 * the commit occurs. This is because doing so would interfere with
 * recovery from partial uploads.
 *
 * It *cannot* include the name of the Change object because
 * cryptographically hashed names and circular reference really don't
 * mix. It could harmlessly name the containing branch, but I'm trying
 * hard to keep all of the pointers going one way (which is probably a
 * mistake for purposes of recoverability, but for now I want to keep
 * things simple). The CommitInfo object *should not* contain the
 * version number of this commit. Here's why:
 *
 * Suppose a commit fails because the branch top has moved. This can
 * occur between the time that the upload starts and the upload
 * completes. Then we would need to re-perform the commit against a
 * different base version. But if we do this:
 *
 *   1. We would then need to change the version number in the
 *      CommitInfo record.
 *   2. This would change it's true name.
 *   3. This would in turn change the content of every Entity and
 *      Change object.
 *   4. Which would in turn change *their* true names.
 *
 * all of which would ultimately force us to perform the whole damned
 * upload again from scratch.
 *
 * Thus: no version number here.
 */

void
commitinfo_show(const void *vp)
{
  const CommitInfo *ci = vp;
  
  xprintf("Date:          %s\n", ci->time);
  xprintf("AuthorURI:     %s\n", ci->authorURI);
  xprintf("BranchURI:     %s\n", ci->branchURI);
  xprintf("Branch Version: %s\n", xunsigned64_str(ci->branchVersion));
  xprintf("Descrip:\n%s\n", buffer_asString(ci->descrip));
}

void
commitinfo_serialize(SDR_stream *strm, const void *vp)
{
  const CommitInfo *ci = vp;

  sdr_w_string("time", strm, ci->time);
  sdr_w_obname("authorURI", strm, ci->authorURI);
  sdr_write("descrip", strm, ci->descrip);
  sdr_w_obname("branchURI", strm, ci->branchURI);
  sdr_w_u64("rev", strm, ci->branchVersion);
}

void *
commitinfo_deserialize(const DeserializeInfo *di, SDR_stream *strm)
{
  CommitInfo *ci = (CommitInfo *) GC_MALLOC(sizeof(CommitInfo));

  ci->serType        = &CommitInfo_SerType;
  SER_MODIFIED(ci);
  
  ci->time           = sdr_r_string("time", strm);
  ci->authorURI      = sdr_r_obname("authorURI", strm);
  ci->descrip        = sdr_read("descrip", strm);
  ci->branchURI      = sdr_r_obname("branchURI", strm);
  ci->branchVersion  = sdr_r_u64("rev", strm);

  return ci;
}

CommitInfo *
commitinfo_create(const Buffer *descrip, const char *authorURI,
		  const char *branchURI, oc_uint64_t rev)
{
  CommitInfo *ci = (CommitInfo *) GC_MALLOC(sizeof(CommitInfo));

  assert(authorURI);

  ci->serType = &CommitInfo_SerType;
  SER_MODIFIED(ci);

  ci->time = os_GetISOTime();
  ci->descrip = descrip;
  ci->authorURI = authorURI;
  ci->branchURI = branchURI;
  ci->branchVersion = rev;
  
  return ci;
}

