#ifndef CLIENT_PENDINGCHANGE_H
#define CLIENT_PENDINGCHANGE_H

/*
 * Copyright (c) 2002, The EROS Group, LLC and Johns Hopkins
 * University. All rights reserved.
 * 
 * This software was developed to support the EROS secure operating
 * system project (http://www.eros-os.org). The latest version of
 * the OpenCM software can be found at http://www.opencm.org.
 * 
 * Redistribution and use in source and binary forms, with or
 * without modification, are permitted provided that the following
 * conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials
 *    provided with the distribution.
 * 
 * 3. Neither the name of the The EROS Group, LLC nor the name of
 *    Johns Hopkins University, nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/* A PendingChange represents the current state of a workspace, which
 * is either the current state of a branch (when the workspace is
 * unmodified) or the future state of a branch.
 *
 * Unlike a Change, the entity list is inline here.
 */

struct PendingChange {
  Serializable ser;

  /* identifies the repository from which this Workspace was built */
  const char *reposURI;

  /* identifies the user (key file name) from which this Workspace was built */
  const char *userName;

  /* identifies the branch/version on the above repository */
  const char *branchURI;

  /* identifies the user who generated this Workspace */
  const char *creator;

  /* Number of revisions that existed on the branch at the time this
   * pending change was checked out: */
  oc_uint64_t nRevisions;
  const char *baseChangeName;	/* truename of that change if any */
  const char *baseCmtInfoName;	/* truename of CommitInfo record for
				   baseChangeName */
  
  /* TrueName of the change that has been merged into this workspace,
   * if any. */
  const char *mergedChange;
  OC_bool        isPartialMerge;
  
  ObVec *entSet;		/* vector of (WsEntity *) */

  const char *notes;		/* notes added by user while underway */

  OC_bool isSorted;		/* not serialized */

  Buffer *filterSet;		/* per-branch fsname filter */
  const char *filterSetName;	/* true name of original filter set */
};

PendingChange *pendingchange_create(Repository *r, 
				    const char *branchURI,
				    oc_uint64_t nRevs);

void pendingchange_RecomputeStatus(PendingChange*, StrVec *names);
void pendingchange_ReportStatus(PendingChange*, StrVec *names);
void pendingchange_diff(PendingChange*, Repository *r, StrVec *nmList);

void pendingchange_InsertEntity(PendingChange*, WsEntity *);
WsEntity *pendingchange_FindEntity(PendingChange*, const char *name);
void pendingchange_RemoveEntity(PendingChange*, WsEntity *);

/* Implemented in Merge.c: */
OC_bool pendingchange_mergeFromX(PendingChange*, Repository *r,
			      Change *other);
void pendingchange_addNote(PendingChange*, const char *);
StrVec *pendingchange_build_description(PendingChange *);

#endif /* CLIENT_PENDINGCHANGE_H */
