/*
 * Copyright (c) 2008 Matthieu Herrb <matthieu@herrb.eu>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
#include <sys/types.h>
#include <signal.h>
#include <stdio.h>
#include <string.h>
#include <termios.h>
#include <unistd.h>

#include <ftdi.h>

int done;

void
interrupt(int sig)
{
	done = 1;
}

int main(int argc, char **argv)
{
	char buf[4096];
	struct ftdi_context ftdic;
	struct termios tio, saved_tio;
	size_t n;
	int ret;

	ftdi_init(&ftdic);
	
	if((ret = ftdi_usb_open(&ftdic, 0x1457, 0x5118)) < 0) {
		fprintf(stderr, "unable to open ftdi device: %d (%s)\n", 
		    ret, ftdi_get_error_string(&ftdic));
		exit(2);
	}
	ftdi_set_interface(&ftdic, INTERFACE_B);
	ftdi_set_baudrate(&ftdic, 115200);
	
	tcgetattr(fileno(stdin), &tio);
	memcpy(&saved_tio, &tio, sizeof(struct termios));
	cfmakeraw(&tio);
	tio.c_cc[VMIN] = 0;
	tio.c_cc[VTIME] = 0;
	tio.c_lflag |= ISIG;
	tio.c_cc[VQUIT] = CTRL('\\');
	tio.c_cc[VINTR] = 0;

	tcsetattr(fileno(stdin), TCSAFLUSH, &tio);

	signal(SIGINT, interrupt);
	signal(SIGQUIT, interrupt);

	done = 0;
	while (!done) {
		n = read(fileno(stdin), buf, sizeof(buf));
		if (n > 0)
			ftdi_write_data(&ftdic, buf, n);
		
		n = ftdi_read_data(&ftdic, buf, sizeof(buf));
		if (n < 0) {
			fprintf(stderr, "read error %d (%s)\n",
			    (int)n, ftdi_get_error_string(&ftdic));
			break;
		}
		write(fileno(stdout), buf, n);
	}
	tcsetattr(fileno(stdin), TCSAFLUSH, &saved_tio);
	putchar('\n');
	ftdi_usb_close(&ftdic);
	ftdi_deinit(&ftdic);
	
	return EXIT_SUCCESS;
}
