/*
 * xlibvid.h
 *
 *This file contains the class heirachy for 1)creating windows under xlib.
 *                                          2)displaying data on those windows
 *                                          3)converting image formats.
 *
 * Copyright (c) 1999-2000 Indranet Technologies ltd
 *
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 *
 * The Original Code is Open H323
 *
 * The Initial Developer of the Original Code is Equivalence Pty. Ltd.
 *
 * Contributor(s): Author Derek J Smithies (derek@indranet.co.nz)
 *
 * Note the license of this file was changed to the above with the full
 * permission of the authors.
 *
 * $Log: xlibvid.h,v $
 * Revision 1.13  2003/03/18 07:11:34  robertj
 * Removed openh323 versions of videoio.h classes as PVideoOutputDevice
 *   descendants for NULL and PPM files added to PWLib.
 *
 * Revision 1.12  2002/06/27 02:17:40  robertj
 * Renamed video format 411 to the correct 420P, thanks Mark Cooke
 *
 * Revision 1.11  2001/05/25 01:14:44  dereks
 * Alter SetFrameSize & OpenWindo to use unsigned variables. Change type of
 * other variables to eliminate compiler warnings.
 *
 * Revision 1.10  2000/12/19 22:35:53  dereks
 * Install revised video handling code, so that a video channel is used.
 * Code now better handles the situation where the video grabber could not be opened.
 *
 * Revision 1.9  2000/09/29 02:19:03  craigs
 * Fixed compile problem with Dereks new changes
 *
 * Revision 1.8  2000/09/27 03:06:14  dereks
 * Added lots of PTRACE statements to xlib code.
 * Removed X videoMutex from main.cxx & main.h
 * Removed some weird display issues from X code.
 *
 * Revision 1.7  2000/09/13 23:58:12  dereks
 * Corrected bug in video display. Now correctly handles 8, 16, 32 bit colour
 * Correctly handles 8 bit grayscale.
 *
 * Revision 1.6  2000/08/07 03:47:42  dereks
 * Add picture in picture option (only for  X window display), better handling
 * of X windows. Handles situation where user selects cross on a X window.
 *
 * Revision 1.5  2000/07/03 06:35:20  craigs
 * Added ability for video to work on 24, 16 and 8 bit visuals
 *
 * Revision 1.4  2000/05/02 04:32:25  robertj
 * Fixed copyright notice comment.
 *
 */

#include <X11/X.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/keysym.h>
#include <X11/extensions/XShm.h>
#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/shm.h>
#include <sys/types.h>
#include <sys/ipc.h>
#include <signal.h>
void segvhandler(int signo);

#ifndef __XLIBVID_H
#define __XLIBVID_H

#include <ptlib.h>

/*Each of the publically accessible components in GenericXlibVideoDevice
 *are protected by the mutex videoMutex. 
 *Thus, it is only possible for one task to access the X server at one
 *time. The primary purpose of videoMutex is to save the display routine.
 *Suppose a controlling task closes the xlibvideodevice, which closes the
 *associated memory regions. At the same time, an image is being
 *rendered. The rendering of an image will abort, with a seg v fault.
 *
 *The publically accessible functions which can be used to close windows,
 * the constructor, destructor, Close, Redraw, SetFrameSize are all protected
 * by videoMutex. None are virtual, so a child class cannot change the 
 *definition (and accidentally remove the videoMutex protection.
 *
 *All xlib classes share the same mutex, videoMutex.
 ************/
/**Traps errors generated by the Xwindow server. One can decode the
   error numbers in the fie /usr/X11R6/include/X11/X.h.
   e.g. 9==BadDrawable */
int  XLibErrorHandler(Display *dpy,XErrorEvent *ee);
 

/**Core class for handling requests to the X server.
  */
class GenericXlibVideoDevice : public PVideoOutputDevice
{
  PCLASSINFO(GenericXlibVideoDevice, PVideoOutputDevice);

 public:
  /**Constructor. Does not make a window. Initializes all variables;
   */
  GenericXlibVideoDevice(const PString & _remoteName,BOOL _isEncoding, BOOL _videoPIP);

  /**Destructor.  Closes window if necessary, (which initializes all variables)
   */
  ~GenericXlibVideoDevice();

  /**Open the device given the device name.
    */
  virtual BOOL Open(
    const PString & deviceName,   /// Device name to open
    BOOL startImmediate = TRUE    /// Immediately start device
  );
      
  /**Synonymous with the destructor.
   */
  BOOL Close();
  
  /**Global test function to determine if this video rendering
     class is open.*/
  BOOL IsOpen();
   
  /**Get a list of all of the drivers available.
   */
  virtual PStringList GetDeviceNames() const;

  /**Get the maximum frame size in bytes.
   */
  virtual PINDEX GetMaxFrameBytes();

   /**Set a section of the output frame buffer.
     */
   virtual BOOL SetFrameData(
     unsigned x,
     unsigned y,
     unsigned width,
     unsigned height,
     const BYTE * data,
     BOOL endFrame = TRUE
   );

   /**Indicate frame may be displayed.
    */
   virtual BOOL EndFrame();

   /**Set size of the window. Closes existing window, opens new window.
    */
   BOOL SetFrameSize (unsigned _width ,unsigned _height);

   /**Name of remote computer, which is used for window title.
    */
   virtual PString GetRemoteName() const
     { return remoteName ; }
    
   /**Name of remote computer, which is used for window title.
    */
   virtual void SetRemoteName(const PString & _remoteName)
     { remoteName = _remoteName; }

   /**Specifies required number of bitplanes in the image.
    */
   void ForceDepth(int d)
     { forceDepth = d; }

  protected:
    // new functions

    /** Finds the preselected visual information
     */
    BOOL DetermineVisualInformation(void);

    /**Free the colormap and graphics context associated with a display
     */
    void FreeColormapInformation(void);

    /**Translate YUV420P image into RGB. Order and size of RGB bytes determined 
       previously when the window was opened.
     */
    void Translate420PInput(BYTE *dest, const void *frame);

    /**Allocate memory regions to hold info on pixels. The pixel memory is later 
       used to construct an image, which is then displayed.
     */
    virtual void OpenRender(void) = 0;

    /** Disposes of allocated memory regions. Has to be able to cope with the
        situation of the allocated memory regions already been closed. 
     */
    virtual void CloseRender() = 0;

    /**Display the image which was previously assembled in an image structure.
     */
    virtual void DisplayImage(XImage *xi) = 0;

    /**Open a window of the specified size. This routine refuses to open a window
      if there is one already present.
      This routine endeavous to use the pixeldepth specified with ForceDepth.
      Appropriate calls are made to allocate memory for the images.
     */
    BOOL OpenWindow(unsigned _width = 0, unsigned _height = 0);

    /**Closes the window associated with this task. Frees associated memory.
       This routine has to cope with the following three scenarios:
        a.Window has been closed previously by a separate task, using this routine.
        b.The window we are about to close was a child window (picture in picture).
          But, the parent window was closed with a call to XDestroyWindow, which 
          deleted this window but did not remove associated memory.
        c.Window is up and active.
    */
    void CloseWindow();

    /**Used in determining the gray scale ramp, for those cases when only a
       8 bit psuedo colour display can be opened.
    */
    void AllocGrays(int num);

    /**Used for determining the number of bits in, and position of, a particular
       colour component.*/
    void MaskCalc(int & pos, int & looseNBits, int mask);

    /**Used to determine the appropriate index into the array of windows
     */
    int xIndex(void)
      { return(isEncoding ? 1 : 0); }

    /**Determines the index into the four element pixel/image array.
      */
    int xIndex(int useSecond)
      { return((xIndex()<<1) + (useSecond ? 1 : 0) ); }

    /**Used to determine the index of the receive window.
     */
    int xReceiveIndex(void)
      { return(0); }

    // member variables
    /**wm_delete_window is used to cope with the user selecting the cross at the
       top right of a wndow */

    /** For freeing the render memory and window associated with local or 
        received video */
    void CloseWindowOfType(BOOL closeEncoder);

    inline const char *DirectionStr(void)
      { return(isEncoding?"Encoding":"Receiving"); }

    Atom     wm_delete_window;


    static int           nWindowsOpen;
    static Window        win[2];
    static Display       *display;
    static GC            gc;
    static Visual        *visual; 
    static Colormap      colormap;

    //videoMutex provides protection from when a logical channel attempts
    //to close down the video display, while the video decode task is
    //attempting to convert and render an image.
    static PMutex        videoMutex;

    static int           pixelDepth;
    static int           byteDepth;
    static BOOL          useGrayScale;
    static int           nGrayColors;
    static unsigned long grayColors[64];
    static float         grayScaleFactor;

    static int           redLoose;
    static int           greenLoose;
    static int           blueLoose;
    static int           redPos;
    static int           greenPos;
    static int           bluePos;

    static XImage        *images[4];
    static BYTE          *pixels[4];

    static int           displayWidth[2];
    static int           displayHeight[2];

    int      screen;
    PString  remoteName;
    int      forceDepth;

    

    BOOL     isEncoding;
    BOOL     videoPIP;

    unsigned     n_bytes; //# bytes in Y plane of image (i.e. width*height)
    unsigned     width,height;

    int    useImage0;
};


/**Class to handle shared memory requests
  */
class ShmXlibVideoDevice : public GenericXlibVideoDevice
{
  PCLASSINFO(ShmXlibVideoDevice, GenericXlibVideoDevice);

  public:
  /**Constructor for X displays with shared memory enabled.
     Shared memory is faster at displaying data than non shared */
    ShmXlibVideoDevice(const PString & s, BOOL _isEncoding, BOOL _videoPIP)
      : GenericXlibVideoDevice(s,_isEncoding, _videoPIP) { }

  /**Destructor.  Calls the close routine.
   */
    ~ShmXlibVideoDevice()
      { Close();
      }

  
  protected:
    // overrides from GenericXlibVideoDevice
    /**Allocate Shared memory regions 
      */
    void OpenRender(void);

    /**Delallocate shared memory regions.
     */
    void CloseRender();

    /**Use Shared memory calls to display an image.
     */
    void DisplayImage(XImage *xi);

    // new functions
    void CreateShmXImage (XImage **image,       XShmSegmentInfo *xsi);
    void DestroyShmXImage(XImage **SourceImage, XShmSegmentInfo *xsi);

    // member variables
    static XShmSegmentInfo shminfo[4];
};

/**Display data on X screen, without the shared memory enhancements
 */
class XlibVideoDevice : public GenericXlibVideoDevice
{
  PCLASSINFO(XlibVideoDevice, GenericXlibVideoDevice);

  public:
    XlibVideoDevice(const PString & s,BOOL _isEncoding, BOOL _videoPIP)
      : GenericXlibVideoDevice(s,_isEncoding, _videoPIP) { }

  /**Destructor.  Calls the close routine.
   */
    ~XlibVideoDevice()
      { Close();
      }


  protected:
    // overrides from GenericXlibVideoDevice
    /**Allocate memory for use in displaying camera generated pictures
       and make call to allocate image structure.
      */
    void OpenRender(void);

    /**Free allocated memory for displaying camera generated pictures
       and make call to destroy image.
      */
    void CloseRender();

    /**Use native X calls to display one image structrue.
     */
    void DisplayImage(XImage *xi);

    // new functions
    /** Allocate image structures to display camera generated pictures.
     */
    void CreateXImage (XImage **image, BYTE ** pixels );

    /**Free image structures created for use in displaying camera pictures.
     */
    void DestroyXImage(XImage **SourceImage);
};

////////////////////////////////////////////////////////////////////////////////
#endif
