#include <ctype.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>

#include <sys/types.h>
#include <sys/wait.h>

#include "vobsub.h"
#include "spudec.h"

static char srtxname[FILENAME_MAX];
static char pgmname[FILENAME_MAX];
static spudec_handle_t *spudec;
static unsigned int sub_idx;
static FILE *fsrtx;

int sub_pos;
int vobsub_id;

static void
usage (void)
{
  fprintf (stderr, "subp2pgm [options] <vobsub basename>\n");
  fprintf (stderr, "  -o <filename> - the output base filename\n");
  fprintf (stderr, "  -t <sid>      - the subtitle id, default 0\n");
}

static void
output_srtx (const char *const fname, unsigned int start, unsigned int end)
{
  int temp, h, m, s, ms;

  fprintf (fsrtx, "%d\n", sub_idx);

  temp = start;
  temp /= 90;
  h = temp / 3600000;
  temp %= 3600000;
  m = temp / 60000;
  temp %= 60000;
  s = temp / 1000;
  temp %= 1000;
  ms = temp;

  fprintf (fsrtx, "%02d:%02d:%02d,%03d --> ", h, m, s, ms);

  temp = end;
  temp /= 90;
  h = temp / 3600000;
  temp %= 3600000;
  m = temp / 60000;
  temp %= 60000;
  s = temp / 1000;
  temp %= 1000;
  ms = temp;

  fprintf (fsrtx, "%02d:%02d:%02d,%03d\n", h, m, s, ms);
  fprintf (fsrtx, "%s.txt\n\n", fname);
  fflush (fsrtx);
}

static void
output_pgm (FILE * f, int w, int h, unsigned char *src, unsigned char *srca,
            int stride)
{
  int x, y;
  fprintf (f, "P5\n" "%d %d\n" "255\n", w, h);
  for (y = 0; y < h; ++y)
  {
    for (x = 0; x < w; ++x)
    {
      int res;
      if (srca[x])
        res = src[x] * (256 - srca[x]);
      else
        res = 0;
      res = (65535 - res) >> 8;
      putc (res & 0xff, f);
    }
    src += stride;
    srca += stride;
  }
  putc ('\n', f);
}

static void
draw_alpha (int x0, int y0, int w, int h, unsigned char *src,
            unsigned char *srca, int stride)
{
  char buf[FILENAME_MAX];
  FILE *f;

  sprintf (buf, "%s%04d.pgm", pgmname, ++sub_idx);

  f = fopen (buf, "w");
  output_pgm (f, w, h, src, srca, stride);
  fclose (f);

  output_srtx (buf, spudec->start_pts, spudec->end_pts);
}

int
main (int argc, char **argv)
{
  const char *vobsubname;
  void *vobsub, *packet;
  unsigned int pts100;
  int packet_len, ch;

  if (argc < 2)
  {
    usage ();
    return EXIT_FAILURE;
  }

  while ((ch = getopt (argc, argv, "t:o:h")) != -1)
  {
    switch (ch)
    {
      case 't':
        if (sscanf (optarg, "%d", &vobsub_id) != 1)
        {
          fprintf (stderr, "vobsub_id for -g option\n");
          return EXIT_FAILURE;
        }
        break;
      case 'o':
        if (sscanf (optarg, "%s", pgmname) != 1)
        {
          fprintf (stderr, "no filename specified to option -i\n");
          return EXIT_FAILURE;
        }
        break;
      case 'h':
        usage ();
        return EXIT_SUCCESS;
        break;
      default:
        fprintf (stderr,
                 "Unknown option. Use -h to list all valid options.\n");
        return EXIT_FAILURE;
        break;
    }
  }

  if (optind >= argc)
  {
    usage ();
    return EXIT_FAILURE;
  }

  vobsubname = argv[optind];
  if (pgmname[0] == 0)
    strncpy (pgmname, vobsubname, FILENAME_MAX - 10);

  strncpy (srtxname, pgmname, FILENAME_MAX - 5);
  strncat (srtxname, ".srtx", FILENAME_MAX - 5);

  fsrtx = fopen (srtxname, "w");
  if (!fsrtx)
  {
    fprintf (stderr, "Error opening %s\n", srtxname);
    return EXIT_FAILURE;
  }

  vobsub = vobsub_open (vobsubname, NULL, 0, &spudec);
  if (!vobsub)
  {
    fprintf (stderr, "Error opening vobsubs\n");
    fclose (fsrtx);
    return EXIT_FAILURE;
  }
  if (!spudec)
  {
    fprintf (stderr, "Error creating spudec handle\n");
    vobsub_close (vobsub);
    fclose (fsrtx);
    return EXIT_FAILURE;
  }

  while ((packet_len =
          vobsub_get_next_packet (vobsub, &packet, &pts100)) >= 0)
  {
    spudec_assemble (spudec, packet, packet_len, pts100);
    if (spudec->queue_head)
    {
      spudec_heartbeat (spudec, spudec->queue_head->start_pts);
      if (spudec_changed (spudec))
        spudec_draw (spudec, draw_alpha);
    }
  }

  if (vobsub)
    vobsub_close (vobsub);

  fprintf (stdout, "%u files generated\n", sub_idx);

  fclose (fsrtx);

  return EXIT_SUCCESS;
}
