/* OGMRip - A DVD Encoder for GNOME
 * Copyright (C) 2004-2006 Olivier Rolland <billl@users.sf.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "config.h"

#ifdef HAVE_ENCHANT_SUPPORT

#include "ogmrip-spell.h"
#include "ogmrip-helper.h"

#include <string.h>
#include <enchant.h>
#include <glade/glade.h>

#define OGMRIP_GLADE_FILE "ogmrip/ogmrip-spell.glade"

enum
{
  OGMRIP_SPELL_RESPONSE_NONE       = GTK_RESPONSE_NONE,
  OGMRIP_SPELL_RESPONSE_CANCEL     = GTK_RESPONSE_CANCEL,
  OGMRIP_SPELL_RESPONSE_REPLACE    = -12,
  OGMRIP_SPELL_RESPONSE_IGNORE     = -13,
  OGMRIP_SPELL_RESPONSE_IGNORE_ALL = -14,
  OGMRIP_SPELL_RESPONSE_ADD_WORD   = -15
};

typedef struct
{
  EnchantBroker *broker;
  EnchantDict *dict;

  GtkWidget *dialog;

  GtkTextBuffer *buffer;

  GtkWidget *word_entry;
  GtkWidget *replace_entry;

  GtkTreeSelection *select;
  GtkListStore *word_store;

  const gchar *word;
} DialogData;

static void 
ogmrip_spell_destroyed (DialogData *data)
{
  enchant_broker_free_dict (data->broker, data->dict);
  enchant_broker_free (data->broker);
  g_free (data);
}

static void
ogmrip_spell_changed (DialogData *data, GtkTreeSelection *select)
{
  GtkTreeModel *model;
  GtkTreeIter iter;

  if (gtk_tree_selection_get_selected (select, &model, &iter))
  {
    gchar *text;

    gtk_tree_model_get (model, &iter, 0, &text, -1);
    gtk_entry_set_text (GTK_ENTRY (data->replace_entry), text);

    g_free (text);
  }
}

static void
ogmrip_spell_replace (DialogData *data)
{
  data->word = gtk_entry_get_text (GTK_ENTRY (data->replace_entry));
  gtk_dialog_response (GTK_DIALOG (data->dialog), OGMRIP_SPELL_RESPONSE_REPLACE);
}

static void
ogmrip_spell_ignore (DialogData *data)
{
  data->word = gtk_entry_get_text (GTK_ENTRY (data->word_entry));
  gtk_dialog_response (GTK_DIALOG (data->dialog), OGMRIP_SPELL_RESPONSE_IGNORE);
}

static void
ogmrip_spell_ignore_all (DialogData *data)
{
  data->word = gtk_entry_get_text (GTK_ENTRY (data->word_entry));
  gtk_dialog_response (GTK_DIALOG (data->dialog), OGMRIP_SPELL_RESPONSE_IGNORE_ALL);
}

static void
ogmrip_spell_add_word (DialogData *data)
{
  data->word = gtk_entry_get_text (GTK_ENTRY (data->word_entry));
  gtk_dialog_response (GTK_DIALOG (data->dialog), OGMRIP_SPELL_RESPONSE_ADD_WORD);
}

static void
ogmrip_spell_row_activated (DialogData *data, GtkTreePath *path, 
    GtkTreeViewColumn  *column)
{
  ogmrip_spell_replace (data);
}

void
ogmrip_spell_set_text (DialogData *data, const gchar *text)
{
  gtk_text_buffer_set_text (data->buffer, text, -1);
}

void
ogmrip_spell_set_word (DialogData *data, const gchar *word, gint offset, gchar **suggs, guint n_suggs)
{
  GtkTreeIter iter;
  guint i;

  data->word = NULL;
  gtk_entry_set_text (GTK_ENTRY (data->word_entry), word);
  gtk_editable_delete_text (GTK_EDITABLE (data->replace_entry), 0, -1);

  if (offset > -1)
  {
    GtkTextIter ins, bound;

    gtk_text_buffer_get_iter_at_offset (data->buffer, &ins, offset);
    gtk_text_buffer_get_iter_at_offset (data->buffer, &bound, offset + g_utf8_strlen (word, -1));
    gtk_text_buffer_select_range (data->buffer, &ins, &bound);
  }

  gtk_list_store_clear (data->word_store);

  for (i = 0; i < n_suggs; i++)
  {
    gtk_list_store_append (data->word_store, &iter);
    gtk_list_store_set (data->word_store, &iter, 0, suggs[i], -1);

    if (i == 0)
      gtk_tree_selection_select_iter (data->select, &iter);
  }

  if (gtk_tree_model_get_iter_first (GTK_TREE_MODEL (data->word_store), &iter))
    gtk_tree_selection_select_iter (data->select, &iter);
  else
    gtk_entry_set_text (GTK_ENTRY (data->replace_entry), word);
}

gchar *
ogmrip_spell_get_word (DialogData *data)
{
  return g_strdup (data->word);
}

GtkWidget *
ogmrip_spell_new (const gchar *language)
{
  DialogData *data;

  GtkCellRenderer *renderer;
  GtkTreeViewColumn *column;
  GtkWidget *widget;

  EnchantBroker *broker;
  EnchantDict *dict;

  GladeXML *xml;

  xml = glade_xml_new (OGMRIP_DATA_DIR "/" OGMRIP_GLADE_FILE, NULL, NULL);
  if (!xml)
  {
    g_warning ("Could not find " OGMRIP_GLADE_FILE "\n");
    return NULL;
  }

  broker = enchant_broker_init ();
  dict = enchant_broker_request_dict (broker, language);
  if (!dict)
  {
    enchant_broker_free (broker);
    g_object_unref (xml);

    return NULL;
  }

  data = g_new0 (DialogData, 1);
  data->broker = broker;
  data->dict = dict;

  data->dialog = glade_xml_get_widget (xml, "spell-dialog");
  g_signal_connect_swapped (data->dialog, "destroy", G_CALLBACK (ogmrip_spell_destroyed), data);

  gtk_window_set_icon_from_stock (GTK_WINDOW (data->dialog), GTK_STOCK_SPELL_CHECK);
  g_object_set_data (G_OBJECT (data->dialog), "__dialog_data__", data);

  widget = glade_xml_get_widget (xml, "text-view");
  data->buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (widget));
  data->word_entry = glade_xml_get_widget (xml, "word-entry");
  data->replace_entry = glade_xml_get_widget (xml, "replace-entry");

  widget = glade_xml_get_widget (xml, "replace-button");
  g_signal_connect_swapped (widget, "clicked", G_CALLBACK (ogmrip_spell_replace), data);

  widget = glade_xml_get_widget (xml, "ignore-button");
  g_signal_connect_swapped (widget, "clicked", G_CALLBACK (ogmrip_spell_ignore), data);

  widget = glade_xml_get_widget (xml, "ignore-all-button");
  g_signal_connect_swapped (widget, "clicked", G_CALLBACK (ogmrip_spell_ignore_all), data);

  widget = glade_xml_get_widget (xml, "add-word-button");
  g_signal_connect_swapped (widget, "clicked", G_CALLBACK (ogmrip_spell_add_word), data);

  widget = glade_xml_get_widget (xml, "word-list");
  g_signal_connect_swapped (widget, "row-activated", G_CALLBACK (ogmrip_spell_row_activated), data);

  data->select = gtk_tree_view_get_selection (GTK_TREE_VIEW (widget));
  g_signal_connect_swapped (data->select, "changed", G_CALLBACK (ogmrip_spell_changed), data);

  data->word_store = gtk_list_store_new (1, G_TYPE_STRING);
  gtk_tree_view_set_model (GTK_TREE_VIEW (widget), GTK_TREE_MODEL (data->word_store));
  g_object_unref (data->word_store);

  renderer = gtk_cell_renderer_text_new ();
  column = gtk_tree_view_column_new_with_attributes ("Word", renderer, "text", 0, NULL);
  gtk_tree_view_append_column (GTK_TREE_VIEW (widget), column);

  g_object_unref (xml);

  return data->dialog;
}

gboolean
ogmrip_spell_check_word (GtkWidget *checker, const gchar *word, gint offset, gchar **corrected)
{
  DialogData *data;
  gboolean status = TRUE;
  size_t len;

  g_return_val_if_fail (checker != NULL, FALSE);
  g_return_val_if_fail (word != NULL, FALSE);
  g_return_val_if_fail (corrected != NULL, FALSE);

  data = g_object_get_data (G_OBJECT (checker), "__dialog_data__");
  g_return_val_if_fail (data != NULL, FALSE);

  *corrected = NULL;
  len = strlen (word);

  if (len && enchant_dict_check (data->dict, word, len))
  {
    gchar **suggs;
    size_t n_suggs;

    suggs = enchant_dict_suggest (data->dict, word, len, &n_suggs);
    ogmrip_spell_set_word (data, word, offset, suggs, n_suggs);
    switch (gtk_dialog_run (GTK_DIALOG (data->dialog)))
    {
      case OGMRIP_SPELL_RESPONSE_NONE:
      case OGMRIP_SPELL_RESPONSE_CANCEL:
        status = FALSE;
        break;
      case OGMRIP_SPELL_RESPONSE_REPLACE:
        *corrected = ogmrip_spell_get_word (data);
        break;
      case OGMRIP_SPELL_RESPONSE_IGNORE_ALL:
        enchant_dict_add_to_session (data->dict, word, len);
        break;
      case OGMRIP_SPELL_RESPONSE_ADD_WORD:
        enchant_dict_add_to_personal (data->dict, word, len);
        break;
      default:
        break;
    }

    if (suggs && n_suggs)
      enchant_dict_free_suggestions (data->dict, suggs);
  }

  return status;
}

gboolean
ogmrip_spell_check_text (GtkWidget *checker, const gchar *text, gchar **corrected)
{
  GString *string;
  DialogData *data;
  gchar *start, *end, *word, *cw;
  gunichar ch, underscore;
  guint offset;

  g_return_val_if_fail (checker != NULL, FALSE);
  g_return_val_if_fail (text != NULL, FALSE);
  g_return_val_if_fail (corrected != NULL, FALSE);

  data = g_object_get_data (G_OBJECT (checker), "__dialog_data__");
  g_return_val_if_fail (data != NULL, FALSE);

  ogmrip_spell_set_text (data, text);

  offset = 0;
  string = g_string_new (NULL);
  start = end = (gchar *) text;
  underscore = g_utf8_get_char ("_");

  while (*start)
  {
    ch = g_utf8_get_char (end);
    if (!g_unichar_isalpha (ch) && ch != underscore)
    {
      if (*start)
      {
        word = g_strndup (start, end - start);
        if (ogmrip_spell_check_word (checker, word, offset, &cw))
        {
          g_string_append (string, cw ? cw : word);
          g_free (cw);
        }
        else
        {
          g_string_free (string, TRUE);
          return FALSE;
        }
        g_free (word);
      }

      offset += g_utf8_strlen (start, end - start);

      while (*end && !g_unichar_isalpha (g_utf8_get_char (end)))
      {
        g_string_append_unichar (string, g_utf8_get_char (end));
        end = g_utf8_next_char (end);
        offset ++;
      }
      start = end;
    }
    else
      end = g_utf8_next_char (end);
  }

  *corrected = g_string_free (string, FALSE);

  return TRUE;
}

#endif

