/* OGMRip - A DVD Encoder for GNOME
 * Copyright (C) 2004-2006 Olivier Rolland <billl@users.sf.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "config.h"

#include "ogmdvd.h"

#include "ogmrip-fs.h"
#include "ogmrip-pref.h"
#include "ogmrip-gconf.h"
#include "ogmrip-helper.h"
#include "ogmrip-init.h"
#include "ogmrip-xml.h"

#include <string.h>
#include <glib/gi18n.h>
#include <glade/glade.h>

#define OGMRIP_GLADE_FILE "ogmrip/ogmrip-pref.glade"

typedef struct
{
  GtkWidget *container_combo;
  GtkWidget *spell_check;

  GtkWidget *video_table;
  GList *video_radio_list;

  GtkWidget *audio_table;
  GList *audio_radio_list;

  GtkWidget *subp_table;
  GList *subp_radio_list;

  GConfClient *client;
} DialogData;

extern gchar *ogmdvd_languages[][3];
extern guint  ogmdvd_nlanguages;

static void
ogmrip_pref_destroyed (DialogData *data)
{
  g_list_free (data->video_radio_list);
  g_list_free (data->audio_radio_list);
  g_list_free (data->subp_radio_list);

  g_object_unref (data->client);

  g_free (data);
}

static void
ogmrip_pref_set_active_lang (GtkComboBox *combo, gint code)
{
  guint i = 0;

  if (code)
  {
    const gchar *lang;

    lang = ogmdvd_get_language_iso639_1 (code);

    for (i = 1; i < ogmdvd_nlanguages; i++)
      if (strcmp (ogmdvd_languages[i + 1][OGMDVD_LANGUAGE_ISO639_1], lang) == 0)
        break;

    if (i == ogmdvd_nlanguages)
      i = 0;
  }

  gtk_combo_box_set_active (GTK_COMBO_BOX (combo), i);
}

static gint
ogmrip_pref_get_active_lang (GtkComboBox *combo)
{
  const gchar *lang;
  guint i;

  i = gtk_combo_box_get_active (combo);
  if (i > 0 && i + 1 < ogmdvd_nlanguages)
  {
    lang = ogmdvd_languages[i + 1][OGMDVD_LANGUAGE_ISO639_1];
    return (lang[0] << 8) | lang[1];
  }

  return 0;
}

static void
ogmrip_pref_set_output_dir (GConfClient *client, GtkWidget *chooser, const gchar *path)
{
  gchar *old_path, *new_path;

  if (path)
    new_path = g_strdup (path);
  else
    new_path = gconf_client_get_string_default (client, OGMRIP_GCONF_OUTPUT_DIR, OGMRIP_DEFAULT_OUTPUT_DIR);

  old_path = gtk_file_chooser_get_current_folder (GTK_FILE_CHOOSER (chooser));
  if (!old_path || strcmp (new_path, old_path) != 0)
    gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER (chooser), new_path);

  g_free (new_path);
  g_free (old_path);
}

static void
ogmrip_pref_set_tmp_dir (GConfClient *client, GtkWidget *chooser, const gchar *path)
{
  gchar *old_path, *new_path;

  if (path)
    new_path = g_strdup (path);
  else
    new_path = gconf_client_get_string_default (client, OGMRIP_GCONF_TMP_DIR, OGMRIP_DEFAULT_TMP_DIR);

  ogmrip_fs_set_tmp_dir (new_path);

  old_path = gtk_file_chooser_get_current_folder (GTK_FILE_CHOOSER (chooser));
  if (!old_path || strcmp (new_path, old_path) != 0)
    gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER (chooser), new_path);

  g_free (new_path);
  g_free (old_path);
}

static void
ogmrip_pref_codec_select_first (GList *list)
{
  GList *link;

  for (link = list; link; link = link->next)
    if (GTK_WIDGET_IS_SENSITIVE (link->data))
      break;

  if (link)
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (link->data), TRUE);
}

static void
ogmrip_pref_check_container_format (DialogData *data, gint active)
{
  GList *link;
  GType container, codec;
  gboolean can_contain, select_other;
  gint i;

  if (active < 0)
    active = gconf_client_get_int_default (data->client, OGMRIP_GCONF_CONTAINER, OGMRIP_DEFAULT_CONTAINER);

  container = ogmrip_get_nth_container (active);

  for (i = 0, select_other = 0, link = data->video_radio_list; link; i++, link = link->next)
  {
    codec = ogmrip_get_nth_video_codec (i);
    can_contain = ogmrip_can_contain_video (container, codec);
    gtk_widget_set_sensitive (GTK_WIDGET (link->data), can_contain);
    select_other = select_other || (!can_contain && gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (link->data)));
  }

  if (select_other)
    ogmrip_pref_codec_select_first (data->video_radio_list);

  for (i = 0, select_other = 0, link = data->audio_radio_list; link; i++, link = link->next)
  {
    codec = ogmrip_get_nth_audio_codec (i);
    can_contain = ogmrip_can_contain_audio (container, codec);
    gtk_widget_set_sensitive (GTK_WIDGET (link->data), can_contain);
    select_other = select_other || (!can_contain && gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (link->data)));
  }

  if (select_other)
    ogmrip_pref_codec_select_first (data->audio_radio_list);

  for (i = 0, select_other = 0, link = data->subp_radio_list; link; i++, link = link->next)
  {
    codec = ogmrip_get_nth_subp_codec (i);
    can_contain = ogmrip_can_contain_subp (container, codec);
    gtk_widget_set_sensitive (GTK_WIDGET (link->data), can_contain);
    select_other = select_other || (!can_contain && gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (link->data)));
  }

  if (select_other)
    ogmrip_pref_codec_select_first (data->subp_radio_list);
}

static void
ogmrip_pref_check_subp_format (DialogData *data, gint id)
{
  GType codec;

  if (id < 0)
    id = gconf_client_get_int_default (data->client, OGMRIP_GCONF_SUBP_FORMAT, OGMRIP_DEFAULT_SUBP_FORMAT);

  codec = ogmrip_get_nth_subp_codec (id);
  if (codec != G_TYPE_NONE)
    gtk_widget_set_sensitive (data->spell_check, ogmrip_get_subp_codec_text (codec));
}

/*
 * Signals
 */

static void
ogmrip_pref_chooser_changed (GtkWidget *button, GConfClient *client)
{
  gchar *key;

  if ((key = g_object_get_data (G_OBJECT (button), "__gconf_key__")))
  {
    gchar *path;

    path = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (button));
    if (path)
      gconf_client_set_string (client, key, path, NULL);
    g_free (path);
  }
}

static void
ogmrip_pref_combo_changed (GtkWidget *combo, GConfClient *client)
{
  gchar *key;

  if ((key = g_object_get_data (G_OBJECT (combo), "__gconf_key__")))
    gconf_client_set_int (client, key, gtk_combo_box_get_active (GTK_COMBO_BOX (combo)), NULL);
}

static void
ogmrip_pref_spin_changed (GtkWidget *spin, GConfClient *client)
{
  gchar *key;

  if ((key = g_object_get_data (G_OBJECT (spin), "__gconf_key__")))
    gconf_client_set_int (client, key, gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (spin)), NULL);
}

static void
ogmrip_pref_toggle_changed (GtkWidget *toggle, GConfClient *client)
{
  gchar *key;

  if ((key = g_object_get_data (G_OBJECT (toggle), "__gconf_key__")))
    gconf_client_set_bool (client, key, gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (toggle)), NULL);
}

static void
ogmrip_pref_radio_changed (GtkWidget *radio, GConfClient *client)
{
  if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (radio)))
  {
    gchar *key;

    if ((key = g_object_get_data (G_OBJECT (radio), "__gconf_key__")))
      gconf_client_set_int (client, key, gtk_radio_button_get_active (GTK_RADIO_BUTTON (radio)), NULL);
  }
}

static void
ogmrip_pref_lang_changed (GtkWidget *combo, GConfClient *client)
{
  gchar *key;

  if ((key = g_object_get_data (G_OBJECT (combo), "__gconf_key__")))
    gconf_client_set_int (client, key, ogmrip_pref_get_active_lang (GTK_COMBO_BOX (combo)), NULL);
}

/*
 * Notifications
 */

static void
ogmrip_pref_combo_notified (GConfClient *client, guint id, GConfEntry *entry, GtkComboBox *combo)
{
  GtkTreeModel *model;

  model = gtk_combo_box_get_model (combo);
  gtk_combo_box_set_active (combo, 
      CLAMP (gconf_value_get_int (entry->value), 0, gtk_tree_model_iter_n_children (model, NULL) - 1));
}

static void
ogmrip_pref_spin_notified (GConfClient *client, guint id, GConfEntry *entry, GtkSpinButton *spin)
{
  gtk_spin_button_set_value (spin, 
      CLAMP (gconf_value_get_int (entry->value), spin->adjustment->lower, spin->adjustment->upper));
}

static void
ogmrip_pref_toggle_notified (GConfClient *client, guint id, GConfEntry *entry, GtkToggleButton *toggle)
{
  gtk_toggle_button_set_active (toggle, gconf_value_get_bool (entry->value));
}

static void
ogmrip_pref_radio_notified (GConfClient *client, guint id, GConfEntry *entry, GtkRadioButton *radio)
{
  GSList *group;

  group = gtk_radio_button_get_group (radio);
  gtk_radio_button_set_active (radio, 
      CLAMP (gconf_value_get_int (entry->value), 0, g_slist_length (group) - 1));
}

static void
ogmrip_pref_lang_notified (GConfClient *client, guint id, GConfEntry *entry, GtkComboBox *combo)
{
  GtkTreeModel *model;

  model = gtk_combo_box_get_model (combo);
  ogmrip_pref_set_active_lang (combo, gconf_value_get_int (entry->value));
}

static void
ogmrip_pref_container_format_notified (GConfClient *client, guint id, GConfEntry *entry, DialogData *data)
{
  ogmrip_pref_check_container_format (data, gconf_value_get_int (entry->value));
}

static void
ogmrip_pref_subp_format_notified (GConfClient *client, guint id, GConfEntry *entry, DialogData *data)
{
  ogmrip_pref_check_subp_format (data, gconf_value_get_int (entry->value));
}

static void
ogmrip_pref_output_dir_notified (GConfClient *client, guint id, GConfEntry *entry, GtkWidget *chooser)
{
  const gchar *path = NULL;

  if (entry->value)
    path = gconf_value_get_string (entry->value);

  ogmrip_pref_set_output_dir (client, chooser, path);
}

static void
ogmrip_pref_tmp_dir_notified (GConfClient *client, guint id, GConfEntry *entry, GtkWidget *chooser)
{
  const gchar *path = NULL;

  if (entry->value)
    path = gconf_value_get_string (entry->value);

  ogmrip_pref_set_tmp_dir (client, chooser, path);
}

static void
ogmrip_pref_chooser (DialogData *data, GtkWidget *widget, const gchar *key)
{
  if (key)
    g_object_set_data (G_OBJECT (widget), "__gconf_key__", (gpointer) key);

  g_signal_connect (widget, "selection-changed", G_CALLBACK (ogmrip_pref_chooser_changed), data->client);
}

static void
ogmrip_pref_combo (DialogData *data, GtkWidget *widget, const gchar *key, gint value)
{
  GtkTreeModel *model;
  gint active, nchild;

  if (key)
    g_object_set_data (G_OBJECT (widget), "__gconf_key__", (gpointer) key);

  model = gtk_combo_box_get_model (GTK_COMBO_BOX (widget));
  nchild = gtk_tree_model_iter_n_children (model, NULL);

  active = gconf_client_get_int_default (data->client, key, value);
  if (active >= nchild)
  {
    gconf_client_set_int (data->client, key, 0, NULL);
    active = 0;
  }

  gtk_combo_box_set_active (GTK_COMBO_BOX (widget), active);
  g_signal_connect (widget, "changed", G_CALLBACK (ogmrip_pref_combo_changed), data->client);
  gconf_client_notify_add (data->client, key,
      (GConfClientNotifyFunc) ogmrip_pref_combo_notified, widget, NULL, NULL);
}

static void
ogmrip_pref_spin (DialogData *data, GtkWidget *widget, const gchar *key, gint value)
{
  if (key)
    g_object_set_data (G_OBJECT (widget), "__gconf_key__", (gpointer) key);

  gtk_spin_button_set_value (GTK_SPIN_BUTTON (widget),
      gconf_client_get_int_default (data->client, key, value));
  g_signal_connect (widget, "value-changed", G_CALLBACK (ogmrip_pref_spin_changed), data->client);
  gconf_client_notify_add (data->client, key,
      (GConfClientNotifyFunc) ogmrip_pref_spin_notified, widget, NULL, NULL);
}

static void
ogmrip_pref_toggle (DialogData *data, GtkWidget *widget, const gchar *key, gint value)
{
  if (key)
    g_object_set_data (G_OBJECT (widget), "__gconf_key__", (gpointer) key);

  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (widget),
      gconf_client_get_bool_default (data->client, key, value));
  g_signal_connect (widget, "toggled", G_CALLBACK (ogmrip_pref_toggle_changed), data->client);
  gconf_client_notify_add (data->client, key,
      (GConfClientNotifyFunc) ogmrip_pref_toggle_notified, widget, NULL, NULL);
}

static void
ogmrip_pref_radio (DialogData *data, GtkWidget *widget, const gchar *key, gint value, gboolean active)
{
  if (key)
    g_object_set_data (G_OBJECT (widget), "__gconf_key__", (gpointer) key);

  g_signal_connect (widget, "toggled", G_CALLBACK (ogmrip_pref_radio_changed), data->client);

  if (active)
  {
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (widget), TRUE);
    gconf_client_notify_add (data->client, key,
        (GConfClientNotifyFunc) ogmrip_pref_radio_notified, widget, NULL, NULL);
  }
}

static void
ogmrip_pref_lang (DialogData *data, GtkWidget *widget, const gchar *key, gint value)
{
  if (key)
    g_object_set_data (G_OBJECT (widget), "__gconf_key__", (gpointer) key);

  ogmrip_pref_set_active_lang (GTK_COMBO_BOX (widget),
      gconf_client_get_int_default (data->client, key, value));
  g_signal_connect (widget, "changed", G_CALLBACK (ogmrip_pref_lang_changed), data->client);
  gconf_client_notify_add (data->client, key,
      (GConfClientNotifyFunc) ogmrip_pref_lang_notified, widget, NULL, NULL);
}

static void
ogmrip_pref_append_container (GType type, const gchar *name, const gchar *description, gpointer widget)
{
  gtk_combo_box_append_text (GTK_COMBO_BOX (widget), description);
}

static GtkWidget *
ogmrip_pref_append_codec (GtkTable *table, const gchar *description)
{
  static GSList *group = NULL;
  static guint row = 0, col = 0;
  GtkWidget *radio;

  if (!table->children)
  {
    group = NULL;
    col = row = 0;
  }

  radio = gtk_radio_button_new_with_label (group, description);
  gtk_table_attach (table, radio, col, col + 1, row, row + 1, GTK_FILL, 0, 0, 0);
  gtk_widget_show (radio);

  group = gtk_radio_button_get_group (GTK_RADIO_BUTTON (radio));

  col = (col + 1) % 2;
  if (!col)
    row ++;

  return radio;
}

static void
ogmrip_pref_append_video_codec (GType type, gchar *name, gchar *description, DialogData *data)
{
  GtkWidget *radio;

  radio = ogmrip_pref_append_codec (GTK_TABLE (data->video_table), description);
  data->video_radio_list = g_list_append (data->video_radio_list, radio);
}

static void
ogmrip_pref_append_audio_codec (GType type, gchar *name, gchar *description, DialogData *data)
{
  GtkWidget *radio;

  radio = ogmrip_pref_append_codec (GTK_TABLE (data->audio_table), description);
  data->audio_radio_list = g_list_append (data->audio_radio_list, radio);
}

static void
ogmrip_pref_append_subp_codec (GType type, gchar *name, gchar *description, DialogData *data)
{
  GtkWidget *radio;

  radio = ogmrip_pref_append_codec (GTK_TABLE (data->subp_table), description);
  data->subp_radio_list = g_list_append (data->subp_radio_list, radio);
}

static void
ogmrip_pref_codec (DialogData *data, GtkWidget *widget, guint type, const gchar *key, gint value)
{
  GList *children, *child;
  gint n, def;

  switch (type)
  {
    case 0:
      n = ogmrip_get_n_video_codecs ();
      break;
    case 1:
      n = ogmrip_get_n_audio_codecs ();
      break;
    default:
      n = ogmrip_get_n_subp_codecs ();
      break;
  }

  gtk_table_resize (GTK_TABLE (widget), (n + 1) / 2, 2);
  gtk_widget_set_sensitive (widget, n > 0);

  switch (type)
  {
    case 0:
      ogmrip_foreach_video_codec ((OGMRipXmlFunc) ogmrip_pref_append_video_codec, data);
      children = data->video_radio_list;
      break;
    case 1:
      ogmrip_foreach_audio_codec ((OGMRipXmlFunc) ogmrip_pref_append_audio_codec, data);
      children = data->audio_radio_list;
      break;
    default:
      ogmrip_foreach_subp_codec ((OGMRipXmlFunc) ogmrip_pref_append_subp_codec, data);
      children = data->subp_radio_list;
      break;
  }

  def = gconf_client_get_int_default (data->client, key, value);
  if (def >= n)
  {
    gconf_client_set_int (data->client, key, 0, NULL);
    def = 0;
  }

  for (child = children, n = 0; child; child = child->next, n++)
    ogmrip_pref_radio (data, GTK_WIDGET (child->data), key, value, n == def);
}

GtkWidget *
ogmrip_pref_new (void)
{
  DialogData *data;
  GtkWidget *dialog;
  GtkWidget *widget;
  GladeXML *xml;

  gchar *str;
  gint i, n;

  xml = glade_xml_new (OGMRIP_DATA_DIR "/" OGMRIP_GLADE_FILE, NULL, NULL);
  if (!xml)
  {
    g_warning ("Could not find " OGMRIP_GLADE_FILE);
    return NULL;
  }

  data = g_new0 (DialogData, 1);
  data->client = gconf_client_get_default ();

  dialog = glade_xml_get_widget (xml, "pref-dialog");
  gtk_window_set_icon_from_stock (GTK_WINDOW (dialog), GTK_STOCK_PREFERENCES);

  g_signal_connect (dialog, "delete-event", G_CALLBACK (gtk_true), NULL);
  g_signal_connect_after (dialog, "response", G_CALLBACK (gtk_widget_hide), NULL);
  g_signal_connect_swapped (dialog, "destroy", G_CALLBACK (ogmrip_pref_destroyed), data);

  widget = glade_xml_get_widget (xml, "output-dir-chooser");
  ogmrip_pref_chooser (data, widget, OGMRIP_GCONF_OUTPUT_DIR);

  gconf_client_notify_add (data->client, OGMRIP_GCONF_OUTPUT_DIR, 
      (GConfClientNotifyFunc) ogmrip_pref_output_dir_notified, widget, NULL, NULL);
  ogmrip_pref_set_output_dir (data->client, widget, NULL);
  gtk_file_chooser_set_action (GTK_FILE_CHOOSER (widget), 
      GTK_FILE_CHOOSER_ACTION_SELECT_FOLDER);

  widget = glade_xml_get_widget (xml, "tmp-dir-chooser");
  ogmrip_pref_chooser (data, widget, OGMRIP_GCONF_TMP_DIR);

  gconf_client_notify_add (data->client, OGMRIP_GCONF_TMP_DIR, 
      (GConfClientNotifyFunc) ogmrip_pref_tmp_dir_notified, widget, NULL, NULL);
  ogmrip_pref_set_tmp_dir (data->client, widget, NULL);
  gtk_file_chooser_set_action (GTK_FILE_CHOOSER (widget), 
      GTK_FILE_CHOOSER_ACTION_SELECT_FOLDER);

  widget = glade_xml_get_widget (xml, "filename-combo");
  ogmrip_pref_combo (data, widget, OGMRIP_GCONF_FILENAME, OGMRIP_DEFAULT_FILENAME);

  n = ogmrip_get_n_containers ();

  data->container_combo = glade_xml_get_widget (xml, "container-combo");
  gtk_combo_box_remove_text (GTK_COMBO_BOX (data->container_combo), 0);
  ogmrip_foreach_container (ogmrip_pref_append_container, data->container_combo);
  gtk_widget_set_sensitive (data->container_combo, n > 0);

  ogmrip_pref_combo (data, data->container_combo, OGMRIP_GCONF_CONTAINER, OGMRIP_DEFAULT_CONTAINER);

  widget = glade_xml_get_widget (xml, "fourcc-combo");
  ogmrip_pref_combo (data, widget, OGMRIP_GCONF_FOURCC, OGMRIP_DEFAULT_FOURCC);

  widget = glade_xml_get_widget (xml, "tnumber-spin");
  ogmrip_pref_spin (data, widget, OGMRIP_GCONF_TNUMBER, OGMRIP_DEFAULT_TNUMBER);

  widget = glade_xml_get_widget (xml, "tsize-spin");
  ogmrip_pref_spin (data, widget, OGMRIP_GCONF_TSIZE, OGMRIP_DEFAULT_TSIZE);

  widget = glade_xml_get_widget (xml, "copy-dvd-check");
  ogmrip_pref_toggle (data, widget, OGMRIP_GCONF_COPY_DVD, OGMRIP_DEFAULT_COPY_DVD);

  widget = glade_xml_get_widget (xml, "keep-tmp-check");
  ogmrip_pref_toggle (data, widget, OGMRIP_GCONF_KEEP_TEMP, OGMRIP_DEFAULT_KEEP_TEMP);

  widget = glade_xml_get_widget (xml, "ensure-sync-check");
  ogmrip_pref_toggle (data, widget, OGMRIP_GCONF_ENSURE_SYNC, OGMRIP_DEFAULT_ENSURE_SYNC);

  data->video_table = glade_xml_get_widget (xml, "video-table");
  ogmrip_pref_codec (data, data->video_table, 0, OGMRIP_GCONF_VIDEO_FORMAT, OGMRIP_DEFAULT_VIDEO_FORMAT);

  widget = glade_xml_get_widget (xml, "quality-combo");
  ogmrip_pref_combo (data, widget, OGMRIP_GCONF_VIDEO_QUALITY, OGMRIP_DEFAULT_VIDEO_QUALITY);

  widget = glade_xml_get_widget (xml, "passes-combo");
  ogmrip_pref_combo (data, widget, OGMRIP_GCONF_VIDEO_PASSES, OGMRIP_DEFAULT_VIDEO_PASSES);

  widget = glade_xml_get_widget (xml, "scaler-combo");
  ogmrip_pref_combo (data, widget, OGMRIP_GCONF_VIDEO_SCALER, OGMRIP_DEFAULT_VIDEO_SCALER);

  widget = glade_xml_get_widget (xml, "trellis-check");
  ogmrip_pref_toggle (data, widget, OGMRIP_GCONF_VIDEO_TRELLIS, OGMRIP_DEFAULT_VIDEO_TRELLIS);

  widget = glade_xml_get_widget (xml, "qpel-check");
  ogmrip_pref_toggle (data, widget, OGMRIP_GCONF_VIDEO_QPEL, OGMRIP_DEFAULT_VIDEO_QPEL);

  widget = glade_xml_get_widget (xml, "turbo-check");
  ogmrip_pref_toggle (data, widget, OGMRIP_GCONF_VIDEO_TURBO, OGMRIP_DEFAULT_VIDEO_TURBO);

  widget = glade_xml_get_widget (xml, "denoise-check");
  ogmrip_pref_toggle (data, widget, OGMRIP_GCONF_VIDEO_DENOISE, OGMRIP_DEFAULT_VIDEO_DENOISE);

  widget = glade_xml_get_widget (xml, "pref-audio-combo");
  for (i = 2; i < ogmdvd_nlanguages; i++)
  {
    str = g_strdup_printf ("%s (%s)", ogmdvd_languages[i][OGMDVD_LANGUAGE_NAME], 
        ogmdvd_languages[i][OGMDVD_LANGUAGE_ISO639_1]);
    gtk_combo_box_append_text (GTK_COMBO_BOX (widget), str);
    g_free (str);
  }
  ogmrip_pref_lang (data, widget, OGMRIP_GCONF_PREF_AUDIO, OGMRIP_DEFAULT_PREF_AUDIO);

  data->audio_table = glade_xml_get_widget (xml, "audio-table");
  ogmrip_pref_codec (data, data->audio_table, 1, OGMRIP_GCONF_AUDIO_FORMAT, OGMRIP_DEFAULT_AUDIO_FORMAT);

  widget = glade_xml_get_widget (xml, "quality-spin");
  ogmrip_pref_spin (data, widget, OGMRIP_GCONF_AUDIO_QUALITY, OGMRIP_DEFAULT_AUDIO_QUALITY);

  widget = glade_xml_get_widget (xml, "normalize-check");
  ogmrip_pref_toggle (data, widget, OGMRIP_GCONF_AUDIO_NORMALIZE, OGMRIP_DEFAULT_AUDIO_NORMALIZE);

  widget = glade_xml_get_widget (xml, "pref-subp-combo");
  for (i = 2; i < ogmdvd_nlanguages; i++)
  {
    str = g_strdup_printf ("%s (%s)", ogmdvd_languages[i][OGMDVD_LANGUAGE_NAME], 
        ogmdvd_languages[i][OGMDVD_LANGUAGE_ISO639_1]);
    gtk_combo_box_append_text (GTK_COMBO_BOX (widget), str);
    g_free (str);
  }
  ogmrip_pref_lang (data, widget, OGMRIP_GCONF_PREF_SUBP, OGMRIP_DEFAULT_PREF_SUBP);

  data->subp_table = glade_xml_get_widget (xml, "subp-table");
  ogmrip_pref_codec (data, data->subp_table, 2, OGMRIP_GCONF_SUBP_FORMAT, OGMRIP_DEFAULT_SUBP_FORMAT);

  data->spell_check = glade_xml_get_widget (xml, "spell-check");
  ogmrip_pref_toggle (data, data->spell_check, OGMRIP_GCONF_SPELL_CHECK, OGMRIP_DEFAULT_SPELL_CHECK);

#if HAVE_SRT_SUPPORT && HAVE_ENCHANT_SUPPORT
  gtk_widget_set_sensitive (widget, TRUE);
#endif

  gconf_client_notify_add (data->client, OGMRIP_GCONF_CONTAINER,
      (GConfClientNotifyFunc) ogmrip_pref_container_format_notified, data, NULL, NULL);
  ogmrip_pref_check_container_format (data, -1);

  gconf_client_notify_add (data->client, OGMRIP_GCONF_SUBP_FORMAT,
      (GConfClientNotifyFunc) ogmrip_pref_subp_format_notified, data, NULL, NULL);
  ogmrip_pref_check_subp_format (data, -1);

  g_object_unref (xml);

  return dialog;
}

