/* OGMRip - A DVD Encoder for GNOME
 * Copyright (C) 2004-2006 Olivier Rolland <billl@users.sf.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "config.h"

#include "ogmdvd.h"

#include "ogmrip-options.h"
#include "ogmrip-helper.h"
#include "ogmrip-crop.h"
#include "ogmrip-lavc.h"

#include <glib/gi18n.h>
#include <glade/glade.h>

#define OGMRIP_GLADE_FILE "ogmrip/ogmrip-options.glade"

#define ROUND(x) ((gint) ((x) + 0.5) != (gint) (x) ? ((gint) ((x) + 0.5)) : ((gint) (x)))

enum
{
  OGMRIP_SCALE_XSMALL,
  OGMRIP_SCALE_SMALL,
  OGMRIP_SCALE_MEDIUM,
  OGMRIP_SCALE_LARGE,
  OGMRIP_SCALE_XLARGE,
  OGMRIP_SCALE_FULL,
  OGMRIP_SCALE_USER
};

typedef struct
{
  OGMDvdTitle *title;

  GtkWidget *dialog;

  GtkWidget *bitrate_check;
  GtkWidget *bitrate_spin;

  GtkWidget *crop_check;
  GtkWidget *crop_left_label;
  GtkWidget *crop_right_label;
  GtkWidget *crop_top_label;
  GtkWidget *crop_bottom_label;

  GtkWidget *scale_check;
  GtkWidget *scale_combo;
  GtkWidget *scale_user_hbox;
  GtkWidget *scale_width_spin;
  GtkWidget *scale_height_spin;
  GtkWidget *autoscale_button;
  GtkListStore *scale_store;

  GtkWidget *more_exp;
  GtkWidget *ntsc_frame;

  GtkWidget *deint_combo;
  GtkWidget *framestep_combo;
  GtkWidget *progressive_check;
  GtkWidget *pullup_check;
} DialogData;

static gint
ogmrip_options_get_crop_internal (DialogData *data, guint *x, guint *y, guint *width, guint *height)
{
  gint retval;

  retval = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->crop_check)) ?
    OGMRIP_OPTIONS_AUTOMATIC : OGMRIP_OPTIONS_MANUAL;

  *x = *y = 0;
  ogmdvd_title_get_size (data->title, width, height);

  if (retval == OGMRIP_OPTIONS_MANUAL)
  {
    *x = gtk_label_get_int (GTK_LABEL (data->crop_left_label));
    *y = gtk_label_get_int (GTK_LABEL (data->crop_top_label));
    *width -= *x + gtk_label_get_int (GTK_LABEL (data->crop_right_label));
    *height -= *y + gtk_label_get_int (GTK_LABEL (data->crop_bottom_label));
  }

  return retval;
}

static gdouble
ogmrip_options_get_aspect (DialogData *data, guint crop_width, guint crop_height)
{
  guint raw_width, raw_height, numerator, denominator;

  ogmdvd_title_get_size (data->title, &raw_width, &raw_height);

  ogmdvd_title_get_aspect_ratio (data->title, &numerator, &denominator);

  return (crop_width * raw_height * numerator) / (gdouble) (crop_height * raw_width * denominator);
}

static void
ogmrip_options_scale (guint size, gdouble aspect, guint *width, guint *height)
{
  const struct { guint mult; guint div; } table[] = 
  { 
    { 1, 8 }, { 1, 4 }, { 1, 2 }, { 3, 4 }, { 5, 6 }, { 1, 1 } 
  };

  *width = *width * table[size].mult / table[size].div;
  *width += *width % 2;

  *width = 16 * ROUND (*width / 16.);
  *height = 16 * ROUND (*width / aspect / 16);
}

static void
ogmrip_options_scale_update (DialogData *data)
{
  GtkTreeIter iter;
  guint scale_width, scale_height;
  guint crop_x, crop_y, crop_width, crop_height;
  gfloat aspect;
  gchar text[16];
  gint i, active;

  ogmrip_options_get_crop_internal (data, &crop_x, &crop_y, &crop_width, &crop_height);
  aspect = ogmrip_options_get_aspect (data, crop_width, crop_height);

  gtk_spin_button_set_range (GTK_SPIN_BUTTON (data->scale_width_spin), 2, crop_width);
  gtk_spin_button_set_range (GTK_SPIN_BUTTON (data->scale_height_spin), 2, crop_height);

  active = gtk_combo_box_get_active (GTK_COMBO_BOX (data->scale_combo));

  for (i = 0; i < OGMRIP_SCALE_USER; i++)
  {
    if (gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (data->scale_store), &iter, NULL, i))
    {
      scale_width = crop_width;
      scale_height = crop_height;

      ogmrip_options_scale (i, aspect, &scale_width, &scale_height);

      snprintf (text, 16, "%u x %u", scale_width, scale_height);
      gtk_list_store_set (data->scale_store, &iter, 1, text, -1);

      if (active == i)
      {
        gtk_spin_button_set_value (GTK_SPIN_BUTTON (data->scale_width_spin), (gdouble) scale_width);
        gtk_spin_button_set_value (GTK_SPIN_BUTTON (data->scale_height_spin), (gdouble) scale_height);
      }
    }
  }
}

static void
ogmrip_options_crop_update (DialogData *data)
{
  gtk_label_set_int (GTK_LABEL (data->crop_left_label), 0);
  gtk_label_set_int (GTK_LABEL (data->crop_top_label), 0);
  gtk_label_set_int (GTK_LABEL (data->crop_right_label), 0);
  gtk_label_set_int (GTK_LABEL (data->crop_bottom_label), 0);
}

static void
ogmrip_options_destroyed (DialogData *data)
{
  if (data->title)
    ogmdvd_title_unref (data->title);
  data->title = NULL;

  g_free (data);
}

static void
ogmrip_options_scale_combo_changed (DialogData *data, GtkWidget *combo)
{
  gint active;

  active = gtk_combo_box_get_active (GTK_COMBO_BOX (combo));

  gtk_widget_set_sensitive (data->scale_user_hbox, active == OGMRIP_SCALE_USER);

  if (active > -1 && active < OGMRIP_SCALE_USER)
  {
    guint x, y, width, height;
    gdouble aspect;

    ogmrip_options_get_crop_internal (data, &x, &y, &width, &height);
    aspect = ogmrip_options_get_aspect (data, width, height);

    ogmrip_options_scale (active, aspect, &width, &height);

    gtk_spin_button_set_value (GTK_SPIN_BUTTON (data->scale_width_spin), width);
    gtk_spin_button_set_value (GTK_SPIN_BUTTON (data->scale_height_spin), height);
  }
}

static void
ogmrip_options_autoscale_clicked (DialogData *data)
{
  OGMJobSpawn *spawn;
  guint scale_width, scale_height;
  guint crop_x, crop_y, crop_width, crop_height;
  guint bitrate;

  spawn = ogmrip_lavc_new (data->title, NULL);

  ogmrip_options_get_crop_internal (data, &crop_x, &crop_y, &crop_width, &crop_height);
  ogmrip_video_set_crop_size (OGMRIP_VIDEO (spawn), crop_x, crop_y, crop_width, crop_height);

  if (!gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->bitrate_check)))
  {
    bitrate = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (data->bitrate_spin));
    ogmrip_video_set_bitrate (OGMRIP_VIDEO (spawn), bitrate);
  }

  ogmrip_video_autoscale (OGMRIP_VIDEO (spawn));
  ogmrip_video_get_scale_size (OGMRIP_VIDEO (spawn), &scale_width, &scale_height);
  g_object_unref (spawn);

  gtk_spin_button_set_value (GTK_SPIN_BUTTON (data->scale_width_spin), (gdouble) scale_width);
  gtk_spin_button_set_value (GTK_SPIN_BUTTON (data->scale_height_spin), (gdouble) scale_height);

  gtk_combo_box_set_active (GTK_COMBO_BOX (data->scale_combo), OGMRIP_SCALE_USER);
}

static void
ogmrip_options_crop_clicked (DialogData *data)
{
  GtkWidget *dialog;
  guint old_left, old_top, old_right, old_bottom;
  guint new_left, new_top, new_right, new_bottom;

  old_left = gtk_label_get_int (GTK_LABEL (data->crop_left_label));
  old_top = gtk_label_get_int (GTK_LABEL (data->crop_top_label));
  old_right = gtk_label_get_int (GTK_LABEL (data->crop_right_label));
  old_bottom = gtk_label_get_int (GTK_LABEL (data->crop_bottom_label));

  dialog = ogmrip_crop_new (data->title, old_left, old_top, old_right, old_bottom);
  if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_OK)
  {
    ogmrip_crop_get_crop (dialog, &new_left, &new_top, &new_right, &new_bottom);
    if (new_left != old_left || new_right != old_right || new_top != old_top || new_bottom != old_bottom)
    {
      gtk_label_set_int (GTK_LABEL (data->crop_left_label), new_left);
      gtk_label_set_int (GTK_LABEL (data->crop_top_label), new_top);
      gtk_label_set_int (GTK_LABEL (data->crop_right_label), new_right);
      gtk_label_set_int (GTK_LABEL (data->crop_bottom_label), new_bottom);

      ogmrip_options_scale_update (data);
    }
  }
  gtk_widget_destroy (dialog);
}

static void
ogmrip_options_autocrop_clicked (DialogData *data)
{
  GtkWidget *dialog;
  OGMJobSpawn *spawn;
  guint old_x, old_y, old_width, old_height;
  guint new_x, new_y, new_width, new_height;
  guint raw_width, raw_height;
  gchar *message;

  ogmrip_options_get_crop_internal (data, &old_x, &old_y, &old_width, &old_height);

  message = g_strdup_printf ("<b>%s</b>\n\n%s", _("Detecting cropping parameters"), _("Please wait"));
  dialog = gtk_message_dialog_new (GTK_WINDOW (data->dialog), 
      GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT, GTK_MESSAGE_INFO, GTK_BUTTONS_NONE, message);
  g_free (message);

  gtk_window_set_icon_from_stock (GTK_WINDOW (dialog), GTK_STOCK_DIALOG_INFO);
  gtk_label_set_use_markup (GTK_LABEL (GTK_MESSAGE_DIALOG (dialog)->label), TRUE);
  gtk_widget_show (dialog);

  spawn = ogmrip_lavc_new (data->title, NULL);
  ogmrip_video_autocrop (OGMRIP_VIDEO (spawn), 0);
  ogmrip_video_get_crop_size (OGMRIP_VIDEO (spawn), &new_x, &new_y, &new_width, &new_height);
  g_object_unref (spawn);

  gtk_widget_destroy (dialog);

  ogmdvd_title_get_size (data->title, &raw_width, &raw_height);

  if (new_width == 0)
    new_width = raw_width;

  if (new_height == 0)
    new_height = raw_height;

  if (new_x != old_x || new_y != old_y || new_width != old_width || new_height != old_height)
  {
    gtk_label_set_int (GTK_LABEL (data->crop_left_label), new_x);
    gtk_label_set_int (GTK_LABEL (data->crop_top_label), new_y);
    gtk_label_set_int (GTK_LABEL (data->crop_right_label), raw_width - new_x - new_width);
    gtk_label_set_int (GTK_LABEL (data->crop_bottom_label), raw_height - new_y - new_height);

    ogmrip_options_scale_update (data);
  }
}

static void
ogmrip_options_check_toggled (GtkWidget *check, GtkWidget *widget)
{
  gtk_widget_set_sensitive (widget, 
      !gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (check)));
}

static void
ogmrip_options_bitrate_or_crop_toggled (DialogData *data)
{
  gboolean active;

  active = (!gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->bitrate_check)) &
    !gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->crop_check)));

  gtk_widget_set_sensitive (data->autoscale_button, active);
}

static void
ogmrip_options_set_title (DialogData *data, OGMDvdTitle *title)
{
  ogmdvd_title_ref (title);
  data->title = title;

  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (data->bitrate_check), TRUE);
  gtk_spin_button_set_value (GTK_SPIN_BUTTON (data->bitrate_spin), 800);

  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (data->crop_check), TRUE);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (data->scale_check), TRUE);

  ogmrip_options_crop_update (data);
  ogmrip_options_scale_update (data);

  gtk_combo_box_set_active (GTK_COMBO_BOX (data->scale_combo), OGMRIP_SCALE_FULL);

  gtk_combo_box_set_active (GTK_COMBO_BOX (data->deint_combo), 0);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (data->progressive_check), FALSE);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (data->pullup_check), FALSE);

  gtk_expander_set_expanded (GTK_EXPANDER (data->more_exp), FALSE);
  gtk_combo_box_set_active (GTK_COMBO_BOX (data->framestep_combo), 0);

  if (ogmdvd_title_get_video_format (title) == OGMDVD_VIDEO_FORMAT_NTSC)
    gtk_widget_show (data->ntsc_frame);
  else
    gtk_widget_hide (data->ntsc_frame);
}

GtkWidget *
ogmrip_options_new (OGMDvdTitle *title)
{
  DialogData *data;
  GtkCellRenderer *renderer;
  GtkWidget *widget;
  GladeXML *xml;
  guint i;

  GtkTreeIter iter;
  gchar *size[] = { N_("Extra Small"), N_("Small"), N_("Medium"), 
    N_("Large"), N_("Extra Large"), N_("Full"), N_("User Defined") };

  xml = glade_xml_new (OGMRIP_DATA_DIR "/" OGMRIP_GLADE_FILE, NULL, NULL);
  if (!xml)
  {
    g_warning ("Could not find " OGMRIP_GLADE_FILE);
    return NULL;
  }

  data = g_new0 (DialogData, 1);

  data->dialog = glade_xml_get_widget (xml, "options-dialog");
  g_object_set_data (G_OBJECT (data->dialog), "__ogmrip_options__", data);
  gtk_window_set_icon_from_stock (GTK_WINDOW (data->dialog), GTK_STOCK_PROPERTIES);

  g_signal_connect (data->dialog, "delete-event", G_CALLBACK (gtk_true), NULL);
  g_signal_connect_after (data->dialog, "response", G_CALLBACK (gtk_widget_hide), NULL);
  g_signal_connect_swapped (data->dialog, "destroy", G_CALLBACK (ogmrip_options_destroyed), data);

  data->bitrate_check = glade_xml_get_widget (xml, "bitrate-check");
  widget = glade_xml_get_widget (xml,"bitrate-vbox");
  g_signal_connect (data->bitrate_check, "toggled", 
      G_CALLBACK (ogmrip_options_check_toggled), widget);
  g_signal_connect_swapped (data->bitrate_check, "toggled", 
      G_CALLBACK (ogmrip_options_bitrate_or_crop_toggled), data);

  data->bitrate_spin = glade_xml_get_widget (xml, "bitrate-spin");

  data->scale_check = glade_xml_get_widget (xml, "scale-check");
  widget = glade_xml_get_widget (xml,"scale-vbox");
  g_signal_connect (data->scale_check, "toggled", 
      G_CALLBACK (ogmrip_options_check_toggled), widget);

  data->autoscale_button = glade_xml_get_widget (xml, "autoscale-button");
  g_signal_connect_swapped (data->autoscale_button, "clicked", 
      G_CALLBACK (ogmrip_options_autoscale_clicked), data);

  data->scale_user_hbox = glade_xml_get_widget (xml, "scale-user-hbox");
  data->scale_width_spin = glade_xml_get_widget (xml, "scale-width-spin");
  data->scale_height_spin = glade_xml_get_widget (xml, "scale-height-spin");

  data->scale_store = gtk_list_store_new (2, G_TYPE_STRING, G_TYPE_STRING);

  data->scale_combo = glade_xml_get_widget (xml, "scale-combo");
  g_signal_connect_swapped (data->scale_combo, "changed",
      G_CALLBACK (ogmrip_options_scale_combo_changed), data);

  for (i = 0; i < G_N_ELEMENTS (size); i++)
  {
    gtk_list_store_append (data->scale_store, &iter);
    gtk_list_store_set (data->scale_store, &iter, 0, _(size[i]), -1);
  }

  renderer = gtk_cell_renderer_text_new ();
  g_object_set (renderer, "xalign", 0.0, NULL);
  gtk_cell_layout_pack_start (GTK_CELL_LAYOUT (data->scale_combo), renderer, TRUE);
  gtk_cell_layout_set_attributes (GTK_CELL_LAYOUT (data->scale_combo), renderer, "text", 0, NULL);

  renderer = gtk_cell_renderer_text_new ();
  g_object_set (renderer, "xalign", 1.0, "xpad", 4, NULL);
  gtk_cell_layout_pack_start (GTK_CELL_LAYOUT (data->scale_combo), renderer, TRUE);
  gtk_cell_layout_set_attributes (GTK_CELL_LAYOUT (data->scale_combo), renderer, "text", 1, NULL);

  gtk_combo_box_set_model (GTK_COMBO_BOX (data->scale_combo), GTK_TREE_MODEL (data->scale_store));

  data->crop_check = glade_xml_get_widget (xml, "crop-check");
  widget = glade_xml_get_widget (xml,"crop-hbox");
  g_signal_connect (data->crop_check, "toggled", 
      G_CALLBACK (ogmrip_options_check_toggled), widget);
  g_signal_connect_swapped (data->crop_check, "toggled", 
      G_CALLBACK (ogmrip_options_bitrate_or_crop_toggled), data);

  data->crop_left_label = glade_xml_get_widget (xml, "crop-left-label");
  data->crop_right_label = glade_xml_get_widget (xml, "crop-right-label");
  data->crop_top_label = glade_xml_get_widget (xml, "crop-top-label");
  data->crop_bottom_label = glade_xml_get_widget (xml, "crop-bottom-label");

  widget = glade_xml_get_widget (xml, "crop-button");
  g_signal_connect_swapped (widget, "clicked", 
      G_CALLBACK (ogmrip_options_crop_clicked), data);

  widget = glade_xml_get_widget (xml, "autocrop-button");
  g_signal_connect_swapped (widget, "clicked", 
      G_CALLBACK (ogmrip_options_autocrop_clicked), data);

  data->more_exp = glade_xml_get_widget (xml, "more-expander");
  data->ntsc_frame = glade_xml_get_widget (xml, "ntsc-frame");

  data->pullup_check = glade_xml_get_widget (xml, "pullup-check");
  data->progressive_check = glade_xml_get_widget (xml, "progressive-check");

  data->deint_combo = glade_xml_get_widget (xml, "deint-combo");
  gtk_combo_box_set_active (GTK_COMBO_BOX (data->deint_combo), 0);

  data->framestep_combo = glade_xml_get_widget (xml, "framestep-combo");
  gtk_combo_box_set_active (GTK_COMBO_BOX (data->framestep_combo), 0);

  ogmrip_options_set_title (data, title);

  g_object_unref (xml);

  return data->dialog;
}

gint
ogmrip_options_get_bitrate (GtkWidget *dialog, guint *bitrate)
{
  DialogData *data;
  gint retval;

  g_return_val_if_fail (GTK_IS_DIALOG (dialog), OGMRIP_OPTIONS_ERROR);
  g_return_val_if_fail (bitrate != NULL, OGMRIP_OPTIONS_ERROR);

  data = g_object_get_data (G_OBJECT (dialog), "__ogmrip_options__");
  g_return_val_if_fail (data != NULL, OGMRIP_OPTIONS_ERROR);

  retval = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->bitrate_check)) ?
      OGMRIP_OPTIONS_AUTOMATIC : OGMRIP_OPTIONS_MANUAL;
  *bitrate = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (data->bitrate_spin)) * 1000;

  return retval;
}

gint
ogmrip_options_get_scale (GtkWidget *dialog, guint *width, guint *height)
{
  DialogData *data;
  gint retval;

  g_return_val_if_fail (GTK_IS_DIALOG (dialog), OGMRIP_OPTIONS_ERROR);
  g_return_val_if_fail (width != NULL, OGMRIP_OPTIONS_ERROR);
  g_return_val_if_fail (height != NULL, OGMRIP_OPTIONS_ERROR);

  data = g_object_get_data (G_OBJECT (dialog), "__ogmrip_options__");
  g_return_val_if_fail (data != NULL, OGMRIP_OPTIONS_ERROR);

  retval = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->scale_check)) ?
      OGMRIP_OPTIONS_AUTOMATIC : OGMRIP_OPTIONS_MANUAL;

  if (retval == OGMRIP_OPTIONS_MANUAL)
  {
    *width = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (data->scale_width_spin));
    *height = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (data->scale_height_spin));
  }

  return retval;
}

gint
ogmrip_options_get_crop (GtkWidget *dialog, guint *x, guint *y, guint *width, guint *height)
{
  DialogData *data;

  g_return_val_if_fail (GTK_IS_DIALOG (dialog), OGMRIP_OPTIONS_ERROR);
  g_return_val_if_fail (x != NULL, OGMRIP_OPTIONS_ERROR);
  g_return_val_if_fail (y != NULL, OGMRIP_OPTIONS_ERROR);
  g_return_val_if_fail (width != NULL, OGMRIP_OPTIONS_ERROR);
  g_return_val_if_fail (height != NULL, OGMRIP_OPTIONS_ERROR);

  data = g_object_get_data (G_OBJECT (dialog), "__ogmrip_options__");
  g_return_val_if_fail (data != NULL, OGMRIP_OPTIONS_ERROR);

  return ogmrip_options_get_crop_internal (data, x, y, width, height);
}

gint
ogmrip_options_get_deinterlacer (GtkWidget *dialog)
{
  DialogData *data;

  data = g_object_get_data (G_OBJECT (dialog), "__ogmrip_options__");
  g_return_val_if_fail (data != NULL, OGMRIP_OPTIONS_ERROR);

  if (!GTK_WIDGET_VISIBLE (data->ntsc_frame))
    return 0;

  return gtk_combo_box_get_active (GTK_COMBO_BOX (data->deint_combo));
}

gboolean
ogmrip_options_get_progressive (GtkWidget *dialog)
{
  DialogData *data;

  data = g_object_get_data (G_OBJECT (dialog), "__ogmrip_options__");
  g_return_val_if_fail (data != NULL, OGMRIP_OPTIONS_ERROR);

  if (!GTK_WIDGET_VISIBLE (data->ntsc_frame))
    return FALSE;

  return gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->progressive_check));
}

gboolean
ogmrip_options_get_pullup (GtkWidget *dialog)
{
  DialogData *data;

  data = g_object_get_data (G_OBJECT (dialog), "__ogmrip_options__");
  g_return_val_if_fail (data != NULL, OGMRIP_OPTIONS_ERROR);

  if (!GTK_WIDGET_VISIBLE (data->ntsc_frame))
    return FALSE;

  return gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->pullup_check));
}

gint
ogmrip_options_get_framestep (GtkWidget *dialog)
{
  DialogData *data;
  gint framestep;

  data = g_object_get_data (G_OBJECT (dialog), "__ogmrip_options__");
  g_return_val_if_fail (data != NULL, OGMRIP_OPTIONS_ERROR);

  framestep = gtk_combo_box_get_active (GTK_COMBO_BOX (data->framestep_combo));

  return framestep + 1;
}

