/* OGMRip - A DVD Encoder for GNOME
 * Copyright (C) 2004-2006 Olivier Rolland <billl@users.sf.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "config.h"
#include "ogmrip-fs.h"
#include "ogmrip-pref.h"
#include "ogmrip-progress.h"
#include "ogmrip-helper.h"
#include "ogmrip-gconf.h"
#include "ogmrip-options.h"

#include "ogmrip-chapter-list.h"
#include "ogmrip-chooser-list.h"

#include <ogmdvd.h>
#include <ogmdvd-gtk.h>

#include <ogmjob.h>
#include <ogmrip.h>

#ifdef HAVE_ENCHANT_SUPPORT
#include "ogmrip-spell.h"
#endif

#include <sys/stat.h>
#include <glib/gi18n.h>
#include <glib/gstdio.h>

#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <math.h>

#define OGMRIP_UI_FILE    "ogmrip/ogmrip-ui.xml"
#define OGMRIP_GLADE_FILE "ogmrip/ogmrip-main.glade"
#define OGMRIP_ICON_FILE  "pixmaps/ogmrip.png"

#define OGMRIP_DEFAULT_FILE_NAME "movie"

typedef struct
{
  OGMDvdDisc *disc;

  GtkWidget *window;

  GtkWidget *pref_dialog;
  GtkWidget *options_dialog;
  GtkWidget *progress_dialog;

  GtkWidget *title_entry;
  GtkWidget *title_chooser;
  GtkWidget *length_label;
  GtkWidget *relative_check;

  GtkWidget *audio_list;
  GtkWidget *subp_list;

  GtkWidget *chapter_list;

  GtkAction *extract_action;
  GtkWidget *extract_button;

  GConfClient *gconf;

  gboolean encoding;

} OGMRipData;

typedef struct
{
  OGMDvdTitle *title;
  GList *audio_streams;
  GList *subp_streams;

  gboolean copy_dvd;
  gboolean ensure_sync;
  gboolean keep_temp;

  guint passes;
  GType container;
  GType video_codec;
  GType audio_codec;
  GType subp_codec;

  guint start_chap;
  gint  end_chap;

  guint target_number;
  guint target_size;

  gint64 rip_size;

  gchar *outfile;

  const gchar *label;
} OGMRipEncode;

#ifdef HAVE_ENCHANT_SUPPORT
static gboolean
ogmrip_main_spell_check (OGMRipData *data, OGMRipSubp *subp)
{
  gboolean retval = FALSE;
  gchar *text, *corrected;
  gchar *new_file = NULL;
  const gchar *old_file;
  gint lang;

  GtkWidget *checker = NULL;
  GIOChannel *input = NULL, *output = NULL;
  GIOStatus status = G_IO_STATUS_NORMAL;

  old_file = ogmrip_codec_get_output (OGMRIP_CODEC (subp));

  lang = ogmdvd_subp_stream_get_language (ogmrip_subp_get_dvd_subp_stream (subp));
  checker = ogmrip_spell_new (ogmdvd_get_language_iso639_1 (lang));
  if (!checker)
  {
    ogmrip_message_dialog (GTK_WINDOW (data->window), GTK_MESSAGE_ERROR, "<b>%s</b>\n\n%s",
        _("Could not create dictionary."), _("Spell will not be checked."));
    goto spell_check_cleanup;
  }

  input = g_io_channel_new_file (old_file, "r", NULL);
  if (!input)
    goto spell_check_cleanup;

  new_file = ogmrip_fs_mktemp ("sub.XXXXXX", NULL);
  if (!new_file)
    goto spell_check_cleanup;

  output = g_io_channel_new_file (new_file, "w", NULL);
  if (!output)
    goto spell_check_cleanup;

  gtk_window_set_parent (GTK_WINDOW (checker), GTK_WINDOW (data->window));
  gtk_widget_show (checker);

  do
  {
    status = g_io_channel_read_line (input, &text, NULL, NULL, NULL);
    if (status == G_IO_STATUS_NORMAL)
    {
      retval = ogmrip_spell_check_text (checker, text, &corrected);
      if (retval)
      {
        do
        {
          status = g_io_channel_write_chars (output, corrected ? corrected : text, -1, NULL, NULL);
        }
        while (status == G_IO_STATUS_AGAIN);

        g_free (corrected);
      }
      g_free (text);
    }
  } 
  while (retval == TRUE && (status == G_IO_STATUS_NORMAL || status == G_IO_STATUS_AGAIN));

  retval &= status == G_IO_STATUS_EOF;

spell_check_cleanup:
  if (checker)
    gtk_widget_destroy (checker);

  if (output)
  {
    g_io_channel_shutdown (output, TRUE, NULL);
    g_io_channel_unref (output);
  }

  if (input)
  {
    g_io_channel_shutdown (input, TRUE, NULL);
    g_io_channel_unref (input);
  }

  if (retval)
    retval = ogmrip_fs_rename (new_file, old_file, NULL);
  else
    g_unlink (new_file);

  g_free (new_file);

  return retval;
}
#endif /* HAVE_ENCHANT_SUPPORT */

/*
 * Makes the log file for 2-pass encoding
 */
static gchar *
ogmrip_main_mklog (void)
{
#if MPLAYER_PRE < 8
  GConfClient *gconf;
  GType type;
  gint codec;
#endif
  gchar *filename;

#if MPLAYER_PRE >= 8
  filename = ogmrip_fs_mktemp ("log.XXXXXX", NULL);
#else /* MPLAYER_PRE < 8 */
  /*
   * Workaround against xvid pass log file
   * Should disappear someday
   */
  gconf = gconf_client_get_default ();
  codec = gconf_client_get_int_default (gconf, OGMRIP_GCONF_VIDEO_FORMAT, OGMRIP_DEFAULT_VIDEO_FORMAT);
  g_object_unref (gconf);

  type = ogmrip_get_nth_video_codec (codec);

  if (type == OGMRIP_TYPE_XVID)
    filename = g_build_filename (g_get_tmp_dir (), "xvid-twopass.stats", NULL);
  else
    filename = ogmrip_fs_mktemp ("log.XXXXXX", NULL);
#endif /* MPLAYER_PRE */

  return filename;
}

/*
 * Builds the output filename
 */
static gchar *
ogmrip_main_get_filename (OGMRipData *data, OGMRipEncode *encode)
{
  gchar *utf8, *filename, *path, *outfile, *ext;
  const gchar *lang = "";

  if (encode->audio_streams)
    lang = ogmdvd_get_language_label (ogmdvd_audio_stream_get_language (encode->audio_streams->data));

  ext = g_utf8_strdown (ogmrip_get_container_name (encode->container), -1);

  switch (gconf_client_get_int_default (data->gconf, OGMRIP_GCONF_FILENAME, OGMRIP_DEFAULT_FILENAME))
  {
    case 0:
      utf8 = g_strdup_printf ("%s.%s", encode->label, ext);
      break;
    case 1:
      utf8 = g_strdup_printf ("%s - %s.%s", encode->label, lang, ext);
      break;
    case 2:
      utf8 = g_strdup_printf ("%s - %s - %s.%s", encode->label, lang, 
          ogmrip_get_video_codec_name (encode->video_codec), ext);
      break;
    case 3:
      utf8 = g_strdup_printf ("%s - %s - %s %s.%s", encode->label, lang,
          ogmrip_get_video_codec_name (encode->video_codec),
          ogmrip_get_audio_codec_name (encode->audio_codec), ext);
      break;
    default:
      utf8 = g_strdup_printf ("%s.%s", OGMRIP_DEFAULT_FILE_NAME, ext);
      break;
  }

  g_free (ext);

  filename = g_filename_from_utf8 (utf8, -1, NULL, NULL, NULL);
  g_free (utf8);

  path = gconf_client_get_filename_default (data->gconf, OGMRIP_GCONF_OUTPUT_DIR, OGMRIP_DEFAULT_OUTPUT_DIR);
  outfile = g_build_filename (path, filename, NULL);
  g_free (filename);
  g_free (path);

  return outfile;
}

/*
 * Returns the estimated rip size
 */
static gint64
ogmrip_main_get_rip_size (OGMRipData *data, OGMRipEncode *encode)
{
  gdouble factor = 1.0;

  if (GTK_WIDGET_SENSITIVE (data->relative_check) &
      gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->relative_check)))
  {
    glong full_len, chap_len;

    full_len = ogmdvd_title_get_length (encode->title, NULL);
    if (full_len < 0)
      return -1;

    chap_len = ogmdvd_title_get_chapters_length (encode->title, encode->start_chap, encode->end_chap, NULL);
    if (chap_len < 0)
      return -1;

    factor = chap_len / (gdouble) full_len;
  }

  return (gint64) ceil (factor * encode->target_size * encode->target_number * 1024 * 1024);
}

/*
 * Returns the estimed audio size used to enforce sync
 */
static gint64
ogmrip_main_get_sync_size (OGMRipData *data, OGMRipEncode *encode)
{
  guint numerator, denominator;
  glong chap_len;

  if (!encode->ensure_sync)
    return 0;

  chap_len = ogmdvd_title_get_chapters_length (encode->title, encode->start_chap, encode->end_chap, NULL);
  if (chap_len < 0)
    return -1;

  ogmdvd_title_get_framerate (encode->title, &numerator, &denominator);

  return chap_len / (gdouble) numerator * denominator * 4000 * 2;
}

/*
 * Returns the VMG + VTS size
 */
static gint64
ogmrip_main_get_dvd_size (OGMRipData *data, OGMRipEncode *encode)
{
  OGMDvdDisc *disc;
  gint64 vts_size;

  if (!encode->copy_dvd)
    return 0;

  vts_size = ogmdvd_title_get_vts_size (encode->title);
  if (vts_size < 0)
    return -1;

  disc = ogmdvd_title_get_disc (encode->title);

  return vts_size + ogmdvd_disc_get_vmg_size (disc);
}

/*
 * Checks if the DVD can be copied
 */
static gboolean
ogmrip_main_get_copy_dvd (OGMRipData *data, GError **error)
{
  gboolean copy_dvd;

  copy_dvd = gconf_client_get_bool_default (data->gconf, OGMRIP_GCONF_COPY_DVD, OGMRIP_DEFAULT_COPY_DVD);

  if (copy_dvd)
  {
    const gchar *device;
    struct stat buf;

    device = ogmdvd_disc_get_device (data->disc);
    if (g_stat (device, &buf) < 0)
    {
      g_set_error (error, OGMRIP_ERROR, 0, _("Cannot stat '%s'"), device);
      return FALSE;
    }

    if (!S_ISBLK (buf.st_mode))
    {
      g_set_error (error, OGMRIP_ERROR, 0, _("'%s' is not a valid block device"), device);
      return FALSE;
    }
  }

  return copy_dvd;
}

/*
 * Checks if container and codecs are compatible
 */
static gboolean
ogmrip_main_check_container (OGMRipData *data, OGMRipEncode *encode, GError **error)
{
  if (encode->container == G_TYPE_NONE)
    return FALSE;

  if (encode->video_codec != G_TYPE_NONE &&
      !ogmrip_can_contain_video (encode->container, encode->video_codec))
  {
    g_set_error (error, OGMRIP_ERROR, 0, "<b>%s</b>\n\n%s",
        _("The container and the video codec are incompatible."), 
        _("Please, choose some others."));
    return FALSE;
  }

  if (encode->audio_codec != G_TYPE_NONE && encode->audio_streams != NULL &&
      !ogmrip_can_contain_audio (encode->container, encode->audio_codec))
  {
    g_set_error (error, OGMRIP_ERROR, 0, "<b>%s</b>\n\n%s",
        _("The container and the audio codec are incompatible."), 
        _("Please, choose some others."));
    return FALSE;
  }

  if (encode->subp_codec != G_TYPE_NONE && encode->subp_streams != NULL &&
      !ogmrip_can_contain_subp (encode->container, encode->subp_codec))
  {
    g_set_error (error, OGMRIP_ERROR, 0, "<b>%s</b>\n\n%s",
        _("The container and the subtitles codec are incompatible."), 
        _("Please, choose some others."));
    return FALSE;
  }

  return TRUE;
}

/*
 * Check if container can contain multiple streams
 */
static gboolean
ogmrip_main_check_streams (OGMRipData *data, OGMRipEncode *encode, GError **error)
{
  GList *stream;
  gint nstreams;

  if (encode->container == G_TYPE_NONE)
    return FALSE;

  nstreams = g_list_length (encode->audio_streams);
  if (!ogmrip_can_contain_n_audio (encode->container, nstreams))
  {
    g_set_error (error, OGMRIP_ERROR, 0, "<b>%s</b>\n\n%s",
        _("The selected container does not support multiple audio streams"), 
        _("Please, choose one audio stream only."));
    return FALSE;
  }

  nstreams = g_list_length (encode->subp_streams);
  if (!ogmrip_can_contain_n_subp (encode->container, nstreams))
  {
    g_set_error (error, OGMRIP_ERROR, 0, "<b>%s</b>\n\n%s",
        _("The selected container does not support multiple subtitles streams"), 
        _("Please, choose one subtitles stream only."));
    return FALSE;
  }

  for (stream = encode->audio_streams; stream; stream = stream->next)
  {
    if (ogmdvd_audio_stream_get_format (stream->data) == OGMDVD_AUDIO_FORMAT_DTS)
    {
#ifndef HAVE_DTS_SUPPORT
      g_set_error (error, OGMRIP_ERROR, 0,
          _("Your version of MPlayer does not support DTS streams")); 
      return FALSE;
#endif
      /*
       * TODO generalize ?
       */
      if (encode->container == OGMRIP_TYPE_OGG)
      {
        g_set_error (error, OGMRIP_ERROR, 0,
            _("The OGM container does not support DTS streams")); 
        return FALSE;
      }
    }
  }

  return TRUE;
}

/*
 * Checks if the output file already exists
 */
static gboolean
ogmrip_main_check_filename (OGMRipData *data, OGMRipEncode *encode)
{
  /*
   * TODO What if multiple targets ?
   */
  if (g_file_test (encode->outfile, G_FILE_TEST_EXISTS))
  {
    gint response;

    response = ogmrip_message_dialog (GTK_WINDOW (data->window), GTK_MESSAGE_QUESTION,
        _("A file called '%s' exists.\nDo you want to overwrite it?"), encode->outfile);

    if (response == GTK_RESPONSE_NO)
      return FALSE;

    g_unlink (encode->outfile);
  }

  return TRUE;
}

/*
 * Checks if there is enough space on the drive
 */
static gboolean
ogmrip_main_check_space (OGMRipData *data, OGMRipEncode *encode, GError **error)
{
  gchar *output_mount_point, *tmp_mount_point;
  gint64 sync_size, dvd_size;
  gboolean retval = FALSE;

  output_mount_point = ogmrip_fs_get_mount_point (encode->outfile);
  tmp_mount_point = ogmrip_fs_get_mount_point (ogmrip_fs_get_tmp_dir ());

  sync_size = ogmrip_main_get_sync_size (data, encode);
  dvd_size = ogmrip_main_get_dvd_size (data, encode);

  if (output_mount_point && tmp_mount_point)
  {
    if (strcmp (tmp_mount_point, output_mount_point) == 0)
    {
      retval = encode->rip_size * 2 + dvd_size + sync_size < ogmrip_fs_get_left_space (encode->outfile);
      if (!retval)
      {
        gint64 needed = (2 * encode->rip_size + dvd_size) / 1024 / 1024;
        gchar *needed_str = g_strdup_printf ("%" G_GUINT64_FORMAT,needed);

        g_set_error (error, OGMRIP_ERROR, 0, 
            _("Not enough space to store output and temporary files (%sMB needed)."), needed_str);
        g_free (needed_str);
      }
    }
    else
    {
      retval = encode->rip_size + dvd_size + sync_size < ogmrip_fs_get_left_space (ogmrip_fs_get_tmp_dir ());
      if (!retval)
      {
        gint64 needed = (encode->rip_size + dvd_size + sync_size) / 1024 / 1024;
        gchar *needed_str = g_strdup_printf ("%" G_GUINT64_FORMAT, needed);

        g_set_error (error, OGMRIP_ERROR, 0, 
            _("Not enough space to store the temporary files (%sMB needed)."), needed_str);
        g_free (needed_str);
      }
      else
      {
        retval = encode->rip_size < ogmrip_fs_get_left_space (encode->outfile);
        if (!retval)
        {
          gint64 needed = encode->rip_size / 1024 / 1024;
          gchar *needed_str = g_strdup_printf ("%" G_GUINT64_FORMAT, needed);

          g_set_error (error, OGMRIP_ERROR, 0, 
              _("Not enough space to store the output file (%sMB needed)."), needed_str);
          g_free (needed_str);
        }
      }
    }
  }

  g_free (output_mount_point);
  g_free (tmp_mount_point);

  return retval;
}

/*
 * Encodes the video stream pass
 */
static gint
ogmrip_main_extract_video_stream_pass (OGMRipData *data, OGMJobSpawn *spawn, OGMDvdTitle *title, guint pass, GError **error)
{
  static gchar *logfile = NULL;
  gchar *message;
  gint result;

  if (pass != 2 && logfile)
  {
    g_unlink (logfile);
    g_free (logfile);
    logfile = NULL;
  }

  ogmrip_video_set_pass (OGMRIP_VIDEO (spawn), pass);

  switch (pass)
  {
    case 1:
      message = g_strdup_printf (_("<b>Encoding video title %d, Pass 1</b>"), ogmdvd_title_get_nr (title) + 1);
      logfile = ogmrip_main_mklog ();
      ogmrip_video_set_log (OGMRIP_VIDEO (spawn), logfile);
      break;
    case 2:
      message = g_strdup_printf (_("<b>Encoding video title %d, Pass 2</b>"), ogmdvd_title_get_nr (title) + 1);
      break;
    default:
      message = g_strdup_printf (_("<b>Encoding video title %d</b>"), ogmdvd_title_get_nr (title) + 1);
      break;
  }
  ogmrip_progress_set_spawn (data->progress_dialog, spawn, message);
  g_free (message);

  result = ogmjob_spawn_run (spawn, error);

  if (pass == 2 && logfile)
  {
    g_unlink (logfile);
    g_free (logfile);
    logfile = NULL;
  }

  return result;
}

/*
 * Encodes the video stream
 */
static gint
ogmrip_main_extract_video_stream (OGMRipData *data, OGMRipEncode *encode, OGMRipContainer *container, GError **error)
{
  OGMJobSpawn *spawn;

  gint64 nonvideo_size, overhead_size;
  guint x, y, width, height, bitrate;
  gchar *output;
  gint result;

  output = ogmrip_fs_mktemp ("video.XXXXXX", error);
  if (!output)
    return OGMJOB_RESULT_ERROR;

  spawn = g_object_new (encode->video_codec, "input", encode->title, "output", output, NULL);
  g_free (output);

  ogmrip_codec_set_unlink_on_unref (OGMRIP_CODEC (spawn), !encode->keep_temp);
  ogmrip_codec_set_chapters (OGMRIP_CODEC (spawn), encode->start_chap, encode->end_chap);

  ogmrip_video_set_quality (OGMRIP_VIDEO (spawn),
      gconf_client_get_int_default (data->gconf, OGMRIP_GCONF_VIDEO_QUALITY, OGMRIP_DEFAULT_VIDEO_QUALITY));
  ogmrip_video_set_denoise (OGMRIP_VIDEO (spawn),
      gconf_client_get_bool_default (data->gconf, OGMRIP_GCONF_VIDEO_DENOISE, OGMRIP_DEFAULT_VIDEO_DENOISE));
  ogmrip_video_set_trellis (OGMRIP_VIDEO (spawn),
      gconf_client_get_bool_default (data->gconf, OGMRIP_GCONF_VIDEO_TRELLIS, OGMRIP_DEFAULT_VIDEO_TRELLIS));
  ogmrip_video_set_turbo (OGMRIP_VIDEO (spawn),
      gconf_client_get_bool_default (data->gconf, OGMRIP_GCONF_VIDEO_TURBO, OGMRIP_DEFAULT_VIDEO_TURBO));
  ogmrip_video_set_qpel (OGMRIP_VIDEO (spawn),
      gconf_client_get_bool_default (data->gconf, OGMRIP_GCONF_VIDEO_QPEL, OGMRIP_DEFAULT_VIDEO_QPEL));
  ogmrip_video_set_scaler (OGMRIP_VIDEO (spawn),
      gconf_client_get_int_default (data->gconf, OGMRIP_GCONF_VIDEO_SCALER, OGMRIP_DEFAULT_VIDEO_SCALER));

  ogmrip_video_set_deinterlacer (OGMRIP_VIDEO (spawn), ogmrip_options_get_deinterlacer (data->options_dialog));
  ogmrip_codec_set_framestep (OGMRIP_CODEC (spawn), ogmrip_options_get_framestep (data->options_dialog));
  ogmrip_video_set_pullup (OGMRIP_VIDEO (spawn), ogmrip_options_get_pullup (data->options_dialog));

  if (ogmrip_options_get_progressive (data->options_dialog))
    ogmrip_codec_set_framerate (OGMRIP_CODEC (spawn), 24000, 1001);

  if (encode->ensure_sync)
    ogmrip_video_set_ensure_sync (OGMRIP_VIDEO (spawn), encode->audio_streams->data);

  ogmrip_container_set_video (container, OGMRIP_VIDEO (spawn));
  g_object_unref (spawn);

  nonvideo_size = ogmrip_container_get_nonvideo_size (container);
  overhead_size = ogmrip_container_get_overhead_size (container);

  ogmrip_progress_set_spawn (data->progress_dialog, spawn, _("<b>Calculating bitrate, cropping, scaling</b>"));

  if (ogmrip_options_get_bitrate (data->options_dialog, &bitrate) == OGMRIP_OPTIONS_AUTOMATIC)
    ogmrip_video_autobitrate (OGMRIP_VIDEO (spawn), nonvideo_size, overhead_size, encode->rip_size);
  else
    ogmrip_video_set_bitrate (OGMRIP_VIDEO (spawn), bitrate);

  if (ogmrip_options_get_crop (data->options_dialog, &x, &y, &width, &height) == OGMRIP_OPTIONS_AUTOMATIC)
    ogmrip_video_autocrop (OGMRIP_VIDEO (spawn), 0);
  else
    ogmrip_video_set_crop_size (OGMRIP_VIDEO (spawn), x, y, width, height);

  if (ogmrip_options_get_scale (data->options_dialog, &width, &height) == OGMRIP_OPTIONS_AUTOMATIC)
    ogmrip_video_autoscale (OGMRIP_VIDEO (spawn));
  else
    ogmrip_video_set_scale_size (OGMRIP_VIDEO (spawn), width, height);

  /*
   * TODO
   * The mp4 container's implementation in mencoder does not support b frames
   * This should be automated in libogmrip
   */
#ifdef HAVE_LAVF_SUPPORT
  if (encode->container == OGMRIP_TYPE_MP4)
    ogmrip_video_set_max_b_frames (OGMRIP_VIDEO (spawn), 0);
#endif

  if (encode->passes == 0)
    result = ogmrip_main_extract_video_stream_pass (data, spawn, encode->title, 0, error);
  else
  {
    result = ogmrip_main_extract_video_stream_pass (data, spawn, encode->title, 1, error);
    if (result == OGMJOB_RESULT_COMPLETED)
      result = ogmrip_main_extract_video_stream_pass (data, spawn, encode->title, 2, error);
  }

  if (result == OGMJOB_RESULT_ERROR && error && !(*error))
    g_set_error (error, OGMRIP_ERROR, 0, "<b>%s</b>\n\n%s",
        _("Unknown error while extracting video stream"),
        _("Please, check http://ogmrip.sf.net to see if this is a known issue"));

  return result;
}

/*
 * Encodes the audio streams
 */
static gint
ogmrip_main_extract_audio_streams (OGMRipData *data, OGMRipEncode *encode, OGMRipContainer *container, GError **error)
{
  OGMRipAudioDemuxer demuxer;
  OGMJobSpawn *spawn;

  GList *audio_stream;
  gint quality, framestep, language, result;
  gboolean normalize, progressive;
  gchar *output, *message;

  quality = gconf_client_get_int_default (data->gconf, OGMRIP_GCONF_AUDIO_QUALITY, OGMRIP_DEFAULT_AUDIO_QUALITY);
  normalize = gconf_client_get_bool_default (data->gconf, OGMRIP_GCONF_AUDIO_NORMALIZE, OGMRIP_DEFAULT_AUDIO_NORMALIZE);
  progressive = ogmrip_options_get_progressive (data->options_dialog);
  framestep = ogmrip_options_get_framestep (data->options_dialog);

  for (audio_stream = encode->audio_streams; audio_stream; audio_stream = audio_stream->next)
  {
    output = ogmrip_fs_mktemp ("audio.XXXXXX", error);
    if (!output)
      return OGMJOB_RESULT_ERROR;

    spawn = g_object_new (encode->audio_codec, "stream", audio_stream->data, "output", output, NULL);
    g_free (output);

    ogmrip_codec_set_unlink_on_unref (OGMRIP_CODEC (spawn), !encode->keep_temp);
    ogmrip_codec_set_chapters (OGMRIP_CODEC (spawn), encode->start_chap, encode->end_chap);
    ogmrip_codec_set_framestep (OGMRIP_CODEC (spawn), framestep);

    ogmrip_audio_set_quality (OGMRIP_AUDIO (spawn), quality);
    ogmrip_audio_set_normalize (OGMRIP_AUDIO (spawn), normalize);

    if (progressive)
      ogmrip_codec_set_framerate (OGMRIP_CODEC (spawn), 24000, 1001);

    demuxer = OGMRIP_AUDIO_DEMUXER_AUTO;
    if (encode->audio_codec == OGMRIP_TYPE_AUDIO_COPY)
    {
      switch (ogmdvd_audio_stream_get_format (audio_stream->data))
      {
        case OGMDVD_AUDIO_FORMAT_AC3:
          demuxer = OGMRIP_AUDIO_DEMUXER_AC3;
          break;
        case OGMDVD_AUDIO_FORMAT_DTS:
          demuxer = OGMRIP_AUDIO_DEMUXER_DTS;
          break;
        default:
          break;
      }
    }

    language = ogmdvd_audio_stream_get_language (audio_stream->data);
    ogmrip_container_add_audio (container, OGMRIP_AUDIO (spawn), demuxer, language);
    g_object_unref (spawn);

    message = g_strdup_printf (_("<b>Extracting audio stream %d</b>"), 
        ogmdvd_audio_stream_get_nr (audio_stream->data) + 1);
    ogmrip_progress_set_spawn (data->progress_dialog, spawn, message);
    g_free (message);

    if ((result = ogmjob_spawn_run (spawn, error)) != OGMJOB_RESULT_COMPLETED)
    {
      if (result == OGMJOB_RESULT_ERROR && error && !(*error))
        g_set_error (error, OGMRIP_ERROR, 0, "<b>%s</b>\n\n%s",
            _("Unknown error while extracting audio stream"),
            _("Please, check http://ogmrip.sf.net to see if this is a known issue"));

      return result;
    }
  }

  return OGMJOB_RESULT_COMPLETED;
}

/*
 * Encodes the subp streams
 */
static gboolean
ogmrip_main_extract_subp_streams (OGMRipData *data, OGMRipEncode *encode, OGMRipContainer *container, GError **error)
{
  OGMRipSubpDemuxer demuxer;
  OGMJobSpawn *spawn;

  GList *subp_stream;
  gint language, result;
  gchar *output, *message;

  for (subp_stream = encode->subp_streams; subp_stream; subp_stream = subp_stream->next)
  {
    output = ogmrip_fs_mktemp ("subp.XXXXXX", error);
    if (!output)
      return OGMJOB_RESULT_ERROR;

    spawn = g_object_new (encode->subp_codec, "stream", subp_stream->data, "output", output, NULL);
    g_free (output);

    ogmrip_codec_set_unlink_on_unref (OGMRIP_CODEC (spawn), !encode->keep_temp);
    ogmrip_codec_set_chapters (OGMRIP_CODEC (spawn), encode->start_chap, encode->end_chap);

    demuxer = OGMRIP_SUBP_DEMUXER_AUTO;
    if (encode->subp_codec == OGMRIP_TYPE_VOBSUB)
      demuxer = OGMRIP_SUBP_DEMUXER_VOBSUB;

    language = ogmdvd_subp_stream_get_language (subp_stream->data);
    ogmrip_container_add_subp (container, OGMRIP_SUBP (spawn), demuxer, language);
    g_object_unref (spawn);

    message = g_strdup_printf (_("<b>Extracting subtitle stream %d</b>"),
        ogmdvd_subp_stream_get_nr (subp_stream->data) + 1);
    ogmrip_progress_set_spawn (data->progress_dialog, spawn, message);
    g_free (message);

    if ((result = ogmjob_spawn_run (spawn, error)) != OGMJOB_RESULT_COMPLETED)
    {
      if (result == OGMJOB_RESULT_ERROR && error && !(*error))
        g_set_error (error, OGMRIP_ERROR, 0, "<b>%s</b>\n\n%s",
            _("Unknown error while extracting subtitle stream"),
            _("Please, check http://ogmrip.sf.net to see if this is a known issue"));

      return result;
    }

#ifdef HAVE_ENCHANT_SUPPORT
    if (ogmrip_get_subp_codec_text (encode->subp_codec) && 
        gconf_client_get_bool_default (data->gconf, OGMRIP_GCONF_SPELL_CHECK, OGMRIP_DEFAULT_SPELL_CHECK))
      ogmrip_main_spell_check (data, OGMRIP_SUBP (spawn));
#endif /* HAVE_ENCHANT_SUPPORT */
  }

  return OGMJOB_RESULT_COMPLETED;
}

/*
 * Extracts chapter information
 */
static gboolean
ogmrip_main_extract_chapters (OGMRipData *data, OGMRipEncode *encode, OGMRipContainer *container, GError **error)
{
  OGMJobSpawn *spawn;
  gchar *output, *message, *label;
  gint result, chapter;

  output = ogmrip_fs_mktemp ("chapters.XXXXXX", error);
  if (!output)
    return FALSE;

  spawn = ogmrip_chapters_new (encode->title, output);
  g_free (output);

  for (chapter = encode->start_chap; chapter <= encode->end_chap; chapter ++)
  {
    if (ogmdvd_chapter_list_get_chapter (OGMDVD_CHAPTER_LIST (data->chapter_list), chapter, &label, NULL))
    {
      ogmrip_chapters_set_label (OGMRIP_CHAPTERS (spawn), chapter, label);
      g_free (label);
    }
  }

  ogmrip_codec_set_unlink_on_unref (OGMRIP_CODEC (spawn), !encode->keep_temp);
  ogmrip_codec_set_chapters (OGMRIP_CODEC (spawn), encode->start_chap, encode->end_chap);

  ogmrip_container_add_chapters (container, OGMRIP_CHAPTERS (spawn));
  g_object_unref (spawn);

  message = g_strdup_printf (_("<b>Extracting chapters information</b>"));
  ogmrip_progress_set_spawn (data->progress_dialog, spawn, message);
  g_free (message);

  if ((result = ogmjob_spawn_run (spawn, error)) != OGMJOB_RESULT_COMPLETED)
  {
    if (result == OGMJOB_RESULT_ERROR && error && !(*error))
      g_set_error (error, OGMRIP_ERROR, 0, "<b>%s</b>\n\n%s",
          _("Unknown error while extracting chapters"),
          _("Please, check http://ogmrip.sf.net to see if this is a known issue"));

    return result;
  }

  return OGMJOB_RESULT_COMPLETED;
}

/*
 * Creates the container
 */
static OGMRipContainer *
ogmrip_main_container_new (OGMRipData *data, OGMRipEncode *encode)
{
  OGMJobSpawn *spawn;
  gint fcc;

  static const gchar *fourcc[] =
  {
    NULL,
    "XVID",
    "DIVX",
    "DX50",
    "FMP4"
  };

  spawn = g_object_new (encode->container, "output", encode->outfile, NULL);

  ogmrip_container_set_split (OGMRIP_CONTAINER (spawn), encode->target_number, encode->target_size);
  ogmrip_container_set_label (OGMRIP_CONTAINER (spawn), encode->label);

  fcc = gconf_client_get_int_default (data->gconf, OGMRIP_GCONF_FOURCC, OGMRIP_DEFAULT_FOURCC);
  if (fcc >= 0 && fcc <= 4)
    ogmrip_container_set_fourcc (OGMRIP_CONTAINER (spawn), fourcc[fcc]);
  else
    ogmrip_container_set_fourcc (OGMRIP_CONTAINER (spawn), NULL);

  return OGMRIP_CONTAINER (spawn);
}

/*
 * Merges the streams together
 */
static gint
ogmrip_main_merge (OGMRipData *data, OGMRipContainer *container, GError **error)
{
  gint result;
#if MPLAYER_PRE < 8
  gchar *cwd;
#endif /* MPLAYER_PRE */

  ogmrip_progress_set_spawn (data->progress_dialog, 
      OGMJOB_SPAWN (container), _("<b>Merging audio and video streams</b>"));

#if MPLAYER_PRE < 8
  cwd = g_get_current_dir ();
  g_chdir (ogmrip_fs_get_tmp_dir ());
#endif /* MPLAYER_PRE */

  result = ogmjob_spawn_run (OGMJOB_SPAWN (container), error);

  if (result == OGMJOB_RESULT_ERROR && error && !(*error))
    g_set_error (error, OGMRIP_ERROR, 0, "<b>%s</b>\n\n%s",
        _("Unknown error while merging"),
        _("Please, check http://ogmrip.sf.net to see if this is a known issue"));

#if MPLAYER_PRE < 8
  g_chdir (cwd);
  g_free (cwd);
#endif /* MPLAYER_PRE */

  return result;
}

/*
 * Extracts a DVD title
 */
static gboolean
ogmrip_main_extract (OGMRipData *data, OGMRipEncode *encode, GError **error)
{
  OGMRipContainer *container;

  gboolean result;

  encode->keep_temp = gconf_client_get_bool_default (data->gconf, OGMRIP_GCONF_KEEP_TEMP, OGMRIP_DEFAULT_KEEP_TEMP);

  container = ogmrip_main_container_new (data, encode);
  if (!container)
    return FALSE;

  result = ogmrip_main_extract_chapters (data, encode, container, error);
  if (result != OGMJOB_RESULT_COMPLETED)
    goto extract_cleanup;

  /*
   * TODO check if the container can contain the subp stream
   */
  if (encode->subp_codec != G_TYPE_NONE && encode->subp_streams != NULL)
  {
    result = ogmrip_main_extract_subp_streams (data, encode, container, error);
    if (result != OGMJOB_RESULT_COMPLETED)
      goto extract_cleanup;
  }

  /*
   * TODO check if the container can contain the audio stream
   */
  if (encode->audio_codec != G_TYPE_NONE && encode->audio_streams != NULL)
  {
    if (!gconf_client_get_bool_default (data->gconf, OGMRIP_GCONF_NOAUDIO, OGMRIP_DEFAULT_NOAUDIO))
    {
      result = ogmrip_main_extract_audio_streams (data, encode, container, error);
      if (result != OGMJOB_RESULT_COMPLETED)
        goto extract_cleanup;
    }
  }

  if (encode->video_codec != G_TYPE_NONE)
  {
    result = ogmrip_main_extract_video_stream (data, encode, container, error);
    if (result != OGMJOB_RESULT_COMPLETED)
      goto extract_cleanup;
  }

  result = ogmrip_main_merge (data, container, error);

extract_cleanup:
  g_object_unref (container);

  return result != OGMJOB_RESULT_ERROR;
}

/*
 * Backups a DVD title
 */
static gboolean
ogmrip_main_backup (OGMRipData *data, OGMRipEncode *encode, GError **error)
{
  OGMDvdDisc *disc;
  OGMDvdAudioStream *astream;
  OGMDvdSubpStream *sstream;
  GList *link;

  OGMJobSpawn *spawn;
  gint result, nr;
  gchar *path;

  path = ogmrip_fs_mkdtemp ("dvdXXXXXX", error);
  if (!path)
    return FALSE;

  spawn = ogmrip_dvdcpy_new (encode->title, path);
  if (!spawn)
  {
    g_free (path);
    return FALSE;
  }

  ogmrip_progress_set_spawn (data->progress_dialog, spawn, _("<b>DVD backup</b>"));
  g_object_unref (spawn);

  result = ogmjob_spawn_run (spawn, error);
  if (result != OGMJOB_RESULT_COMPLETED)
  {
    g_free (path);
    return result != OGMJOB_RESULT_ERROR;
  }

  disc = ogmdvd_disc_open (path, error);
  g_free (path);

  if (!disc)
    return FALSE;

  nr = ogmdvd_title_get_nr (encode->title);
  ogmdvd_title_unref (encode->title);

  encode->title = ogmdvd_disc_get_nth_title (disc, nr);
  ogmdvd_disc_unref (disc);

  if (!encode->title)
    return FALSE;

  for (link = encode->audio_streams; link; link = link->next)
  {
    astream = link->data;
    nr = ogmdvd_audio_stream_get_nr (astream);
    ogmdvd_audio_stream_unref (astream);

    link->data = ogmdvd_title_get_nth_audio_stream (encode->title, nr);
  }

  for (link = encode->subp_streams; link; link = link->next)
  {
    sstream = link->data;
    nr = ogmdvd_subp_stream_get_nr (sstream);
    ogmdvd_subp_stream_unref (sstream);

    link->data = ogmdvd_title_get_nth_subp_stream (encode->title, nr);
  }

  return TRUE;
}

/*
 * Loads a DVD disk or a DVD structure
 */
static void
ogmrip_main_load (OGMRipData *data, const gchar *path)
{
  OGMDvdDisc *disc;
  GError *error = NULL;

  disc = ogmdvd_disc_open (path, &error);
  if (!disc)
  {
    if (error)
    {
      ogmrip_message_dialog (GTK_WINDOW (data->window), GTK_MESSAGE_ERROR, "<b>%s</b>\n\n%s", 
          _("Could not open the DVD"), _(error->message));
      g_error_free (error);
    }
    else
      ogmrip_message_dialog (GTK_WINDOW (data->window), GTK_MESSAGE_ERROR, "<b>%s</b>\n\n%s",
          _("Could not read the DVD"), _("Unknown error"));
  }
  else
  {
    gchar *label = NULL;
    gint nvid;

    if (data->disc)
      ogmdvd_disc_unref (data->disc);
    data->disc = disc;

    ogmdvd_title_chooser_set_disc (OGMDVD_TITLE_CHOOSER (data->title_chooser), disc);

    nvid = ogmdvd_disc_get_n_titles (disc);
    if (nvid > 0)
      label = ogmdvd_disc_get_label (disc);

    gtk_widget_set_sensitive (data->title_chooser, nvid > 0);
    gtk_widget_set_sensitive (data->title_entry, nvid > 0);
    gtk_widget_set_sensitive (data->extract_button, nvid > 0);
    gtk_action_set_sensitive (data->extract_action, nvid > 0);

    gtk_entry_set_text (GTK_ENTRY (data->title_entry), label ? label : "");
    g_free (label);
  }
}

/*
 * Clear the GUI
 */
static void
ogmrip_main_clear (OGMRipData *data)
{
  ogmdvd_title_chooser_set_disc (OGMDVD_TITLE_CHOOSER (data->title_chooser), NULL);

  gtk_widget_set_sensitive (data->title_chooser, FALSE);
  gtk_widget_set_sensitive (data->title_entry, FALSE);
  gtk_widget_set_sensitive (data->extract_button, FALSE);
  gtk_action_set_sensitive (data->extract_action, FALSE);

  gtk_entry_set_text (GTK_ENTRY (data->title_entry), "");
}

/*
 * Events
 */

static gboolean
ogmrip_main_delete_event (OGMRipData *data)
{
  if (data->encoding)
    return TRUE;

  return FALSE;
}

static void
ogmrip_main_destroyed (OGMRipData *data)
{
  if (data->gconf)
    g_object_unref (data->gconf);
  data->gconf = NULL;

  if (data->disc)
    ogmdvd_disc_unref (data->disc);
  data->disc = NULL;

  if (data->pref_dialog)
    gtk_widget_destroy (data->pref_dialog);

  g_free (data);
}

static void
ogmrip_main_options_dialog_construct (OGMRipData *data, OGMRipEncode *encode)
{
  data->options_dialog = ogmrip_options_new (encode->title);
  gtk_window_set_parent (GTK_WINDOW (data->options_dialog), GTK_WINDOW (data->window));
}

static void
ogmrip_main_progress_dialog_construct (OGMRipData *data, OGMRipEncode *encode)
{
  guint steps;

  steps = encode->passes + 1 +              /* video */
    (encode->target_number > 1 ? 1 : 0) +   /* split */
    g_list_length (encode->audio_streams) + /* audio */
    g_list_length (encode->subp_streams) +  /* subp  */
    encode->copy_dvd +                      /* copy  */
    3;   /* chapters + bitrate, crop, scale  + merge */

  data->progress_dialog = ogmrip_progress_new (encode->label, steps);
  g_signal_connect (data->progress_dialog, "destroy", G_CALLBACK (gtk_widget_destroyed), &data->progress_dialog);
  gtk_window_set_parent (GTK_WINDOW (data->progress_dialog), GTK_WINDOW (data->window));
}

static void
ogmrip_main_extract_activated (OGMRipData *data)
{
  OGMRipEncode encode;
  GError *error = NULL;
  gboolean result = TRUE;
  gint value;

  encode.title = ogmdvd_title_chooser_get_active (OGMDVD_TITLE_CHOOSER (data->title_chooser));

  ogmrip_main_options_dialog_construct (data, &encode);
  if (gtk_dialog_run (GTK_DIALOG (data->options_dialog)) == GTK_RESPONSE_OK)
  {
    encode.label = gtk_entry_get_text (GTK_ENTRY (data->title_entry));

    value = gconf_client_get_int_default (data->gconf, OGMRIP_GCONF_CONTAINER, OGMRIP_DEFAULT_CONTAINER);
    encode.container = ogmrip_get_nth_container (value);

    value = gconf_client_get_int_default (data->gconf, OGMRIP_GCONF_VIDEO_FORMAT, OGMRIP_DEFAULT_VIDEO_FORMAT);
    encode.video_codec = ogmrip_get_nth_video_codec (value);

    encode.audio_streams = ogmrip_chooser_list_get_streams (OGMRIP_CHOOSER_LIST (data->audio_list));
    value = gconf_client_get_int_default (data->gconf, OGMRIP_GCONF_AUDIO_FORMAT, OGMRIP_DEFAULT_AUDIO_FORMAT);
    encode.audio_codec = ogmrip_get_nth_audio_codec (value);

    encode.subp_streams = ogmrip_chooser_list_get_streams (OGMRIP_CHOOSER_LIST (data->subp_list));
    value = gconf_client_get_int_default (data->gconf, OGMRIP_GCONF_SUBP_FORMAT, OGMRIP_DEFAULT_SUBP_FORMAT);
    encode.subp_codec = ogmrip_get_nth_subp_codec (value);

    if ((result = ogmrip_main_check_container (data, &encode, &error)) &&
        (result = ogmrip_main_check_streams (data, &encode, &error)))
    {
      encode.outfile = ogmrip_main_get_filename (data, &encode);
      if (ogmrip_main_check_filename (data, &encode))
      {
        encode.ensure_sync = gconf_client_get_bool_default (data->gconf, OGMRIP_GCONF_ENSURE_SYNC, OGMRIP_DEFAULT_ENSURE_SYNC);
        encode.target_number = gconf_client_get_int_default (data->gconf, OGMRIP_GCONF_TNUMBER, OGMRIP_DEFAULT_TNUMBER);
        encode.target_size = gconf_client_get_int_default (data->gconf, OGMRIP_GCONF_TSIZE, OGMRIP_DEFAULT_TSIZE);
    
        ogmrip_chapter_list_get_selected (OGMRIP_CHAPTER_LIST (data->chapter_list), &encode.start_chap, &encode.end_chap);

        encode.rip_size = ogmrip_main_get_rip_size (data, &encode);
    
        encode.copy_dvd = ogmrip_main_get_copy_dvd (data, &error);
        if (!error)
        {
          if ((result = ogmrip_main_check_space (data, &encode, &error)))
          {
            encode.passes = gconf_client_get_int_default (data->gconf, OGMRIP_GCONF_VIDEO_PASSES, OGMRIP_DEFAULT_VIDEO_PASSES);

            ogmrip_main_progress_dialog_construct (data, &encode);
            gtk_widget_show (data->progress_dialog);

            if (encode.copy_dvd)
              result = ogmrip_main_backup (data, &encode, &error);

            if (result)
              result = ogmrip_main_extract (data, &encode, &error);
          }
        }
      }

      g_free (encode.outfile);
    }

    g_list_foreach (encode.audio_streams, (GFunc) ogmdvd_audio_stream_unref, NULL);
    g_list_free (encode.audio_streams);

    g_list_foreach (encode.subp_streams, (GFunc) ogmdvd_subp_stream_unref, NULL);
    g_list_free (encode.subp_streams);
  }

  if (encode.title)
    ogmdvd_title_unref (encode.title);

  if (data->progress_dialog)
    gtk_widget_destroy (data->progress_dialog);
  data->progress_dialog = NULL;

  if (data->options_dialog)
    gtk_widget_destroy (data->options_dialog);
  data->options_dialog = NULL;

  if (!result || error)
  {
    if (!error)
      ogmrip_message_dialog (GTK_WINDOW (data->window), GTK_MESSAGE_ERROR, _("Unknown error"));
    else
    {
      ogmrip_message_dialog (GTK_WINDOW (data->window), GTK_MESSAGE_ERROR, error->message);
      g_error_free (error);
    }
  }
}

static void
ogmrip_main_eject_activated (OGMRipData *data, GtkWidget *dialog)
{
  if (data->disc)
  {
    NautilusBurnDrive *drive;
    const gchar *device;

    drive = ogmdvd_drive_dialog_get_drive (OGMDVD_DRIVE_DIALOG (dialog));
    if (drive)
    {
      device = ogmdvd_disc_get_device (data->disc);
      if (strcmp (device, drive->device) == 0)
      {
        ogmdvd_disc_unref (data->disc);
        data->disc = NULL;

        ogmrip_main_clear (data);
      }
      g_object_unref (drive);
    }
  }
}

static void
ogmrip_main_load_activated (OGMRipData *data)
{
  if (!data->encoding)
  {
    GtkWidget *dialog;

    dialog = ogmdvd_drive_dialog_new ();
    gtk_window_set_icon_from_stock (GTK_WINDOW (dialog), GTK_STOCK_REFRESH);
    gtk_window_set_parent (GTK_WINDOW (dialog), GTK_WINDOW (data->window));

    g_signal_connect_swapped (dialog, "eject", G_CALLBACK (ogmrip_main_eject_activated), data);

    if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_OK)
    {
      NautilusBurnDrive *drive;

      drive = ogmdvd_drive_dialog_get_drive (OGMDVD_DRIVE_DIALOG (dialog));
      if (drive)
      {
        ogmrip_main_load (data, drive->device);
        g_object_unref (drive);
      }
    }
    gtk_widget_destroy (dialog);
  }
}

static void
ogmrip_main_open_activated (OGMRipData *data)
{
  if (!data->encoding)
  {
    GtkWidget *dialog;

    dialog = gtk_file_chooser_dialog_new (_("Open DVD Structure"), 
        GTK_WINDOW (data->window), GTK_FILE_CHOOSER_ACTION_SELECT_FOLDER, 
        GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL, GTK_STOCK_OPEN, GTK_RESPONSE_OK, NULL);
    gtk_window_set_icon_from_stock (GTK_WINDOW (dialog), GTK_STOCK_OPEN);
    gtk_window_set_parent (GTK_WINDOW (dialog), GTK_WINDOW (data->window));

    if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_OK)
    {
      gchar *dir;

      dir = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));
      if (dir)
        ogmrip_main_load (data, dir);
    }
    gtk_widget_destroy (dialog);
  }
}

static void
ogmrip_main_select_all_activated (OGMRipData *data)
{
  OGMDvdTitle *title;
  OGMDvdTime time_;

  ogmrip_chapter_list_select_all (OGMRIP_CHAPTER_LIST (data->chapter_list));

  title = ogmdvd_title_chooser_get_active (OGMDVD_TITLE_CHOOSER (data->title_chooser));
  if (ogmdvd_title_get_length (title, &time_) > 0)
  {
    gchar *str;

    str = g_strdup_printf ("%02d:%02d:%02d", time_.hour, time_.min, time_.sec);
    gtk_label_set_text (GTK_LABEL (data->length_label), str);
    g_free (str);
  }
  ogmdvd_title_unref (title);

  gtk_widget_set_sensitive (data->extract_button, TRUE);
  gtk_widget_set_sensitive (data->relative_check, FALSE);
  gtk_action_set_sensitive (data->extract_action, TRUE);
}

static void
ogmrip_main_deselect_all_activated (OGMRipData *data)
{
  ogmrip_chapter_list_deselect_all (OGMRIP_CHAPTER_LIST (data->chapter_list));

  gtk_label_set_text (GTK_LABEL (data->length_label), "");
  gtk_widget_set_sensitive (data->extract_button, FALSE);
  gtk_widget_set_sensitive (data->relative_check, FALSE);
  gtk_action_set_sensitive (data->extract_action, FALSE);
}

static void
ogmrip_main_pref_activated (OGMRipData *data)
{
  gtk_widget_show (data->pref_dialog);
}

static void
ogmrip_main_about_activated (OGMRipData *data)
{
  static GdkPixbuf *icon = NULL;

  const gchar *authors[] = 
  { 
    "Olivier Rolland <billl@users.sf.net>", 
    NULL 
  };
  gchar *translator_credits = _("translator-credits");

  const gchar *documenters[] = 
  { 
    "Olivier Brisson <olivier.brisson@gmail.com>",
    NULL 
  };

  if (!icon)
    icon = gdk_pixbuf_new_from_file (OGMRIP_DATA_DIR "/" OGMRIP_ICON_FILE, NULL);

  if (strcmp (translator_credits, "translator-credits") == 0)
    translator_credits = NULL;

  gtk_show_about_dialog (GTK_WINDOW (data->window), 
      "name", PACKAGE_NAME,
      "version", PACKAGE_VERSION,
      "comments", _("A DVD Encoder for GNOME"),
      "copyright", "(c) 2004-2006 Olivier Rolland",
      "website", "http://ogmrip.sourceforge.net",
      "translator-credits", translator_credits,
      "documenters", documenters,
      "authors", authors,
      "logo", icon,
      NULL);
}

static void
ogmrip_main_title_chooser_changed (OGMRipData *data)
{
  GtkWidget *audio_chooser;
  GtkWidget *subp_chooser;

  OGMDvdTitle *title;
  OGMDvdTime time_;
  gchar *str;

  ogmrip_chooser_list_clear (OGMRIP_CHOOSER_LIST (data->audio_list));

  audio_chooser = ogmdvd_audio_chooser_new ();
  gtk_container_add (GTK_CONTAINER (data->audio_list), audio_chooser);
  gtk_widget_show (audio_chooser);

  gtk_widget_set_sensitive (data->audio_list, data->disc != NULL);

  ogmrip_chooser_list_clear (OGMRIP_CHOOSER_LIST (data->subp_list));

  subp_chooser = ogmdvd_subtitle_chooser_new ();
  gtk_container_add (GTK_CONTAINER (data->subp_list), subp_chooser);
  gtk_widget_show (subp_chooser);

  gtk_widget_set_sensitive (data->subp_list, data->disc != NULL);

  ogmdvd_chapter_list_clear (OGMDVD_CHAPTER_LIST (data->chapter_list));

  gtk_label_set_text (GTK_LABEL (data->length_label), "");

  if (data->disc)
  {
    gint def, pref;
    GType container, codec;

    title = ogmdvd_title_chooser_get_active (OGMDVD_TITLE_CHOOSER (data->title_chooser));

    if (ogmdvd_title_get_length (title, &time_) > 0)
    {
      str = g_strdup_printf ("%02d:%02d:%02d", time_.hour, time_.min, time_.sec);
      gtk_label_set_text (GTK_LABEL (data->length_label), str);
      g_free (str);
    }

    def = gconf_client_get_int_default (data->gconf, 
        OGMRIP_GCONF_CONTAINER, OGMRIP_DEFAULT_CONTAINER);
    container = ogmrip_get_nth_container (def);

    pref = gconf_client_get_int_default (data->gconf, 
        OGMRIP_GCONF_PREF_AUDIO, OGMRIP_DEFAULT_PREF_AUDIO);
    ogmdvd_audio_chooser_set_title (OGMDVD_AUDIO_CHOOSER (audio_chooser), title, pref);

    pref = gconf_client_get_int_default (data->gconf, 
        OGMRIP_GCONF_PREF_SUBP, OGMRIP_DEFAULT_PREF_SUBP);
    ogmdvd_subtitle_chooser_set_title (OGMDVD_SUBTITLE_CHOOSER (subp_chooser), title, pref);

    /* Only one audio stream in an AVI container */
    ogmrip_chooser_list_set_max (OGMRIP_CHOOSER_LIST (data->audio_list), 
        ogmrip_get_container_max_audio (container));

    /* Only one subp stream in an AVI container */
    ogmrip_chooser_list_set_max (OGMRIP_CHOOSER_LIST (data->subp_list), 
        ogmrip_get_container_max_subp (container));

    codec = ogmrip_get_nth_audio_codec (gconf_client_get_int_default (data->gconf,
          OGMRIP_GCONF_AUDIO_FORMAT, OGMRIP_DEFAULT_AUDIO_FORMAT));
    gtk_widget_set_sensitive (data->audio_list, ogmrip_can_contain_audio (container, codec));

    codec = ogmrip_get_nth_subp_codec (gconf_client_get_int_default (data->gconf,
          OGMRIP_GCONF_SUBP_FORMAT, OGMRIP_DEFAULT_SUBP_FORMAT));
    gtk_widget_set_sensitive (data->subp_list, ogmrip_can_contain_subp (container, codec));

    ogmdvd_chapter_list_set_title (OGMDVD_CHAPTER_LIST (data->chapter_list), title);
    ogmrip_chapter_list_select_all (OGMRIP_CHAPTER_LIST (data->chapter_list));

    ogmdvd_title_unref (title);

    gtk_widget_set_sensitive (data->relative_check, FALSE);
  }
}

static void
ogmrip_main_audio_chooser_added (OGMRipData *data, OGMDvdAudioChooser *chooser)
{
  OGMDvdTitle *title;

  title = ogmdvd_title_chooser_get_active (OGMDVD_TITLE_CHOOSER (data->title_chooser));
  if (title)
  {
    ogmdvd_audio_chooser_set_title (chooser, title, 0);
    ogmdvd_title_unref (title);
  }
}

static void
ogmrip_main_subp_chooser_added (OGMRipData *data, OGMDvdSubtitleChooser *chooser)
{
  OGMDvdTitle *title;

  title = ogmdvd_title_chooser_get_active (OGMDVD_TITLE_CHOOSER (data->title_chooser));
  if (title)
  {
    ogmdvd_subtitle_chooser_set_title (chooser, title, 0);
    ogmdvd_title_unref (title);
  }
}

static void
ogmrip_main_chapter_selection_changed (OGMRipData *data)
{
  OGMDvdTitle *title;
  OGMDvdTime time_;

  guint start_chap;
  gint end_chap;

  gboolean sensitive;
  gchar *str;

  sensitive = ogmrip_chapter_list_get_selected (OGMRIP_CHAPTER_LIST (data->chapter_list), &start_chap, &end_chap);
  if (sensitive)
  {
    title = ogmdvd_title_chooser_get_active (OGMDVD_TITLE_CHOOSER (data->title_chooser));
    if (ogmdvd_title_get_chapters_length (title, start_chap, end_chap, &time_) > 0)
    {
      str = g_strdup_printf ("%02d:%02d:%02d", time_.hour, time_.min, time_.sec);
      gtk_label_set_text (GTK_LABEL (data->length_label), str);
      g_free (str);
    }
    ogmdvd_title_unref (title);
  }

  gtk_widget_set_sensitive (data->extract_button, sensitive);
  gtk_action_set_sensitive (data->extract_action, sensitive);

  gtk_widget_set_sensitive (data->relative_check, sensitive && (start_chap > 0 || end_chap != -1));
}

static void
ogmrip_main_container_notified (GConfClient *gconf, guint id, GConfEntry *entry, OGMRipData *data)
{
  if (data->disc)
  {
    GType container, codec;
    gint max, value;

    value = gconf_value_get_int (entry->value);
    container = ogmrip_get_nth_container (value);
    max = ogmrip_get_container_max_subp (container);

    /* Disable extra audio streams if in an AVI container */
    ogmrip_chooser_list_set_max (OGMRIP_CHOOSER_LIST (data->audio_list), max);

    /* Disable extra subp streams if in an AVI container */
    ogmrip_chooser_list_set_max (OGMRIP_CHOOSER_LIST (data->subp_list), max);

    codec = ogmrip_get_nth_audio_codec (gconf_client_get_int_default (data->gconf,
          OGMRIP_GCONF_AUDIO_FORMAT, OGMRIP_DEFAULT_AUDIO_FORMAT));
    gtk_widget_set_sensitive (data->audio_list, ogmrip_can_contain_audio (container, codec));

    codec = ogmrip_get_nth_subp_codec (gconf_client_get_int_default (data->gconf,
          OGMRIP_GCONF_SUBP_FORMAT, OGMRIP_DEFAULT_SUBP_FORMAT));
    gtk_widget_set_sensitive (data->subp_list, ogmrip_can_contain_subp (container, codec));
  }
}

static void
ogmrip_main_item_selected (GtkWidget *item, GtkWidget *statusbar)
{
  gchar *hint;

  hint = g_object_get_data (G_OBJECT (item), "__menu_hint__");
  if (hint)
  {
    guint context_id;

    context_id = gtk_statusbar_get_context_id (GTK_STATUSBAR (statusbar), "__menu_hint__");
    gtk_statusbar_push (GTK_STATUSBAR (statusbar), context_id, hint);
  }
}

static void
ogmrip_main_item_deselected (GtkWidget *item, GtkWidget *statusbar)
{
  guint context_id;

  context_id = gtk_statusbar_get_context_id (GTK_STATUSBAR (statusbar), "__menu_hint__");
  gtk_statusbar_pop (GTK_STATUSBAR (statusbar), context_id);
}

static void
ogmrip_main_connect_proxy (GtkUIManager *uimanager, GtkAction *action, GtkWidget *proxy, GtkWidget *statusbar)
{
  if (GTK_IS_MENU_ITEM (proxy))
  {
    gchar *hint;

    g_object_get (action, "tooltip", &hint, NULL);
    if (hint)
    {
      g_object_set_data_full (G_OBJECT (proxy), "__menu_hint__", hint, (GDestroyNotify) g_free);

      g_signal_connect (proxy, "select", G_CALLBACK (ogmrip_main_item_selected), statusbar);
      g_signal_connect (proxy, "deselect", G_CALLBACK (ogmrip_main_item_deselected), statusbar);
    }
  }
}

static OGMRipData *
ogmrip_main_new (void)
{
  GtkActionEntry action_entries[] =
  {
    { "FileMenu",    NULL,                  N_("_File"),         NULL,          NULL,                           NULL },
    { "Load",        GTK_STOCK_CDROM,       N_("_Load"),         "<ctl>L",      N_("Load a DVD disk"),          NULL },
    { "Open",        GTK_STOCK_OPEN,        NULL,                NULL,          N_("Open a DVD structure"),     NULL },
    { "Extract",     GTK_STOCK_CONVERT,     N_("E_xtract"),      "<ctl>Return", N_("Extract selected streams"), NULL },
    { "Quit",        GTK_STOCK_QUIT,        NULL,                NULL,          N_("Exit OGMRip"),              NULL },
    { "EditMenu",    NULL,                  N_("_Edit"),         NULL,          NULL,                           NULL },
    { "SelectAll",   NULL,                  N_("Select _All"),   "<ctl>A",      N_("Select all chapters"),      NULL },
    { "DeselectAll", NULL,                  N_("_Deselect All"), "<ctl>D",      N_("Deselect all chapters"),    NULL },
    { "Preferences", GTK_STOCK_PREFERENCES, NULL,                NULL,          N_("Edit the preferences"),     NULL },
    { "HelpMenu",    NULL,                  N_("_Help"),         NULL,          NULL,                           NULL },
    { "About",       GTK_STOCK_ABOUT,       N_("_About"),        NULL,          N_("About OGMRip"),             NULL },
  };

  OGMRipData *data;
  GtkWidget *widget;
  GladeXML *xml;

  GtkAction *action;
  GtkActionGroup *action_group;
  GtkAccelGroup *accel_group;
  GtkUIManager *ui_manager;

  xml = glade_xml_new (OGMRIP_DATA_DIR "/" OGMRIP_GLADE_FILE, NULL, NULL);
  if (!xml)
  {
    g_warning ("Could not find " OGMRIP_GLADE_FILE);
    return NULL;
  }

  data = g_new0 (OGMRipData, 1);

  data->gconf = gconf_client_get_default ();
  gconf_client_add_dir (data->gconf, OGMRIP_GCONF_ROOT, 
      GCONF_CLIENT_PRELOAD_ONELEVEL, NULL);

  gconf_client_notify_add (data->gconf, OGMRIP_GCONF_CONTAINER, 
      (GConfClientNotifyFunc) ogmrip_main_container_notified, data, NULL, NULL);

  data->window = glade_xml_get_widget (xml, "main-window");
  gtk_window_set_default_size (GTK_WINDOW (data->window), 350, 500);
  gtk_window_set_icon_from_file (GTK_WINDOW (data->window), OGMRIP_DATA_DIR "/" OGMRIP_ICON_FILE, NULL);

  g_signal_connect_swapped (data->window, "delete-event", G_CALLBACK (ogmrip_main_delete_event), data);
  g_signal_connect_swapped (data->window, "destroy", G_CALLBACK (ogmrip_main_destroyed), data);
  g_signal_connect (data->window, "destroy", G_CALLBACK (gtk_main_quit), NULL);

  action_group = gtk_action_group_new ("MenuActions");
  gtk_action_group_set_translation_domain (action_group, GETTEXT_PACKAGE);
  gtk_action_group_add_actions (action_group, action_entries, G_N_ELEMENTS (action_entries), NULL);

  ui_manager = gtk_ui_manager_new ();

  widget = glade_xml_get_widget (xml, "statusbar");
  g_signal_connect (ui_manager, "connect-proxy", G_CALLBACK (ogmrip_main_connect_proxy), widget);

  gtk_ui_manager_insert_action_group (ui_manager, action_group, 0);
  gtk_ui_manager_add_ui_from_file (ui_manager, OGMRIP_DATA_DIR "/" OGMRIP_UI_FILE, NULL);

  accel_group = gtk_ui_manager_get_accel_group (ui_manager);
  gtk_window_add_accel_group (GTK_WINDOW (data->window), accel_group);

  widget = gtk_ui_manager_get_widget (ui_manager, "/Menubar");
  gtk_box_pack_start (GTK_BOX (GTK_BIN (data->window)->child), widget, FALSE, FALSE, 0);

  action =  gtk_action_group_get_action (action_group, "Load");
  g_signal_connect_swapped (action, "activate", G_CALLBACK (ogmrip_main_load_activated), data);

  action =  gtk_action_group_get_action (action_group, "Open");
  g_signal_connect_swapped (action, "activate", G_CALLBACK (ogmrip_main_open_activated), data);

  action =  gtk_action_group_get_action (action_group, "Quit");
  g_signal_connect_swapped (action, "activate", G_CALLBACK (gtk_widget_destroy), data->window);

  action =  gtk_action_group_get_action (action_group, "Preferences");
  g_signal_connect_swapped (action, "activate", G_CALLBACK (ogmrip_main_pref_activated), data);

  action =  gtk_action_group_get_action (action_group, "SelectAll");
  g_signal_connect_swapped (action, "activate", G_CALLBACK (ogmrip_main_select_all_activated), data);

  action =  gtk_action_group_get_action (action_group, "DeselectAll");
  g_signal_connect_swapped (action, "activate", G_CALLBACK (ogmrip_main_deselect_all_activated), data);

  action =  gtk_action_group_get_action (action_group, "About");
  g_signal_connect_swapped (action, "activate", G_CALLBACK (ogmrip_main_about_activated), data);

  data->extract_action =  gtk_action_group_get_action (action_group, "Extract");
  g_signal_connect_swapped (data->extract_action, "activate", G_CALLBACK (ogmrip_main_extract_activated), data);
  gtk_action_set_sensitive (data->extract_action, FALSE);

  widget = glade_xml_get_widget (xml, "load-button");
  g_signal_connect_swapped (widget, "clicked", G_CALLBACK (ogmrip_main_load_activated), data);

  widget = glade_xml_get_widget (xml, "open-button");
  g_signal_connect_swapped (widget, "clicked", G_CALLBACK (ogmrip_main_open_activated), data);

  data->extract_button = glade_xml_get_widget (xml, "extract-button");
  g_signal_connect_swapped (data->extract_button, "clicked", G_CALLBACK (ogmrip_main_extract_activated), data);

  widget = data->title_chooser = glade_xml_get_widget (xml, "title-chooser");
  gtk_widget_set_sensitive (widget, FALSE);
  gtk_widget_show (widget);

  g_signal_connect_swapped (widget, "changed", G_CALLBACK (ogmrip_main_title_chooser_changed), data);

  widget = glade_xml_get_widget (xml, "table");

  data->audio_list = ogmrip_chooser_list_new (OGMDVD_TYPE_AUDIO_CHOOSER);
  g_signal_connect_swapped (data->audio_list, "add", G_CALLBACK (ogmrip_main_audio_chooser_added), data);
  gtk_table_attach (GTK_TABLE (widget), data->audio_list, 1, 3, 2, 3, GTK_EXPAND | GTK_FILL, 0, 0, 0);
  gtk_widget_set_sensitive (data->audio_list, FALSE);
  gtk_widget_show (data->audio_list);

  data->subp_list = ogmrip_chooser_list_new (OGMDVD_TYPE_SUBTITLE_CHOOSER);
  g_signal_connect_swapped (data->subp_list, "add", G_CALLBACK (ogmrip_main_subp_chooser_added), data);
  gtk_table_attach (GTK_TABLE (widget), data->subp_list, 1, 3, 3, 4, GTK_EXPAND | GTK_FILL, 0, 0, 0);
  gtk_widget_set_sensitive (data->subp_list, FALSE);
  gtk_widget_show (data->subp_list);

  widget = ogmdvd_audio_chooser_new ();
  gtk_container_add (GTK_CONTAINER (data->audio_list), widget);
  gtk_widget_show (widget);

  widget = ogmdvd_subtitle_chooser_new ();
  gtk_container_add (GTK_CONTAINER (data->subp_list), widget);
  gtk_widget_show (widget);

  data->length_label = glade_xml_get_widget (xml, "length-label");
  data->relative_check = glade_xml_get_widget (xml, "relative-check");
  data->title_entry = glade_xml_get_widget (xml, "title-entry");

  data->chapter_list = glade_xml_get_widget (xml, "chapter-list");
  g_signal_connect_swapped (data->chapter_list, "selection-changed", 
      G_CALLBACK (ogmrip_main_chapter_selection_changed), data);
  gtk_widget_show (data->chapter_list);

  g_object_unref (xml);

  return data;
}

int
main (int argc, char *argv[])
{
  OGMRipData *data;

#ifdef ENABLE_NLS
  bindtextdomain (GETTEXT_PACKAGE, LOCALEDIR);
  bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
  textdomain (GETTEXT_PACKAGE);
#endif /* ENABLE_NLS */

  gtk_init (&argc, &argv);

  ogmrip_init ();

  data = ogmrip_main_new ();

  if (argc > 1)
    if (g_file_test (argv[1], G_FILE_TEST_EXISTS))
      ogmrip_main_load (data, argv[1]);

  data->pref_dialog = ogmrip_pref_new ();
  gtk_window_set_parent (GTK_WINDOW (data->pref_dialog), GTK_WINDOW (data->window));

  gtk_widget_show (data->window);

  gtk_main ();

  ogmrip_uninit ();

  return 0;
}

