/* OGMRip - A library for DVD ripping and encoding
 * Copyright (C) 2004-2006 Olivier Rolland <billl@users.sf.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "ogmrip-vobsub.h"
#include "ogmrip-backend.h"

#include "ogmjob-queue.h"
#include "ogmjob-enums.h"
#include "ogmjob-exec.h"

#include <fcntl.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>

#include <glib/gstdio.h>

static gint ogmrip_vobsub_run      (OGMJobSpawn *spawn);
static void ogmrip_vobsub_finalize (GObject     *gobject);

G_DEFINE_TYPE (OGMRipVobSub, ogmrip_vobsub, OGMRIP_TYPE_SUBP)

static void
ogmrip_vobsub_class_init (OGMRipVobSubClass *klass)
{
  GObjectClass *gobject_class;
  OGMJobSpawnClass *spawn_class;

  gobject_class = G_OBJECT_CLASS (klass);
  spawn_class = OGMJOB_SPAWN_CLASS (klass);

  gobject_class->finalize = ogmrip_vobsub_finalize;
  spawn_class->run = ogmrip_vobsub_run;
}

static void
ogmrip_vobsub_init (OGMRipVobSub *vobsub)
{
}

static void
ogmrip_vobsub_finalize (GObject *gobject)
{
  const gchar *output;

  output = ogmrip_codec_get_output (OGMRIP_CODEC (gobject));
  if (output)
  {
    if (ogmrip_codec_get_unlink_on_unref (OGMRIP_CODEC (gobject)))
    {
      gchar *filename;

      filename = g_strconcat (output, ".idx", NULL);
      if (g_file_test (filename, G_FILE_TEST_IS_REGULAR))
        g_unlink (filename);
      g_free (filename);

      filename = g_strconcat (output, ".sub", NULL);
      if (g_file_test (filename, G_FILE_TEST_IS_REGULAR))
        g_unlink (filename);
      g_free (filename);
    }
  }

  G_OBJECT_CLASS (ogmrip_vobsub_parent_class)->finalize (gobject);
}

static gint
ogmrip_vobsub_run (OGMJobSpawn *spawn)
{
  OGMJobSpawn *child;
  gchar **argv;
  gint result;

  argv = ogmrip_backend_vobsub_command (OGMRIP_SUBP (spawn), NULL, NULL);
  if (!argv)
    return OGMJOB_RESULT_ERROR;

  child = ogmjob_exec_newv (argv);
  ogmjob_exec_add_watch_full (OGMJOB_EXEC (child), (OGMJobWatch) ogmrip_backend_vobsub_watch, spawn, TRUE, FALSE, FALSE);
  ogmjob_container_add (OGMJOB_CONTAINER (spawn), child);
  g_object_unref (child);

  result = OGMJOB_SPAWN_CLASS (ogmrip_vobsub_parent_class)->run (spawn);
  if (result == OGMJOB_RESULT_COMPLETED)
  {
    struct stat buf; 
    const gchar *basename;
    gchar *idxname, *subname;

    basename = ogmrip_codec_get_output (OGMRIP_CODEC (spawn));
    idxname = g_strconcat (basename, ".idx", NULL);
    subname = g_strconcat (basename, ".sub", NULL);

    if ((g_file_test (idxname, G_FILE_TEST_IS_REGULAR) && g_stat (idxname, &buf) == 0 && buf.st_size > 0) &&
        (g_file_test (subname, G_FILE_TEST_IS_REGULAR) && g_stat (subname, &buf) == 0 && buf.st_size > 0))
    {
      gint fd;

      fd = g_open (basename, O_WRONLY);
      write (fd, "foo", 3);
      close (fd);
    }

    g_free (idxname);
    g_free (subname);
  }

  ogmjob_container_remove (OGMJOB_CONTAINER (spawn), child);

  return result;
}

OGMJobSpawn *
ogmrip_vobsub_new (OGMDvdSubpStream *subp, const gchar *output)
{
  OGMJobSpawn *spawn;

  spawn = g_object_new (OGMRIP_TYPE_VOBSUB, NULL);
  ogmrip_codec_set_output (OGMRIP_CODEC (spawn), output);
  ogmrip_subp_set_dvd_subp_stream (OGMRIP_SUBP (spawn), subp);

  return spawn;
}

