/* OGMRip - A library for DVD ripping and encoding
 * Copyright (C) 2004-2006 Olivier Rolland <billl@users.sf.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "ogmrip-video.h"
#include "ogmrip-backend.h"

#define OGMRIP_VIDEO_GET_PRIVATE(o) \
  (G_TYPE_INSTANCE_GET_PRIVATE ((o), OGMRIP_TYPE_VIDEO, OGMRipVideoPriv))

#define ROUND(x) ((gint) ((x) + 0.5) != (gint) (x) ? ((gint) ((x) + 0.5)) : ((gint) (x)))

struct _OGMRipVideoPriv
{
  gchar *log;
  guint angle;
  guint bitrate;
  guint nframes;
  guint pass;
  guint crop_x;
  guint crop_y;
  guint crop_width;
  guint crop_height;
  guint scale_width;
  guint scale_height;
  guint max_b_frames;
  gboolean grayscale;
  gboolean cartoon;
  gboolean denoise;
  gboolean trellis;
  gboolean pullup;
  gboolean turbo;
  gboolean qpel;
  gboolean v4mv;
  OGMDvdAudioStream *astream;
  OGMRipQualityType quality;
  OGMRipScalerType scaler;
  OGMRipDeintType deint;
};

static void ogmrip_video_finalize (GObject *gobject);

G_DEFINE_ABSTRACT_TYPE (OGMRipVideo, ogmrip_video, OGMRIP_TYPE_CODEC)

static void
ogmrip_video_class_init (OGMRipVideoClass *klass)
{
  GObjectClass *gobject_class;

  gobject_class = G_OBJECT_CLASS (klass);

  gobject_class->finalize = ogmrip_video_finalize;

  g_type_class_add_private (klass, sizeof (OGMRipVideoPriv));
}

static void
ogmrip_video_init (OGMRipVideo *video)
{
  video->priv = OGMRIP_VIDEO_GET_PRIVATE (video);
  video->priv->scaler = OGMRIP_SCALER_GAUSS;
  video->priv->astream = NULL;
  video->priv->bitrate = 800000;
  video->priv->trellis = TRUE;
  video->priv->turbo = TRUE;
  video->priv->v4mv = TRUE;
  video->priv->max_b_frames = 2;
  video->priv->nframes = 25;
  video->priv->angle = 1;
}

static void
ogmrip_video_finalize (GObject *gobject)
{
  OGMRipVideo *video;

  video = OGMRIP_VIDEO (gobject);

  g_free (video->priv->log);
  video->priv->log = NULL;

  G_OBJECT_CLASS (ogmrip_video_parent_class)->finalize (gobject);
}

OGMDvdAudioStream * 
ogmrip_video_get_ensure_sync (OGMRipVideo *video)
{
  g_return_val_if_fail (OGMRIP_IS_VIDEO (video), FALSE);

  return video->priv->astream;
}

void
ogmrip_video_set_ensure_sync (OGMRipVideo *video, OGMDvdAudioStream *stream)
{
  g_return_if_fail (OGMRIP_IS_VIDEO (video));

  if (stream)
    ogmdvd_audio_stream_ref (stream);

  if (video->priv->astream)
    ogmdvd_audio_stream_unref (video->priv->astream);

  video->priv->astream = stream;
}

void
ogmrip_video_set_angle (OGMRipVideo *video, guint angle)
{
  OGMDvdTitle *title;

  g_return_if_fail (OGMRIP_IS_VIDEO (video));

  title = ogmrip_codec_get_input (OGMRIP_CODEC (video));

  g_return_if_fail (angle > 0 && angle <= ogmdvd_title_get_n_angles (title));

  video->priv->angle = angle;
}

gint
ogmrip_video_get_angle (OGMRipVideo *video)
{
  g_return_val_if_fail (OGMRIP_IS_VIDEO (video), -1);

  return video->priv->angle;
}

G_CONST_RETURN gchar *
ogmrip_video_get_log (OGMRipVideo *video)
{
  g_return_val_if_fail (OGMRIP_IS_VIDEO (video), NULL);

  return video->priv->log;
}

void
ogmrip_video_set_log (OGMRipVideo *video, const gchar *log)
{
  g_return_if_fail (OGMRIP_IS_VIDEO (video));

  g_free (video->priv->log);
  video->priv->log = log ? g_strdup (log) : NULL;
}

void
ogmrip_video_set_bitrate (OGMRipVideo *video, guint bitrate)
{
  g_return_if_fail (OGMRIP_IS_VIDEO (video));

  video->priv->bitrate = CLAMP (bitrate, 4000, 24000000);
}

gint
ogmrip_video_get_bitrate (OGMRipVideo *video)
{
  g_return_val_if_fail (OGMRIP_IS_VIDEO (video), -1);

  return video->priv->bitrate;
}

void
ogmrip_video_set_pass (OGMRipVideo *video, guint pass)
{
  g_return_if_fail (OGMRIP_IS_VIDEO (video));

  video->priv->pass = MIN (pass, 2);
}

gint
ogmrip_video_get_pass (OGMRipVideo *video)
{
  g_return_val_if_fail (OGMRIP_IS_VIDEO (video), -1);

  return video->priv->pass;
}

void
ogmrip_video_set_scaler (OGMRipVideo *video, OGMRipScalerType scaler)
{
  g_return_if_fail (OGMRIP_IS_VIDEO (video));

  video->priv->scaler = scaler;
}

gint
ogmrip_video_get_scaler (OGMRipVideo *video)
{
  g_return_val_if_fail (OGMRIP_IS_VIDEO (video), -1);

  return video->priv->scaler;
}

void
ogmrip_video_set_deinterlacer (OGMRipVideo *video, OGMRipDeintType deint)
{
  g_return_if_fail (OGMRIP_IS_VIDEO (video));

  video->priv->deint = deint;
}

gint
ogmrip_video_get_deinterlacer (OGMRipVideo *video)
{
  g_return_val_if_fail (OGMRIP_IS_VIDEO (video), -1);

  return video->priv->deint;
}

void
ogmrip_video_set_pullup (OGMRipVideo *video, gboolean pullup)
{
  g_return_if_fail (OGMRIP_IS_VIDEO (video));

  video->priv->pullup = pullup;
}

gboolean
ogmrip_video_get_pullup (OGMRipVideo *video)
{
  g_return_val_if_fail (OGMRIP_IS_VIDEO (video), FALSE);

  return video->priv->pullup;
}

void
ogmrip_video_set_trellis (OGMRipVideo *video, gboolean trellis)
{
  g_return_if_fail (OGMRIP_IS_VIDEO (video));

  video->priv->trellis = trellis;
}

gboolean
ogmrip_video_get_trellis (OGMRipVideo *video)
{
  g_return_val_if_fail (OGMRIP_IS_VIDEO (video), FALSE);

  return video->priv->trellis;
}

void
ogmrip_video_set_4mv (OGMRipVideo *video, gboolean v4mv)
{
  g_return_if_fail (OGMRIP_IS_VIDEO (video));

  video->priv->v4mv = v4mv;
}

gboolean
ogmrip_video_get_4mv (OGMRipVideo *video)
{
  g_return_val_if_fail (OGMRIP_IS_VIDEO (video), FALSE);

  return video->priv->v4mv;
}

void
ogmrip_video_set_qpel (OGMRipVideo *video, gboolean qpel)
{
  g_return_if_fail (OGMRIP_IS_VIDEO (video));

  video->priv->qpel = qpel;
}

gboolean
ogmrip_video_get_qpel (OGMRipVideo *video)
{
  g_return_val_if_fail (OGMRIP_IS_VIDEO (video), FALSE);

  return video->priv->qpel;
}

void
ogmrip_video_set_turbo (OGMRipVideo *video, gboolean turbo)
{
  g_return_if_fail (OGMRIP_IS_VIDEO (video));

  video->priv->turbo = turbo;
}

gboolean
ogmrip_video_get_turbo (OGMRipVideo *video)
{
  g_return_val_if_fail (OGMRIP_IS_VIDEO (video), FALSE);

  return video->priv->turbo;
}

void
ogmrip_video_set_grayscale (OGMRipVideo *video, gboolean grayscale)
{
  g_return_if_fail (OGMRIP_IS_VIDEO (video));

  video->priv->grayscale = grayscale;
}

gboolean
ogmrip_video_get_grayscale (OGMRipVideo *video)
{
  g_return_val_if_fail (OGMRIP_IS_VIDEO (video), FALSE);

  return video->priv->grayscale;
}

void
ogmrip_video_set_cartoon (OGMRipVideo *video, gboolean cartoon)
{
  g_return_if_fail (OGMRIP_IS_VIDEO (video));

  video->priv->cartoon = cartoon;
}

gboolean
ogmrip_video_get_cartoon (OGMRipVideo *video)
{
  g_return_val_if_fail (OGMRIP_IS_VIDEO (video), FALSE);

  return video->priv->cartoon;
}

void
ogmrip_video_set_denoise (OGMRipVideo *video, gboolean denoise)
{
  g_return_if_fail (OGMRIP_IS_VIDEO (video));

  video->priv->denoise = denoise;
}

gboolean
ogmrip_video_get_denoise (OGMRipVideo *video)
{
  g_return_val_if_fail (OGMRIP_IS_VIDEO (video), FALSE);

  return video->priv->denoise;
}

void
ogmrip_video_set_max_b_frames (OGMRipVideo *video, guint max_b_frames)
{
  g_return_if_fail (OGMRIP_IS_VIDEO (video));

  video->priv->max_b_frames = MIN (max_b_frames, 4);
}

gint
ogmrip_video_get_max_b_frames (OGMRipVideo *video)
{
  g_return_val_if_fail (OGMRIP_IS_VIDEO (video), -1);

  return video->priv->max_b_frames;
}

void
ogmrip_video_set_quality (OGMRipVideo *video, OGMRipQualityType quality)
{
  g_return_if_fail (OGMRIP_IS_VIDEO (video));

  video->priv->quality = MIN (quality, OGMRIP_QUALITY_FAST);
}

gint
ogmrip_video_get_quality (OGMRipVideo *video)
{
  g_return_val_if_fail (OGMRIP_IS_VIDEO (video), -1);

  return video->priv->quality;
}

void
ogmrip_video_get_raw_size (OGMRipVideo *video, guint *width, guint *height)
{
  OGMDvdTitle *title;

  g_return_if_fail (OGMRIP_IS_VIDEO (video));
  g_return_if_fail (width != NULL);
  g_return_if_fail (height != NULL);

  title = ogmrip_codec_get_input (OGMRIP_CODEC (video));

  g_return_if_fail (title != NULL);

  ogmdvd_title_get_size (title, width, height);
}

void
ogmrip_video_get_crop_size (OGMRipVideo *video, guint *x, guint *y, guint *width, guint *height)
{
  g_return_if_fail (OGMRIP_IS_VIDEO (video));
  g_return_if_fail (x != NULL);
  g_return_if_fail (y != NULL);
  g_return_if_fail (width != NULL);
  g_return_if_fail (height != NULL);

  *x = video->priv->crop_x;
  *y = video->priv->crop_y;
  *width = video->priv->crop_width;
  *height = video->priv->crop_height;
}

void
ogmrip_video_set_crop_size (OGMRipVideo *video, guint x, guint y, guint width, guint height)
{
  g_return_if_fail (OGMRIP_IS_VIDEO (video));

  if (width > 0 && height > 0)
  {
    video->priv->crop_x = x;
    video->priv->crop_y = y;
    video->priv->crop_width = width;
    video->priv->crop_height = height;
  }
}

void
ogmrip_video_get_scale_size (OGMRipVideo *video, guint *width, guint *height)
{
  g_return_if_fail (OGMRIP_IS_VIDEO (video));
  g_return_if_fail (width != NULL);
  g_return_if_fail (height != NULL);

  *width = video->priv->scale_width;
  *height = video->priv->scale_height;
}

void
ogmrip_video_set_scale_size (OGMRipVideo *video, guint width, guint height)
{
  g_return_if_fail (OGMRIP_IS_VIDEO (video));
  g_return_if_fail (height > 0);
  g_return_if_fail (width > 0);

  video->priv->scale_width = width;
  video->priv->scale_height = height;
}

void
ogmrip_video_autoscale (OGMRipVideo *video)
{
  OGMDvdTitle *title;
  guint anumerator, adenominator;
  guint rnumerator, rdenominator;
  guint crop_width, crop_height;
  guint raw_width, raw_height;
  gfloat ratio, bpp;

  g_return_if_fail (OGMRIP_IS_VIDEO (video));

  title = ogmrip_codec_get_input (OGMRIP_CODEC (video));
  g_return_if_fail (title != NULL);

  ogmrip_video_get_raw_size (video, &raw_width, &raw_height);

  crop_width = video->priv->crop_width > 0 ? video->priv->crop_width : raw_width;
  crop_height = video->priv->crop_height > 0 ? video->priv->crop_height : raw_height;

  ogmdvd_title_get_aspect_ratio (title, &anumerator, &adenominator);
  ogmdvd_title_get_framerate (title, &rnumerator, &rdenominator);

  ratio = (crop_width * raw_height * anumerator) / (gdouble) (crop_height * raw_width * adenominator);

  for (video->priv->scale_width = 320; video->priv->scale_width <= 720; video->priv->scale_width += 16)
  {
    video->priv->scale_height = 16 * ROUND (video->priv->scale_width / ratio / 16);

    bpp = (video->priv->bitrate * rdenominator) / 
      (gdouble) (video->priv->scale_width * video->priv->scale_height * rnumerator);

    if (bpp < 0.25)
      break;
  }

  video->priv->scale_width = MIN (video->priv->scale_width, 720);
}

void
ogmrip_video_autobitrate (OGMRipVideo *video, guint64 nonvideo_size, guint64 overhead_size, guint64 total_size)
{
  OGMDvdTitle *title;
  guint numerator, denominator;
  gdouble video_size, length;

  g_return_if_fail (OGMRIP_IS_VIDEO (video));

  title = ogmrip_codec_get_input (OGMRIP_CODEC (video));
  g_return_if_fail (title != NULL);

  video_size = total_size - nonvideo_size - overhead_size;
  length = ogmrip_codec_get_length (OGMRIP_CODEC (video), NULL);
  ogmdvd_title_get_framerate (title, &numerator, &denominator);

  ogmrip_video_set_bitrate (video, (video_size * numerator * 8.) / (length * denominator));
}

void
ogmrip_video_autocrop (OGMRipVideo *video, guint nframes)
{
  OGMJobSpawn *child;
  gchar **argv;

  g_return_if_fail (OGMRIP_IS_VIDEO (video));

  argv = ogmrip_backend_crop_command (video, NULL, NULL);

  if (nframes > 0)
    video->priv->nframes = nframes + 5;

  child = ogmjob_exec_newv (argv);
  ogmjob_exec_add_watch_full (OGMJOB_EXEC (child), (OGMJobWatch) ogmrip_backend_crop_watch, video, TRUE, FALSE, TRUE);

  ogmjob_spawn_run (child, NULL);

  g_object_unref (child);
}

